#!/usr/bin/env python3

# Copyright 2021 RnD Center "ELVEES", JSC

import argparse
import json
import subprocess
import sys
from typing import Dict, List

kHz = 1000
MHz = 1000 * kHz


def PLL(x):
    return x


CLOCKS = {
    "SERVICE": {
        "ppolicy": None,
        "pll_addr": [0x1F00_1000],
        "ucgs": [
            {
                "addr": 0x1F02_0000,
                "name": "UCG",
                "names": [
                    "service_apb_clk",
                    "service_core_clk",
                    "qspi0_clk",
                    "bpam_clk",
                    "risc0_clk",
                    "mfbsp0_clk",
                    "mfbsp1_clk",
                    "mailbox0_clk",
                    "pvtctr_clk",
                    "i2c4_clk",
                    "trng_clk",
                    "spiotp_clk",
                    "i2c4_ext_clk",
                    "qspi0_ext_clk",
                    "clkout_clk",
                    "risc0_tck_clk",
                ],
            },
        ],
    },
    "CPU": {
        "ppolicy": 0x1F000000,
        "pll_addr": [0x100_0050],
        "ucgs": [
            {
                "addr": 0x108_0000,
                "name": "UCG",
                "names": ["cpu_sys_clk", "cpu_core_clk", "cpu_dbus_clk"],
            },
        ],
    },
    "DDR": {
        "ppolicy": 0x1F000038,
        "pll_addr": [0xC00_0000, 0xC00_0008],
        "ucgs": [
            {
                "addr": 0xC01_0000,
                "name": "UCG0",
                "names": ["ddr0_clk", "ddr0_x4_clk", "ddr1_clk", "ddr1_x4_clk"],
                "pll": 0,
            },
            {
                "addr": 0xC02_0000,
                "name": "UCG1",
                "names": [
                    "ddr_sys_clk",
                    "ddr_sdr_clk",
                    "ddr_pcie_clk",
                    "ddr_isp_clk",
                    "ddr_gpu_clk",
                    "ddr_vpu_clk",
                    "ddr_dp_clk",
                    "ddr_cpu_clk",
                    "ddr_service_clk",
                    "ddr_hsperiph_clk",
                    "ddr_lsperiph0_clk",
                    "ddr_lsperiph1_clk",
                ],
                "pll": 1,
            },
        ],
    },
    "HSPERIPH": {
        "ppolicy": 0x1F000020,
        "pll_addr": [0x1040_0000],
        "ucgs": [
            {
                "addr": 0x1041_0000,
                "name": "UCG0",
                "names": [
                    "hsp_sys_clk",
                    "hsp_dma_clk",
                    "hsp_ctr_clk",
                    "spram_clk",
                    "emac0_clk",
                    "emac1_clk",
                    "usb0_clk",
                    "usb1_clk",
                    "nfc_clk",
                    "pdma2_clk",
                    "sdmmc0_clk",
                    "sdmmc1_clk",
                    "qspi_clk",
                ],
            },
            {
                "addr": 0x1042_0000,
                "name": "UCG1",
                "names": [
                    "sdmmc0_xin_clk",
                    "sdmmc1_xin_clk",
                    "nfc_clk_flash",
                    "qspi_ext_clk",
                    "ust_clk",
                ],
            },
            {
                "addr": 0x1043_0000,
                "name": "UCG2",
                "names": ["emac0_clk_1588", "emac0_rgmii_txc", "emac1_clk_1588", "emac1_rgmii_txc"],
            },
            {
                "addr": 0x1044_0000,
                "name": "UCG3",
                "names": [
                    "usb0_ref_alt_clk",
                    "usb0_suspend_clk",
                    "usb1_ref_alt_clk",
                    "usb1_suspend_clk",
                ],
            },
        ],
        "ref": [PLL(0), 125 * MHz, None, None],
        "refclk_addr": 0x1040_000C,
    },
    "LSPERIPH0": {
        "ppolicy": 0x1F000028,
        "pll_addr": [0x168_0000],
        "ucgs": [
            {
                "addr": 0x169_0000,
                "name": "UCG0",
                "names": [
                    "lsp0_sys_clk",
                    "uart3_clk",
                    "uart1_clk",
                    "uart2_clk",
                    "spi0_clk",
                    "i2c0_clk",
                    "gpio0_dbclk",
                ],
            },
        ],
    },
    "LSPERIPH1": {
        "ppolicy": 0x1F000030,
        "pll_addr": [0x17E_0000],
        "ucgs": [
            {
                "addr": 0x17C_0000,
                "name": "UCG",
                "names": [
                    "lsp1_sys_clk",
                    "i2c1_clk",
                    "i2c2_clk",
                    "i2c3_clk",
                    "gpio1_dbclk",
                    "ssi1_clk",
                    "uart0_clk",
                    "timers0_clk",
                    "pwm0_clk",
                    "wdt1_clk",
                ],
            },
            {"addr": 0x17D_0000, "name": "I2S_UCG", "names": ["lsp1_i2s_clk"]},
        ],
    },
    "MEDIA": {
        "ppolicy": 0x1F000010,
        "pll_addr": [0x132_0000, 0x132_0010, 0x132_0020, 0x132_0030],
        "ucgs": [
            {
                "addr": 0x132_0040,
                "name": "UCG0",
                "names": ["media_sys_aclk", "isp_sys_clk"],
                "pll": 0,
            },
            {
                "addr": 0x132_00C0,
                "name": "UCG1",
                "names": ["disp_aclk", "disp_mclk", "disp_pxlclk"],
                "pll": 1,
            },
            {
                "addr": 0x132_0140,
                "name": "UCG2",
                "names": ["gpu_sys_clk", "gpu_mem_clk", "gpu_core_clk"],
                "pll": 2,
            },
            {
                "addr": 0x132_01C0,
                "name": "UCG3",
                "names": [
                    "mipi_rx_ref_clk",
                    "mipi_rx0_cfg_clk",
                    "mipi_rx1_cfg_clk",
                    "mipi_tx_ref_clk",
                    "mipi_tx_cfg_clk",
                    "cmos0_clk",
                    "cmos1_clk",
                    "mipi_txclkesc",
                    "vpu_clk",
                ],
                "pll": 3,
            },
        ],
    },
    "SDR": {
        "ppolicy": 0x1F000008,
        "pll_addr": [0x191_0000, 0x191_0008, 0x191_0010],
        "ucgs": [
            {
                "addr": 0x190_0000,
                "name": "UCG0",
                "names": [
                    "sdr_cfg_clk",
                    "sdr_ext_clk",
                    "sdr_int_clk",
                    "pci_clk",
                    "vcu_aclk",
                    "acc0_clk",
                    "acc1_clk",
                    "acc2_clk",
                    "aux_pci_clk",
                    "gnss_clk",
                    "dfe_alt_clk",
                    "vcu_tclk",
                    "lvds_clk",
                ],
                "pll": 0,
            },
            {
                "addr": 0x190_8000,
                "name": "UCG_PCI0_REF",
                "names": ["ref_alt_clk"],
                "pll": 0,
            },
            {
                "addr": 0x19E_0000,
                "name": "UCG_JESD0_TX",
                "names": [None, "jesd0_tx_sample_clk", "jesd0_tx_character_clk"],
                "pll": 0,
            },
            {
                "addr": 0x19E_8000,
                "name": "UCG_JESD0_RX",
                "names": [None, "jesd0_rx_sample_clk", "jesd0_rx_character_clk"],
                "pll": 0,
            },
            {
                "addr": 0x1C0_8000,  # ???
                "name": "UCG_N_DFE or UCG2",
                "names": ["dfe_nels_clk"],
                "pll": 1,
            },
            {
                "addr": 0x1C0_0000,
                "name": "UCG_PCI1_REF",
                "names": ["ref_alt_clk"],
                "pll": 0,
            },
            {
                "addr": 0x1CE_0000,
                "name": "UCG_JESD1_TX",
                "names": [None, "jesd1_tx_sample_clk", "jesd1_tx_character_clk"],
                "pll": 0,
            },
            {
                "addr": 0x1CE_8000,
                "name": "UCG_JESD1_RX",
                "names": [None, "jesd1_rx_sample_clk", "jesd1_rx_character_clk"],
                "pll": 0,
            },
        ],
    },
}


def read_reg(addr):
    res = subprocess.check_output(["devmem", f"{addr:#x}"]).decode("utf-8")
    return int(res, 0)


def get_pll_freq(addr, in_freq):
    """Get output frequency of PLL
    addr - address of pllcfg register
    in_freq - PLL input frequency
    """
    pll = read_reg(addr)

    # In bypass mode output frequency is equal to input frequncy
    is_bypass = False if pll & 0xFF else True
    if not is_bypass and (pll >> 9) & 0x1:
        nr = (pll >> 27) & 0xF
        nf = (pll >> 14) & 0x1FFF
        od = (pll >> 10) & 0xF
        k = (nf + 1) / (nr + 1) / (od + 1)
    else:
        k = (pll & 0xFF) + 1

    # PLL set LOCK bit only if PLL is not in bypass mode
    is_lock = is_bypass or bool((pll >> 31) & 0x1)
    freq = 0 if not is_lock else in_freq * k
    return {"is_lock": is_lock, "freq": freq}


def get_ucg_freq(ucg, pll_freq, xti_freq):
    """Get frequncies for all channels of UCG.
    ucg - dict from CLOCKS
    pll_freq - frequency of source PLL
    xti_freq - frequency in bypass mode
    Return list of dict
    """
    addr = ucg["addr"]
    bp = read_reg(addr + 0x40)
    ucg_result: List[Dict] = []
    for idx, name in enumerate(ucg["names"]):
        if name is None:
            ucg_result.append({})
            continue
        ctr = read_reg(addr + (idx * 4))
        coeff = (ctr >> 10) & 0xFFFFF
        coeff = 1 if coeff == 0 else coeff
        is_enabled = bool((ctr >> 1) & 0x1)
        is_bypass = bool((bp >> idx) & 0x1)
        freq = xti_freq if is_bypass else pll_freq / coeff
        res = {"name": name, "freq": freq, "is_enabled": is_enabled, "is_bypass": is_bypass}
        ucg_result.append(res)

    return ucg_result


def get_subsystem_freq(subsys, name, xti_freq):
    """Get frequencies for all PLLs and UCGs in subsystem.
    subsys - dict from CLOCKS
    name - name of subsystem
    xti_freq - frequency in bypass mode
    Return dict with PLLs and UCGs setings
    """
    PP_ON = 0x10
    if subsys["ppolicy"] is not None and read_reg(subsys["ppolicy"]) != PP_ON:
        return {"subsystem": name, "pll": [], "ucg": [], "status": "disabled"}

    ref = subsys["ref"] if "ref" in subsys.keys() else [0] * len(subsys["ucgs"])
    refclk_addr = subsys["refclk_addr"] if "refclk_addr" in subsys else None
    refclk = read_reg(refclk_addr) if refclk_addr is not None else 0
    plls = [get_pll_freq(pll_addr, xti_freq) for pll_addr in subsys["pll_addr"]]
    ucgs: List[Dict] = []
    for idx, ucg in enumerate(subsys["ucgs"]):
        if "pll" in ucg:
            ref_value = ucg["pll"]
        else:
            ref_id = (refclk >> (idx * 2)) & 0x3
            if ref_id >= len(ref) or ref[ref_id] is None:
                print(f"Invalid value in refclk for UCG{idx} in subsys {name}", file=sys.stderr)
                ucgs.append({})
                continue

            ref_value = ref[ref_id]

        if ref_value < 4:
            ucg_in_freq = plls[ref_value]["freq"] if plls[ref_value]["is_lock"] else 0
            ucg_in = "pll"
            ucg_in_pll = ref_value
        else:
            ucg_in_freq = ref_value
            ucg_in = "fixed"
            ucg_in_pll = None

        ucg_list = get_ucg_freq(ucg, ucg_in_freq, xti_freq)
        ucgs.append(
            {
                "name": ucg["name"],
                "input_source": ucg_in,
                "input_pll": ucg_in_pll,
                "input_freq": ucg_in_freq,
                "channels": ucg_list,
            }
        )

    return {"subsystem": name, "pll": plls, "ucg": ucgs}


def show_result_text(result):
    for subsys in result:
        status = f" ({subsys['status']})" if "status" in subsys.keys() else ""
        print(f"Subsystem: {subsys['subsystem']}{status}")
        for idx, pll in enumerate(subsys["pll"]):
            pll_status = f"{pll['freq']/MHz:0.3f} MHz"
            if not pll["is_lock"]:
                pll_status = f"NOT LOCKED ({pll_status})"
            print(f"  PLL{idx}: {pll_status}")
        for ucg in subsys["ucg"]:
            if ucg["input_source"] == "pll":
                source = f"PLL{ucg['input_pll']}"
            else:
                source = f"Fixed ({ucg['input_freq']})"

            print(f"  {ucg['name']}, source: {source}")
            max_name_len = max([len(x["name"]) if "name" in x else 0 for x in ucg["channels"]])
            for idx, channel in enumerate(ucg["channels"]):
                channel_prefix = f"    channel {idx:2d}: "
                if not list(channel.keys()):
                    print(f"{channel_prefix}-")
                    continue

                if channel["freq"] % kHz:
                    freq = f"{channel['freq'] / kHz:0.3f} kHz"
                elif channel["freq"]:
                    freq = f"{channel['freq'] / MHz:0.3f} MHz"

                if channel["is_enabled"]:
                    freq = f"{freq} ON"
                else:
                    freq = f"{freq} OFF"

                if channel["is_bypass"]:
                    freq = f"{freq} (bypass)"

                name = f"{{:{max_name_len+1}}}".format(channel["name"] + ":")
                print(f"{channel_prefix}{name} {freq}")

        print("")


def main():
    parser = argparse.ArgumentParser(description="Tool to read clocks of subsystems of MCom-03 SoC")
    parser.add_argument("--json", action="store_true", help="Output in JSON format")
    parser.add_argument("--xti", type=int, default=27000000, help="XTI frequency")
    args = parser.parse_args()

    result = []
    for name, subsys in CLOCKS.items():
        result.append(get_subsystem_freq(subsys, name, args.xti))

    if args.json:
        json.dump(result, sys.stdout, indent=4)
        print("")
    else:
        show_result_text(result)


if __name__ == "__main__":
    main()
