#!/bin/bash
#
# Copyright 2023 RnD Center "ELVEES", JSC
#
set -e

print_help() {
    cat << END
Script to test display output switching between HDMI and DSI on MCom03 buildroot disto.

Before switching, this script checks whenever both HDMI and DSI are connected,
and if DSI is not - then it means that DSI mcom03 encoder driver is not enabled if DT.
Then it try to load DTB overlay with DSI encoder enabled.

After all encoders enabled it perfrom infinite loop of display switchings:
each switch will disable previous output, enable new, and display picture on it.

Also utility commands available:
    help   - show this help
    unload - unload dtbo
    load   - load dtbo
    hdmi   - display sample image on HDMI
    dsi    - display sample image on DSI

and options:
    --method=*    - set test utility : modetest, mpv or all
    --dtb=*       - path to DTBO file
    --count=<num> - number of test runs (display switchings)
END
}

# Device tree overlay to support Rasberri Pie Waveshare 7inch DSI display
DTBO="${DTBO:-rbpi-7inch-overlay.dtbo}"
TEST_METHOD="${TEST_METHOD:-modetest}"
TEST_COUNT="${TEST_COUNT:-3}"
CFGFS="/sys/kernel/config/device-tree/overlays/test"

MDTST="modetest -M mali-dp"
HDMI_CON="HDMI-A-1"
DSI_CON="DSI-1"
# time to show image after display switch
SHOWDURATION_SEC=4

error() {
    echo "$1" >&2
    exit 1
}

sig_handler() {
    error "SIGINT: wait last command to exit ..."
}

# check that DRM encoder have "connected" status
check_connected() {
    $MDTST | grep $1 | grep -qw connected
}

assert_configfs () {
    if ! zcat /proc/config.gz | grep -q CONFIG_OF_CONFIGFS=y; then
        error "Unable to load dtbo at configfs: kernel not configured for this!"
    fi
}

# This will try to load DTB overlay with DSI encoder enabled.
load_dtbo() {
    assert_configfs

    mkdir -p "$CFGFS"

    # try to load DTBO file from current directory, or from /lib/firmware
    if test -f "$DTBO"; then
        cat "$DTBO" > "$CFGFS/dtbo" ||
            error "unable to load DTBO fw file to configfs DT-overlay node"
    elif [ -n "$(find /lib/firmware -name $DTBO -print -quit)" ]; then
        echo "$DTBO" > "$CFGFS/path"  ||
            error "unable to load DTBO fw file to configfs DT-overlay node"
    else
        error "Unable to find DTBO file $DTBO: must be in current dir or in /lib/firmare/"
    fi

    # new devices was created, so rebind DRM master to take them
    echo "1300000.dp" >  /sys/bus/platform/drivers/mali-dp/unbind
    echo "1300000.dp" >  /sys/bus/platform/drivers/mali-dp/bind
}

# This will unload DTB overlay
unload_dtbo() {
    assert_configfs

    # unload drm master to remove drivers
    echo "1300000.dp" >  /sys/bus/platform/drivers/mali-dp/unbind
    # this will delete overlay devices from DTS
    rmdir $CFGFS

    # devices was deleted, so rebind DRM master to take them
    echo "1300000.dp" >  /sys/bus/platform/drivers/mali-dp/bind
}

# this will display png on display by MPV player
test_mpv() {
    local TESTIMAGE_FILE="/usr/share/pixmaps/gradient.png"
    mpv --drm-connector="$1" --vo=drm --image-display-duration="$SHOWDURATION_SEC" \
        "$TESTIMAGE_FILE" &> /dev/null
}

# this will switch to $1 output and display modetest basic test on it
test_modetest() {
    local CHECK_CMD="$MDTST -s ${1}:#0"

    timeout --signal=SIGINT $SHOWDURATION_SEC $CHECK_CMD || local error=$?
    if [ 0${error} -eq 124 ]; then
        # OK, it is timeout as expected
        return
    else
        error "modetest did not timeout. Errcode $error"
    fi
}

test_connector() {
    check_connected $1 ||
        error "$1 is not connected! Please plug $1"

    case "$TEST_METHOD" in
        modetest)
            test_modetest $1
            ;;
        mpv)
            test_mpv $1
            ;;
        all)
            test_modetest $1
            test_mpv $1
            ;;
        *)
            error "Unknown test method: \"$TEST_METHOD\""
    esac
}

test_switching() {
    check_connected $HDMI_CON ||
        error "HDMI is not connected! Please plug HDMI"

    echo "$HDMI_CON is connected"

    while ! check_connected $DSI_CON; do
        echo "try to load DSI encoder driver"
        load_dtbo
    done
    echo "$DSI_CON is connected"

    trap sig_handler SIGINT
    echo "Will perform $TEST_COUNT display switching, press CTRL+C to stop"

    for i in $(seq $TEST_COUNT); do
        echo "test $i"
        for c in $DSI_CON $HDMI_CON; do
            test_connector $c
        done
    done
}

for arg in "$@"; do
    case "$arg" in
        --dtb=*)
            DTBO="${arg#*=}"
            ;;

        --method=*)
            TEST_METHOD="${arg#*=}"
            ;;

        --count=*)
            TEST_COUNT="${arg#*=}"
            ;;

        help|--help|-h)
            print_help
            exit 0
            ;;

        ld|load)
            load_dtbo
            exit 0
            ;;

        uld|unload)
            unload_dtbo
            exit 0
            ;;

        hdmi)
            test_connector $HDMI_CON
            exit 0
            ;;

        dsi)
            test_connector $DSI_CON
            exit 0
            ;;

        *)
            error "Unknown argument $arg"
    esac
done

test_switching
