/*!
 * @defgroup embedded_cli_support Поддержка embedded CLI
 *
 * @brief Набор функций для поддержки команд embedded CLI
 *
 */

/*!
 * @addtogroup embedded_cli_support
 * @{
 */

/*!
 * @file cli_time_support.h
 *
 * @brief Набор функций для поддержки команд времени (settime, timenow) CLI
 *
 */
#ifndef CLI_TIME_SUPPORT_H
#define CLI_TIME_SUPPORT_H

#include "eliot1_board.h"
#include "embedded_cli.h"
#include "hal_rwc.h"

/*!
 * @brief Возвращаемые значения функций обработки даты/времени
 *
 */
enum dtime_status_t
{
    Dtime_Status_NoParameters           = 0U,  /*!< Не были переданы параметры */
    Dtime_Status_TooMuchParameters      = 1U,  /*!< Были переданы лишние параметры */
    Dtime_Status_SetDateOk              = 2U,  /*!< Успешная установка даты */
    Dtime_Status_ForbiddenDayValue      = 3U,  /*!< Неверно указан день */
    Dtime_Status_ForbiddenMonthValue    = 4U,  /*!< Неверно указан месяц */
    Dtime_Status_ForbiddenYearValue     = 5U,  /*!< Неверно указан год */
    Dtime_Status_IncorrectDayOfMonth    = 6U,  /*!< Неверно указан день месяца */
    Dtime_Status_SetTimeOk              = 7U,  /*!< Успешная установка времени */
    Dtime_Status_ForbiddenSecValue      = 8U,  /*!< Неверно указаны секунды */
    Dtime_Status_ForbiddenMinValue      = 9U,  /*!< Неверно указаны минуты */
    Dtime_Status_ForbiddenHrsValue      = 10U, /*!< Неверно указаны часы */
    Dtime_Status_UnsupportedParameter   = 11U, /*!< Неподдерживаемый параметр */
    Dtime_Status_NonDigitInput          = 12U, /*!< В значении для даты/времени присутствует посторонний символ */
    Dtime_status_IllegalFormat          = 13U, /*!< Неподдерживаемый формат введенных данных */
};

/**
 * @brief Функция сравнения времени
 *
 * Сравнивает переданные структуры даты и времени.
 *
 * @param t1 1-я структура времени для сравнения
 * @param t2 2-я структура времени для сравнения
 *
 * @retval 1 структуры идентичны
 * @retval 0 структуры отличаются
 */
static uint32_t CompareTime(const rtc_datetime_t *t1, const rtc_datetime_t *t2)
{
    if (t1->year != t2->year) return 0;
    if (t1->month != t2->month) return 0;
    if (t1->day != t2->day) return 0;
    if (t1->hour != t2->hour) return 0;
    if (t1->minute != t2->minute) return 0;
    if (t1->second != t2->second) return 0;
    return 1;
}

/*!
 * @brief Получение информационного сообщения о статусе
 *
 * @param status возвращаемый статус
 *
 * @return информационное сообщение о переданном статусе
 */
static const char *InfoDtimeStatus (enum dtime_status_t *status) {
    switch (*status) {
        case 0U:
            return "no parameters was specified. Use help settime to see what parameters are available";
        case 1U:
            return "too much parameters was specified. Use help settime to see what parameters are available";
        case 2U:
            return "date set successfully!";
        case 3U:
            return "day value is incorrect";
        case 4U:
            return "month value is incorrect";
        case 5U:
            return "year value is incorrect";
        case 6U:
            return "day not exists in specified month";
        case 7U:
            return "time set successfully!";
        case 8U:
            return "seconds value is incorrect";
        case 9U:
            return "minutes value is incorrect";
        case 10U:
            return "hours value is incorrect";
        case 11U:
            return "unsupported mode. Use help settime to see what modes are available";
        case 12U:
            return "inputted parameters include non-digit symbols";
        case 13U:
            return "inputted parameters dont fit format of command. Use help settime to see what format is allowed";
        default:
            return "unsupported status";
    }
}

/*!
 * @brief Проверка корректности даты
 *
 * Проверяет, содержится ли передаваемый день в указанных месяце и году.
 *
 * @param day   день
 * @param month месяц
 * @param year  год
 *
 * @retval 1 дата указана верно
 * @retval 0 дата указана неверно
 */
static uint32_t CorrectDate (const uint8_t *day, const uint8_t *month, 
    const uint16_t *year)
{
    switch (*month) {
        case 4:
        case 6:
        case 9:
        case 11:
            if (*day > 30) {
                return 0;
            } else {
                return 1;
            }
        case 2:
            if (*year % 4 == 0) {
                if (*day > 28) {
                    return 0;
                } else {
                    return 1;
                }
            } else {
                if (*day > 27) {
                    return 0;
                } else {
                    return 1;
                }
            }
        default:
            if (*day > 31) {
                return 0;
            } else {
                return 1;
            }
    };
}

/*!
 * @brief Проверка на цифру (0-9)
 *
 * Функция проверяет, является ли переданный символ цифрой 0-9.
 *
 * @param ch проверяемый символ
 *
 * @retval 1 является цифрой 0-9
 * @retval 0 не является цифрой 0-9
 */
static uint32_t IsDigit(const char ch)
{
    if ('0' <= ch && ch <= '9') {
        return 1;
    } else {
        return 0;
    }
}

/*!
 * @brief Проверка на нулевое значение
 *
 * Функция проверяет, состоит ли переданная строка только из 0.
 *
 * @param str проверяемая строка символов
 *
 * @retval 1 строка состоит только из 0
 * @retval 0 строка состоит не только из 0
 */
static uint32_t IsZeroValue (const char *str)
{
    for(uint32_t i = 0; i < strlen(str); i++) {
        if(IsDigit(str[i]) == 0) {
            return 0;
        } else {
            if (str[i] != '0') {
                return  0;
            }
        }
    }

    return 1;
}

/*!
 * @brief Проверка на наличие символов кроме 0-9 в строке
 *
 * Функция проверяет наличие в строке символов, отличных от цифр 0-9
 *
 * @param str проверяемая строка символов
 *
 * @retval 1 строка содержит не только цифры 0-9
 * @retval 0 строка содержит только цифры 0-9
 */
uint32_t ContainsNonDigits (const char *str)
{
    for(uint32_t i = 0; i < strlen(str); i++) {
        if (IsDigit(str[i]) == 0) {
            return 1;
        }
    }
    return 0;
}

/*!
 * @brief Проверка на наличие необходимых символов-разделителей
 *
 * Функция проверяет, удовлетворяет ли переданная пользователем строка
 * поддерживаемым форматам даты/времени, сравнивая количество указанных
 * символов в строке с заданным.
 * 
 * @param str   строка с данными для проверки
 * @param delim заданный символ-разделитель
 * @param count требуемое количество символов-разделитей в строке
 *
 * @return 1 строка соответствует формату данных
 * @return 0 строка не соответствует формату данных
 */
uint32_t ContainsRequiredDelims(const char *str, const char delim, const uint8_t count)
{
    uint8_t current_amount = 0;
    for(uint32_t i = 0; i < strlen(str); i++) {
        if (str[i] == delim) {
            current_amount++;
        }
    }
    
    return (current_amount == count) ? 1 : 0;
}

/*!
 * @brief Обработка времени
 *
 * Последовательно выделяет единицы времени из передаваемой строки
 * (часы, минуты, секунды), проверяет их на соответствие ограничениям и
 * сохраняет изменения.
 *
 * @param tokenized_time строка в формате hh:mm:ss
 * @param hrs            часы
 * @param min            минуты
 * @param sec            секунды
 *
 * @retval #Dtime_Status_SetTimeOk
 * @retval #Dtime_Status_ForbiddenHrsValue
 * @retval #Dtime_Status_ForbiddenHrsValue
 * @retval #Dtime_Status_ForbiddenSecValue
 * @retval #Dtime_status_IllegalFormat
 * @retval #Dtime_Status_NonDigitInput
 */
enum dtime_status_t ProcessTime(char *tokenized_time, uint8_t *hrs,
    uint8_t *min, uint8_t *sec)
{
    uint8_t tmp_sec, tmp_min, tmp_hrs;
    char *str_hrs, *str_min, *str_sec, *str_tmp;

    if (!(ContainsRequiredDelims(tokenized_time, ':', 2))) {
        return Dtime_status_IllegalFormat;
    }

    if ((str_hrs = strtok(tokenized_time, ":")) != NULL) {
        if (ContainsNonDigits(str_hrs)) {
            return Dtime_Status_NonDigitInput;
        }
        if (strlen(str_hrs) != 2) {
            return Dtime_status_IllegalFormat;
        }
        tmp_hrs = (IsZeroValue(str_hrs) == 0) ? (uint8_t) atoi(str_hrs) : 0;
        if (tmp_hrs > 23) {
            return Dtime_Status_ForbiddenHrsValue;
        }
    } else {
        return Dtime_Status_ForbiddenHrsValue;
    }

    if ((str_min = strtok(NULL, ":")) != NULL) {
        if (ContainsNonDigits(str_min)) {
            return Dtime_Status_NonDigitInput;
        }
        if (strlen(str_min) != 2) {
            return Dtime_status_IllegalFormat;
        }
        tmp_min = (IsZeroValue(str_min) == 0) ? (uint8_t) atoi(str_min) : 0;
        if (tmp_min > 59) {
            return Dtime_Status_ForbiddenMinValue;
        }
    } else {
        return Dtime_Status_ForbiddenMinValue;
    }

    if ((str_sec = strtok(NULL, ":")) != NULL) {
        if (ContainsNonDigits(str_sec)) {
            return Dtime_Status_NonDigitInput;
        }
        if (strlen(str_sec) != 2) {
            return Dtime_status_IllegalFormat;
        }
        tmp_sec = (IsZeroValue(str_sec) == 0) ? (uint8_t) atoi(str_sec) : 0;
        if (tmp_sec > (uint8_t) 59) {
            return Dtime_Status_ForbiddenSecValue;
        }
    } else {
        return Dtime_Status_ForbiddenSecValue;
    }

    if ((str_tmp = strtok(NULL, ".")) != NULL) {
        return Dtime_status_IllegalFormat;
    }

    *hrs = tmp_hrs;
    *min = tmp_min;
    *sec = tmp_sec;

    return Dtime_Status_SetTimeOk;
}

/*!
 * @brief Обработка даты
 *
 * Последовательно выделяет единицы даты из передаваемой строки
 * (день, месяц, год), проверяет их на соответствие ограничениям и
 * сохраняет изменения.
 *
 * @param tokenized_date строка в формате dd.mm.yyyy
 * @param day            день
 * @param month          месяц
 * @param year           год
 *
 * @retval #Dtime_Status_SetDateOk
 * @retval #Dtime_Status_ForbiddenDayValue
 * @retval #Dtime_Status_ForbiddenMonthValue
 * @retval #Dtime_Status_ForbiddenYearValue
 * @retval #Dtime_status_IllegalFormat
 * @retval #Dtime_Status_NonDigitInput
 * @retval #Dtime_Status_IncorrectDayOfMonth
 */
enum dtime_status_t ProcessDate(char *tokenized_date, uint8_t *day,
    uint8_t *month, uint16_t *year)
{
    uint8_t tmp_day, tmp_month;
    uint16_t tmp_year;
    char *str_day, *str_month, *str_year, *str_tmp;

    if (!(ContainsRequiredDelims(tokenized_date, '.', 2))) {
        return Dtime_status_IllegalFormat;
    }

    if ((str_day = strtok(tokenized_date, ".")) != NULL) {
        if (ContainsNonDigits(str_day)) {
            return Dtime_Status_NonDigitInput;
        }
        if (strlen(str_day) != 2) {
            return Dtime_status_IllegalFormat;
        }
        tmp_day = (uint8_t) atoi(str_day);
        if (tmp_day == 0 || tmp_day > 31) {
            return Dtime_Status_ForbiddenDayValue;
        }
    } else {
        return Dtime_Status_ForbiddenDayValue;
    }

    if ((str_month = strtok(NULL, ".")) != NULL) {
        if (ContainsNonDigits(str_month)) {
            return Dtime_Status_NonDigitInput;
        }
        if (strlen(str_month) != 2) {
            return Dtime_status_IllegalFormat;
        }
        tmp_month = (uint8_t) atoi(str_month);
        if (tmp_month == 0 || tmp_month > 12) {
            return Dtime_Status_ForbiddenMonthValue;
        }
    } else {
        return Dtime_Status_ForbiddenMonthValue;
    }

    if ((str_year = strtok(NULL, ".")) != NULL) {
        if (ContainsNonDigits(str_year)) {
            return Dtime_Status_NonDigitInput;
        }
        if (strlen(str_year) != 4) {
            return Dtime_status_IllegalFormat;
        }
        tmp_year = (uint16_t) atoi(str_year);
        if (tmp_year == 0) {
            return Dtime_Status_ForbiddenYearValue;
        }
    } else {
        return Dtime_Status_ForbiddenYearValue;
    }

    if ((str_tmp = strtok(NULL, ".")) != NULL) {
        return Dtime_status_IllegalFormat;
    }

    if (CorrectDate(&tmp_day, &tmp_month, &tmp_year) == 0) {
        return Dtime_Status_IncorrectDayOfMonth;
    }

    *day = tmp_day;
    *month = tmp_month;
    *year = tmp_year;

    return Dtime_Status_SetDateOk;
}

/*!
 * @brief Установка даты/времени
 *
 * В зависимости от введенных пользователем параметров
 * выполняет обработку даты/времени
 *
 * @param datetime2set сохраняемая дата
 * @param args         введенные пользователем параметры команды в виде строки
 *
 * @retval #Dtime_Status_NoParameters
 * @retval #Dtime_Status_TooMuchParameters
 * @retval #Dtime_Status_SetDateOk
 * @retval #Dtime_Status_ForbiddenDayValue
 * @retval #Dtime_Status_ForbiddenMonthValue
 * @retval #Dtime_Status_ForbiddenYearValue
 * @retval #Dtime_Status_IncorrectDayOfMonth
 * @retval #Dtime_Status_SetTimeOk
 * @retval #Dtime_Status_ForbiddenSecValue
 * @retval #Dtime_Status_ForbiddenMinValue
 * @retval #Dtime_Status_ForbiddenHrsValue
 * @retval #Dtime_Status_UnsupportedParameter
 * @retval #Dtime_Status_NonDigitInput
 * @retval #Dtime_status_IllegalFormat
 */
enum dtime_status_t SetDatetime (rtc_datetime_t *datetime2set, char *args)
{
    uint8_t sec, min, hrs, day, month;
    uint16_t year;

    const uint32_t amount_tokens = embeddedCliGetTokenCount(args);
    char *param1 = (char *) embeddedCliGetToken(args, 1);
    char *param2 = (char *) embeddedCliGetToken(args, 2);

    enum dtime_status_t ret_status;

    if (amount_tokens == 0) {
        return Dtime_Status_NoParameters;
    }

    if (amount_tokens > 2) {
        return Dtime_Status_TooMuchParameters;
    }

    /* Обработка введенной даты в режиме date. */
    if (strcmp(param1, "date") == 0) {
        ret_status = ProcessDate(param2, &day, &month, &year);
        if (ret_status == Dtime_Status_SetDateOk) {
            datetime2set->day = day;
            datetime2set->month = month;
            datetime2set->year = year;
        }

        return ret_status;
    }

    /* Обработка введенного времени в режиме time. */
    if (strcmp(param1, "time") == 0) {
        ret_status = ProcessTime(param2, &hrs, &min, &sec);
        if (ret_status == Dtime_Status_SetTimeOk) {
            datetime2set->hour = hrs;
            datetime2set->minute = min;
            datetime2set->second = sec;
        }

        return ret_status;
    }

    /* Обработка введенных данных в режиме изменения даты и времени. */
    ret_status = ProcessDate(param1, &day, &month, &year);
    if (ret_status == Dtime_Status_SetDateOk) {
        datetime2set->day = day;
        datetime2set->month = month;
        datetime2set->year = year;
    } else {
        return ret_status;
    }

    ret_status = ProcessTime(param2, &hrs, &min, &sec);
    if (ret_status == Dtime_Status_SetTimeOk) {
        datetime2set->hour = hrs;
        datetime2set->minute = min;
        datetime2set->second = sec;
    }

    return ret_status;
}

#endif /* CLI_TIME_SUPPORT_H */

/*!
 * @}
 *
 */
