python


import gdb
import os
import struct
import sys


# Script constants
SCRIPT_NAME = "set_boot_from_otp.sh"
SCRIPT_PATH = os.path.realpath(sys.argv[0]) + "/"
TCL_UTILS_FILE = SCRIPT_PATH + "utils.tcl"
DUMP_DIR = SCRIPT_PATH + "memory_dump/"

WORD_SIZE              = 4                           # word size for ELIOT1
OTP_ADDRESS            = 0x5008a000                  # OTP start address (NVM view)
OTP_SIZE               = 0x400                       # 1024 bytes (256 words)
OTP_BOOT_ADDR          = 0x5008a0b0                  # Register address for boot address value
OTP_BOOT_ADDR_RELATIVE = OTP_BOOT_ADDR - OTP_ADDRESS # Relative register address for boot address value
OTP_BOOT_ADDR_VALUE    = 0x1e000001                  # Boot from the beginning of the OTP User Area


# Function: print error and exit
def error_exit(errstr):
    print >> sys.stderr, ("{}: {}".format(SCRIPT_NAME, errstr))
    exit(1)


# Function: read binary file, return byte array
def read_binary_file(fname):
    if not os.path.isfile(fname):
        error_exit("No such image file: \'{}\'".format(fname))

    f = open(fname, "rb")
    fdata = bytearray(f.read())
    f.close()
    return fdata


# Function: convert byte array to int array
def bytearr_to_intarr(arr):
    count = len(arr) // WORD_SIZE
    return struct.unpack("{}I".format(count), arr)


# Function: dump memory
def dump_memory(addr, size):
    if (size % WORD_SIZE) != 0:
        error_exit("Invalid memory size to read: {} (must be a divisor of a word size - {})".format(size, WORD_SIZE))

    tmpfile = DUMP_DIR + "tmp.bin"
    gdb.execute("dump memory {} 0x{:08x} 0x{:08x}"
        .format(tmpfile, addr, addr + size))
    dump = read_binary_file(tmpfile)
    os.remove(tmpfile)

    return bytearr_to_intarr(dump)


# Function: dump memory to file
def dump_memory_to_file(fname, addr, size):
    gdb.execute("dump memory {} 0x{:08x} 0x{:08x}"
        .format(fname, addr, addr + size)
    )


# Main

## check utility file
if not os.path.exists(TCL_UTILS_FILE):
    error_exit("No such utility file: {}".format(TCL_UTILS_FILE))

## create a directory for memory dump files if not exist
try:
    os.mkdir(DUMP_DIR)
except OSError:
    if not os.path.isdir(DUMP_DIR):
        raise

## connect to the board
gdb.execute("set pagination off")
gdb.execute("set confirm off")
gdb.execute("target extended-remote :3333")
gdb.execute("monitor source {}".format(TCL_UTILS_FILE))

## welcome message
print ("\nThe {} script is ready to program the OTP memory.".format(SCRIPT_NAME))
print ("Warning: The programming process of OTP is irreversible.")

## get OTP_BOOT_ADDR register value
otp_dump = dump_memory(OTP_ADDRESS, OTP_SIZE)
curr_reg_value = otp_dump[0xb0 // WORD_SIZE]
if curr_reg_value != 0:
    error_exit("Boot address is set already: 0x{:08x}".format(curr_reg_value))

## prompt before writting
print ("\nCurrent value of OPT_BOOT_ADDR register is: 0x{:08x}".format(curr_reg_value))
print ("Are you sure you want to set boot address to OTP User Area (0x1e000000)? [y / n (any other symbol)]")
answer = sys.stdin.readline()
if (answer != "y\n") and (answer != "y\r\n"):
    error_exit("Boot address setting has cancelled")

## set boot address to OTP User Area
dump_memory_to_file(DUMP_DIR + "otp-dump-before-set-otp-boot-addr.bin",
    OTP_ADDRESS, OTP_SIZE)

gdb.execute("monitor otp_write_init 1 1 16")
gdb.execute("monitor otp_write {} {}".format(OTP_BOOT_ADDR_RELATIVE, OTP_BOOT_ADDR_VALUE))

dump_memory_to_file(DUMP_DIR + "otp-dump-after-set-otp-boot-addr.bin",
    OTP_ADDRESS, OTP_SIZE)


end


quit
