/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @addtogroup watchdog_driver
 * @{
 */

/*!
 * @file hal_wdt.c
 *
 * @brief Имплементация драйвера сторожевого таймера
 */

#include "hal_wdt.h"

#define WDT_REGS_WR_EN  (0x1ACCE551UL) /*!< Разрешение записей в регистры */
#define WDT_INT_CLR_CMD (0)            /*!< Команда сброса прерывания */

static enum wdt_status last_error = WDT_Status_Ok;

static const WDT_Type *const wdt_timers[WDT_NUMBER_OF_TIMERS] = {
    NSWDT, SWDT_Secure, LPWDT_Secure
};

/*!
 * @brief Проверка наличия таймера
 *
 * @param base   Таймер
 *
 * @retval #WDT_Status_Ok
 * @retval #WDT_Status_BadConfigure
 */
static enum wdt_status WDT_TestBase(WDT_Type *base)
{
    uint32_t i;
    for (i = 0; i < WDT_NUMBER_OF_TIMERS; i++)
        if (wdt_timers[i] == base)
            return WDT_Status_Ok;
    return WDT_Status_BadConfigure;
}

enum wdt_status WDT_GetDefaultConfig(struct wdt_config *config)
{
    if (NULL == config)
        return WDT_Status_InvalidArgument;

    config->load = 0xFFFFFFFF;
    config->resen = WDT_ResenDisable;
    config->inten = WDT_IntenDisable;

    return WDT_Status_Ok;
}

enum wdt_status WDT_Init(WDT_Type *base,
    const struct wdt_config *config)
{
    if ((WDT_Status_Ok != WDT_TestBase(base)) || (NULL == config))
        return WDT_Status_InvalidArgument;

    /* Разрешение записи в регистры */
    SET_VAL_MSK(base->WDOGLOCK, WDT_WDOGLOCK_ENABLEREGISTERWRITES_Msk,
        WDT_WDOGLOCK_ENABLEREGISTERWRITES_Pos, WDT_REGS_WR_EN);

    SET_VAL_MSK(base->WDOGLOAD, WDT_WDOGLOAD_WDOGLOAD_Msk,
        WDT_WDOGLOAD_WDOGLOAD_Pos, config->load);
    SET_VAL_MSK(base->WDOGCONTROL, WDT_WDOGCONTROL_RESEN_Msk,
        WDT_WDOGCONTROL_RESEN_Pos, config->resen);
    SET_VAL_MSK(base->WDOGCONTROL, WDT_WDOGCONTROL_INTEN_Msk,
        WDT_WDOGCONTROL_INTEN_Pos, config->inten);

    /* Запрет записи в регистры */
    SET_VAL_MSK(base->WDOGLOCK, WDT_WDOGLOCK_ENABLEREGISTERWRITES_Msk,
        WDT_WDOGLOCK_ENABLEREGISTERWRITES_Pos, ~WDT_REGS_WR_EN);

    return WDT_Status_Ok;
}

enum wdt_status WDT_Deinit(WDT_Type *base)
{
    if (WDT_Status_Ok != WDT_TestBase(base))
        return WDT_Status_InvalidArgument;

    /* Разрешение записи в регистры */
    SET_VAL_MSK(base->WDOGLOCK, WDT_WDOGLOCK_ENABLEREGISTERWRITES_Msk,
        WDT_WDOGLOCK_ENABLEREGISTERWRITES_Pos, WDT_REGS_WR_EN);

    SET_VAL_MSK(base->WDOGCONTROL, WDT_WDOGCONTROL_INTEN_Msk,
        WDT_WDOGCONTROL_INTEN_Pos, WDT_IntenDisable);
    SET_VAL_MSK(base->WDOGCONTROL, WDT_WDOGCONTROL_RESEN_Msk,
        WDT_WDOGCONTROL_RESEN_Pos, WDT_ResenDisable);


    /* Запрет записи в регистры */
    SET_VAL_MSK(base->WDOGLOCK, WDT_WDOGLOCK_ENABLEREGISTERWRITES_Msk,
        WDT_WDOGLOCK_ENABLEREGISTERWRITES_Pos, ~WDT_REGS_WR_EN);

    return WDT_Status_Ok;
}

enum wdt_status WDT_Enable(WDT_Type *base)
{
    if (WDT_Status_Ok != WDT_TestBase(base))
        return WDT_Status_InvalidArgument;

    /* Разрешение записи в регистры */
    SET_VAL_MSK(base->WDOGLOCK, WDT_WDOGLOCK_ENABLEREGISTERWRITES_Msk,
        WDT_WDOGLOCK_ENABLEREGISTERWRITES_Pos, WDT_REGS_WR_EN);

    SET_VAL_MSK(base->WDOGCONTROL, WDT_WDOGCONTROL_INTEN_Msk,
        WDT_WDOGCONTROL_INTEN_Pos, WDT_IntenEnable);

    /* Запрет записи в регистры */
    SET_VAL_MSK(base->WDOGLOCK, WDT_WDOGLOCK_ENABLEREGISTERWRITES_Msk,
        WDT_WDOGLOCK_ENABLEREGISTERWRITES_Pos, ~WDT_REGS_WR_EN);

    return WDT_Status_Ok;
}

enum wdt_status WDT_Disable(WDT_Type *base)
{
    if (WDT_Status_Ok != WDT_TestBase(base))
        return WDT_Status_InvalidArgument;

    /* Разрешение записи в регистры */
    SET_VAL_MSK(base->WDOGLOCK, WDT_WDOGLOCK_ENABLEREGISTERWRITES_Msk,
        WDT_WDOGLOCK_ENABLEREGISTERWRITES_Pos, WDT_REGS_WR_EN);

    SET_VAL_MSK(base->WDOGCONTROL, WDT_WDOGCONTROL_INTEN_Msk,
        WDT_WDOGCONTROL_INTEN_Pos, WDT_IntenDisable);

    /* Запрет записи в регистры */
    SET_VAL_MSK(base->WDOGLOCK, WDT_WDOGLOCK_ENABLEREGISTERWRITES_Msk,
        WDT_WDOGLOCK_ENABLEREGISTERWRITES_Pos, ~WDT_REGS_WR_EN);

    return WDT_Status_Ok;
}

uint32_t WDT_GetStatusFlagsRaw(WDT_Type *base)
{
    last_error = WDT_Status_Ok;

    if (WDT_Status_Ok != WDT_TestBase(base)) {
        last_error = WDT_Status_InvalidArgument;
        return 0;
    }

    return GET_VAL_MSK(base->WDOGRIS, WDT_WDOGRIS_RAWWATCHDOGINTERRUPT_Msk,
            WDT_WDOGRIS_RAWWATCHDOGINTERRUPT_Pos);
}

uint32_t WDT_GetStatusFlagsMsk(WDT_Type *base)
{
    last_error = WDT_Status_Ok;

    if (WDT_Status_Ok != WDT_TestBase(base)) {
        last_error = WDT_Status_InvalidArgument;
        return 0;
    }

    return GET_VAL_MSK(base->WDOGMIS, WDT_WDOGRIS_RAWWATCHDOGINTERRUPT_Msk,
            WDT_WDOGRIS_RAWWATCHDOGINTERRUPT_Pos);
}

enum wdt_status WDT_ClearStatusFlags(WDT_Type *base, uint32_t mask)
{
    UNUSED(mask);
    if (WDT_Status_Ok != WDT_TestBase(base))
        return WDT_Status_InvalidArgument;

    /* Разрешение записи в регистры */
    SET_VAL_MSK(base->WDOGLOCK, WDT_WDOGLOCK_ENABLEREGISTERWRITES_Msk,
        WDT_WDOGLOCK_ENABLEREGISTERWRITES_Pos, WDT_REGS_WR_EN);

    SET_VAL_MSK(base->WDOGINTCLR, WDT_WDOGINTCLR_WDOGINTCLR_Msk,
        WDT_WDOGINTCLR_WDOGINTCLR_Pos, WDT_INT_CLR_CMD);

    /* Запрет записи в регистры */
    SET_VAL_MSK(base->WDOGLOCK, WDT_WDOGLOCK_ENABLEREGISTERWRITES_Msk,
        WDT_WDOGLOCK_ENABLEREGISTERWRITES_Pos, ~WDT_REGS_WR_EN);

    return WDT_Status_Ok;
}

enum wdt_status WDT_SetWarningValue(WDT_Type *base, uint32_t warning_value)
{
    if ((WDT_Status_Ok != WDT_TestBase(base)) ||
        (warning_value < 1))
        return WDT_Status_InvalidArgument;

    /* Разрешение записи в регистры */
    SET_VAL_MSK(base->WDOGLOCK, WDT_WDOGLOCK_ENABLEREGISTERWRITES_Msk,
        WDT_WDOGLOCK_ENABLEREGISTERWRITES_Pos, WDT_REGS_WR_EN);

    SET_VAL_MSK(base->WDOGLOAD, WDT_WDOGLOAD_WDOGLOAD_Msk,
        WDT_WDOGLOAD_WDOGLOAD_Pos, warning_value);

    /* Запрет записи в регистры */
    SET_VAL_MSK(base->WDOGLOCK, WDT_WDOGLOCK_ENABLEREGISTERWRITES_Msk,
        WDT_WDOGLOCK_ENABLEREGISTERWRITES_Pos, ~WDT_REGS_WR_EN);

    return WDT_Status_Ok;
}

uint32_t WDT_GetWarningValue(WDT_Type *base)
{
    last_error = WDT_Status_Ok;
    if (WDT_Status_Ok != WDT_TestBase(base)) {
        last_error = WDT_Status_InvalidArgument;
        return 0;
    }
    return GET_VAL_MSK(base->WDOGVALUE, WDT_WDOGVALUE_VALUE_Msk,
            WDT_WDOGVALUE_VALUE_Pos);
}

enum wdt_status WDT_SetTimeoutValue(WDT_Type *base, uint32_t timeout_count)
{
    if ((WDT_Status_Ok != WDT_TestBase(base)) ||
        (timeout_count & 1) || (timeout_count < 2))
        return WDT_Status_InvalidArgument;

    /* Разрешение записи в регистры */
    SET_VAL_MSK(base->WDOGLOCK, WDT_WDOGLOCK_ENABLEREGISTERWRITES_Msk,
        WDT_WDOGLOCK_ENABLEREGISTERWRITES_Pos, WDT_REGS_WR_EN);

    SET_VAL_MSK(base->WDOGLOAD, WDT_WDOGLOAD_WDOGLOAD_Msk,
        WDT_WDOGLOAD_WDOGLOAD_Pos, timeout_count / 2);

    /* Запрет записи в регистры */
    SET_VAL_MSK(base->WDOGLOCK, WDT_WDOGLOCK_ENABLEREGISTERWRITES_Msk,
        WDT_WDOGLOCK_ENABLEREGISTERWRITES_Pos, ~WDT_REGS_WR_EN);

    return WDT_Status_Ok;
}

enum wdt_status WDT_Refresh(WDT_Type *base)
{
    if (WDT_Status_Ok != WDT_TestBase(base))
        return WDT_Status_InvalidArgument;

    /* Разрешение записи в регистры */
    SET_VAL_MSK(base->WDOGLOCK, WDT_WDOGLOCK_ENABLEREGISTERWRITES_Msk,
        WDT_WDOGLOCK_ENABLEREGISTERWRITES_Pos, WDT_REGS_WR_EN);

    SET_VAL_MSK(base->WDOGINTCLR, WDT_WDOGINTCLR_WDOGINTCLR_Msk,
        WDT_WDOGINTCLR_WDOGINTCLR_Pos, WDT_INT_CLR_CMD);

    SET_VAL_MSK(base->WDOGLOAD, WDT_WDOGLOAD_WDOGLOAD_Msk,
        WDT_WDOGLOAD_WDOGLOAD_Pos, GET_VAL_MSK(base->WDOGLOAD,
            WDT_WDOGLOAD_WDOGLOAD_Msk, WDT_WDOGLOAD_WDOGLOAD_Pos));

    /* Запрет записи в регистры */
    SET_VAL_MSK(base->WDOGLOCK, WDT_WDOGLOCK_ENABLEREGISTERWRITES_Msk,
        WDT_WDOGLOCK_ENABLEREGISTERWRITES_Pos, ~WDT_REGS_WR_EN);

    return WDT_Status_Ok;
}

enum wdt_status WDT_GetLastAPIStatus()
{
    return last_error;
}

/*!
 * @}
 */
