/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @addtogroup vtu_driver
 * @{
 */

/*!
 * @file hal_vtu.c
 *
 * @brief Имплементация драйвера универсального блока таймеров
 */

#include "hal_vtu.h"
#include "hal_ioim.h"

#define VTU_NUMBER_OF_SYSTEM             2 /*!< Количество систем (блоков) VTU */
#define VTU_NUMBER_OF_SUBSYSTEM          4 /*!< Количество подсистем (таймеров) в VTU */
#define VTU_NUMBER_OF_TIMER_IN_SUBSYSTEM 2 /*!< Количество сдвоенных таймеров */

static enum vtu_status last_error = VTU_Status_Ok;

/* Блоки VTU */
static const VTU_Type * const vtu_timers[VTU_NUMBER_OF_SYSTEM] = {
    VTU0, VTU1
};

/* Инициализированность подсистемы */
static uint32_t vtu_subsystem_init[VTU_NUMBER_OF_SYSTEM]
    [VTU_NUMBER_OF_SUBSYSTEM] = {
    {0, 0, 0, 0}, 
    {0, 0, 0, 0}
};

/* Инициализированность таймера */
static int32_t vtu_timer_init[VTU_NUMBER_OF_SYSTEM][VTU_NUMBER_OF_SUBSYSTEM]
    [VTU_NUMBER_OF_TIMER_IN_SUBSYSTEM] = {
    {
        {0, 0}, {0, 0}, {0, 0}, {0, 0}
    },
    {
        {0, 0}, {0, 0}, {0, 0}, {0, 0}
    }
};

static vtu_callback s_user_callback[VTU_NUMBER_OF_TIMER_IN_SUBSYSTEM
    * VTU_NUMBER_OF_SUBSYSTEM];

/*!
 * @brief Обработчик прерываний VTU
 * 
 * @param base Базовый адрес контроллера
 * @param handle Указатель на контекст драйвера, передаваемый из ioim
 */
static void VTU_IRQHandler(VTU_Type *base, void *handle);

/*!
 * @brief Получение индекса подсистемы в блоке
 *
 * @param timer_index Индекс таймера
 *
 * @return Индекс посистемы
 */
static uint32_t VTU_GetSubsystem(uint32_t timer_index)
{
    return timer_index;
}

/*!
 * @brief Получение индекса субтаймера в подсистеме
 *
 * @param timer_index Индекс таймера
 *
 * @return Индекс субтаймера
 */
static uint32_t VTU_GetSubtimer(uint32_t timer_index)
{
    return timer_index % VTU_NUMBER_OF_TIMER_IN_SUBSYSTEM;
}

/*!
 * @brief Получение масок и смещений прерываний
 *
 * @param subsystem Индекс подсистемы
 * @param mode      Режим работы подсистемы
 * @param msk       Маска прерываний
 * @param pos       Смещение прерываний 
 */
static void VTU_GetIntMskPos(uint32_t subsystem, enum vtu_mode mode,
    enum vtu_interrupt_control *msk, uint32_t *pos)
{
    switch (mode) {
        case VTU_PWM16Bit:
            *msk = VTU_DutyCycleMatch
                | VTU_PeriodMatch;
            break;
        case VTU_Capture:
            *msk = VTU_CaptureToPERCAPx
                | VTU_CaptureToDTYCAPx
                | VTU_CounterOverflow;
            break;
        default : /* VTU_PWMDual8Bit и VTU_LowPower */
            *msk = VTU_LowByteDutyCycleMatch
                | VTU_LowBytePeriodMatch
                | VTU_HighByteDutyCycleMatch
                | VTU_HighBytePeriodMatch;
    }
    *pos = VTU_INTCTL_I1AEN_Pos
        + ((VTU_INTCTL_I2AEN_Pos - VTU_INTCTL_I1AEN_Pos) * subsystem);
}

/*!
 * @brief Получение адреса IOXCTL
 *
 * @param base        Блок таймеров
 * @param timer_index Индекс таймера
 *
 * @return Адрес IOXCTL
 */
static uint32_t * VTU_GetPIOXCtl(VTU_Type *base, uint32_t timer_index)
{
    if ((timer_index == 0) || (timer_index == 2))
        return (uint32_t *) &base->IO1CTL;
    else
        return (uint32_t *) &base->IO2CTL;
}

/*!
 * @brief Получение адреса CLKXPS
 *
 * @param base      Блок таймеров
 * @param subsystem Индекс подсистемы
 *
 * @return Адрес CLKXPS
 */
static uint32_t * VTU_GetPClkXPs(VTU_Type *base, uint32_t subsystem)
{
    switch (subsystem)
    {
    case 0:
    case 1:
        return (uint32_t *) &base->CLK1PS;
    default:
        return (uint32_t *) &base->CLK2PS;
    }
}

/*!
 * @brief Получение адреса COUNTX
 *
 * @param base      Блок таймеров
 * @param subsystem Индекс подсистемы
 *
 * @return Адрес COUNTX
 */
static uint32_t * VTU_GetPCountX(VTU_Type *base, uint32_t subsystem)
{
    switch (subsystem) {
        case 0:
            return (uint32_t *) &base->COUNT1;
        case 1:
            return (uint32_t *) &base->COUNT2;
        case 2:
            return (uint32_t *) &base->COUNT3;
        default:
            return (uint32_t *) &base->COUNT4;
    }
}

/*!
 * @brief Получение адреса PERCAPX
 *
 * @param base      Блок таймеров
 * @param subsystem Индекс подсистемы
 *
 * @return Адрес PERCAPX
 */
static uint32_t * VTU_GetPPerCapX(VTU_Type *base, uint32_t subsystem)
{
    switch (subsystem) {
        case 0:
            return (uint32_t *) &base->PERCAP1;
        case 1:
            return (uint32_t *) &base->PERCAP2;
        case 2:
            return (uint32_t *) &base->PERCAP3;
        default:
            return (uint32_t *) &base->PERCAP4;
    }
}

/*!
 * @brief Получение адреса DTYCAPX
 *
 * @param base      Блок таймеров
 * @param subsystem Индекс подсистемы
 *
 * @return Адрес DTYCAPX
 */
static uint32_t * VTU_GetPDtyCapX(VTU_Type *base, uint32_t subsystem)
{
    switch (subsystem) {
        case 0:
            return (uint32_t *) &base->DTYCAP1;
        case 1:
            return (uint32_t *) &base->DTYCAP2;
        case 2:
            return (uint32_t *) &base->DTYCAP3;
        default:
            return (uint32_t *) &base->DTYCAP4;
    }
}

/*!
 * @brief Получение масок и смещений регистров DTYCAPX, PERCAPX, COUNTX
 *
 * @param subtimer Индекс субтаймера
 * @param extended Признак расширенного(16-битного) режима
 * @param msk      Маска
 * @param pos      Позиция
 */
static void VTU_GetCountPerCapDtyCapMskPos(uint32_t subtimer, bool extended,
    uint32_t *msk, uint32_t *pos)
{
    if (extended) {
        *msk = 0xFFFF;
        *pos = 0;
    } else {
        switch (subtimer) {
            case 0:
                *msk = 0xFF;
                *pos = 0;
                break;
            default:
                *msk = 0xFF00;
                *pos = 8;
                break;
        }
    }
}

/*!
 * @brief Получение маски и смещения поля mode
 *
 * @param subtimer Индекс подсистемы
 * @param msk      Маска
 * @param pos      Позиция
 */
static void VTU_GetModeMskPos(uint32_t subsystem,
    uint32_t *msk, uint32_t *pos)
{
    *msk = VTU_MODE_TMOD1_Msk
        << ((VTU_MODE_TMOD2_Pos - VTU_MODE_TMOD1_Pos) * subsystem);
    *pos = VTU_MODE_TMOD1_Pos
        + ((VTU_MODE_TMOD2_Pos - VTU_MODE_TMOD1_Pos) * subsystem);
}

/*!
 * @brief Получение масок и смещений поля run
 *
 * @param subtimer Индекс подсистемы
 * @param msk1     Маска младшего таймера
 * @param pos1     Позиция младшего таймера
 * @param msk2     Маска старшего таймера
 * @param pos2     Позиция старшего таймера
 */
static void VTU_GetRunMskPos(uint32_t subsystem,
    uint32_t *msk1, uint32_t *pos1,
    uint32_t *msk2, uint32_t *pos2)
{
    *msk1 = VTU_MODE_T1RUN_Msk
        << ((VTU_MODE_T3RUN_Pos - VTU_MODE_T1RUN_Pos) * subsystem);
    *pos1 = VTU_MODE_T1RUN_Pos
        + ((VTU_MODE_T3RUN_Pos - VTU_MODE_T1RUN_Pos) * subsystem);
    *msk2 = VTU_MODE_T2RUN_Msk
        << ((VTU_MODE_T3RUN_Pos - VTU_MODE_T1RUN_Pos) * subsystem);
    *pos2 = VTU_MODE_T2RUN_Pos
        + ((VTU_MODE_T3RUN_Pos - VTU_MODE_T1RUN_Pos) * subsystem);
}

/*!
 * @brief Получение маски и смещения поля pxpol
 *
 * @param timer_index Индекс таймера
 * @param extended    Признак расширенного(16-битного) режима
 * @param msk         Маска младшего таймера
 * @param pos         Позиция младшего таймера
 */
static void VTU_GetPxpolMskPos(uint32_t timer_index, bool extended,
    uint32_t *msk, uint32_t *pos)
{
    uint32_t p;
    p = timer_index & 0x3;
    *msk = VTU_IO1CTL_P1POL_Msk
        << ((VTU_IO1CTL_P2POL_Pos - VTU_IO1CTL_P1POL_Pos) * (p));
    *pos = VTU_IO1CTL_P1POL_Pos
        + ((VTU_IO1CTL_P2POL_Pos - VTU_IO1CTL_P1POL_Pos) * (p));
    if (extended && !(timer_index & 1)) {
        *msk |= VTU_IO1CTL_P2POL_Msk
            << ((VTU_IO1CTL_P2POL_Pos - VTU_IO1CTL_P1POL_Pos) * (p));
    }
}

/*!
 * @brief Получение маски и смещения поля cxedg
 *
 * @param timer_index Индекс таймера
 * @param extended    Признак расширенного(16-битного) режима
 * @param msk         Маска младшего таймера
 * @param pos         Позиция младшего таймера
 */
static void VTU_GetCxedgMskPos(uint32_t timer_index, bool extended,
    uint32_t *msk, uint32_t *pos)
{
    uint32_t p;
    p = timer_index & 0x3;
    *msk = VTU_IO1CTL_C1EDG_Msk 
        << ((VTU_IO1CTL_C2EDG_Pos - VTU_IO1CTL_C1EDG_Pos) * (p));
    *pos = VTU_IO1CTL_C1EDG_Pos
        + ((VTU_IO1CTL_C2EDG_Pos - VTU_IO1CTL_C1EDG_Pos) * (p));
    if (extended && !(timer_index & 1)) {
        *msk |= VTU_IO1CTL_C2EDG_Msk
            << ((VTU_IO1CTL_C2EDG_Pos - VTU_IO1CTL_C1EDG_Pos) * (p));
    }
}

/*!
 * @brief Получение маски и смещения поля clkxps
 *
 * @param timer_index Индекс подсистемы
 * @param msk         Маска младшего таймера
 * @param pos         Позиция младшего таймера
 */
static void VTU_GetClkxpsMskPos(uint32_t subsystem, uint32_t *msk,
    uint32_t *pos)
{
    *msk = VTU_CLK1PS_C1PRSC_Msk
        << ((VTU_CLK1PS_C2PRSC_Pos - VTU_CLK1PS_C1PRSC_Pos) * (subsystem & 1));
    *pos = VTU_CLK1PS_C1PRSC_Pos
        + ((VTU_CLK1PS_C2PRSC_Pos - VTU_CLK1PS_C1PRSC_Pos) * (subsystem & 1));
}

/*!
 * @brief Проверка наличия системы таймера VTU, и получение его индекса 
 *
 * @param base  Таймер
 * @param index Указатель на индекс системы таймера, если результат выполнения
 *              функции VTU_Status_Ok
 *
 * @retval #VTU_Status_Ok
 * @retval #VTU_Status_BadConfigure
 */
static enum vtu_status VTU_TestBase(VTU_Type *base, uint32_t *index)
{
    uint32_t i;
    for (i = 0; i < VTU_NUMBER_OF_SYSTEM; i++)
        if (vtu_timers[i] == base) {
            *index = i;
            return VTU_Status_Ok;
        }
    *index = i;
    return VTU_Status_BadConfigure;
}

enum vtu_status VTU_GetLastAPIStatus(void)
{
    return last_error;
}

enum vtu_status VTU_GetDefaultConfig(struct vtu_config *config)
{
    if (config == NULL)
        return VTU_Status_InvalidArgument;
    config->mode = VTU_LowPower;
    config->capture_edge_control1 = VTU_CaptureRisingEdgeResetNo;
    config->capture_edge_control2 = VTU_CaptureRisingEdgeResetNo;
    config->pwm_polarity = VTU_One;
    config->pwm_polarity2 = VTU_One;
    config->interrupt_control = VTU_NoInterrupt;
    config->prescaler = 0;
    config->counter = 0;
    config->period = 0;
    config->duty_cycle_capture = 0;
    return VTU_Status_Ok;
}

enum vtu_status VTU_Init(VTU_Type *base, uint32_t timer,
    struct vtu_config *config)
{
    uint32_t system_index; /* Индекс системы */
    uint32_t timer_index;  /* Индекс таймера */
    uint32_t subsystem;    /* Индекс подсистемы */
    uint32_t subtimer;     /* Индекс субтаймера */
    uint32_t mode_msk, mode_pos;
    uint32_t run1_msk, run1_pos;
    uint32_t run2_msk, run2_pos;
    uint32_t cxedg_msk, cxedg_pos;
    uint32_t pxpol_msk, pxpol_pos;
    uint32_t *ioxctl;
    enum vtu_mode mode_field;
    enum vtu_interrupt_control all_interrupt_msk;
    uint32_t all_interrupt_pos;
    uint32_t *clkxps;
    uint32_t cxprsc_msk, cxprsc_pos;
    uint32_t complement_timer_is_run;
    uint32_t count_percap_dtycap_msk, count_percap_dtycap_pos;
    uint32_t *countx;
    uint32_t *percapx;
    uint32_t *dtycapx;

    if (NULL == config)
        return VTU_Status_InvalidArgument;
    /* Проверка базового адреса. */
    if (VTU_Status_Ok != VTU_TestBase(base, &system_index))
        return VTU_Status_InvalidArgument;
    /* Проверка режима. */
    if (config->mode == VTU_LowPower)
        return VTU_Status_InvalidArgument;
    /* Проверка номера таймера. */
    if ((timer == 0) || (timer >
        VTU_NUMBER_OF_SUBSYSTEM * VTU_NUMBER_OF_TIMER_IN_SUBSYSTEM))
    {
        return VTU_Status_InvalidArgument;
    }
    timer_index = timer - 1;
    /* Вычисление индексов подсистемы и таймера в подсистеме. */
    subsystem = VTU_GetSubsystem(timer_index);
    subtimer = VTU_GetSubtimer(timer_index);
    /* Индекс таймера для режимов VTU_PWM16Bit и
     * VTU_Capture может быть только четным. */
    if (subtimer
        && ((config->mode == VTU_PWM16Bit) || (config->mode == VTU_Capture)))
    {
        return VTU_Status_BadConfigure;
    }
    /* Проверка корректности запрошенных прерываний. */
    VTU_GetIntMskPos(subsystem, config->mode, 
        &all_interrupt_msk, &all_interrupt_pos);
    if ((config->interrupt_control & ~all_interrupt_msk) != 0)
        return VTU_Status_BadConfigure; /* Недопустимые прерывания */
    /* Маски, смещения и базовые адреса */
    VTU_GetModeMskPos(subsystem, &mode_msk, &mode_pos);
    VTU_GetRunMskPos(subsystem, &run1_msk, &run1_pos, &run2_msk, &run2_pos);
    VTU_GetPxpolMskPos(timer_index, (config->mode == VTU_PWMDual8Bit) ? 0 : 1,
        &pxpol_msk, &pxpol_pos);
    VTU_GetCxedgMskPos(timer_index, (config->mode == VTU_PWMDual8Bit) ? 0 : 1,
        &cxedg_msk, &cxedg_pos);
    VTU_GetClkxpsMskPos(subsystem, &cxprsc_msk, &cxprsc_pos);
    /* Вычисление адресов регистров ioxctl для выбранного таймера. */
    ioxctl = VTU_GetPIOXCtl(base, timer_index);
    UNUSED(ioxctl);
    /* Вычисление адреса регистра clkxps для выбранного таймера. */
    clkxps = VTU_GetPClkXPs(base, subsystem);
    /* Вычисление адресов регистров countx, percapx, dtycapx,
       а так же масок полей для выбранного таймера. */
    countx =  VTU_GetPCountX(base, subsystem);
    percapx = VTU_GetPPerCapX(base, subsystem);
    dtycapx = VTU_GetPDtyCapX(base, subsystem);
    /* Формирование маски для регистров счетчиков. */
    VTU_GetCountPerCapDtyCapMskPos(subtimer,
        (config->mode == VTU_PWMDual8Bit) ? 0 : 1,
        &count_percap_dtycap_msk, &count_percap_dtycap_pos);
    /* Установка режима. */
    if (vtu_subsystem_init[system_index][subsystem] != 0) { /* Подсистема инициализирована. */
        /* Проверка на возможность использования сдвоенного режима. */
        if (config->mode != VTU_PWMDual8Bit) /* Если это не сдвоенный режим, то инициализация невозможна. */
            return VTU_Status_TimerBusy;
        mode_field = GET_VAL_MSK(base->MODE, mode_msk, mode_pos);
        if (mode_field != VTU_PWMDual8Bit) /* Несовместимые режимы */
            return VTU_Status_TimerBusy;
        if (vtu_timer_init[system_index][subsystem][subtimer] != 0) /* Именно запрашиваемый таймер занят. */
            return VTU_Status_TimerBusy;
        /* Проверка совпадения предделителей. */
        if (config->prescaler != GET_VAL_MSK(*clkxps, cxprsc_msk, cxprsc_pos))
            return VTU_Status_BadConfigure;
        if (config->prescaler != GET_VAL_MSK(*clkxps, cxprsc_msk, cxprsc_pos))
            return VTU_Status_BadConfigure;
        /* Проверка рабочего состояния сдвоенного таймера. */
        switch (subtimer) {
            case 0:
                complement_timer_is_run = GET_VAL_MSK(base->MODE, run2_msk,
                    run2_pos);
                break;
            case 1:
                complement_timer_is_run = GET_VAL_MSK(base->MODE, run1_msk,
                    run1_pos);
                break;
            default :
                return VTU_Status_BadConfigure;
        }
        if (complement_timer_is_run)
            return VTU_Status_DualTimerNotCanRun;
        vtu_timer_init[system_index][subsystem][subtimer] = 1; /* Заняли таймер в подсистеме. */
        /* Останавливаем указанный таймер в подсистеме. */
        switch (subtimer) {
            case 0:
                SET_VAL_MSK(base->MODE, run1_msk, run1_pos, 0);
                break;
            case 1:
                SET_VAL_MSK(base->MODE, run2_msk, run2_pos, 0);
                break;
            default :
                return VTU_Status_BadConfigure;
        }
        /* Установка полярности и фронта захвата. */
        VTU_SetCaptureEdgeCtrl(base, timer,
            config->capture_edge_control1, config->capture_edge_control2, 0);
        VTU_SetPWMPolarity(base, timer,
            config->pwm_polarity, config->pwm_polarity2, 0);
        /* Установка прерываний. */
        SET_VAL_MSK(base->INTCTL, all_interrupt_msk, 
            all_interrupt_pos, config->interrupt_control);
        /* Установка значений регистров. */
        SET_VAL_MSK(*countx, count_percap_dtycap_msk, count_percap_dtycap_pos,
            config->counter);
        SET_VAL_MSK(*percapx, count_percap_dtycap_msk, count_percap_dtycap_pos,
            config->period);
        SET_VAL_MSK(*dtycapx, count_percap_dtycap_msk, count_percap_dtycap_pos,
            config->duty_cycle_capture);
    } else { /* Подсистема не инициализирована. */
        vtu_subsystem_init[system_index][subsystem] = 1;
        /* Остановка всех таймеров в подсистеме. */
        SET_VAL_MSK(base->MODE, run1_msk, run1_pos, 0);
        SET_VAL_MSK(base->MODE, run2_msk, run2_pos, 0);   
        /* Установка режима подсистемы. */
        SET_VAL_MSK(base->MODE, mode_msk, mode_pos, config->mode);
        /* Управление VTU_GetTimerIRQ захватом и полярностью. */
        VTU_SetCaptureEdgeCtrl(base, timer,
            config->capture_edge_control1, config->capture_edge_control2,
            (config->mode == VTU_PWMDual8Bit) ? 0 : 1);
        VTU_SetPWMPolarity(base, timer,
            config->pwm_polarity, config->pwm_polarity2,
            (config->mode == VTU_PWMDual8Bit) ? 0 : 1);
        SET_VAL_MSK(base->INTCTL, all_interrupt_msk, all_interrupt_pos,
            config->interrupt_control);
        SET_VAL_MSK(*clkxps, cxprsc_msk, cxprsc_pos, config->prescaler);
        SET_VAL_MSK(*countx, count_percap_dtycap_msk, count_percap_dtycap_pos,
            config->counter);
        SET_VAL_MSK(*percapx, count_percap_dtycap_msk, count_percap_dtycap_pos,
            config->period);
        SET_VAL_MSK(*dtycapx, count_percap_dtycap_msk, count_percap_dtycap_pos,
            config->duty_cycle_capture);
        /* Захват флага таймера в подсистеме. */
        vtu_timer_init[system_index][subsystem][subtimer] = 1;
    }

    if (IOIM_ClearIRQHandler(base) != IOIM_Status_Ok) {
        return VTU_Status_DriverError;
    }

    if (IOIM_SetIRQHandler(base, VTU_IRQHandler, NULL) != IOIM_Status_Ok) {
        return VTU_Status_DriverError;
    }

    return VTU_Status_Ok;
}

enum vtu_status VTU_Deinit(VTU_Type *base, uint32_t timer)
{
    uint32_t system_index; /* Индекс системы */
    uint32_t timer_index;  /* Индекс таймера */
    uint32_t subsystem;    /* Индекс подсистемы */
    uint32_t subtimer;     /* Индекс субтаймера */
    enum vtu_mode mode_field;
    uint32_t i;
    uint32_t mode_msk, mode_pos;
    uint32_t run1_msk, run1_pos;
    uint32_t run2_msk, run2_pos;

    /* Проверка базового адреса. */
    if (VTU_Status_Ok != VTU_TestBase(base, &system_index))
        return VTU_Status_InvalidArgument;
    /* Проверка номера таймера. */
    if ((timer == 0) || (timer >
        VTU_NUMBER_OF_SUBSYSTEM*VTU_NUMBER_OF_TIMER_IN_SUBSYSTEM))
    {
        return VTU_Status_InvalidArgument;
    }
    timer_index = timer - 1;
    /* Вычисление индексов подсистемы и субтаймера. */
    subsystem = VTU_GetSubsystem(timer_index);
    subtimer = VTU_GetSubtimer(timer_index);
    /* Маски, смещения и базовые адреса */
    VTU_GetModeMskPos(subsystem, &mode_msk, &mode_pos);       
    VTU_GetRunMskPos(subsystem, &run1_msk, &run1_pos, &run2_msk, &run2_pos);
    mode_field = GET_VAL_MSK(base->MODE, mode_msk, mode_pos);
    if (mode_field == VTU_PWMDual8Bit) {
        /* Останов таймерa. */
        switch (subtimer) {
            case 0:
                SET_VAL_MSK(base->MODE, run1_msk, run1_pos, 0);
                break;
            case 1:
                SET_VAL_MSK(base->MODE, run2_msk, run2_pos, 0);
                break;
            default :
                return VTU_Status_DriverError;
        }
        /* Признак освобождения таймера */
        vtu_timer_init[system_index][subsystem][subtimer] = 0;
        /* Анализ освобождения подсистемы */
        for (i = 0; i < VTU_NUMBER_OF_TIMER_IN_SUBSYSTEM; i++)
            if (vtu_timer_init[system_index][subsystem][i] != 0)
                break;
        if (i == VTU_NUMBER_OF_TIMER_IN_SUBSYSTEM) {            
            SET_VAL_MSK(base->MODE, mode_msk, mode_pos, VTU_LowPower);
            vtu_subsystem_init[system_index][subsystem] = 0;
        }
    } else {
        /* Останов и перевод таймера в сберегающий режим. */
        SET_VAL_MSK(base->MODE, run2_msk, run2_pos, 0);
        SET_VAL_MSK(base->MODE, run1_msk, run1_pos, 0);
        SET_VAL_MSK(base->MODE, mode_msk, mode_pos, VTU_LowPower);
        /* Признак освобождения таймера и подсистемы. */
        vtu_timer_init[system_index][subsystem][subtimer] = 0;
        vtu_subsystem_init[system_index][subsystem] = 0;
    }
    return VTU_Status_Ok;
}

enum vtu_status VTU_EnableTimer(VTU_Type *base, uint32_t timer, bool enable)
{
    uint32_t system_index; /* Индекс системы */
    uint32_t timer_index;  /* Индекс таймера */
    uint32_t subsystem;    /* Индекс подсистемы */
    uint32_t subtimer;     /* Индекс субтаймера */
    uint32_t run;
    uint32_t run1_msk, run1_pos;
    uint32_t run2_msk, run2_pos;

    /* Проверка базового адреса. */
    if (VTU_Status_Ok != VTU_TestBase(base, &system_index))
        return VTU_Status_InvalidArgument;
    /* Проверка номера таймера. */
    if ((timer == 0) || (timer >
        VTU_NUMBER_OF_SUBSYSTEM*VTU_NUMBER_OF_TIMER_IN_SUBSYSTEM))
    {
        return VTU_Status_InvalidArgument;
    }
    timer_index = timer - 1;
    /* Вычисление индексов подсистемы и таймера в подсистеме. */
    subsystem = VTU_GetSubsystem(timer_index);
    subtimer = VTU_GetSubtimer(timer_index);
    /* Маски, смещения и базовые адреса */
    VTU_GetRunMskPos(subsystem, &run1_msk, &run1_pos, &run2_msk, &run2_pos);
    if (vtu_timer_init[system_index][subsystem][subtimer] == 0) {
        return VTU_Status_TimerNotInit;
    }
    /* Разрешение работы таймерa */
    run = enable ? 1 : 0;
    switch (subtimer) {
        case 0:
            SET_VAL_MSK(base->MODE, run1_msk, run1_pos, run);
            break;
        case 1:
            SET_VAL_MSK(base->MODE, run2_msk, run2_pos, run);
            break;
        default :
            return VTU_Status_DriverError;
    }
    return VTU_Status_Ok;
}

enum vtu_status VTU_SetCounter(VTU_Type *base, uint32_t timer,
    uint16_t value, bool extended)
{
    uint32_t system_index; /* Индекс системы */
    uint32_t timer_index;  /* Индекс таймера */
    uint32_t subsystem;    /* Индекс подсистемы */
    uint32_t subtimer;     /* Индекс субтаймера */
    uint32_t count_percap_dtycap_msk, count_percap_dtycap_pos;
    uint32_t *countx;

    /* Проверка базового адреса */
    if (VTU_Status_Ok != VTU_TestBase(base, &system_index))
        return VTU_Status_InvalidArgument;
    /* Проверка номера таймера */
    if ((timer == 0) || (timer >
        VTU_NUMBER_OF_SUBSYSTEM*VTU_NUMBER_OF_TIMER_IN_SUBSYSTEM))
    {
        return VTU_Status_InvalidArgument;
    }
    timer_index = timer - 1;
    /* Вычисление индексов подсистемы и таймера в подсистеме */
    subsystem = VTU_GetSubsystem(timer_index);
    subtimer = VTU_GetSubtimer(timer_index);
    /* Проверка флага инициализации таймера */
    if (vtu_timer_init[system_index][subsystem][subtimer] == 0) {
        return VTU_Status_TimerNotInit;
    }
    if (subtimer && extended)
    {
        return VTU_Status_BadConfigure;
    }
    /* TODO Проверку запущенности*/
    /* Формирование маски для регистров счетчиков.. */
    VTU_GetCountPerCapDtyCapMskPos(subtimer, extended,
        &count_percap_dtycap_msk, &count_percap_dtycap_pos);
    /* Вычисление адресов регистров countx, 
       а так же масок полей для выбранного таймера. */
    countx = VTU_GetPCountX(base, subsystem);
    SET_VAL_MSK(*countx, count_percap_dtycap_msk, count_percap_dtycap_pos,
        value);
    return VTU_Status_Ok;
}

uint16_t VTU_GetCounter(VTU_Type *base, uint32_t timer, bool extended)
{
    uint32_t system_index; /* Индекс системы */
    uint32_t timer_index;  /* Индекс таймера */
    uint32_t subsystem;    /* Индекс подсистемы */
    uint32_t subtimer;     /* Индекс субтаймера */
    uint32_t count_percap_dtycap_msk, count_percap_dtycap_pos;
    uint32_t *countx;

    last_error = VTU_Status_Ok;
    /* Проверка базового адреса. */
    if (VTU_Status_Ok != VTU_TestBase(base, &system_index)) {
        last_error = VTU_Status_InvalidArgument;
        return 0;
    }
    /* Проверка номера таймера. */
    if ((timer == 0) || (timer >
        VTU_NUMBER_OF_SUBSYSTEM*VTU_NUMBER_OF_TIMER_IN_SUBSYSTEM))
    {
        last_error = VTU_Status_InvalidArgument;
        return 0;
    }
    timer_index = timer - 1;
    /* Вычисление индексов подсистемы и таймера в подсистеме. */
    subsystem = VTU_GetSubsystem(timer_index);
    subtimer = VTU_GetSubtimer(timer_index);
    /* Проверка флага инициализации таймера. */
    if (vtu_timer_init[system_index][subsystem][subtimer] == 0) {
        last_error = VTU_Status_InvalidArgument;
        return 0;
    }
    /* Проверка совместимости индекса таймера и режима. */
    if (subtimer && extended)
    {
        last_error = VTU_Status_BadConfigure;
        return 0;
    }
    /* Формирование маски для регистров счетчиков.. */
    VTU_GetCountPerCapDtyCapMskPos(subtimer, extended,
        &count_percap_dtycap_msk, &count_percap_dtycap_pos);
    /* Вычисление адресов регистров countx, 
       а так же масок полей для выбранного таймера. */
    countx = VTU_GetPCountX(base, subsystem);
    return GET_VAL_MSK(*countx, count_percap_dtycap_msk,
        count_percap_dtycap_pos);
}

enum vtu_status VTU_SetPrescaler(VTU_Type *base, uint32_t timer, uint8_t value)
{
    uint32_t system_index; /* Индекс системы */
    uint32_t timer_index;  /* Индекс таймера */
    uint32_t subsystem;    /* Индекс подсистемы */
    uint32_t subtimer;     /* Индекс субтаймера */
    uint32_t *clkxps;      /* Указатель на предделитель */
    uint32_t cxprsc_msk, cxprsc_pos;
    uint8_t prescaler;
    /* Проверка базового адреса. */
    if (VTU_Status_Ok != VTU_TestBase(base, &system_index))
        return VTU_Status_InvalidArgument;
    /* Проверка номера таймера. */
    if ((timer == 0) || (timer >
        VTU_NUMBER_OF_SUBSYSTEM*VTU_NUMBER_OF_TIMER_IN_SUBSYSTEM))
    {
        return VTU_Status_InvalidArgument;
    }
    timer_index = timer - 1;
    /* Вычисление индексов подсистемы и таймера в подсистеме. */
    subsystem = VTU_GetSubsystem(timer_index);
    subtimer = VTU_GetSubtimer(timer_index);
    /* Проверка флага инициализации таймера. */
    if (vtu_timer_init[system_index][subsystem][subtimer] == 0) {
        return VTU_Status_TimerNotInit;
    }
    VTU_GetClkxpsMskPos(subsystem, &cxprsc_msk, &cxprsc_pos);
    /* Вычисление адреса регистра clkxps для выбранного таймера. */
    clkxps = VTU_GetPClkXPs(base, subsystem);
    prescaler = GET_VAL_MSK(*clkxps, cxprsc_msk, cxprsc_pos); /* Получение значения предделителя. */

    /* Проверка инициализированности "соседнего" субтаймера. */
    if (vtu_timer_init[system_index][subsystem][subtimer ^ 1] != 0) {
        if (prescaler != value)
            return VTU_Status_BadConfigure; /* Противоречивая конфигурация */
        else
            return VTU_Status_Ok;
    }

    SET_VAL_MSK(*clkxps, cxprsc_msk, cxprsc_pos, value);
    return VTU_Status_Ok;
}

uint8_t VTU_GetPrescaler(VTU_Type *base, uint32_t timer)
{
    uint32_t system_index; /* Индекс системы */
    uint32_t timer_index;  /* Индекс таймера */
    uint32_t subsystem;    /* Индекс подсистемы */
    uint32_t subtimer;     /* Индекс субтаймера */
    uint32_t *clkxps;      /* Указатель на предделитель */
    uint32_t cxprsc_msk, cxprsc_pos;

    last_error = VTU_Status_Ok;
    /* Проверка базового адреса. */
    if (VTU_Status_Ok != VTU_TestBase(base, &system_index)) {
        last_error = VTU_Status_InvalidArgument;
        return 0;
    }
    /* Проверка номера таймера. */
    if ((timer == 0) || (timer >
        VTU_NUMBER_OF_SUBSYSTEM*VTU_NUMBER_OF_TIMER_IN_SUBSYSTEM))
    {
        last_error = VTU_Status_InvalidArgument;
        return 0;
    }
    timer_index = timer - 1;
    /* Вычисление индексов подсистемы и таймера в подсистеме. */
    subsystem = VTU_GetSubsystem(timer_index);
    subtimer = VTU_GetSubtimer(timer_index);
    /* Проверка флага инициализации таймера. */
    if (vtu_timer_init[system_index][subsystem][subtimer] == 0) {
        last_error = VTU_Status_TimerNotInit;
        return 0;
    }
    VTU_GetClkxpsMskPos(subsystem, &cxprsc_msk, &cxprsc_pos);
    /* Вычисление адреса регистра clkxps для выбранного таймера. */
    clkxps = VTU_GetPClkXPs(base, subsystem);
    return GET_VAL_MSK(*clkxps, cxprsc_msk, cxprsc_pos);
}

enum vtu_status VTU_SetPeriodCapture(VTU_Type *base, uint32_t timer,
    uint16_t value, bool extended)
{
    uint32_t system_index; /* Индекс системы */
    uint32_t timer_index;  /* Индекс таймера */
    uint32_t subsystem;    /* Индекс подсистемы */
    uint32_t subtimer;     /* Индекс субтаймера */
    uint32_t count_percap_dtycap_msk, count_percap_dtycap_pos;
    uint32_t *percapx;

    /* Проверка базового адреса. */
    if (VTU_Status_Ok != VTU_TestBase(base, &system_index))
        return VTU_Status_InvalidArgument;
    /* Проверка номера таймера. */
    if ((timer == 0) || (timer >
        VTU_NUMBER_OF_SUBSYSTEM*VTU_NUMBER_OF_TIMER_IN_SUBSYSTEM))
    {
        return VTU_Status_InvalidArgument;
    }
    timer_index = timer - 1;
    /* Вычисление индексов подсистемы и таймера в подсистеме. */
    subsystem = VTU_GetSubsystem(timer_index);
    subtimer = VTU_GetSubtimer(timer_index);
    /* Проверка флага инициализации таймера. */
    if (vtu_timer_init[system_index][subsystem][subtimer] == 0) {
        return VTU_Status_TimerNotInit;
    }
    if (subtimer && extended)
    {
        return VTU_Status_BadConfigure;
    }
    /* Формирование маски для регистров счетчиков.. */
    VTU_GetCountPerCapDtyCapMskPos(subtimer, extended, 
        &count_percap_dtycap_msk, &count_percap_dtycap_pos);
    /* Вычисление адресов регистров percapx, 
       а так же масок полей для выбранного таймера. */
    percapx = VTU_GetPPerCapX(base, subsystem);
    SET_VAL_MSK(*percapx, count_percap_dtycap_msk, count_percap_dtycap_pos,
        value);
    return VTU_Status_Ok;
}

uint16_t VTU_GetPeriodCapture(VTU_Type *base, uint32_t timer,
    bool extended)
{
    uint32_t system_index; /* Индекс системы */
    uint32_t timer_index;  /* Индекс таймера */
    uint32_t subsystem;    /* Индекс подсистемы */
    uint32_t subtimer;     /* Индекс субтаймера */
    uint32_t count_percap_dtycap_msk, count_percap_dtycap_pos;
    uint32_t *percapx;

    last_error = VTU_Status_Ok;
    /* Проверка базового адреса. */
    if (VTU_Status_Ok != VTU_TestBase(base, &system_index)) {
        last_error = VTU_Status_InvalidArgument;
        return 0;
    }
    /* Проверка номера таймера. */
    if ((timer == 0) || (timer >
        VTU_NUMBER_OF_SUBSYSTEM*VTU_NUMBER_OF_TIMER_IN_SUBSYSTEM))
    {
        last_error = VTU_Status_InvalidArgument;
        return 0;
    }
    timer_index = timer - 1;
    /* Вычисление индексов подсистемы и таймера в подсистеме. */
    subsystem = VTU_GetSubsystem(timer_index);
    subtimer = VTU_GetSubtimer(timer_index);
    /* Проверка флага инициализации таймера. */
    if (vtu_timer_init[system_index][subsystem][subtimer] == 0) {
        last_error = VTU_Status_TimerNotInit;
        return 0;
    }
    if (subtimer && extended)
    {
        last_error = VTU_Status_BadConfigure;
        return 0;
    }
    /* Формирование маски для регистров счетчиков.. */
    VTU_GetCountPerCapDtyCapMskPos(subtimer, extended, 
        &count_percap_dtycap_msk, &count_percap_dtycap_pos);
    /* Вычисление адресов регистров percapx, 
       а так же масок полей для выбранного таймера. */
    percapx = VTU_GetPPerCapX(base, subsystem);
    return GET_VAL_MSK(*percapx, count_percap_dtycap_msk,
        count_percap_dtycap_pos);
}

enum vtu_status VTU_SetDutyCycleCapture(VTU_Type *base, uint32_t timer,
    uint16_t value, bool extended)
{
    uint32_t system_index; /* Индекс системы */
    uint32_t timer_index; /* Индекс таймера */
    uint32_t subsystem; /* Индекс подсистемы */
    uint32_t subtimer; /* Индекс таймера в подсистеме */
    uint32_t count_percap_dtycap_msk, count_percap_dtycap_pos;
    uint32_t *dtycapx;

    /* Проверка базового адреса. */
    if (VTU_Status_Ok != VTU_TestBase(base, &system_index))
        return VTU_Status_InvalidArgument;
    /* Проверка номера таймера. */
    if ((timer == 0) || (timer >
        VTU_NUMBER_OF_SUBSYSTEM*VTU_NUMBER_OF_TIMER_IN_SUBSYSTEM))
    {
        return VTU_Status_InvalidArgument;
    }
    timer_index = timer - 1;
    /* Вычисление индексов подсистемы и таймера в подсистеме. */
    subsystem = VTU_GetSubsystem(timer_index);
    subtimer = VTU_GetSubtimer(timer_index);
    /* Проверка флага инициализации таймера. */
    if (vtu_timer_init[system_index][subsystem][subtimer] == 0) {
        return VTU_Status_TimerNotInit;
    }
    if (subtimer && extended)
    {
        return VTU_Status_BadConfigure;
    }
    VTU_GetCountPerCapDtyCapMskPos(subtimer, extended, 
        &count_percap_dtycap_msk, &count_percap_dtycap_pos);
    dtycapx = VTU_GetPDtyCapX(base, subsystem);
    SET_VAL_MSK(*dtycapx, count_percap_dtycap_msk, count_percap_dtycap_pos,
        value);
    return VTU_Status_Ok;
}

uint16_t VTU_GetDutyCycleCapture(VTU_Type *base, uint32_t timer,
    bool extended)
{
    uint32_t system_index; /* Индекс системы */
    uint32_t timer_index; /* Индекс таймера */
    uint32_t subsystem; /* Индекс подсистемы */
    uint32_t subtimer; /* Индекс субтаймера */
    uint32_t count_percap_dtycap_msk, count_percap_dtycap_pos;
    uint32_t *dtycapx;

    last_error = VTU_Status_Ok;
    /* Проверка базового адреса. */
    if (VTU_Status_Ok != VTU_TestBase(base, &system_index)) {
        last_error = VTU_Status_InvalidArgument;
        return 0;
    }
    /* Проверка номера таймера. */
    if ((timer == 0) || (timer >
        VTU_NUMBER_OF_SUBSYSTEM*VTU_NUMBER_OF_TIMER_IN_SUBSYSTEM))
    {
        last_error = VTU_Status_InvalidArgument;
        return 0;
    }
    timer_index = timer - 1;
    /* Вычисление индексов подсистемы и таймера в подсистеме. */
    subsystem = VTU_GetSubsystem(timer_index);
    subtimer = VTU_GetSubtimer(timer_index);
    /* Проверка флага инициализации таймера. */
    if (vtu_timer_init[system_index][subsystem][subtimer] == 0) {
        last_error = VTU_Status_TimerNotInit;
        return 0;
    }
    if (subtimer && extended)
    {
        last_error = VTU_Status_BadConfigure;
        return 0;
    }
    VTU_GetCountPerCapDtyCapMskPos(subtimer, extended, 
        &count_percap_dtycap_msk, &count_percap_dtycap_pos);
    dtycapx = VTU_GetPDtyCapX(base, subsystem);
    return GET_VAL_MSK(*dtycapx, count_percap_dtycap_msk,
        count_percap_dtycap_pos);
}

enum vtu_status VTU_EnableTimerIRQ(VTU_Type *base, uint32_t timer,
    enum vtu_interrupt_control values, bool enable, enum vtu_mode mode)
{
    uint32_t system_index; /* Индекс системы */
    uint32_t timer_index; /* Индекс таймера */
    uint32_t subsystem; /* Индекс подсистемы */
    enum vtu_interrupt_control all_interrupt_msk;
    uint32_t all_interrupt_pos;
    uint32_t interrupts;

    /* Проверка базового адреса. */
    if (VTU_Status_Ok != VTU_TestBase(base, &system_index))
        return VTU_Status_InvalidArgument;
    /* Проверка номера таймера. */
    if ((timer == 0) || (timer >
        VTU_NUMBER_OF_SUBSYSTEM*VTU_NUMBER_OF_TIMER_IN_SUBSYSTEM))
    {
        return VTU_Status_InvalidArgument;
    }
    timer_index = timer - 1;
    subsystem = VTU_GetSubsystem(timer_index);
    /* Подготовка маски прерываний. */
    VTU_GetIntMskPos(subsystem, mode, 
        &all_interrupt_msk, &all_interrupt_pos);
    interrupts = GET_VAL_MSK(base->INTCTL, all_interrupt_msk,
        all_interrupt_pos);
    if (enable)
        values = interrupts | values;
    else
        values = interrupts & (~values);
    SET_VAL_MSK(base->INTCTL,
        (all_interrupt_msk << (subsystem * VTU_NUMBER_OF_SUBSYSTEM)),
        all_interrupt_pos, values);
    return VTU_Status_Ok;
}

enum vtu_status VTU_ClearTimerIRQ(VTU_Type *base, uint32_t timer,
    enum vtu_interrupt_control values, enum vtu_mode mode)
{
    uint32_t system_index; /* Индекс системы */
    uint32_t timer_index; /* Индекс таймера */
    uint32_t subsystem; /* Индекс подсистемы */
    enum vtu_interrupt_control all_interrupt_msk;
    uint32_t all_interrupt_pos;

    /* Проверка базового адреса. */
    if (VTU_Status_Ok != VTU_TestBase(base, &system_index))
        return VTU_Status_InvalidArgument;
    /* Проверка номера таймера. */
    if ((timer == 0) || (timer >
        VTU_NUMBER_OF_SUBSYSTEM*VTU_NUMBER_OF_TIMER_IN_SUBSYSTEM))
    {
        return VTU_Status_InvalidArgument;
    }
    timer_index = timer - 1;
    subsystem = VTU_GetSubsystem(timer_index);
    VTU_GetIntMskPos(subsystem, mode, 
        &all_interrupt_msk, &all_interrupt_pos);
    SET_VAL_MSK(base->INTPND, all_interrupt_msk, all_interrupt_pos, values);

    return VTU_Status_Ok;
}

enum vtu_interrupt_control VTU_GetTimerIRQ(VTU_Type *base, uint32_t timer,
    enum vtu_mode mode)
{
    uint32_t system_index; /* Индекс системы */
    uint32_t timer_index; /* Индекс таймера */
    uint32_t subsystem; /* Индекс подсистемы */
    enum vtu_interrupt_control all_interrupt_msk;
    uint32_t all_interrupt_pos;

    last_error = VTU_Status_Ok;
    /* Проверка базового адреса. */
    if (VTU_Status_Ok != VTU_TestBase(base, &system_index)) {
        last_error = VTU_Status_InvalidArgument;
        return VTU_NoInterrupt;
    }
    /* Проверка номера таймера. */
    if ((timer == 0) || (timer >
        VTU_NUMBER_OF_SUBSYSTEM*VTU_NUMBER_OF_TIMER_IN_SUBSYSTEM))
    {
        last_error = VTU_Status_InvalidArgument;
        return VTU_NoInterrupt;
    }
    if (mode == VTU_LowPower) {
        last_error = VTU_Status_BadConfigure;
        return VTU_NoInterrupt;
    }
    timer_index = timer - 1;
    subsystem = VTU_GetSubsystem(timer_index);
    /* Подготовка маски прерываний. */
    VTU_GetIntMskPos(subsystem, mode, 
        &all_interrupt_msk, &all_interrupt_pos);
    return GET_VAL_MSK(base->INTPND, all_interrupt_msk, all_interrupt_pos);
}

enum vtu_status VTU_SetPWMPolarity(VTU_Type *base, uint32_t timer,
    enum vtu_pwm_polarity value1, enum vtu_pwm_polarity value2,
    bool use_value2)
{
    uint32_t system_index; /* Индекс системы */
    uint32_t timer_index;  /* Индекс таймера */
    uint32_t subtimer;     /* Индекс таймера в подсистеме */
    uint32_t pxpol_msk, pxpol_pos;
    uint32_t *ioxctl;
    /* Проверка базового адреса. */
    if (VTU_Status_Ok != VTU_TestBase(base, &system_index))
        return VTU_Status_InvalidArgument;
    /* Проверка номера таймера. */
    if ((timer == 0) || (timer >
        VTU_NUMBER_OF_SUBSYSTEM*VTU_NUMBER_OF_TIMER_IN_SUBSYSTEM))
    {
        return VTU_Status_InvalidArgument;
    }
    timer_index = timer - 1;
    subtimer = VTU_GetSubtimer(timer_index);
    if (subtimer && use_value2) 
        return VTU_Status_BadConfigure; /* Расширенный режим для старшего субтаймера не имеет смысла. */
    VTU_GetPxpolMskPos(timer_index, use_value2, &pxpol_msk, &pxpol_pos);
    ioxctl = VTU_GetPIOXCtl(base, timer_index);
    SET_VAL_MSK(*ioxctl, pxpol_msk, pxpol_pos,
        value1 | (value2 << (VTU_IO1CTL_P2POL_Pos - VTU_IO1CTL_P1POL_Pos)));
    return VTU_Status_Ok;
}

enum vtu_status VTU_GetPWMPolarity(VTU_Type *base, uint32_t timer,
    enum vtu_pwm_polarity *value1, enum vtu_pwm_polarity *value2,
    bool use_value2)
{
    uint32_t system_index; /* Индекс системы */
    uint32_t timer_index;  /* Индекс таймера */
    uint32_t subtimer;     /* Индекс таймера в подсистеме */
    uint32_t pxpol_msk, pxpol_pos;
    uint32_t *ioxctl;
    uint32_t polarity;

    /* Проверка базового адреса. */
    if (VTU_Status_Ok != VTU_TestBase(base, &system_index))
        return VTU_Status_InvalidArgument;
    /* Проверка номера таймера. */
    if ((timer == 0) || (timer >
        VTU_NUMBER_OF_SUBSYSTEM*VTU_NUMBER_OF_TIMER_IN_SUBSYSTEM))
    {
        return VTU_Status_InvalidArgument;
    }
    if ((value1 == NULL) || ((value2 == NULL) && use_value2))
        return VTU_Status_InvalidArgument;
    timer_index = timer - 1;
    subtimer = VTU_GetSubtimer(timer_index);
    if (subtimer && use_value2) 
        return VTU_Status_BadConfigure; /* Расширенный режим для старшего субтаймера не имеет смысла. */
    VTU_GetPxpolMskPos(timer_index, use_value2, &pxpol_msk, &pxpol_pos);
    ioxctl = VTU_GetPIOXCtl(base, timer_index);
    polarity = GET_VAL_MSK(*ioxctl, pxpol_msk, pxpol_pos);
    *value1 = polarity & VTU_IO1CTL_P1POL_Msk;
    if (use_value2)
        *value2 = (polarity & VTU_IO1CTL_P2POL_Msk) >> VTU_IO1CTL_P2POL_Pos;
    return VTU_Status_Ok;
}

enum vtu_status VTU_SetCaptureEdgeCtrl(VTU_Type *base, uint32_t timer,
    enum vtu_capture_edge_control value1,
    enum vtu_capture_edge_control value2,
    bool use_value2)
{
    uint32_t system_index; /* Индекс системы */
    uint32_t timer_index;  /* Индекс таймера */
    uint32_t subtimer;     /* Индекс таймера в подсистеме */
    uint32_t cxedg_msk, cxedg_pos;
    uint32_t *ioxctl;

    /* Проверка базового адреса. */
    if (VTU_Status_Ok != VTU_TestBase(base, &system_index))
        return VTU_Status_InvalidArgument;
    /* Проверка номера таймера. */
    if ((timer == 0) || (timer >
        VTU_NUMBER_OF_SUBSYSTEM*VTU_NUMBER_OF_TIMER_IN_SUBSYSTEM))
    {
        return VTU_Status_InvalidArgument;
    }
    timer_index = timer - 1;
    subtimer = VTU_GetSubtimer(timer_index);
    if (subtimer && use_value2) 
        return VTU_Status_BadConfigure; /* Расширенный режим для старшего субтаймера не имеет смысла. */
    VTU_GetCxedgMskPos(timer_index, use_value2, &cxedg_msk, &cxedg_pos);
    ioxctl = VTU_GetPIOXCtl(base, timer_index);
    SET_VAL_MSK(*ioxctl, cxedg_msk, cxedg_pos,
        value1 | (value2 << (VTU_IO1CTL_P2POL_Pos - VTU_IO1CTL_P1POL_Pos)));
    return VTU_Status_Ok;
}

enum vtu_status VTU_GetCaptureEdgeCtrl(VTU_Type *base, uint32_t timer,
    enum vtu_capture_edge_control *value1,
    enum vtu_capture_edge_control *value2,
    bool use_value2)
{
    uint32_t system_index; /* Индекс системы */
    uint32_t timer_index;  /* Индекс таймера */
    uint32_t subtimer;     /* Индекс таймера в подсистеме */
    uint32_t cxedg_msk, cxedg_pos;
    uint32_t *ioxctl;
    uint32_t edges;

    /* Проверка базового адреса. */
    if (VTU_Status_Ok != VTU_TestBase(base, &system_index))
        return VTU_Status_InvalidArgument;
    /* Проверка номера таймера. */
    if ((timer == 0) || (timer >
        VTU_NUMBER_OF_SUBSYSTEM*VTU_NUMBER_OF_TIMER_IN_SUBSYSTEM))
    {
        return VTU_Status_InvalidArgument;
    }
    if ((value1 == NULL) || ((value2 == NULL) && use_value2))
        return VTU_Status_InvalidArgument;
    timer_index = timer - 1;
    subtimer = VTU_GetSubtimer(timer_index);
    if (subtimer && use_value2) 
        return VTU_Status_BadConfigure; /* Расширенный режим для старшего субтаймера не имеет смысла. */
    VTU_GetCxedgMskPos(timer_index, use_value2, &cxedg_msk, &cxedg_pos);
    /* Вычисление адресов регистров ioxctl для выбранного таймера. */
    ioxctl = VTU_GetPIOXCtl(base, timer_index);
    edges = GET_VAL_MSK(*ioxctl, cxedg_msk, cxedg_pos);
    *value1 = edges & VTU_IO1CTL_C1EDG_Msk;
    if (use_value2)
        *value2 = (edges & VTU_IO1CTL_C2EDG_Msk) >> VTU_IO1CTL_C2EDG_Pos;
    return VTU_Status_Ok;
}

void VTU_SetCallback(uint32_t timer, vtu_callback callback)
{
    assert(callback != NULL);
    assert(timer > 0);
    assert(timer <= VTU_NUMBER_OF_SUBSYSTEM
        * VTU_NUMBER_OF_TIMER_IN_SUBSYSTEM);
    s_user_callback[timer] = callback;
}

static void VTU_IRQHandler(VTU_Type *base, void *unused)
{
    UNUSED(unused);

    uint32_t interrupt_index = 0;
    uint32_t interrupt_enabled = base->INTCTL & base->INTPND;

    uint32_t timer = 0;
    uint32_t interrupt_bits = VTU_NUMBER_OF_SUBSYSTEM * 4;

    uint32_t mode = base->MODE;

    enum vtu_interrupt_control match;

    for (interrupt_index = 0; interrupt_index < interrupt_bits; interrupt_index++) {
        if ((interrupt_enabled >> interrupt_index) & 1) {
            for (uint32_t i = 0; i < interrupt_bits; i++) {
                if ((mode >> i) & 1) {
                    switch (mode)
                    {
                        case VTU_Timer0Mode8bit:
                        case VTU_Timer0Mode16bit:
                            timer = 1;
                            break;
                        case VTU_Timer1Mode8bit:
                            timer = 2;
                            break;
                        case VTU_Timer2Mode8bit:
                        case VTU_Timer2Mode16bit:
                            timer = 3;
                            break;
                        case VTU_Timer3Mode8bit:
                            timer = 4;
                            break;
                        default:
                            break;
                    }
                    break;
                }
            }
            if ((interrupt_index + 1) % 4 == 0) {
                match = VTU_HighBytePeriodMatch;
            } else if ((interrupt_index + 1) % 2 == 0) {
                match = VTU_LowBytePeriodMatch;
            } else if ((interrupt_index + 1) % 3 == 0) {
                match = VTU_HighByteDutyCycleMatch;
            } else {
                match = VTU_LowByteDutyCycleMatch;
            }
            if (s_user_callback[timer] != NULL) {
                (s_user_callback[timer])(base, timer, match);
            }
        }
        base->INTPND &= (1 << interrupt_index);
    }    
}

/*!
 * @}
 */
