/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @addtogroup timer_driver
 * @{
 */

/*!
 * @file hal_timer.c
 *
 * @brief Имплементация драйвера таймеров общего назначения
 */

#include "hal_timer.h"

#define TIMER_ADDRESS_SEC_MSK (0x10000000UL) /*!< Маска для доверенного/недоверенного таймера */

/*!
 * @brief Конфигурация таймера общего назначения
 */
struct timer_handle {
    uint32_t                     ini_flag; /*!< Флаг инициализированности */
    TIM_Type                        *base; /*!< Базовый адрес таймера */
    struct timer_hardware_config hard_cfg; /*!< Конфигурация аппаратной части таймера */
    enum timer_work_mode         mode;     /*!< Режим работы таймера */
    callback_t                   callback; /*!< Функция обратного вызова */
    volatile uint32_t            ticks_h;  /*!< Количество тиков */
};

/*!
 * @brief Получение экземпляра таймера общего назначения
 *
 * @note Функция #TIMER_GetAPIStatus позволяет проверить корректность
 * получения экземпляра класса. Вызываться #TIMER_GetAPIStatus должна сразу
 * после #TIMER_GetInstance.
 *
 * @param base Базовый адрес таймера
 *
 * @return Экземпляр таймера общего назначения
 */
static uint32_t TIMER_GetInstance(TIM_Type *base);

/* Базовые адреса таймеров общего назначения */
static const uint32_t timer_bases[TIMER_COUNT] = {
    TIM0_BASE, TIM1_BASE, LPTIM_BASE
};

/* Номера прерываний таймеров в NVIC */
static const IRQn_Type timer_irqn[TIMER_COUNT] = {
    TIMER0_IRQn, TIMER1_IRQn, LP_TIMER_IRQn
};

/* Дескрипторы таймеров */
static struct timer_handle timers[TIMER_COUNT] = { 0 };

/*
 * Статус выполнения последней функции, тип возвращаемого значения которой
 * отличен от #timer_status.
 */
static enum timer_status last_status = TIMER_Status_Ok;

/*!
 * @brief Общий обработчик прерываний таймеров общего назначения
 *
 * @param base Базовый адрес таймера
 */
static void TIMER_Handler(TIM_Type *base)
{
    uint32_t instance = TIMER_GetInstance(base);

    base->INTCLEAR = TIM_INTCLEAR_INTCLEAR_Msk;

    switch (timers[instance].mode) {
        case TIMER_Hardware:
            if (timers[instance].callback != NULL) {
                timers[instance].callback(base);
            }
            break;
        case TIMER_Software:
            timers[instance].ticks_h--;
            if (timers[instance].callback != NULL) {
                timers[instance].callback(base);
            }
            break;
        default:
            break;
    }
}

/*!
 * @brief Обработчик прерываний таймера TIM
 */
void TIM0_DriverHandler(void)
{
    TIMER_Handler(timers[0].base);
    __DSB();
}

/*!
 * @brief Обработчик прерываний таймера TIM1
 */
void TIM1_DriverHandler(void)
{
    TIMER_Handler(timers[1].base);
    __DSB();
}

/*!
 * @brief Обработчик прерываний таймера LPTIM
 */
void LPTIM_DriverHandler(void)
{
    TIMER_Handler(timers[2].base);
    __DSB();
}

static uint32_t TIMER_GetInstance(TIM_Type *base)
{
    uint32_t instance;
    uint32_t base_addr = ((uint32_t) base) & ~TIMER_ADDRESS_SEC_MSK;

    last_status = TIMER_Status_Ok;

    for (instance = 0; instance < TIMER_COUNT; instance++) {
        if (timer_bases[instance] == base_addr) {
            break;
        }
    }

    if (instance >= TIMER_COUNT) {
        last_status = TIMER_Status_InvalidArgument;
    }

    return instance;
}

enum timer_status TIMER_Init(TIM_Type *base,
    struct timer_hardware_config config, enum timer_work_mode mode,
    callback_t callback, uint32_t ticks_h)
{
    uint32_t temp_ctrl;
    uint32_t instance = TIMER_GetInstance(base);
    enum timer_status temp_status = TIMER_GetAPIStatus();

    if (TIMER_Status_Ok != temp_status) {
        return temp_status;
    }

    if (timers[instance].ini_flag) {
        return TIMER_Status_TimerBusy;
    }

    timers[instance].ini_flag = 1;
    timers[instance].base = base;
    timers[instance].mode = mode;
    timers[instance].callback = callback;
    timers[instance].hard_cfg.work_type = config.work_type;
    timers[instance].hard_cfg.start_enable = config.start_enable;

    switch (mode) {
        case TIMER_Hardware:
            /* Сохранение конфигурации. */
            timers[instance].hard_cfg.start_value = config.start_value;
            timers[instance].hard_cfg.reload_value = config.reload_value;
            timers[instance].hard_cfg.interrupt_enable =
                config.interrupt_enable;

            /* Настройка таймера. */
            base->VALUE = config.start_value;
            base->RELOAD = config.reload_value;
            base->INTCLEAR = TIM_INTCLEAR_INTCLEAR_Msk;
            temp_ctrl = 0;
            if (config.interrupt_enable) {
                SET_VAL_MSK(temp_ctrl, TIM_CTRL_TIMERINTEN_Msk,
                    TIM_CTRL_TIMERINTEN_Pos, 1);
                NVIC_ClearPendingIRQ(timer_irqn[instance]);
                NVIC_EnableIRQ(timer_irqn[instance]);
            }

            switch (config.work_type) {
                case TIMER_Work:
                    break;
                case TIMER_Debug:
                    SET_VAL_MSK(temp_ctrl, TIM_CTRL_SELECTEXTEN_Msk,
                        TIM_CTRL_SELECTEXTEN_Pos, 1);
                    break;
                default:
                    return TIMER_Status_InvalidArgument;
            }

            if (config.start_enable) {
                SET_VAL_MSK(temp_ctrl, TIM_CTRL_ENABLE_Msk,
                    TIM_CTRL_ENABLE_Pos, 1);
            }

            base->CTRL = temp_ctrl;
            break;
        case TIMER_Software:
            /* Сохранение конфигурации для TIMER_Software. */
            timers[instance].hard_cfg.start_value = 0xFFFFFFFF;
            timers[instance].hard_cfg.reload_value = 0xFFFFFFFF;
            timers[instance].hard_cfg.interrupt_enable = 1;
            timers[instance].ticks_h = ticks_h;

            /* Настройка таймера. */
            base->VALUE = 0xFFFFFFFF;
            base->RELOAD = 0xFFFFFFFF;
            base->INTCLEAR = TIM_INTCLEAR_INTCLEAR_Msk;
            temp_ctrl = TIM_CTRL_TIMERINTEN_Msk;
            NVIC_ClearPendingIRQ(timer_irqn[instance]);
            NVIC_EnableIRQ(timer_irqn[instance]);
            switch (config.work_type) {
                case TIMER_Work:
                    break;
                case TIMER_Debug:
                    SET_VAL_MSK(temp_ctrl, TIM_CTRL_SELECTEXTEN_Msk,
                        TIM_CTRL_SELECTEXTEN_Pos, 1);
                    break;
                default:
                    return TIMER_Status_InvalidArgument;
            }

            if (config.start_enable) {
                SET_VAL_MSK(temp_ctrl, TIM_CTRL_ENABLE_Msk,
                    TIM_CTRL_ENABLE_Pos, 1);
            }

            base->CTRL = temp_ctrl;
            break;
        default:
            return TIMER_Status_InvalidArgument;
    }

    return TIMER_Status_Ok;
}

enum timer_status TIMER_Deinit(TIM_Type *base)
{
    uint32_t instance = TIMER_GetInstance(base);
    enum timer_status temp_status = TIMER_GetAPIStatus();

    if (TIMER_Status_Ok != temp_status) {
        return temp_status;
    }

    if (GET_VAL_MSK(base->CTRL, TIM_CTRL_ENABLE_Msk, TIM_CTRL_ENABLE_Pos)) {
        SET_VAL_MSK(base->CTRL, TIM_CTRL_ENABLE_Msk, TIM_CTRL_ENABLE_Pos, 0);
    }

    timers[instance].ini_flag = 0;

    return TIMER_Status_Ok;
}

enum timer_status TIMER_Run(TIM_Type *base)
{
    uint32_t instance = TIMER_GetInstance(base);
    enum timer_status temp_status = TIMER_GetAPIStatus();

    if (TIMER_Status_Ok != temp_status) {
        return temp_status;
    }

    if (!(timers[instance].ini_flag)) {
        return TIMER_Status_NotIni;
    }

    SET_VAL_MSK(base->CTRL, TIM_CTRL_ENABLE_Msk, TIM_CTRL_ENABLE_Pos, 1);

    return TIMER_Status_Ok;
}

enum timer_status TIMER_Stop(TIM_Type *base)
{
    uint32_t instance = TIMER_GetInstance(base);
    enum timer_status temp_status = TIMER_GetAPIStatus();

    if (TIMER_Status_Ok != temp_status) {
        return temp_status;
    }

    if (!(timers[instance].ini_flag)) {
        return TIMER_Status_NotIni;
    }

    SET_VAL_MSK(base->CTRL, TIM_CTRL_ENABLE_Msk, TIM_CTRL_ENABLE_Pos, 0);

    return TIMER_Status_Ok;
}

enum timer_status TIMER_Reset(TIM_Type *base)
{
    uint32_t instance = TIMER_GetInstance(base);
    enum timer_status temp_status = TIMER_GetAPIStatus();

    UNUSED(instance);

    if (TIMER_Status_Ok != temp_status) {
        return temp_status;
    }

    return TIMER_Status_NotSupport;
}

uint64_t TIMER_GetTicks(TIM_Type *base)
{
    uint64_t tick_h;
    uint64_t tick_l;
    uint32_t instance;
    enum timer_status temp_status;

    last_status = TIMER_Status_Ok;

    instance = TIMER_GetInstance(base);
    temp_status = TIMER_GetAPIStatus();

    if (TIMER_Status_Ok != temp_status) {
        last_status = temp_status;
        return 0;
    }

    if (!(timers[instance].ini_flag)) {
        last_status = TIMER_Status_NotIni;
        return 0;
    }

    switch (timers[instance].mode) {
        case TIMER_Hardware:
            return (uint64_t) (TIMER_HARDWARE_FIELD_MAX - base->VALUE);
        case TIMER_Software:
            do {
                tick_h = (uint64_t) (timers[instance].ticks_h);
                tick_l = base->VALUE;
            } while (tick_h != (uint64_t) (timers[instance].ticks_h));
            return (TIMER_SOFTWARE_FIELD_MAX -
                ((tick_h << TIMER_SOFTWARE_FIELD_HIGH_OFFSET) | tick_l));
        default:
            last_status = TIMER_Status_InvalidArgument;
            return 0;
    }
}

enum timer_status TIMER_SetTick(TIM_Type *base, uint64_t ticks)
{
    UNUSED(base);
    UNUSED(ticks);

    return TIMER_Status_NotSupport;
}

enum timer_status TIMER_GetAPIStatus()
{
    return last_status;
}

enum timer_status TIMER_SetConfig(TIM_Type *base,
    struct timer_hardware_config config, enum timer_work_mode mode,
    callback_t callback, uint32_t ticks_h)
{
    uint32_t instance = TIMER_GetInstance(base);
    enum timer_status temp_status = TIMER_GetAPIStatus();

    if (TIMER_Status_Ok != temp_status) {
        return temp_status;
    }

    timers[instance].ini_flag = 1;
    timers[instance].base = base;
    timers[instance].mode = mode;
    timers[instance].callback = callback;
    timers[instance].hard_cfg.work_type = config.work_type;
    timers[instance].hard_cfg.start_enable = config.start_enable;

    switch (mode) {
        case TIMER_Hardware:
            /* Сохранение конфигурации. */
            timers[instance].hard_cfg.start_value = config.start_value;
            timers[instance].hard_cfg.reload_value = config.reload_value;
            timers[instance].hard_cfg.interrupt_enable =
                config.interrupt_enable;

            if (config.interrupt_enable) {
                NVIC_ClearPendingIRQ(timer_irqn[instance]);
                NVIC_EnableIRQ(timer_irqn[instance]);
            }

            switch (config.work_type) {
                case TIMER_Work:
                    break;
                case TIMER_Debug:
                    break;
                default:
                    return TIMER_Status_InvalidArgument;
            }
            break;
        case TIMER_Software:
            /* Сохранение конфигурации для TIMER_Software. */
            timers[instance].hard_cfg.start_value = 0xFFFFFFFF;
            timers[instance].hard_cfg.reload_value = 0xFFFFFFFF;
            timers[instance].hard_cfg.interrupt_enable = 1;
            timers[instance].ticks_h = ticks_h;

            NVIC_ClearPendingIRQ(timer_irqn[instance]);
            NVIC_EnableIRQ(timer_irqn[instance]);
            switch (config.work_type) {
                case TIMER_Work:
                    break;
                case TIMER_Debug:
                    break;
                default:
                    return TIMER_Status_InvalidArgument;
            }
            break;
        default:
            return TIMER_Status_InvalidArgument;
    }

    return TIMER_Status_Ok;
}

enum timer_status TIMER_IRQEnable(TIM_Type *base)
{
    uint32_t instance = TIMER_GetInstance(base);
    enum timer_status temp_status = TIMER_GetAPIStatus();

    UNUSED(instance);

    if (TIMER_Status_Ok != temp_status) {
        return temp_status;
    }

    SET_VAL_MSK(base->CTRL, TIM_CTRL_TIMERINTEN_Msk, TIM_CTRL_TIMERINTEN_Pos,
        1);

    return TIMER_Status_Ok;
}

enum timer_status TIMER_IRQDisable(TIM_Type *base)
{
    uint32_t instance = TIMER_GetInstance(base);
    enum timer_status temp_status = TIMER_GetAPIStatus();

    UNUSED(instance);

    if (TIMER_Status_Ok != temp_status) {
        return temp_status;
    }

    SET_VAL_MSK(base->CTRL, TIM_CTRL_TIMERINTEN_Msk, TIM_CTRL_TIMERINTEN_Pos,
        0);

    return TIMER_Status_Ok;
}

uint32_t TIMER_IRQGetStatus(TIM_Type *base)
{
    uint32_t instance = TIMER_GetInstance(base);
    enum timer_status temp_status = TIMER_GetAPIStatus();

    UNUSED(instance);

    last_status = TIMER_Status_Ok;

    if (TIMER_Status_Ok != temp_status) {
        last_status = temp_status;
        return 0;
    }

    return GET_VAL_MSK(base->INTSTATUS, TIM_INTSTATUS_INTSTATUS_Msk,
        TIM_INTSTATUS_INTSTATUS_Pos);
}

enum timer_status TIMER_IRQClear(TIM_Type *base)
{
    uint32_t instance = TIMER_GetInstance(base);
    enum timer_status temp_status = TIMER_GetAPIStatus();

    UNUSED(instance);

    if (TIMER_Status_Ok != temp_status) {
        return temp_status;
    }

    SET_VAL_MSK(base->INTCLEAR, TIM_INTCLEAR_INTCLEAR_Msk,
        TIM_INTCLEAR_INTCLEAR_Pos, 1);

    return TIMER_Status_Ok;
}

/*!
 * @}
 */
