/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @addtogroup spi_driver
 * @{
 */

/*!
 * @file hal_spi_dma.h
 *
 * @brief Дополнение драйвера SPI с пересылкой данных через DMA
 */

#ifndef HAL_SPI_DMA_H
#define HAL_SPI_DMA_H

#include "hal_dma.h"
#include "hal_spi.h"

/*! @brief Версия драйвера */
#define HAL_SPI_DMA_DRIVER_VERSION (MAKE_VERSION(1, 0, 0))

/*!
 * @brief Дескриптор SPI-DMA
 */
typedef struct _spi_dma_handle spi_dma_handle_t;

/*! @brief Функция обратного вызова */
typedef void (*spi_dma_callback_t)(SPI_Type *base, spi_dma_handle_t *handle,
    void *user_data, dma_irq_t inttype);

/*!
 * @brief Дескриптор SPI-DMA
 */
struct _spi_dma_handle {
    volatile bool tx_in_progress; /*!< Статус передачи DMA: 1 - в процессе, 0 - завершена */
    volatile bool rx_in_progress; /*!< Статус приема DMA: 1 - в процессе, 0 - завершена */
    dma_handle_t *tx_handle;      /*!< Контекст драйвера DMA для отправки */
    dma_descriptor_t *tx_desc;    /*!< Указатель на дескриптор многоблочной передачи для отправки */
    dma_handle_t *rx_handle;      /*!< Контекст драйвера DMA для приема */
    dma_descriptor_t *rx_desc;    /*!< Указатель на дескриптор многоблочной передачи для приема */
    uint8_t bytes_per_frame;      /*!< Количество байт во фрейме SPI */
    spi_dma_callback_t callback;  /*!< Функция обратного вызова */
    void *user_data;              /*!< Пользовательские данные */
    enum spi_trans_status state;  /*!< Текущее состояние SPI DMA передачи */
    size_t transfer_size;         /*!< Количество байт для передачи */
    void *dummy_data;             /*!< Указатель на фиктивные данные */
};

#if defined(__cplusplus)
extern "C" {
#endif

/*!
 * @brief Функция инициализации дескриптора SPI-DMA
 *
 * @param base      Базовый адрес SPI master
 * @param handle    Дескриптор SPI-DMA
 * @param callback  Функция обратного вызова
 * @param user_data Пользовательские данные
 * @param tx_handle DMA дескриптор отправки
 * @param rx_handle DMA дескриптор приема
 *
 * @retval #SPI_DMA_Status_InvalidArgument
 * @retval #SPI_DMA_Status_Success
 */
enum spi_status SPI_MasterTransferCreateHandleDMA(SPI_Type *base,
    spi_dma_callback_t callback, void *user_data, spi_dma_handle_t *handle,
    dma_handle_t *tx_handle, dma_handle_t *rx_handle);

/*!
 * @brief Функция для SPI master приема/передачи данных в полнодуплексном режиме
 *        через DMA каналы в порт SPI
 *
 * @param base   Базовый адрес SPI master
 * @param handle Дескриптор SPI-DMA
 * @param xfer   Параметры передачи SPI
 *
 * @retval #SPI_DMA_Status_InvalidArgument
 * @retval #SPI_DMA_Status_Busy
 * @retval #SPI_DMA_Status_SPINotExist
 * @retval #SPI_DMA_Status_Success
 */
enum spi_status SPI_MasterTransferDMA(SPI_Type *base,
    spi_dma_handle_t *handle, spi_transfer_t *xfer);

/*!
 * @brief Функция приема/передачи данных в полудуплексном режиме через DMA
 *        каналы в порт SPI
 *
 * @param base   Базовый адрес SPI master
 * @param handle Дескриптор передачи
 * @param xfer   Параметры полудуплексной передачи
 *
 * @retval #SPI_DMA_Status_InvalidArgument
 * @retval #SPI_DMA_Status_Busy
 * @retval #SPI_DMA_Status_SPINotExist
 * @retval #SPI_DMA_Status_Fail
 * @retval #SPI_DMA_Status_Success
 */
enum spi_status SPI_MasterHalfDuplexTransferDMA(SPI_Type *base,
    spi_dma_handle_t *handle, spi_half_duplex_transfer_t *xfer);

/*!
 * @brief Функция инициализации дескриптора SPI-DMA
 *
 * @param base      Базовый адрес SPI slave
 * @param handle    Дескриптор SPI-DMA
 * @param callback  Функция обратного вызова
 * @param user_data Пользовательские данные
 * @param tx_handle DMA дескриптор отправки
 * @param rx_handle DMA дескриптор приема
 *
 * @retval #SPI_DMA_Status_InvalidArgument
 * @retval #SPI_DMA_Status_Success
 */
static inline enum spi_status SPI_SlaveTransferCreateHandleDMA(
    SPI_Type *base, spi_dma_callback_t callback, void *user_data,
    spi_dma_handle_t *handle, dma_handle_t *tx_handle, dma_handle_t *rx_handle)
{
    return SPI_MasterTransferCreateHandleDMA(base, callback, user_data,
            handle, tx_handle, rx_handle);
}

/*!
 * @brief Функция для SPI slave приема/передачи данных в полнодуплексном режиме
 *        через DMA каналы в порт SPI
 *
 * @param base   Базовый адрес SPI slave
 * @param handle Дескриптор SPI-DMA
 * @param xfer   Параметры передачи SPI
 *
 * @retval #SPI_DMA_Status_InvalidArgument
 * @retval #SPI_DMA_Status_Busy
 * @retval #SPI_DMA_Status_SPINotExist
 * @retval #SPI_DMA_Status_Success
 */
static inline enum spi_status SPI_SlaveTransferDMA(SPI_Type *base,
    spi_dma_handle_t *handle, spi_transfer_t *xfer)
{
    return SPI_MasterTransferDMA(base, handle, xfer);
}

/*!
 * @brief Прекращение передачи SPI
 *
 * @param handle Дескриптор SPI-DMA
 */
void SPI_MasterTransferAbortDMA(spi_dma_handle_t *handle);

/*!
 * @brief Прекращение передачи SPI
 *
 * @param handle Дескриптор SPI-DMA
 */
static inline void SPI_SlaveTransferAbortDMA(spi_dma_handle_t *handle)
{
    SPI_MasterTransferAbortDMA(handle);
}

/*!
 * @brief Инициализация дескрипторов DMA для многоблочной передачи TX
 *
 * @param base       Базовый адрес SPI
 * @param desc       Указатель на массив дескрипторов
 * @param count      Количество DMA дескрипторов
 * @param data_size  Общее число передаваемых данных
 * @param data_width Ширина одного передаваемого слова
 * @param src_addr   Адрес Источника
 */
static inline void SPI_DMADescriptorInitTX(SPI_Type *base,
    dma_descriptor_t *desc, uint32_t count, uint32_t data_size,
    uint8_t src_incr, uint32_t data_width, void *src_addr)
{
    dma_multiblock_config_t config = {
        .count          = count,
        .data_size      = data_size,
        .transfer_type  = DMA_MemoryToPeripheral_DMA,
        .scatter_en     = false,
        .gather_en      = false,
        .src_burst_size = DMA_BurstSize32,
        .dst_burst_size = DMA_BurstSize1,
        .src_incr       = src_incr,
        .dst_incr       = DMA_NoChange,
        .src_data_width = data_width,
        .dst_data_width = data_width,
        .src_addr       = src_addr,
        .dst_addr       = (void *) &base->DR0,
        .int_en         = true
    };

    DMA_InitMultiblockDescriptor(desc, &config);
}

/*!
 * @brief Инициализация дескрипторов DMA для многоблочной передачи RX
 *
 * @param base       Базовый адрес SPI
 * @param desc       Указатель на массив дескрипторов
 * @param count      Количество DMA дескрипторов
 * @param data_size  Общее число передаваемых данных
 * @param data_width Ширина 1ого передаваемого слова
 * @param dst_addr   Адрес Приемника
 */
static inline void SPI_DMADescriptorInitRX(SPI_Type *base,
    dma_descriptor_t *desc, uint32_t count, uint32_t data_size,
    uint8_t dst_incr, uint32_t data_width, void *dst_addr)
{
    dma_multiblock_config_t config = {
        .count          = count,
        .data_size      = data_size,
        .transfer_type  = DMA_PeripheralToMemory_DMA,
        .scatter_en     = false,
        .gather_en      = false,
        .src_burst_size = DMA_BurstSize1,
        .dst_burst_size = DMA_BurstSize32,
        .src_incr       = DMA_NoChange,
        .dst_incr       = dst_incr,
        .src_data_width = data_width,
        .dst_data_width = data_width,
        .src_addr       = (void *) &base->DR0,
        .dst_addr       = dst_addr,
        .int_en         = true
    };

    DMA_InitMultiblockDescriptor(desc, &config);
}

#if defined(__cplusplus)
}
#endif

#endif /* HAL_SPI_DMA_H */

/*!
 * @}
 */
