/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @defgroup smc_driver Драйвер модуля SMC
 *
 * @brief Драйвер внешней статической памяти
 *
 * Драйвер модуля внешней статической памяти управляет внешней статической памяти .
 */

/*!
 * @addtogroup smc_driver
 * @{
 */

/*!
 * @file hal_smc.h
 *
 * @brief Интерфейс драйвера внешней статической памяти
 */

#ifndef HAL_SMC_H
#define HAL_SMC_H

#ifdef __cplusplus
extern "C" {
#endif

#include "hal_common.h"

/*!
 * @brief Статусы драйвера SMC
 */
enum smc_status {
    SMC_Status_Ok              = 0, /*!< Нет ошибок */
    SMC_Status_InvalidArgument = 1, /*!< Недопустимый аргумент */
};

/*!
 * @brief Тип конфигурационной команды
 */
enum smc_cmd_type {
    SMC_CmdTypeUpdateRegsAndAHBCommand = 0, /*!< UpdateRegs + AHB command */
    SMC_CmdTypeModeReg                 = 1, /*!< ModeReg */
    SMC_CmdTypeUpdateRegs              = 2, /*!< UpdateRegs */
    SMC_CmdTypeModeRegAndUpdateRegs    = 3, /*!< ModeReg + UpdateRegs */
};

/*!
 * @brief Значение выхода CRE при выполнении команды ModeReg
 */
enum smc_cre {
    SMC_CRE0 = 0, /*!< Выход CRE сброшен */
    SMC_CRE1 = 1, /*!< Выход CRE установлен */
};

/*!
 * @brief Граница пакета памяти
 */
enum smc_burst_align {
    SMC_BurstAlignNo  = 0, /*!< Нет границы */
    SMC_BurstAlign32  = 0, /*!< 32 слова по 16 бит */
    SMC_BurstAlign64  = 0, /*!< 64 слова по 16 бит */
    SMC_BurstAlign128 = 0, /*!< 128 слов по 16 бит */
    SMC_BurstAlign256 = 0, /*!< 256 слов по 16 бит */
};

/*!
 * @brief Поведение выводов SMC_NBLS
 */
enum smc_bls {
    SMC_BlsAsNcs = 0, /*!< Выводы SMC_NBLS переключаются так же, как выводы SMC_NCS. */
    SMC_BlsAsNwe = 1, /*!< Выводы SMC_NBLS переключаются так же, как вывод SMC_NWE. */
};

/*!
 * @brief Использование сигнала NADV
 */
enum smc_adv {
    SMC_AdvNotUsed = 0,              /*!< Сигнал не используется. */
    SMC_AdvUsed    = 1,              /*!< Сигнал используется, шины адреса и данных мультиплексируются. */
    SMC_AdvLcd     = SMC_AdvNotUsed, /*!< При работе с LCD дисплеем ADV должен быть сброшен. */
    SMC_AdvMemory  = SMC_AdvUsed,    /*!< ADV должен быть установлен при работе с микросхемами памяти. */
};

/*!
 * @brief Длина пакета данных в 16-битных словах при записи или чтении
 */
enum smc_packet_lenght {
    SMC_PacketLenght1       = 0, /*!< 1 слово */
    SMC_PacketLenght4       = 1, /*!< 4 слово */
    SMC_PacketLenght8       = 2, /*!< 8 слово */
    SMC_PacketLenghtEndless = 5, /*!< Непрерывный пакет */
};

/*!
 * @brief Тип интерфейса при записи или чтении
 */
enum smc_rd_wr_type {
    SMC_RdWrAsync = 0, /*!< Асинхронный */
    SMC_RdWrSync  = 1, /*!< Синхронный */
};

/*!
 * @brief Разрядность интерфейса памяти
 */
enum smc_bit_depth {
    SMC_BitDepth8  = 0, /*!< 8 бит - не поддержано */
    SMC_BitDepth16 = 1, /*!< 16 бит */
    SMC_BitDepth24 = 2, /*!< 24 бит - не поддержано */
    SMC_BitDepth32 = 3, /*!< 32 бит - не поддержано */
};

/*!
 * @brief Управление преобразованием AHB-пакетов типа INCR в пакеты типа INCR4
 */
enum smc_incr_to_incr4 {
    SMC_IncrToIncr4Enable  = 0, /*!< Пакеты типа INCR преобразуются в пакеты типа INCR4. */
    SMC_IncrToIncr4Disable = 1, /*!< Пакеты типа INCR обрабатываются как одиночные обращения типа SINGLE. */
};

/*!
 * @brief Деактивация не выполняется
 */
#define SMC_NO_DEACTIVATION 0

/*!
 * @name Интерфейс драйвера
 * @{
 */

/*!
 * @brief Включение энергосберегающего режима
 *
 * Инициирует запрос на переход контроллера в энергосберегающее состояние.
 * Не дожидается завершения перехода в энергосберегающее состояние.
 *
 * @param  base Адрес блока SMC
 *
 * @retval #SMC_Status_Ok
 * @retval #SMC_Status_InvalidArgument
 */
enum smc_status SMC_PowerSaveOn(SMC_Type *base);

/*!
 * @brief Выключение энергосберегающего режима
 *
 * Инициирует запрос на переход контроллера из энергосберегающего состояния.
 * Не дожидается завершения перехода из энергосберегающего состояния.
 *
 * @param base Адрес блока SMC
 *
 * @retval #SMC_Status_Ok
 * @retval #SMC_Status_InvalidArgument
 */
enum smc_status SMC_PowerSaveOff(SMC_Type *base);

/*!
 * @brief Отправка конфигурационных команд
 *
 * Предназначена для отправки конфигурационных команд во
 * внешнюю память и для управления обновлением конфигурационных регистров
 * контроллера значениями из регистров SMC_SET_OPMODE и SMC_SET_CYCLES
 *
 * @param base        Адрес блока SMC
 * @param chip_select Банк памяти. Может быть 0 или 1.
 * @param cmd_type    Тип конфигурационной команды.
 * @param set_cre     При выполнении команды ModeReg задает значение
 *                    выхода SMC_CRE
 * @param addr        При выполнении команды ModeReg поле используется
 *                    в качестве разрядов [19:0] адреса внешней памяти.
 *                    При выполнении команды UpdateRegs+AHB command
 *                    поле ADDR[15:0] используется для сопоставления
 *                    со значением на шине SMC_DA[15:0] для
 *                    синхронного обновления конфигурации.
 *
 * @retval #SMC_Status_Ok
 * @retval #SMC_Status_InvalidArgument
 */
enum smc_status SMC_DirectCmd(SMC_Type *base, uint32_t chip_select,
    enum smc_cmd_type cmd_type, enum smc_cre set_cre, uint32_t addr);

/*!
 * @brief Хранение новой конфигурации временных параметров
 * интерфейса.
 *
 * Предназначена для хранения новой конфигурации временных параметров
 * интерфейса в регистре SMC_SET_CYCLES. Значение этого регистра переписывается
 * в регистр SMC_CYCLESx выбранного банка при выполнении команды UpdateRegs
 * ( @ref SMC_DirectCmd )
 *
 * @param base  Адрес блока SMC
 * @param ttr   Задержка между последовательными пакетами (turnaround)
 * @param tpc   Длительность цикла доступа к странице
 * @param twp   Задержка активации вывода CMS_NWE
 * @param tceoe Задержка активации вывода SMC_NOE
 * @param twc   Длительность цикла записи
 * @param trc   Длительность цикла чтения
 *
 * @retval #SMC_Status_Ok
 * @retval #SMC_Status_InvalidArgument
 */
enum smc_status SMC_SetCycles(SMC_Type *base, uint32_t ttr, uint32_t tpc,
    uint32_t twp, uint32_t tceoe, uint32_t twc, uint32_t trc);

/*!
 * @brief Установка регистра SET_OPMODE
 *
 * Устанавливает все поля регистра SET_OPMODE. Если хотя бы одно поле имеет
 * недопустимое значение, то записи в регистр не происходит.
 *
 * @note  Используется перед началом работы с блоком
 *
 * @param base      Адрес блока SMC
 * @param align     Выравнивание
 * @param bls       Бит задает поведение выводов SMC_NBLS[1:0]
 * @param adv       Задает использование сигнала SMC_NADV
 * @param wr_lenght Задает длину пакета данных при записи
 * @param wr_sync   Задает тип интерфейса при записи
 * @param rd_lenght Задает длину пакета данных при чтении
 * @param rd_sync   Задает тип интерфейса при чтении
 * @param depth     Задает разрядность интерфейса памяти
 *
 * @retval #SMC_Status_Ok
 * @retval #SMC_Status_InvalidArgument
 */
enum smc_status SMC_SetOpmode(SMC_Type *base, enum smc_burst_align align,
    enum smc_bls bls, enum smc_adv adv, enum smc_packet_lenght wr_lenght,
    enum smc_rd_wr_type wr_sync, enum smc_packet_lenght rd_lenght,
    enum smc_rd_wr_type rd_sync, enum smc_bit_depth depth);

/*!
 * @brief Управление деактивацией микросхемы
 *
 * Задает количество последовательных пакетов перед деактивацией микросхемы
 * памяти для возможности ее обновления
 *
 * @note  Используется перед началом работы с блоком
 *
 * @param base   Адрес блока SMC
 * @param period Количество пакетов перед деактивацией. Допустимые значения:
 *               SMC_NO_DEACTIVATION - нет деактивации,
 *               от 1 до 15 - деактивация после 1, .. 15 пакета
 *
 * @retval #SMC_Status_Ok
 * @retval #SMC_Status_InvalidArgument
 */
enum smc_status SMC_RefreshPeriod(SMC_Type *base, uint32_t period);

/*!
 * @brief Запись в регистр USER_CONFIG
 *
 * Управляет преобразованием AHB-пакетов типа INCR в пакеты типа INCR4.
 * Задает коэффициент деления для формирования тактового сигнала.
 *
 * @note  Используется перед началом работы с блоком
 *
 * @param base      Адрес блока SMC
 * @param bank0     Разрешение преобразования AHB-пакетов типа INCR в пакеты
 *                  типа INCR4 для банка памяти 1.
 * @param bank1     Разрешение преобразования AHB-пакетов типа INCR в пакеты
 *                  типа INCR4 для банка памяти 0.
 * @param smcclkdiv Коэффициент деления SMCCLKDIV. Допустимые значения от 1 до 32
 *
 * @retval #SMC_Status_Ok
 * @retval #SMC_Status_InvalidArgument
 */
enum smc_status SMC_UserConfig(SMC_Type *base, enum smc_incr_to_incr4 bank0,
    enum smc_incr_to_incr4 bank1, uint32_t smcclkdiv);

/*!
 * @brief Проверка завершения установки конфигурации
 *
 * @param base        Адрес блока SMC
 * @param chip_select Банк памяти. Может быть 0 или 1.
 * @param cycles      Временные параметры
 * @param mode        Режима работы
 *
 * @retval true  Конфигурация установлена
 * @retval false Конфигурация не установлена
 *
 * @note При некорректном base возвращается false
 */
bool SMC_CheckConfigure(SMC_Type *base, uint32_t chip_select,
    uint32_t cycles, uint32_t mode);

/*!
 * @}
 */

#ifdef __cplusplus
}
#endif

#endif /* HAL_SMC_H */

/*!
 * @}
 */
