/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @addtogroup smc_driver
 * @{
 */

/*!
 * @file hal_smc.c
 *
 * @brief Имплементация драйвера внешней статической памяти
 */

#include "hal_smc.h"

enum smc_status SMC_PowerSaveOn(SMC_Type* base)
{
    if (base == NULL)
        return SMC_Status_InvalidArgument;
    SMC_MEMC_CFG_SET_LPREQ_PTYPE_SET(base, 1);
    return SMC_Status_Ok;
}

enum smc_status SMC_PowerSaveOff(SMC_Type* base)
{
    if (base == NULL)
        return SMC_Status_InvalidArgument;
    SMC_MEMC_CFG_CLR_LPEXIT_PTYPE_SET(base, 1);
    return SMC_Status_Ok;
}

enum smc_status SMC_DirectCmd(SMC_Type* base, uint32_t chip_select,
    enum smc_cmd_type cmd_type, enum smc_cre set_cre, uint32_t addr)
{
    if (base == NULL)
        return SMC_Status_InvalidArgument;
    if ((chip_select > 1)
        || (cmd_type > 4)
        || (set_cre > 1)
        || (addr > 0xFFFFF))
    {
        return SMC_Status_InvalidArgument;
    }

    uint32_t reg;
    reg = (chip_select << SMC_DIRECT_CMD_CHIP_SELECT_Pos) |
          (cmd_type << SMC_DIRECT_CMD_CMD_TYPE_Pos) |
          (set_cre << SMC_DIRECT_CMD_SET_CRE_Pos) |
          (addr << SMC_DIRECT_CMD_ADDR_Pos);

    base->DIRECT_CMD = reg;
    return SMC_Status_Ok;
}

enum smc_status SMC_SetCycles(SMC_Type* base, uint32_t ttr, uint32_t tpc,
    uint32_t twp, uint32_t tceoe, uint32_t twc, uint32_t trc)
{
    if (base == NULL)
        return SMC_Status_InvalidArgument + 0;
    if (ttr > (SMC_SET_CYCLES_TTR_Msk >> SMC_SET_CYCLES_TTR_Pos))
        return SMC_Status_InvalidArgument + 1;
    if (tpc > (SMC_SET_CYCLES_TPC_Msk >> SMC_SET_CYCLES_TPC_Pos))
        return SMC_Status_InvalidArgument + 2;
    if (twp > (SMC_SET_CYCLES_TWP_Msk >> SMC_SET_CYCLES_TWP_Pos))
        return SMC_Status_InvalidArgument + 3;
    if (tceoe > (SMC_SET_CYCLES_TCEOE_Msk >> SMC_SET_CYCLES_TCEOE_Pos))
        return SMC_Status_InvalidArgument + 4;
    if (twc > (SMC_SET_CYCLES_TWC_Msk >> SMC_SET_CYCLES_TWC_Pos))
        return SMC_Status_InvalidArgument + 5;
    if (trc > (SMC_SET_CYCLES_TRC_Msk >> SMC_SET_CYCLES_TRC_Pos))
        return SMC_Status_InvalidArgument + 6;

    uint32_t reg;
    reg = (ttr << SMC_SET_CYCLES_TTR_Pos)
        | (tpc << SMC_SET_CYCLES_TPC_Pos)
        | (twp << SMC_SET_CYCLES_TWP_Pos)
        | (tceoe << SMC_SET_CYCLES_TCEOE_Pos)
        | (twc << SMC_SET_CYCLES_TWC_Pos)
        | (trc << SMC_SET_CYCLES_TRC_Pos);

    base->SET_CYCLES = reg;
    return SMC_Status_Ok;
}

enum smc_status SMC_SetOpmode(SMC_Type* base, enum smc_burst_align align,
    enum smc_bls bls, enum smc_adv adv, enum smc_packet_lenght wr_lenght,
    enum smc_rd_wr_type wr_sync, enum smc_packet_lenght rd_lenght,
    enum smc_rd_wr_type rd_sync, enum smc_bit_depth depth)
{
    if (base == NULL)
        return SMC_Status_InvalidArgument;
    if ((align > 4)
        || (bls > 1)
        || (adv > 1)
        || ((wr_lenght > 2) && (wr_lenght != 5))
        || (wr_sync > 1)
        || ((rd_lenght > 2) && (rd_lenght != 5))
        || (rd_sync > 1)
        || (depth != 1))
        return SMC_Status_InvalidArgument;

    uint32_t reg;
    reg = (align << SMC_SET_OPMODE_SET_BURST_ALIGN_Pos)
        | (bls << SMC_SET_OPMODE_SET_BLS_Pos)
        | (adv << SMC_SET_OPMODE_SET_ADV_Pos)
        | (wr_lenght << SMC_SET_OPMODE_SET_WR_BL_Pos)
        | (wr_sync << SMC_SET_OPMODE_SET_RW_SYNC_Pos)
        | (rd_lenght << SMC_SET_OPMODE_SET_RD_BL_Pos)
        | (rd_sync << SMC_SET_OPMODE_SET_RD_SYNC_Pos)
        | (depth << SMC_SET_OPMODE_SET_MW_Pos);

    base->SET_OPMODE = reg;
    return SMC_Status_Ok;
}

enum smc_status SMC_RefreshPeriod(SMC_Type* base, uint32_t period)
{
    if (base == NULL)
        return SMC_Status_InvalidArgument;
    if (period > 15)
        return SMC_Status_InvalidArgument;

    base->REFRESH_PERIOD = period << SMC_REFRESH_PERIOD_PERIOD_Pos;

    return SMC_Status_Ok;
}

enum smc_status SMC_UserConfig(SMC_Type* base, enum smc_incr_to_incr4 bank0,
    enum smc_incr_to_incr4 bank1, uint32_t smcclkdiv)
{
    uint32_t reg;

    if (base == NULL)
        return SMC_Status_InvalidArgument;
    if ((bank1 > 1) || (bank0 > 1))
        return SMC_Status_InvalidArgument;
    if ((smcclkdiv > 32) || (smcclkdiv == 0))
        return SMC_Status_InvalidArgument;

    reg = (bank1 << SMC_USER_CONFIG_CONV_INCR_DIS1_Pos)
        | (bank0 << SMC_USER_CONFIG_CONV_INCR_DIS0_Pos)
        | ((smcclkdiv - 1) << SMC_USER_CONFIG_SMCCLKDIV_Pos);
    base->USER_CONFIG = reg;
    // while ((base->USER_STATUS & SMC_USER_CONFIG_SMCCLKDIV_Msk)
    //     !=  (reg & SMC_USER_CONFIG_SMCCLKDIV_Msk));

    return SMC_Status_Ok;
}

bool SMC_CheckConfigure(SMC_Type* base, uint32_t chip,
    uint32_t cycles, uint32_t mode)
{
    if (base == NULL)
        return false;
    if (chip)
        return !((cycles != base->CYCLES1) || (mode != (base->OPMODE1 & 0xFFFF)));
    else
        return !((cycles != base->CYCLES0) || (mode != (base->OPMODE0 & 0xFFFF)));
}

/*!
 * @}
 */
