/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @addtogroup qspi_driver
 * @{
 */

/*!
 * @file hal_qspi.c
 *
 * @brief Имплементация драйвера модуля QSPI
 */

#include "hal_qspi.h"

void QSPI_GetDefaultConfig(qspi_config_t *config)
{
    assert(config != NULL);

    (void)memset(config, 0, sizeof(qspi_config_t));

    config->delay_en          = 0x0;
    config->cpol              = 0x0;
    config->cpha              = 0x0;
    config->msb               = 0x1;
    config->cont_trans_en     = 0x1;
    config->cont_transfer_ext = 0x0;
    config->spi_mode          = QSPI_NormalSPI;
    config->slave_select      = 0x1;
    config->slave_pol         = 0x0;
    config->bit_size          = QSPI_FRAME_BITS_8;
    config->mode              = 0x1;
    config->dma_en            = 0x0;
    config->inhibit_din       = 0x0;
    config->inhibit_dout      = 0x0;
#if defined(QSPI_DRIVER_VERSION_1_6_0)
    config->xfer_format       = QSPI_FORMAT_MOTOROLA;
#endif    
}

void QSPI_Init(QSPI_Type *base, const qspi_config_t *config)
{
    assert(config != NULL);
    assert(base != NULL);

    /* Отключение контроллера QSPI. */
    base->ENABLE = 0x0;
    /* Сброс всех прерываний. */
    base->INTR_CLR = QSPI_INTR_CLR_XFERDONEPULSE_Msk
        | QSPI_INTR_CLR_RXFULLPULSE_Msk
        | QSPI_INTR_CLR_RXWMARKPULSE_Msk
        | QSPI_INTR_CLR_TXWMARKPULSE_Msk
        | QSPI_INTR_CLR_TXEMPTYPULSE_Msk;
    /* Отключение всех прерываний. */
    base->INTR_EN = 0x0;
    /* Настройка полярности сигнала SS. */
    base->SS_POL = config->slave_pol;
    /* Выбор slave-устройства. */
    base->SS = config->slave_select;
    /* Настройка задержки между передачами. */
    QSPI_CTRL_MWAITEN_PTYPE_SET(base, config->delay_en);
    /* Настройка DMA. */
    QSPI_CTRL_DMA_PTYPE_SET(base, config->dma_en);
    /* Настройка режима работы контроллера (Master/Slave). */
    QSPI_CTRL_MASTER_PTYPE_SET(base, config->mode);
    /* Настройка полярности тактового сигнала. */
    QSPI_CTRL_CPOL_PTYPE_SET(base, config->cpol);
    /* Настройка фазы тактового сигнала. */
    QSPI_CTRL_CPHA_PTYPE_SET(base, config->cpha);
    /* Настройка порядка передачи бит. */
    QSPI_CTRL_MSB1ST_PTYPE_SET(base, config->msb);
    /* Настройка бита непрерывной передачи. */
    QSPI_CTRL_CONTXFER_PTYPE_SET(base, config->cont_trans_en);
    /* Настройка бита продление непрерывной передачи. */
    QSPI_CTRL_AUX_CONTXFEREXTEND_PTYPE_SET(base, config->cont_transfer_ext);
    /* Настройка количества битов в передаче. */
    QSPI_CTRL_AUX_BITSIZE_PTYPE_SET(base, config->bit_size);
    /* Настройка запрета записи в RX FIFO. */
    QSPI_CTRL_AUX_INHIBITDIN_PTYPE_SET(base, config->inhibit_din);
    /* Настройка запрет чтение из TX FIFO. */
    QSPI_CTRL_AUX_INHIBITDOUT_PTYPE_SET(base, config->inhibit_dout);
    /* Настройка режима SPI. */
    QSPI_CTRL_AUX_QMODE_PTYPE_SET(base, config->spi_mode);
#if defined (QSPI_DRIVER_VERSION_1_6_0)
    /* Настройка формата SPI. */
    QSPI_CTRL_AUX_XFERFORMAT_PTYPE_SET(base, config->xfer_format);
#endif
    /* Включение контроллера QSPI. */
    QSPI_Enable(base);
}

#if defined (QSPI_DRIVER_VERSION_1_6_0)
void QSPI_SetFormatSPI(QSPI_Type *base, qspi_format_spi_t xfer_format)
{
    assert(base != NULL);
    QSPI_CTRL_AUX_XFERFORMAT_PTYPE_SET(base, xfer_format);
}
#endif

void QSPI_SetBitSize(QSPI_Type *base, qspi_bit_size_t bit_size)
{
    assert(base != NULL);
    QSPI_CTRL_AUX_BITSIZE_PTYPE_SET(base, bit_size);
}

void QSPI_SetQMode(QSPI_Type *base, qspi_qmode_t spi_mode)
{
    assert(base != NULL);
    if (QSPI_CTRL_AUX_QMODE_PTYPE_GET(base) == spi_mode)
        return;
    /* Ожидание завершения всех передач. */
    while ((QSPI_GetStatusFlag(base) & QSPI_STAT_XFERIP_Msk)
        || ((QSPI_GetStatusFlag(base) & QSPI_STAT_TXEMPTY_Msk) == 0))
        ;
    QSPI_CTRL_AUX_QMODE_PTYPE_SET(base, spi_mode);
    while (QSPI_CTRL_AUX_QMODE_PTYPE_GET(base) != spi_mode)
        ;
}

void QSPI_SetInhibitDin(QSPI_Type *base, bool inhibit_din)
{
    /* Ожидание завершения всех передач. */
    while ((QSPI_GetStatusFlag(base) & QSPI_STAT_XFERIP_Msk)
        || ((QSPI_GetStatusFlag(base) & QSPI_STAT_TXEMPTY_Msk) == 0))
        ;
    if (inhibit_din) {
        base->CTRL_AUX |= QSPI_CTRL_AUX_INHIBITDIN_Msk;
    } else {
        base->CTRL_AUX &= ~QSPI_CTRL_AUX_INHIBITDIN_Msk;
    }
    while (QSPI_CTRL_AUX_INHIBITDIN_PTYPE_GET(base) != inhibit_din)
        ;
}

void QSPI_SetInhibitDout(QSPI_Type *base, bool inhibit_dout)
{
    /* Ожидание завершения всех передач. */
    while ((QSPI_GetStatusFlag(base) & QSPI_STAT_XFERIP_Msk)
        || ((QSPI_GetStatusFlag(base) & QSPI_STAT_TXEMPTY_Msk) == 0))
        ;
    if (inhibit_dout) {
        base->CTRL_AUX |= QSPI_CTRL_AUX_INHIBITDOUT_Msk;
    } else {
        base->CTRL_AUX &= ~QSPI_CTRL_AUX_INHIBITDOUT_Msk;
    }
    while (QSPI_CTRL_AUX_INHIBITDOUT_PTYPE_GET(base) != inhibit_dout)
        ;
}

/*!
 * @}
 */
