/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @defgroup pwm_driver Драйвер модуля PWM
 *
 * @brief Драйвер широтно-импульсного модулятора
 *
 * Драйвер модуля широтно-импульсного модулятора управляет блоком генерации
 * широтно-импульсного модулированного сигнала
 */

/*!
 * @addtogroup pwm_driver
 * @{
 */

/*!
 * @file hal_pwm.h
 *
 * @brief Интерфейс драйвера модуля широтно-импульсного модулятора
 */

#ifndef HAL_PWM_H
#define HAL_PWM_H

#ifdef __cplusplus
extern "C" {
#endif

#include "hal_common.h"

#define PWM_CHANNEL_COUNT 4
#define PWM_UNITS { (PWM_Type *)0x40111000, (PWM_Type *)0x40111100, (PWM_Type *)0x40111200, (PWM_Type *)0x40111300 }

/*!
 * @brief Макрос для преобразования частоты в период
 */
#define PWM_FREQ_HZ_TO_PERIOD_US(x) (1000000 / (x))
/*!
 * @brief Макрос для преобразования времени из милисекунд в микросекунды
 */
#define PWM_PERIOD_MS(x)            ((x) * 1000UL)
/*!
 * @brief Макрос для преобразования времени из секунд в микросекунды
 */
#define PWM_PERIOD_S(x)             ((x) * 1000000UL)

/*!
 * @brief Выходы каналов ШИМ
 */
enum {
    PWM_OutA = 0x0UL, /*!< Канал А */
    PWM_OutB = 0x1UL, /*!< Канал B */
};

/*!
 * @brief Частота возникновения прерываний ШИМ
 */
enum pwm_eventprd {
    PWM_EventPrdNo    = 0UL, /*!< Нет генерации прерывания */
    PWM_EventPrdOne   = 1UL, /*!< Генерация прерывания каждый раз при возникновении события */
    PWM_EventPrdTwo   = 2UL, /*!< Генерация прерывания каждый второй раз при возникновении события */
    PWM_EventPrdThree = 3UL, /*!< Генерация прерывания каждый третий раз при возникновении события */
};

/*!
 * @brief Источники прерывания ШИМ
 */
enum pwm_int_source {
    PWM_IntSourceCtrcntEquZero    = 2UL, /*!< Признак равенства счетчика ШИМ нулю */
    PWM_IntSourceCtrcntEquCtrprd  = 3UL, /*!< Признак равенства счетчика ШИМ периоду счета */
    PWM_IntSourceCtrcntEquCmpAInc = 4UL, /*!< Признак равенства счетчика ШИМ регистру сравнения выхода A во время инкремента */
    PWM_IntSourceCtrcntEquCmpADec = 5UL, /*!< Признак равенства счетчика ШИМ регистру сравнения выхода A во время декремента */
    PWM_IntSourceCtrcntEquCmpBInc = 6UL, /*!< Признак равенства счетчика ШИМ регистру сравнения выхода B во время инкремента */
    PWM_IntSourceCtrcntEquCmpBDec = 7UL, /*!< Признак равенства счетчика ШИМ регистру сравнения выхода B во время декремента */
};

/*!
 * @brief Скважность дробящего сигнала
 */
enum pwm_chopper_duty {
    PWM_ChopperDuty_1_8 = 0, /*!< 1/8 */
    PWM_ChopperDuty_2_8 = 1, /*!< 2/8 */
    PWM_ChopperDuty_3_8 = 2, /*!< 3/8 */
    PWM_ChopperDuty_4_8 = 3, /*!< 4/8 */
    PWM_ChopperDuty_5_8 = 4, /*!< 5/8 */
    PWM_ChopperDuty_6_8 = 5, /*!< 6/8 */
    PWM_ChopperDuty_7_8 = 6, /*!< 7/8 */
};

/*!
 * @brief Частота дробящего сигнала
 */
enum pwm_chopper_freq {
    PWM_ChopperFreqClk_8  = 0, /*!< CLK/8  */
    PWM_ChopperFreqClk_16 = 1, /*!< CLK/16 */
    PWM_ChopperFreqClk_24 = 2, /*!< CLK/24 */
    PWM_ChopperFreqClk_32 = 3, /*!< CLK/32 */
    PWM_ChopperFreqClk_40 = 4, /*!< CLK/40 */
    PWM_ChopperFreqClk_48 = 5, /*!< CLK/48 */
    PWM_ChopperFreqClk_56 = 6, /*!< CLK/56 */
    PWM_ChopperFreqClk_64 = 7, /*!< CLK/64 */
};

/*!
 * @brief Ширина первого импульса
 */
enum pwm_chopper_first_width {
    PWM_ChopperFirstWidth_0_8  = 0,  /*!< 0xCLK/8 */
    PWM_ChopperFirstWidth_1_8  = 1,  /*!< 1xCLK/8 */
    PWM_ChopperFirstWidth_2_8  = 2,  /*!< 2xCLK/8 */
    PWM_ChopperFirstWidth_3_8  = 3,  /*!< 3xCLK/8 */
    PWM_ChopperFirstWidth_4_8  = 4,  /*!< 4xCLK/8 */
    PWM_ChopperFirstWidth_5_8  = 5,  /*!< 5xCLK/8 */
    PWM_ChopperFirstWidth_6_8  = 6,  /*!< 6xCLK/8 */
    PWM_ChopperFirstWidth_7_8  = 7,  /*!< 7xCLK/8 */
    PWM_ChopperFirstWidth_8_8  = 8,  /*!< 8xCLK/8 */
    PWM_ChopperFirstWidth_9_8  = 9,  /*!< 9xCLK/8 */
    PWM_ChopperFirstWidth_10_8 = 10, /*!< 10xCLK/8 */
    PWM_ChopperFirstWidth_11_8 = 11, /*!< 11xCLK/8 */
    PWM_ChopperFirstWidth_12_8 = 12, /*!< 12xCLK/8 */
    PWM_ChopperFirstWidth_13_8 = 13, /*!< 13xCLK/8 */
    PWM_ChopperFirstWidth_14_8 = 14, /*!< 14xCLK/8 */
    PWM_ChopperFirstWidth_15_8 = 15, /*!< 15xCLK/8 */
};

/*!
 * @brief Конфигурация дробления сигнала ШИМ
 */
typedef struct _pwm_chopper {
    bool chopper_en;
    enum pwm_chopper_duty chopper_duty;
    enum pwm_chopper_freq chopper_freq;
    enum pwm_chopper_first_width chopper_first_width;
} pwm_chopper_cfg_t;

/*!
 * @brief Полярность OUTx после генерации запрещенной зоны
 */
enum pwm_dz_outx_inv {
    PWM_DzSignalOutxInvOff = 0, /*!< Нет инверсии */
    PWM_DzSignalOutxInvOn  = 1, /*!< Есть инверсия */
};

/*!
 * @brief Выбор режима работы блока запрещенной зоны при формирования OUTx
 */
enum pwm_dz_mode {
    PWM_DzModeOff = 0, /*!< Генератор выключен */
    PWM_DzModeOn  = 1, /*!< Генератор включен */
};

/*!
 * @brief Конфигурация мертвой зоны ШИМ
 */
typedef struct _pwm_dz_cfg {
    bool dz_en;
    enum pwm_dz_outx_inv inv;
} pwm_dz_cfg_t;

/*!
 * @brief Реакции на событие блока trip unit
 */
enum pwm_trip_unit_action {
    PWM_TripUnitActionHigh = 0, /*!< OUTx переводится в высокоомное состояние */
    PWM_TripUnitActionOne  = 1, /*!< OUTx подтягивается к 1 */
    PWM_TripUnitActionZero = 2, /*!< OUTx подтягивается к 0 */
    PWM_TripUnitActionNo   = 3, /*!< Действий не производится */
};

typedef struct _pwm_trip_unit_cfg {
    bool trip_unit_en;
    enum pwm_trip_unit_action action;
} pwm_trip_unit_cfg_t;

/*!
 * @brief Контекст драйвера ШИМ
 */
typedef struct _pwm_handle {
    uint32_t channel;                               /*!< Номер канала */
    uint32_t out;                                   /*!< Выход канала */

    uint32_t ref_clk;                               /*!< Опорная частота контроллера ШИМ */

    float duty_cycle;                               /*!< Ширина импульса от 0.0 до 1.0 */
    uint32_t period_us;                             /*!< Период в микросекундах */

    bool int_en;                                    /*!< Разрешение прерываний */
    enum pwm_int_source int_source;                 /*!< Источник прерывания */
    enum pwm_eventprd int_freq;                     /*!< Частота возникновения прерываний */
    void (*pwm_callback)(struct _pwm_handle *hpwm); /*!< Функция обратного вызова при возникновении прерываний */

    pwm_chopper_cfg_t chopper_cfg;
    pwm_dz_cfg_t deadzone_cfg;
    pwm_trip_unit_cfg_t trip_unit_cfg;
} pwm_handle_t;

/*!
 * @brief Инициализация канала ШИМ и выводов
 * 
 * @param hpwm Контекст драйвера ШИМ
 */
void PWM_Init(pwm_handle_t *hpwm);

/*!
 * @brief Включение вывода канала ШИМ
 * 
 * @param hpwm Контекст драйвера ШИМ
 * @param bool 1 - включение, 0 - выключение
 */
void PWM_Enable(pwm_handle_t *hpwm, bool enable);

/*!
 * @brief Общая функция обратного вызова при возникновении прерываний
 * 
 * @param hpwm Контекст драйвера ШИМ
 */
void PWM_IntCallback(pwm_handle_t *hpwm);

#ifdef __cplusplus
}
#endif

#endif /* HAL_PWM_H */

/*!
 * @}
 */
