/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @addtogroup pwm_driver
 * @{
 */

/*!
 * @file hal_pwm.c
 *
 * @brief Имплементация драйвера широтно-импульсного модулятора
 */

#include "hal_common.h"
#include "hal_pwm.h"

#define PWM_CHANNEL_OFFSET (0x100) /*!< Смещение канала */
#define PWM_CHANNEL_NUMBER (4)     /*!< Количество каналов */

enum pwm_status PWM_GetChannelDefaultConfig(struct pwm_channel_config * cfg)
{
    if (cfg == NULL)
        return PWM_Status_InvalidArgument;

    cfg->channel = 0;
    /* Группа 1 */
    /* Предделитель */
    cfg->prescaler_mode = pwm_PrescModeTimerIsRun;
    cfg->prescaler_cmd = pwm_PrescCmdReset;
    cfg->prescaler_syncrst = pwm_PrescalerSyncRstDis;
    cfg->prescaler = 0;
    cfg->prescaler_divmux = pwm_PrescalerDivMux1;
    /* Основной счетчик */
    cfg->cntmode = pwm_CntModeUpDown;
    cfg->counter = 0;
    cfg->period = 0;
    cfg->loadprd = pwm_LoadPrdDirect;
    cfg->ctrphs = 0;
    cfg->syncphsen = pwm_SyncPhsEnDis;
    /* Блок сравнения */
    cfg->cmpa = 0;
    cfg->cmpb = 0;
    cfg->scmpamode = pwm_SCmpxModeReg;
    cfg->scmpbmode = pwm_SCmpxModeReg;
    cfg->ldamode = 0;
    cfg->ldbmode = 0;
    /* Группа 2 */
    /* Блок реакции на событие */
    cfg->cnt_eq_prd_outa = pwm_OutxCmdNo;
    cfg->cnt_eq_prd_outb = pwm_OutxCmdNo;
    cfg->cnt_eq_cmpa_dec_outa = pwm_OutxCmdNo;
    cfg->cnt_eq_cmpa_inc_outa = pwm_OutxCmdNo;
    cfg->cnt_eq_cmpa_dec_outb = pwm_OutxCmdNo;
    cfg->cnt_eq_cmpa_inc_outb = pwm_OutxCmdNo;
    cfg->cnt_eq_cmpb_dec_outa = pwm_OutxCmdNo;
    cfg->cnt_eq_cmpb_inc_outa = pwm_OutxCmdNo;
    cfg->cnt_eq_cmpb_dec_outb = pwm_OutxCmdNo;
    cfg->cnt_eq_cmpb_inc_outb = pwm_OutxCmdNo;
    cfg->cnt_eq_zero_outa = pwm_OutxCmdNo;
    cfg->cnt_eq_zero_outb = pwm_OutxCmdNo;
    cfg->sw_forced_outa = pwm_OutxCmdNo;
    cfg->sw_forced_outb = pwm_OutxCmdNo;
    cfg->sw_forced_long_outa = pwm_OutxCmdNo;
    cfg->sw_forced_long_outb = pwm_OutxCmdNo;
    cfg->ldcswrf = pwm_LdcswrfCtrcntZero;
    /* Блок прерываний */
    cfg->pwm_int_enable = pwm_IntEnNo;
    cfg->pwm_int_source = pwm_IntSourceNo;
    cfg->eventprd = pwm_EventPrdNo;
    //cfg->pwm_soft_int_enable = pwm_IntEnNo;
    /* Группа 3 */
    /* Генератор запретной зоны */
    cfg->dz_rising_edge_delay_clk = 0;
    cfg->dz_falling_edge_delay_clk = 0;
    cfg->dz_rising_edge_source = pwm_DzSignalOutA;
    cfg->dz_falling_edge_source = pwm_DzSignalOutA;
    cfg->dz_rising_edge_outa_inv = pwm_DzSignalOutxInvOff;
    cfg->dz_falling_edge_outb_inv = pwm_DzSignalOutxInvOff;
    cfg->dz_outa_enable = pwm_DzModeOff;
    cfg->dz_outb_enable = pwm_DzModeOff;
    /* Блок дробления выходного сигнала */
    cfg->chopper_duty = pwm_ChopperDuty_1_8;
    cfg->chopper_freq = pwm_ChopperFreqClk_8;
    cfg->chopper_first_width = pwm_ChopperFirstWidth_0_8;
    cfg->chopper_work = pwm_ChopperWorkOff;
    /* Блок реакции на внешнее воздействие */
    cfg->inputs_mask_one = 0;
    cfg->inputs_mask_mult = 0;
    cfg->trip_unit_action_outa = pwm_TripUnitActionHigh;
    cfg->trip_unit_action_outb = pwm_TripUnitActionHigh;
    cfg->pwmtu_int_one = pwm_IntEnNo;
    cfg->pwmtu_int_mult = pwm_IntEnNo;
    //cfg->pwmtu_soft_int_one = pwm_IntEnNo;
    //cfg->pwmtu_soft_int_mult = pwm_IntEnNo;

    /* Запуск/останов канала*/
    cfg->cmd = pwm_RunCmdStop;

    return PWM_Status_Ok;
}

enum pwm_status PWM_InitChannel(PWM_Type *base,
    struct pwm_channel_config cfg)
{
    PWM_Type *channel_offset;
    uint32_t run_pos, run_mask;
    uint32_t pres_mode_pos, pres_mode_mask;
    uint32_t pres_cmd_pos, pres_cmd_mask;
    
    
    if (base == NULL)
        return PWM_Status_InvalidArgument;
    if ((base != PWM) && (base != PWM_Secure))
        return PWM_Status_InvalidArgument;
    if (cfg.channel >= PWM_CHANNEL_NUMBER)
        return PWM_Status_BadConfigure;

    channel_offset = (PWM_Type *)((uint32_t)base + cfg.channel * PWM_CHANNEL_OFFSET);
    run_pos = (PWM_CTRRUN_RUN1_Pos - PWM_CTRRUN_RUN0_Pos) * cfg.channel + PWM_CTRRUN_RUN0_Pos;
    run_mask = PWM_CTRRUN_RUN0_Msk << run_pos;
    pres_mode_pos = (PWM_CTRRUN_PRESMODE1_Pos - PWM_CTRRUN_PRESMODE0_Pos) * cfg.channel + PWM_CTRRUN_PRESMODE0_Pos;
    pres_mode_mask = PWM_CTRRUN_PRESMODE0_Msk << pres_mode_pos; 
    pres_cmd_pos = (PWM_CTRRUN_PRESRST1_Pos - PWM_CTRRUN_PRESRST0_Pos) * cfg.channel + PWM_CTRRUN_PRESRST0_Pos;
    pres_cmd_mask = PWM_CTRRUN_PRESRST1_Msk << pres_cmd_pos; 

    /* Останов канала */
    SET_VAL_MSK(channel_offset->CTRRUN, run_mask, run_pos, pwm_RunCmdStop);
    /* Нужно исследовать while (PWM_GetCntStat(base, cfg.channel)); */

    /* Предделитель */
    SET_VAL_MSK(channel_offset->CTRRUN, pres_mode_mask, pres_mode_pos, cfg.prescaler_mode);
    SET_VAL_MSK(channel_offset->CTRRUN, pres_cmd_mask, pres_cmd_pos, cfg.prescaler_cmd);
    PWM_CLKCTL_SYNCRST_PTYPE_SET(channel_offset, cfg.prescaler_syncrst);
    PWM_CLKCTL_PRESPRD_PTYPE_SET(channel_offset, cfg.prescaler);
    PWM_CLKCTL_DEVMUX_PTYPE_SET(channel_offset, cfg.prescaler_divmux);
    /* Основной счетчик */
    PWM_CLKCTL_CNTMODE_PTYPE_SET(channel_offset, cfg.cntmode);
    PWM_CTRCNT_CTRCNT_PTYPE_SET(channel_offset, cfg.counter);
    PWM_CLKCTL_LOADPRD_PTYPE_SET(channel_offset, cfg.loadprd);
    PWM_CTRPRD_CTRPRD_PTYPE_SET(channel_offset, cfg.period);
    PWM_CTRPHS_CTRPHS_PTYPE_SET(channel_offset, cfg.ctrphs);
    PWM_CLKCTL_SYNCPHSEN_PTYPE_SET(channel_offset, cfg.syncphsen);
    /* Блок сравнения */
    PWM_CMPCTL_SCMPAMODE_PTYPE_SET(channel_offset, cfg.scmpamode);
    PWM_CMPCTL_SCMPBMODE_PTYPE_SET(channel_offset, cfg.scmpbmode);
    PWM_CMPCTL_LDAMODE_PTYPE_SET(channel_offset, cfg.ldamode);
    PWM_CMPCTL_LDBMODE_PTYPE_SET(channel_offset, cfg.ldbmode);
    PWM_CMPA_CMPA_PTYPE_SET(channel_offset, cfg.cmpa);
    PWM_CMPB_CMPB_PTYPE_SET(channel_offset, cfg.cmpb);
    /* Блок реакции на событие */
    PWM_EMCTLA_EPRD_PTYPE_SET(channel_offset, cfg.cnt_eq_prd_outa);
    PWM_EMCTLB_EPRD_PTYPE_SET(channel_offset, cfg.cnt_eq_prd_outb);
    PWM_EMCTLA_ECMPAD_PTYPE_SET(channel_offset, cfg.cnt_eq_cmpa_dec_outa);
    PWM_EMCTLA_ECMPAI_PTYPE_SET(channel_offset, cfg.cnt_eq_cmpa_inc_outa);
    PWM_EMCTLB_ECMPAD_PTYPE_SET(channel_offset, cfg.cnt_eq_cmpa_dec_outb);
    PWM_EMCTLB_ECMPAI_PTYPE_SET(channel_offset, cfg.cnt_eq_cmpa_inc_outb);
    PWM_EMCTLA_ECMPBD_PTYPE_SET(channel_offset, cfg.cnt_eq_cmpb_dec_outa);
    PWM_EMCTLA_ECMPBI_PTYPE_SET(channel_offset, cfg.cnt_eq_cmpb_inc_outa);
    PWM_EMCTLB_ECMPBD_PTYPE_SET(channel_offset, cfg.cnt_eq_cmpb_dec_outb);
    PWM_EMCTLB_ECMPBI_PTYPE_SET(channel_offset, cfg.cnt_eq_cmpb_inc_outb);
    PWM_EMCTLA_EZRO_PTYPE_SET(channel_offset, cfg.cnt_eq_zero_outa);
    PWM_EMCTLB_EZRO_PTYPE_SET(channel_offset, cfg.cnt_eq_zero_outb);
    PWM_EMSWFR_ACTSFA_PTYPE_SET(channel_offset, cfg.sw_forced_outa);
    PWM_EMSWFR_ACTSFB_PTYPE_SET(channel_offset, cfg.sw_forced_outb);
    PWM_EMCSWFR_LONGSFA_PTYPE_SET(channel_offset, cfg.sw_forced_long_outa);
    PWM_EMCSWFR_LONGSFB_PTYPE_SET(channel_offset, cfg.sw_forced_long_outb);
    PWM_EMSWFR_LDCSWRF_PTYPE_SET(channel_offset, cfg.ldcswrf);
    /* Блок прерываний */
    PWM_ICSEL_INTEN_PTYPE_SET(channel_offset, cfg.pwm_int_enable);
    PWM_ICSEL_INTSEL_PTYPE_SET(channel_offset, cfg.pwm_int_source);
    PWM_ICCTL_EVENTPRD_PTYPE_SET(channel_offset, cfg.eventprd);
    /* Генератор запретной зоны */
    PWM_DZRPER_PER_PTYPE_SET(channel_offset, cfg.dz_rising_edge_delay_clk);
    PWM_DZFPER_PER_PTYPE_SET(channel_offset, cfg.dz_falling_edge_delay_clk);
    PWM_DZCTL_INMUXR_PTYPE_SET(channel_offset, cfg.dz_rising_edge_source);
    PWM_DZCTL_INMUXF_PTYPE_SET(channel_offset, cfg.dz_falling_edge_source);
    PWM_DZCTL_INVMUXA_PTYPE_SET(channel_offset, cfg.dz_rising_edge_outa_inv);
    PWM_DZCTL_INVMUXB_PTYPE_SET(channel_offset, cfg.dz_falling_edge_outb_inv);
    PWM_DZCTL_OUTMUXA_PTYPE_SET(channel_offset, cfg.dz_outa_enable);
    PWM_DZCTL_OUTMUXB_PTYPE_SET(channel_offset, cfg.dz_outb_enable);
    /* Блок дробления выходного сигнала */
    PWM_CHCTL_CHDUTY_PTYPE_SET(channel_offset, cfg.chopper_duty);
    PWM_CHCTL_CHCLKDEV_PTYPE_SET(channel_offset, cfg.chopper_freq);
    PWM_CHCTL_FIRSTWTH_PTYPE_SET(channel_offset, cfg.chopper_first_width);
    PWM_CHCTL_CHEN_PTYPE_SET(channel_offset, cfg.chopper_work);
    /* Блок реакции на внешнее воздействие */
    PWM_TUSEL_ONE_PTYPE_SET(channel_offset, cfg.inputs_mask_one);
    PWM_TUSEL_MULT_PTYPE_SET(channel_offset, cfg.inputs_mask_mult);
    PWM_TUCTL_TUA_PTYPE_SET(channel_offset, cfg.trip_unit_action_outa);
    PWM_TUCTL_TUB_PTYPE_SET(channel_offset, cfg.trip_unit_action_outb);
    PWM_TUINTM_ONE_PTYPE_SET(channel_offset, cfg.pwmtu_int_one);
    PWM_TUINTM_MULT_PTYPE_SET(channel_offset, cfg.pwmtu_int_mult);

    SET_VAL_MSK(channel_offset->CTRRUN, run_mask, run_pos, cfg.cmd);

    return PWM_Status_Ok;
}

enum pwm_status PWM_Deinit(PWM_Type *base)
{
    if (base == NULL)
        return PWM_Status_InvalidArgument;

    SET_VAL_MSK(base->CLKCTL, PWM_CLKCTL_CNTMODE_Msk, PWM_CLKCTL_CNTMODE_Pos, 0);
    SET_VAL_MSK(base->CTRRUN, PWM_CTRRUN_RUN0_Msk, PWM_CTRRUN_RUN0_Pos, 0);
    return PWM_Status_Ok;
}

enum pwm_status PWM_Enable(PWM_Type *base, uint32_t channel, enum pwm_run_command cmd)
{
    PWM_Type *channel_offset;
    uint32_t run_pos, run_mask;

    if (base == NULL)
        return PWM_Status_InvalidArgument;
    if (channel >= PWM_CHANNEL_NUMBER)
        return PWM_Status_BadConfigure;
    if (cmd > pwm_RunCmdMax)
        return PWM_Status_BadConfigure;

    channel_offset = (PWM_Type *)((uint32_t)base + channel * PWM_CHANNEL_OFFSET);
    run_pos = (PWM_CTRRUN_RUN1_Pos - PWM_CTRRUN_RUN0_Pos) * channel + PWM_CTRRUN_RUN0_Pos;
    run_mask = PWM_CTRRUN_RUN0_Msk << run_pos;

    SET_VAL_MSK(channel_offset->CTRRUN, run_mask, run_pos, cmd);
    return PWM_Status_Ok;
}

enum pwm_status PWM_CmdForAllChannels(PWM_Type *base,
    uint32_t channel_mask,
    enum pwm_run_command cmd0,
    enum pwm_run_command cmd1,
    enum pwm_run_command cmd2,
    enum pwm_run_command cmd3
    )
{
    uint32_t chen;
    uint32_t run_pos = 0, run_mask = 0;
    uint32_t cmd;

    if (base == NULL)
        return PWM_Status_InvalidArgument;

    /* Собираем все команды, не нужные обрежутся по маске далее */
    cmd = ((cmd0 << PWM_CTRRUN_RUN0_Pos) & PWM_CTRRUN_RUN0_Msk) |
          ((cmd1 << PWM_CTRRUN_RUN1_Pos) & PWM_CTRRUN_RUN1_Msk) |
          ((cmd2 << PWM_CTRRUN_RUN2_Pos) & PWM_CTRRUN_RUN2_Msk) |
          ((cmd3 << PWM_CTRRUN_RUN3_Pos) & PWM_CTRRUN_RUN3_Msk);
    /* Готовим маску */
    for (chen = 0; chen < 4; chen++) {
        if (channel_mask & 1) {
            run_pos = (PWM_CTRRUN_RUN1_Pos - PWM_CTRRUN_RUN0_Pos) * chen + PWM_CTRRUN_RUN0_Pos;
            run_mask |= PWM_CTRRUN_RUN0_Msk << run_pos;
        }
        channel_mask = channel_mask >> 1;
    }
    run_pos = PWM_CTRRUN_RUN0_Pos;
    SET_VAL_MSK(base->CTRRUN, run_mask, run_pos, cmd);

    return PWM_Status_Ok;
}

enum pwm_status PWM_ApplySoftOuts(PWM_Type *base, uint32_t channel, 
    int8_t mask_outs)
{
    PWM_Type *channel_offset;

    if (base == NULL)
        return PWM_Status_InvalidArgument;
    if (channel >= PWM_CHANNEL_NUMBER)
        return PWM_Status_BadConfigure;
    channel_offset = (PWM_Type *)((uint32_t)base + channel * PWM_CHANNEL_OFFSET);
    if (mask_outs & 1)
        PWM_EMSWFR_ONESFA_PTYPE_SET(channel_offset, 1);
    if (mask_outs & 2)
        PWM_EMSWFR_ONESFB_PTYPE_SET(channel_offset, 1);
    return PWM_Status_Ok;
}

enum pwm_status PWM_ApplyLongSoftOuts(PWM_Type *base, uint32_t channel, 
    int8_t mask_outs, enum pwm_outx_cmd outa, enum pwm_outx_cmd outb)
{
    PWM_Type *channel_offset;

    if (base == NULL)
        return PWM_Status_InvalidArgument;
    if (channel >= PWM_CHANNEL_NUMBER)
        return PWM_Status_BadConfigure;
    channel_offset = (PWM_Type *)((uint32_t)base + channel * PWM_CHANNEL_OFFSET);
    if (mask_outs & 1)
        PWM_EMCSWFR_LONGSFA_PTYPE_SET(channel_offset, outa);
    if (mask_outs & 2)
        PWM_EMCSWFR_LONGSFB_PTYPE_SET(channel_offset, outb);
    return PWM_Status_Ok;
}

uint32_t PWM_GetCntStat(PWM_Type *base, uint32_t channel)
{
    PWM_Type *channel_offset;
    uint32_t stat_pos = 0, stat_mask = 0;
    
    if (base == NULL)
        return PWM_Status_InvalidArgument;
    if (channel >= PWM_CHANNEL_NUMBER)
        return PWM_Status_BadConfigure;
    channel_offset = (PWM_Type *)((uint32_t)base + channel * PWM_CHANNEL_OFFSET);
    stat_pos = (PWM_CTRSTS_RUN_STS1_Pos - PWM_CTRSTS_RUN_STS0_Pos) * channel + PWM_CTRSTS_RUN_STS0_Pos;
    stat_mask |= PWM_CTRSTS_RUN_STS0_Msk << stat_pos;
    
    return GET_VAL_MSK(channel_offset->CTRSTS, stat_mask, stat_pos);
}

enum pwm_status PWM_SetPeriod(PWM_Type *base, uint32_t channel, 
    uint32_t period)
{
    PWM_Type *channel_offset;   
    
    if (base == NULL)
        return PWM_Status_InvalidArgument;
    if ((base != PWM) && (base != PWM_Secure))
        return PWM_Status_InvalidArgument;
    if (channel >= PWM_CHANNEL_NUMBER)
        return PWM_Status_BadConfigure;

    channel_offset = (PWM_Type *)((uint32_t)base + channel * PWM_CHANNEL_OFFSET);
    PWM_CTRPRD_CTRPRD_PTYPE_SET(channel_offset, period);
    return PWM_Status_Ok;
}

/*!
 * @}
 */
