/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @addtogroup jtm_driver
 * @{
 */

/*!
 * @file hal_jtm.c
 *
 * @brief Имплементация драйвера модуля JTM
 */

#include "hal_jtm.h"
#include "hal_ioim.h"

/*!
 * @brief Режимы преобразования
 */
enum {
    JTM_ModeTemper = 0U, /*!< Измерение температуры */
    JTM_ModeVfs10  = 1U, /*!< Дифференциальное измерение, Vfs = 1.0В */
    JTM_ModeVfs05  = 2U, /*!< Дифференциальное измерение, Vfs = 0.5В */
    JTM_ModeVfs20  = 3U, /*!< Дифференциальное измерение, Vfs = 2.0В */
};

/*!
 * @brief Значения милливольт, соответствующие режимам измерения
 */
enum {
    JTM_ModeVfs05Millivolts = 500U,  /*!< 0.5В */
    JTM_ModeVfs10Millivolts = 1000U, /*!< 1.0В */
    JTM_ModeVfs20Millivolts = 2000U, /*!< 2.0В */
};

static const uint32_t JTM_adc_code_size = 4096U; /*!< Ширина кодового диапазона АЦП */
static const int32_t JTM_temper_step = 250U; /*!< Цена младшего разряда АЦП при измерении температуры */

void JTM_Init(JTM_Type *base, jtm_config_t *config)
{
    assert(base != NULL);
    assert(config != NULL);

    SET_VAL_MSK(base->CFG0, JTM_CFG0_TCAL_Msk, JTM_CFG0_TCAL_Pos,
            config->tcal);
    SET_VAL_MSK(base->CFG0, JTM_CFG0_WCAL_Msk, JTM_CFG0_WCAL_Pos,
            config->wcal);
    SET_VAL_MSK(base->CFG1, JTM_CFG1_WTCALCONF_Msk, JTM_CFG1_WTCALCONF_Pos,
            config->wtcalconf);
    SET_VAL_MSK(base->CFG1, JTM_CFG1_WTCONF_Msk, JTM_CFG1_WTCONF_Pos,
            config->wtconf);
}

/*!
 * @brief Запуск преобразования АЦП JTM
 *
 * Функция выполняет проверку корректности запроса параметра и запускает преобразование АЦП.
 *
 * @param base      Базовый адрес контроллера
 * @param parameter Тип параметра для чтения
 *
 * @retval #JTM_Status_Ok
 * @retval #JTM_Status_BadParameter
 */
static jtm_status_t JTM_StartReading(JTM_Type *base, jtm_parameter_t parameter)
{
    SET_VAL_MSK(base->CTR, JTM_CTR_CHSEL_Msk, JTM_CTR_CHSEL_Pos, parameter);

    switch (parameter) {
        case JTM_Temperature:
            SET_VAL_MSK(base->CTR, JTM_CTR_MODE_Msk, JTM_CTR_MODE_Pos,
                    JTM_ModeTemper);
            break;
        case JTM_Vcasn:
        case JTM_Vcore:
            SET_VAL_MSK(base->CTR, JTM_CTR_MODE_Msk, JTM_CTR_MODE_Pos,
                    JTM_ModeVfs20);
            break;
        default:
            return JTM_Status_BadParameter;
    }

    SET_VAL_MSK(base->CTR, JTM_CTR_SOC_Msk, JTM_CTR_SOC_Pos, 1U);
    return JTM_Status_Ok;
}

/*!
 * @brief Чтение результата преобразования АЦП JTM
 *
 * Функция выполняет считывание параметра из регистра АЦП и его математическое приведение.
 *
 * @param base      Базовый адрес контроллера
 * @param parameter Тип параметра для чтения
 *
 * @return          Значение параметра
 */
static int32_t JTM_GetValue(JTM_Type *base, jtm_parameter_t parameter)
{
    switch (parameter) {
        case JTM_Temperature:
        {
            int32_t raw = GET_VAL_MSK(base->STAT, JTM_STAT_TEMP_Msk,
                    JTM_STAT_TEMP_Pos);
            /* Распространение знакового разряда. */
            if (raw & 0x800)
                raw |= 0xFFFFF000;
            return raw * JTM_temper_step;
        }
        case JTM_Vcasn:
        case JTM_Vcore:
        default:
            return GET_VAL_MSK(base->STAT, JTM_STAT_CODE_Msk,
                    JTM_STAT_CODE_Pos) * JTM_ModeVfs20Millivolts
                    / JTM_adc_code_size;
    }
}

jtm_status_t JTM_GetParameterValue(JTM_Type *base, jtm_parameter_t parameter,
        int32_t *value)
{
    assert(base != NULL);
    assert(value != NULL);

    jtm_status_t status;

    /* Проверка, не идет ли уже преобразование. */
    if (GET_VAL_MSK(base->STAT, JTM_STAT_READY_Msk, JTM_STAT_READY_Pos) == 0U)
        return JTM_Status_Busy;

    status = JTM_StartReading(base, parameter);
    if (status != JTM_Status_Ok)
        return status;

    /* Ожидание завершения преобразования. */
    while (GET_VAL_MSK(base->STAT, JTM_STAT_READY_Msk, JTM_STAT_READY_Pos)
            == 0U)
        ;

    *value = JTM_GetValue(base, parameter);
    return JTM_Status_Ok;
}

void JTM_HandleIRQ(JTM_Type *base, jtm_handle_t *handle)
{
    /* Запрет прерывания. */
    SET_VAL_MSK(base->INTEN, JTM_INTEN_READY_Msk, JTM_INTEN_READY_Pos, 0U);

    handle->callback(handle, handle->parameter,
            JTM_GetValue(base, handle->parameter), handle->user_data);

    /* Сброс флага прерывания. */
    SET_VAL_MSK(base->INTCLR, JTM_INTCLR_READY_Msk, JTM_INTCLR_READY_Pos, 1U);
}

jtm_status_t JTM_CreateHandle(JTM_Type *base, jtm_handle_t *handle,
        jtm_callback_t callback, void *user_data)
{
    assert(base != NULL);
    assert(handle != NULL);

    /* Установка обратного вызова и пользовательских данных. */
    handle->callback = callback;
    handle->user_data = user_data;

    /* Установка обработчика прерывания. */
    if (IOIM_SetIRQHandler(base, JTM_HandleIRQ, handle) != IOIM_Status_Ok)
        return JTM_Status_Fail;

    return JTM_Status_Ok;
}

jtm_status_t JTM_GetParameterValueNonBlocking(JTM_Type *base,
        jtm_handle_t *handle, jtm_parameter_t parameter)
{
    assert(base != NULL);
    assert(handle != NULL);

    /* Проверка, не идет ли уже преобразование. */
    if (GET_VAL_MSK(base->STAT, JTM_STAT_READY_Msk, JTM_STAT_READY_Pos) == 0U)
        return JTM_Status_Busy;

    handle->parameter = parameter;

    /* Разрешение прерывания. */
    SET_VAL_MSK(base->INTEN, JTM_INTEN_READY_Msk, JTM_INTEN_READY_Pos, 1U);

    return JTM_StartReading(base, parameter);
}
