/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @defgroup i2s_driver Драйвер модуля I2S
 *
 * @brief Драйвер последовательной шины для соединения цифровых аудиоустройств
 *
 * Драйвер содержит функции управления контроллером I2S микросхемы ELIOT1.
 * Контроллер интерфейса собран со следующими параметрами:
 * - COMP_PARAM1 = 0x024C003A
 * - COMP_PARAM2 = 0x00000489
 */

/*!
 * @addtogroup i2s_driver
 * @{
 */

/*!
 * @file hal_i2s.h
 *
 * @brief Интерфейс драйвера модуля I2S
 */

#ifndef HAL_I2S_H
#define HAL_I2S_H

#ifdef __cplusplus
extern "C" {
#endif

#include "hal_common.h"

/*!
 * @brief Версия драйвера I2S
 */
#define HAL_I2S_DRIVER_VERSION (MAKE_VERSION(0, 1, 0))

/*!
 * @brief Коды возврата функций драйвера I2S
 */
typedef enum _i2s_status {
    I2S_Status_Ok                 = 0U, /*!< Успешно */
    I2S_Status_Fail               = 1U, /*!< Провал */
    I2S_Status_InvalidArgument    = 2U, /*!< Неверный аргумент */
    I2S_Status_TxBusy             = 3U, /*!< Передатчик занят */
    I2S_Status_UnsupportedBitRate = 4U, /*!< Комбинация параметров задана так, что невозможно установить правильную битовую частоту */
} i2s_status_t;

/*!
 * @brief Флаги прерываний I2S
 */
typedef enum _i2s_flag {
    I2S_FlagTxFifoEmpty   =  4U, /*!< Опустошение очереди выдачи */
    I2S_FlagTxFifoOverrun =  5U, /*!< Переполнение очереди выдачи */
} i2s_flag_t;

#if defined(__CC_ARM)
#pragma anon_unions
#endif

/*!
 * @brief Перечисление возможных значений числа синхроимпульсов sclk на
 *        левый и правый поток
 */
typedef enum _i2s_sclk_per_sample {
    I2S_SclkCycles_16 = 0U, /*!< 16 синхроимпульсов на значение (левое/правое) */
    I2S_SclkCycles_24 = 1U, /*!< 24 синхроимпульсов на значение (левое/правое) */
    I2S_SclkCycles_32 = 2U, /*!< 32 синхроимпульсов на значение (левое/правое) */
} i2s_sclk_per_sample_t;

/*!
 * @brief Перечисление возможных значений обрезания числа синхроимпульсов sclk
 *
 * Если разрешение канала I2S меньше, чем размер слова, то часть импульсов sclk
 * может быть обрезана на выходе с помощью установки значения из этого
 * перечисления в параметре драйвер @ref i2s_config_t.sclk_gating.
 */
typedef enum _i2s_sclk_gating {
    I2S_NoSclkGating        = 0U, /*!< Нет обрезания sclk */
    I2S_SclkGatingCycles_12 = 1U, /*!< Обрезание после 12 импульсов sclk */
    I2S_SclkGatingCycles_16 = 2U, /*!< Обрезание после 16 импульсов sclk */
    I2S_SclkGatingCycles_20 = 3U, /*!< Обрезание после 20 импульсов sclk */
    I2S_SclkGatingCycles_24 = 4U, /*!< Обрезание после 24 импульсов sclk */
} i2s_sclk_gating_t;

/*!
 * @brief Разрядность данных I2S
 */
typedef enum _i2s_resolution {
    I2S_ResolutionDefault = 0U, /*!< По умолчанию (32 бита) */
    I2S_Resolution_12     = 1U, /*!< 12 бит */
    I2S_Resolution_16     = 2U, /*!< 16 бит */
    I2S_Resolution_20     = 3U, /*!< 20 бит */
    I2S_Resolution_24     = 4U, /*!< 24 бита */
    I2S_Resolution_32     = 5U, /*!< 32 бита */
} i2s_resolution_t;

/*!
 * @brief Уровни срабатывания прерывания по опустошению очереди I2S
 */
typedef enum _i2s_interrupt_level {
    I2S_InterruptTriggerLevel_1  = 0U,  /*!< Прерывание при 1-м оставшемся слове в очереди */
    I2S_InterruptTriggerLevel_2  = 1U,  /*!< Прерывание при 2-х оставшихся словах в очереди */
    I2S_InterruptTriggerLevel_3  = 2U,  /*!< Прерывание при 3-х оставшихся словах в очереди */
    I2S_InterruptTriggerLevel_4  = 3U,  /*!< Прерывание при 4-х оставшихся словах в очереди */
    I2S_InterruptTriggerLevel_5  = 4U,  /*!< Прерывание при 5-и оставшихся словах в очереди */
    I2S_InterruptTriggerLevel_6  = 5U,  /*!< Прерывание при 6-и оставшихся словах в очереди */
    I2S_InterruptTriggerLevel_7  = 6U,  /*!< Прерывание при 7-и оставшихся словах в очереди */
    I2S_InterruptTriggerLevel_8  = 7U,  /*!< Прерывание при 8-и оставшихся словах в очереди */
    I2S_InterruptTriggerLevel_9  = 8U,  /*!< Прерывание при 9-и оставшихся словах в очереди */
    I2S_InterruptTriggerLevel_10 = 9U,  /*!< Прерывание при 10-и оставшихся словах в очереди */
    I2S_InterruptTriggerLevel_11 = 10U, /*!< Прерывание при 11-и оставшихся словах в очереди */
    I2S_InterruptTriggerLevel_12 = 11U, /*!< Прерывание при 12-и оставшихся словах в очереди */
    I2S_InterruptTriggerLevel_13 = 12U, /*!< Прерывание при 13-и оставшихся словах в очереди */
    I2S_InterruptTriggerLevel_14 = 13U, /*!< Прерывание при 14-и оставшихся словах в очереди */
    I2S_InterruptTriggerLevel_15 = 14U, /*!< Прерывание при 15-и оставшихся словах в очереди */
    I2S_InterruptTriggerLevel_16 = 15U, /*!< Прерывание при 16-и оставшихся словах в очереди */
} i2s_interrupt_level_t;

/*!
 * @brief Структура конфигурации контроллера I2S
 */
typedef struct _i2s_config {
    uint32_t              sample_rate;     /*!< Частота аудиоданных */
    i2s_sclk_per_sample_t sclk_per_sample; /*!< Количество синхроимпульсов sclk на левое/правое значение */
    i2s_sclk_gating_t     sclk_gating;     /*!< Обрезание синхроимпульсов sclk */
    i2s_resolution_t      resolution;      /*!< Разрядность данных */
    i2s_interrupt_level_t interrupt_level; /*!< Уровень опустошения очереди, при котором происходит прерывание */
} i2s_config_t;

/*!
 * @brief Структура для передачи данных I2S в неблокирующем режиме (по
 *        прерыванию)
 */
typedef struct _i2s_transfer {
    uint32_t *left_samples;  /*!< Буфер с данными для передачи в левый поток */
    uint32_t *right_samples; /*!< Буфер с данными для передачи в правый поток */
    size_t   nb_samples;     /*!< Количество значений в буфере */
} i2s_transfer_t;

/*!
 * @brief Сокращённое название типа для структуры обработчика драйвера I2S
 */
typedef struct _i2s_handle i2s_handle_t;

/*!
 * @brief Функция обратного вызова I2S
 *
 * @param base           Базовый адрес контроллера
 * @param handle         Указатель на обработчик
 * @param interrupt_flag Причина вызова
 * @param user_data      Данные пользователя
 */
typedef void (*i2s_transfer_callback_t)(I2S_Type *base, i2s_handle_t *handle,
    i2s_flag_t interrupt_flag, void *user_data);

/*!
 * @brief Структура обработчика драйвера I2S
 */
struct _i2s_handle {
    i2s_transfer_callback_t callback;       /*!< Функция обратного вызова */
    void                    *user_data;     /*!< Данные пользователя */

    uint32_t                *left_samples;  /*!< Буфер с данными для левого потока */
    uint32_t                *right_samples; /*!< Буфер с данными для правого потока */
    size_t                  nb_samples;     /*!< Количество сэмплов */
};

/*!
 * @name Инициализация и деинициализация
 * @{
 */

/*!
 * @brief Инициализация драйвера I2S
 *
 * Функция инициализирует модуль I2S в соответствии с заданными
 * пользовательскими параметрами.
 *
 * @param base            Базовый адрес контроллера
 * @param config          Структура с параметрами конфигурации
 * @param source_clock_hz Опорная частота в Гц
 *
 * @retval #I2S_Status_Ok
 * @retval #I2S_Status_UnsupportedBitRate
 */
i2s_status_t I2S_Init(I2S_Type *base, const i2s_config_t *config,
    uint32_t source_clock_hz);

/*!
 * @brief Деинициализация драйвера I2S
 *
 * Функция деинициализирует контроллер I2S.
 *
 * @param base Базовый адрес контроллера
 */
void I2S_Deinit(I2S_Type *base);

/*!
 * @brief Получение параметров драйвера I2S по умолчанию
 *
 * @param config Структура с параметрами конфигурации
 */
void I2S_GetDefaultConfig(i2s_config_t *config);

/*!
 * @}
 */

/*!
 * @name Управление прерываниями
 * @{
 */

/*!
 * @brief Разрешение прерывания
 *
 * @param base Базовый адрес контроллера
 * @param idx  Номер флага состояния/прерывания
 */
static inline void I2S_EnableInterrupt(I2S_Type *base, i2s_flag_t idx)
{
    base->IMR0 &= ~(1 << idx);
}

/*!
 * @brief Разрешение прерываний по маске
 *
 * @param base Базовый адрес контроллера
 * @param mask Маска флагов прерываний
 */
static inline void I2S_EnableInterruptMask(I2S_Type *base, uint32_t mask)
{
   base->IMR0 &= ~mask;
}

/*!
 * @brief Запрос - разрешено ли прерывание I2S
 *
 * @param base Базовый адрес I2S
 * @param idx  Номер флага состояния/прерывания
 *
 * @retval true  Прерывание разрешено
 * @retval false Прерывание запрещено
 */
static inline bool I2S_IsInterruptEnabled(I2S_Type *base, i2s_flag_t idx)
{
    return !(base->IMR0 & (1 << idx));
}


/*!
 * @brief Запрос маски разрешенных прерываний I2S
 *
 * Запрос маски разрешенных прерываний I2S, единицы в соответствующих
 * разрядах соответствуют включенным прерываниям.
 *
 * @param base Базовый адрес I2S
 *
 * @return Маска разрешенных прерываний
 */
static inline uint32_t I2S_GetEnabledInterruptMask(I2S_Type *base)
{
    return base->IMR0;
}

/*!
 * @brief Запрет прерывания
 *
 * @param base Базовый адрес контроллера
 * @param idx  Номер флага состояния/прерывания
 */
static inline void I2S_DisableInterrupt(I2S_Type *base, i2s_flag_t idx)
{
    base->IMR0 |= (1 << idx);
}

/*!
 * @brief Запрет прерываний по маске
 *
 * @param base Базовый адрес контроллера
 * @param mask Маска флагов прерываний
 */
static inline void I2S_DisableInterruptMask(I2S_Type *base, uint32_t mask)
{
    base->IMR0 |= mask;
}

/*!
 * @brief Получение состояния флага прерывания
 *
 * @param base   Базовый адрес контроллера
 * @param idx    Номер флага состояния/прерывания
 *
 * @retval true  Флаг прерывания установлен
 * @retval false Флаг прерывания сброшен
 */
static inline bool I2S_GetInterruptStatus(I2S_Type *base, i2s_flag_t idx)
{
    return base->ISR0 & (1 << idx);
}

/*!
 * @brief Получение маски активных прерываний
 *
 * @param base Базовый адрес контроллера
 *
 * @return Маска активных прерываний
 */
static inline uint32_t I2S_GetInterruptStatusMask(I2S_Type *base)
{
    return base->ISR0;
}

/*!
 * @brief Сброс флага переполнения очереди выдачи
 *
 * @param base Базовый адрес контроллера
 */
static inline void I2S_ClearDataOverrunFlag(I2S_Type *base)
{
    (void) base->TOR0;
}

/*!
 * @}
 */

/*!
 * @name Прямое управление выдачей
 * @{
 */

/*!
 * @brief Запись сэмпла в левый канал I2S
 *
 * @param base   Базовый адрес контроллера
 * @param sample Значение сэмпла
 */
void I2S_WriteLeftFifo(I2S_Type *base, uint32_t sample);

/*!
 * @brief Запись сэмпла в правый канал I2S
 *
 * @param base   Базовый адрес контроллера
 * @param sample Значение сэмпла
 */
void I2S_WriteRightFifo(I2S_Type *base, uint32_t sample);

/*!
 * @brief Отмена выдачи по I2S
 *
 * @param base Базовый адрес контроллера
 */
static inline void I2S_AbortTx(I2S_Type *base)
{
    base->TXFFR = (1 << I2S_TXFFR_TXFFR_Pos);
}

/*!
 * @brief Включение передатчика I2S
 *
 * @param base I2S Базовый адрес контроллера
 */
static inline void I2S_EnableTx(I2S_Type *base)
{
    SET_VAL_MSK(base->ITER, I2S_ITER_TXEN_Msk, I2S_ITER_TXEN_Pos, 1U);
    SET_VAL_MSK(base->CER, I2S_CER_CLKEN_Msk, I2S_CER_CLKEN_Pos, 1U);
}

/*!
 * @brief Выключение передатчика I2S
 *
 * @param base I2S Базовый адрес контроллера
 */
static inline void I2S_DisableTx(I2S_Type *base)
{
    SET_VAL_MSK(base->CER, I2S_CER_CLKEN_Msk, I2S_CER_CLKEN_Pos, 0U);
    SET_VAL_MSK(base->ITER, I2S_ITER_TXEN_Msk, I2S_ITER_TXEN_Pos, 0U);
}

/*!
 * @}
 */

/*!
 * @name Неблокирующая выдача по I2S
 * @{
 */

/*!
 * @brief Инициализация обработчика событий I2S
 *
 * @param base      Базовый адрес контроллера
 * @param handle    Обработчик
 * @param callback  Функция обратного вызова
 * @param user_data Данные пользователя
 *
 * @retval #I2S_Status_Ok
 * @retval #I2S_Status_Fail
 */
i2s_status_t I2S_TransferCreateHandle(I2S_Type *base, i2s_handle_t *handle,
    i2s_transfer_callback_t callback, void *user_data);

/*!
 * @brief Неблокирующая выдача через высокоприоритетный буфер
 *
 * Функция выдает кадр I2S по прерыванию. Это неблокирующая функция, она
 * возвращает управление сразу. По выдачи кадра в шину вызывает функция
 * обратного вызова.
 *
 * @param base   Базовый адрес контроллера
 * @param handle Обработчик
 * @param xfer   Структура для передачи кадра в неблокирующем режиме
 *
 * @retval #I2S_Status_Ok
 * @retval #I2S_Status_Fail
 * @retval #I2S_Status_TxBusy
 */
i2s_status_t I2S_TransferNonBlocking(I2S_Type *base,
    i2s_handle_t *handle, i2s_transfer_t *xfer);

/*!
 * @brief Отмена выдачи из высокоприоритетного буфера
 *
 * @param base   Базовый адрес контроллера
 * @param handle Обработчик
 */
void I2S_TransferAbort(I2S_Type *base, i2s_handle_t *handle);

/*!
 * @brief Установка обработчика на прерывания от I2S, не связанных с
 *        приемом/выдачей
 *
 * @param base   Базовый адрес контроллера
 * @param handle Обработчик
 */
void I2S_TransferHandleIRQ(I2S_Type *base, i2s_handle_t *handle);

/*!
 * @}
 */

#ifdef __cplusplus
}
#endif

#endif /* HAL_I2S_H */

/*!
 * @}
 */
