/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @addtogroup I2C Driver
 * @{
 */

/*!
 * @file hal_i2c.c
 *
 * @brief Имплементация драйвера модуля I2C.
 */

#include "hal_i2c.h"
#include "hal_ioim.h"

#define I2C_UNITS_NUM 2 /*!< TODO */
#define I2C_UNITS_BASE_ADDR { I2C0_BASE, I2C1_BASE } /*!< TODO */

#define SECURE_BIT_MSK 0x10000000

#define SYSWAIT(x)  ;  /*!< При использовании драйвера I2C с OS сюда необходимо поместить
                            системную функцию ожидания со временем равным длительности 10
                            тактам частоты SCL, при работе в baremetal нужную задержку можно
                            реализовать через I2C_RETRY_TIMES_FOR_DISABLE_UNITS */

#define I2C_RETRY_TIMES_FOR_DISABLE_UNITS 1000 /*!< Количество циклов опроса состояния модуля I2C при ожидании отключении */

#define I2C_FIFO_DEPTH                    8    /*!< Глубина Fifo */

#define I2C_SLAVE_ADDR_DEFAULT            0x12  /*!< Адрес модуля в Slave-режиме, который будет присвоен по умолчанию   */
#define I2C_SLAVE_DUMMY_DATA              0x00  /* Фиктивные данные*/


/*!
 * @brief Флаги в функции I2C_PendingStatus()
 */
enum i2c_pending_status_flags {
    I2C_PendStat_Active          = I2C_Stat_Active,                /*!< Ожидать пока на шине активность */
    I2C_PendStat_NotActive       = I2C_Stat_Active << 8,           /*!< Ожидать пока на шине НЕТ активности */

    I2C_PendStat_TxFifo_Full     = I2C_Stat_TxFifo_NotFull << 8,   /*!< Ожидать пока TxFifo полон */
    I2C_PendStat_TxFifo_NotFull  = I2C_Stat_TxFifo_NotFull,        /*!< Ожидать пока TxFifo НЕ полон */

    I2C_PendStat_TxFifo_Empty    = I2C_Stat_TxFifo_Empty,          /*!< Ожидать пока TxFifo пуст */
    I2C_PendStat_TxFifo_NotEmpty = I2C_Stat_TxFifo_Empty << 8,     /*!< Ожидать пока TxFifo НЕ пуст */

    I2C_PendStat_RxFifo_Empty    = I2C_Stat_RxFifo_NotEmpty << 8,  /*!< Ожидать пока RxFifo пуст */
    I2C_PendStat_RxFifo_NotEmpty = I2C_Stat_RxFifo_NotEmpty,       /*!< Ожидать пока RxFifo НЕ пуст */

    I2C_PendStat_RxFifo_Full     = I2C_Stat_RxFifo_Full,           /*!< Ожидать пока RxFifo полон */
    I2C_PendStat_RxFifo_NotFull  = I2C_Stat_RxFifo_Full << 8,      /*!< Ожидать пока RxFifo НЕ полон */

    I2C_PendStat_Master_Active   = I2C_Stat_Master_Active,         /*!< Ожидать пока ЕСТЬ активность в режиме Master */
    I2C_PendStat_Master_NotActive = I2C_Stat_Master_Active << 8,   /*!< Ожидать пока НЕТ активность в режиме Master */

    I2C_PendStat_Slave_Active    = I2C_Stat_Slave_Active,          /*!< Ожидать пока ЕСТЬ активность в режиме Slave */
    I2C_PendStat_Slave_NotActive = I2C_Stat_Slave_Active << 8,     /*!< Ожидать пока НЕТ активность в режиме Slave */
};



i2c_speed_mode_t CalculateSpeedMode(uint32_t baudrate)
{
    i2c_speed_mode_t speed_mode;

    if (baudrate <= 100000) {        /* Standard-speed режим (0 to 100 Кб/с) */
        speed_mode = I2C_StandardSpeedMode;
    } else if (baudrate <= 400000) { /* Fast-speed режиме (≤ 400 Кб/с) */
        speed_mode = I2C_FastSpeedMode;
    } else if (baudrate <= 3400000) { /* High-speed режим (≤ 3.4 Мб/с) */
        speed_mode = I2C_HighSpeedMode;
    } else {
        speed_mode = I2C_UndefinedSpeedMode;
    }

    return speed_mode;
}

/*! @brief Массив базовых адресов I2C модулей */
static const uint32_t i2c_base_addrs[I2C_UNITS_NUM] = I2C_UNITS_BASE_ADDR;


i2c_status_t I2C_Enable(I2C_Type *base, bool enable)
{
    /* Требуемый режим уже установлен. */
    if (I2C_IsEnable(base) == enable)
        return I2C_Status_Ok;

    if (enable) { /* Включает модуль I2C. */

        SET_VAL_MSK(base->IC_ENABLE, I2C_IC_ENABLE_ENABLE_Msk,
            I2C_IC_ENABLE_ENABLE_Pos, 1U);
    } else { /* Отключает модуль I2C. */

        SET_VAL_MSK(base->IC_ENABLE, I2C_IC_ENABLE_ENABLE_Msk,
            I2C_IC_ENABLE_ENABLE_Pos, 0U);
        {
            volatile uint32_t count = I2C_RETRY_TIMES_FOR_DISABLE_UNITS;

            while (I2C_IsEnable(base) && (--count != 0U)) {
                SYSWAIT();
            }

            if (count == 0)
                return I2C_Status_HwError;
        }
    }
    return I2C_Status_Ok;
}

i2c_status_t I2C_Reset(I2C_Type *base)
{
    i2c_status_t ret;

    /* Если модуль в отключенном состоянии, то ничего не делаем. */
    if (I2C_IsEnable(base) == false)
        return I2C_Status_Ok;

    ret = I2C_Enable(base, false);
    if (ret != I2C_Status_Ok)
        return ret;

    return I2C_Enable(base, true);
}

void I2C_RxFifoClear(I2C_Type *base)
{
    uint32_t i = I2C_FIFO_DEPTH;

    while (GET_VAL_MSK(base->IC_RXFLR, I2C_IC_RXFLR_RXFLR_Msk,
            I2C_IC_RXFLR_RXFLR_Pos) != 0U) {

        volatile uint32_t temp;
        UNUSED(temp);
        temp = GET_VAL_MSK(base->IC_DATA_CMD, I2C_IC_DATA_CMD_DAT_Msk,
                I2C_IC_DATA_CMD_DAT_Pos);

        i--;
        if (0 == i) {
            break;
        }
    }
}

uint32_t I2C_GetInstance(I2C_Type *base)
{
    uint32_t i;

    for (i = 0; i < (uint32_t) I2C_UNITS_NUM; i++) {
        if (((uint32_t) base & ~SECURE_BIT_MSK) == i2c_base_addrs[i]) {

            break;
        }
    }
    assert(i < (uint32_t)I2C_UNITS_NUM);
    return i;
}


void I2C_MasterGetDefaultConfig(i2c_master_config_t *master_config)
{
    (void)memset(master_config, 0, sizeof(i2c_master_config_t));

    master_config->enable_master = true;
    master_config->baudrate_bps  = 100000U;
    master_config->sda_setup     = 2;
    master_config->sda_hold      = 10;
}

i2c_status_t I2C_MasterInit(I2C_Type *base,
    const i2c_master_config_t *master_config, uint32_t src_clock_hz)
{
    i2c_status_t ret = I2C_Status_Ok;

    /* Отключение модуля I2C. */
    ret = I2C_Enable(base, false);

    if (ret != I2C_Status_Ok)
        return ret;

    /* Установка регистра кофигурации модуля I2C. */
    {
        uint32_t reg = 0;

        /* Включение Master-режима. */
        SET_VAL_MSK(reg, I2C_IC_CON_MASTER_MODE_Msk,
            I2C_IC_CON_MASTER_MODE_Pos, 1U);

        /* Установка режима скорости. */
        SET_VAL_MSK(reg, I2C_IC_CON_SPEED_Msk, I2C_IC_CON_SPEED_Pos,
            CalculateSpeedMode(master_config->baudrate_bps));

        /* Модуль может отправить RESTART. */
        SET_VAL_MSK(reg, I2C_IC_CON_IC_RESTART_EN_Msk,
            I2C_IC_CON_IC_RESTART_EN_Pos, 1U);

        /* Выключение Slave-режима. */
        SET_VAL_MSK(reg, I2C_IC_CON_IC_SLAVE_DISABLE_Msk,
            I2C_IC_CON_IC_SLAVE_DISABLE_Pos, 1U);

        base->IC_CON = reg;
    }

    /* Количество тактов удерживания SDA после переднего фронта SCL. */
    SET_VAL_MSK(base->IC_SDA_SETUP, I2C_IC_SDA_SETUP_SDA_SETUP_Msk,
        I2C_IC_SDA_SETUP_SDA_SETUP_Pos,  master_config->sda_setup);

    /* Количество тактов удерживания SDA после заднего фронта SCL. */
    SET_VAL_MSK(base->IC_SDA_HOLD, I2C_IC_SDA_HOLD_IC_SDA_TX_HOLD_Msk,
        I2C_IC_SDA_HOLD_IC_SDA_TX_HOLD_Pos,  master_config->sda_hold);

    /* Установка частоты шины для модуля I2C Master-режима. */
    I2C_MasterSetBaudRate(base, master_config->baudrate_bps, src_clock_hz);

    /* Включать или нет модуль I2C. */
    return I2C_Enable(base, master_config->enable_master);
}


i2c_status_t I2C_MasterDeinit(I2C_Type *base)
{
    i2c_status_t ret = I2C_Status_Ok;

    /* Отключение модуля. */
    ret = I2C_Enable(base, false);

    return ret;
}

i2c_status_t I2C_MasterSetBaudRate(I2C_Type *base, uint32_t baudrate_bps,
    uint32_t src_clock_hz)
{
#define FILTER_CONDITION    7

    uint32_t         p2, p4;
    i2c_speed_mode_t speed_mode;
    bool             i2c_module_enable;
    i2c_status_t     ret = I2C_Status_Ok;

    /*
     *
     *          SCL_Time_Hi   +  SCL_Time_Low;
     *       _________________                   _________________
     *      |                 |                 |                 |
     *      |  HCNT+SPKLEN+7  |                 |                 |
     *      |                 |      LCNT+1     |                 |
     * SCL _|                 |_________________|                 |________
     *                        .                 .
     *                        .         ________.__________
     *                        .        |        .          |
     *                        .        |        .          |
     *                        .        |        .          |
     * SDA ___________________.________|        .          |_______________
     *                        .        .        .          .
     *                        .        .        .          .
     *                        .SDA_HOLD.        .SDA_SETUP .
     *                        .        .        .          .
     *
     *    В режиме SS и FS:
     *        ■ Значения регистров IC_SS_SCL_LCNT и IC_FS_SCL_LCNT должны быть больше, чем IC_FS_SPKLEN + 7.
     *        ■ Значения регистров IC_SS_SCL_HCNT и IC_FS_SCL_HCNT должны быть больше, чем IC_FS_SPKLEN + 5.
     *
     *    В режиме HS:
     *        ■ значение регистра IC_HS_SCL_LCNT должно быть больше чем IC_HS_SPKLEN + 7.
     *        ■ значение регистра IC_HS_SCL_HCNT должно быть больше чем IC_HS_SPKLEN + 5.
     */

    /* p2 содержит 1/2 периода в тактах. */
    p2 = ((uint32_t)(src_clock_hz / baudrate_bps)) >> 1;

    /* p4 содержит 1/4 периода в тактах. */
    p4 = p2 >> 1;

    /* Вычисление режима работы модуля I2C Master. */
    speed_mode = CalculateSpeedMode(baudrate_bps);

    if (speed_mode == I2C_UndefinedSpeedMode)
        return I2C_Status_InvalidParameter;

    /* Текущее состояние модуля I2C. */
    i2c_module_enable = I2C_IsEnable(base);

    /* Отключение модуля I2C. */
    ret = I2C_Enable(base, false);
    if (ret != I2C_Status_Ok)
        return ret;

    /* Установка режима скорости. */
    SET_VAL_MSK(base->IC_CON, I2C_IC_CON_SPEED_Msk,
        I2C_IC_CON_SPEED_Pos, speed_mode);

    /* Настройка регистров частоты и фильтра помех .*/
    switch (speed_mode) {

        case I2C_StandardSpeedMode:
            /* Длительность четверти периода больше максимально-
             * возможной длительноси задержки фильтра помех?
             */
            if (p4 > I2C_IC_FS_SPKLEN_IC_FS_SPKLEN_Msk + FILTER_CONDITION) {

                /* Максимальная степень фильрации */
                SET_VAL_MSK(base->IC_FS_SPKLEN,
                    I2C_IC_FS_SPKLEN_IC_FS_SPKLEN_Msk,
                    I2C_IC_FS_SPKLEN_IC_FS_SPKLEN_Pos,
                    I2C_IC_FS_SPKLEN_IC_FS_SPKLEN_Msk);

                /* (1/2 Hi-периода) - (длительность фильтрации) - FILTER_CONDITION */
                SET_VAL_MSK(base->IC_SS_SCL_HCNT,
                    I2C_IC_SS_SCL_HCNT_IC_SS_SCL_HCNT_Msk,
                    I2C_IC_SS_SCL_HCNT_IC_SS_SCL_HCNT_Pos,
                    p2 - I2C_IC_FS_SPKLEN_IC_FS_SPKLEN_Msk - FILTER_CONDITION);

            } else {

                /* 1/4 периода - FILTER_CONDITION отдается под фильтр */
                SET_VAL_MSK(base->IC_FS_SPKLEN,
                    I2C_IC_FS_SPKLEN_IC_FS_SPKLEN_Msk,
                    I2C_IC_FS_SPKLEN_IC_FS_SPKLEN_Pos, p4 - FILTER_CONDITION);

                /* (1/4 Hi-периода) */
                SET_VAL_MSK(base->IC_SS_SCL_HCNT,
                    I2C_IC_SS_SCL_HCNT_IC_SS_SCL_HCNT_Msk,
                    I2C_IC_SS_SCL_HCNT_IC_SS_SCL_HCNT_Pos, p4);
            }

            /* (пол Low-периода) - 1 */
            SET_VAL_MSK(base->IC_SS_SCL_LCNT,
                I2C_IC_SS_SCL_LCNT_IC_SS_SCL_LCNT_Msk,
                I2C_IC_SS_SCL_LCNT_IC_SS_SCL_LCNT_Pos, p2 - 1);
            break;

        case I2C_FastSpeedMode:
            /* Длительность четверти периода больше максимально-
             * возможной длительности задержки фильтра помех?
             */
            if (p4 > I2C_IC_FS_SPKLEN_IC_FS_SPKLEN_Msk + FILTER_CONDITION) {

                /* Максимальная степень фильрации. */
                SET_VAL_MSK(base->IC_FS_SPKLEN,
                    I2C_IC_FS_SPKLEN_IC_FS_SPKLEN_Msk,
                    I2C_IC_FS_SPKLEN_IC_FS_SPKLEN_Pos,
                    I2C_IC_FS_SPKLEN_IC_FS_SPKLEN_Msk);

                /* (1/2 Hi-периода) - (длительность фильтрации) - FILTER_CONDITION */
                SET_VAL_MSK(base->IC_FS_SCL_HCNT,
                    I2C_IC_FS_SCL_HCNT_IC_FS_SCL_HCNT_Msk,
                    I2C_IC_FS_SCL_HCNT_IC_FS_SCL_HCNT_Pos,
                    p2 - I2C_IC_FS_SPKLEN_IC_FS_SPKLEN_Msk - FILTER_CONDITION);

            } else {

                /* 1/4 периода - FILTER_CONDITION отдается под фильтр. */
                SET_VAL_MSK(base->IC_FS_SPKLEN,
                    I2C_IC_FS_SPKLEN_IC_FS_SPKLEN_Msk,
                    I2C_IC_FS_SPKLEN_IC_FS_SPKLEN_Pos, p4 - FILTER_CONDITION);

                /* (1/4 Hi-периода)  */
                SET_VAL_MSK(base->IC_FS_SCL_HCNT,
                    I2C_IC_FS_SCL_HCNT_IC_FS_SCL_HCNT_Msk,
                    I2C_IC_FS_SCL_HCNT_IC_FS_SCL_HCNT_Pos, p4);
            }

            /* (пол Low-периода) - 1 */
            SET_VAL_MSK(base->IC_FS_SCL_LCNT,
                I2C_IC_FS_SCL_LCNT_IC_FS_SCL_LCNT_Msk,
                I2C_IC_FS_SCL_LCNT_IC_FS_SCL_LCNT_Pos, p2 - 1);
            break;

        case I2C_HighSpeedMode:
            /* Длительность четверти периода больше максимально-
             * возможной длительности задержки фильтра помех?
             */
            if (p4 > I2C_IC_HS_SPKLEN_IC_HS_SPKLEN_Msk + FILTER_CONDITION) {

                /* Максимальная степень фильрации. */
                SET_VAL_MSK(base->IC_HS_SPKLEN,
                    I2C_IC_HS_SPKLEN_IC_HS_SPKLEN_Msk,
                    I2C_IC_HS_SPKLEN_IC_HS_SPKLEN_Pos,
                    I2C_IC_HS_SPKLEN_IC_HS_SPKLEN_Msk);

                /* (1/2 Hi-периода) - (длительность фильтрации) - FILTER_CONDITION */
                SET_VAL_MSK(base->IC_HS_SCL_HCNT,
                    I2C_IC_HS_SCL_HCNT_IC_HS_SCL_HCNT_Msk,
                    I2C_IC_HS_SCL_HCNT_IC_HS_SCL_HCNT_Pos,
                    p2 - I2C_IC_HS_SPKLEN_IC_HS_SPKLEN_Msk - FILTER_CONDITION);

            } else {

                /* 1/4 периода - FILTER_CONDITION отдается под фильтр. */
                SET_VAL_MSK(base->IC_HS_SPKLEN,
                    I2C_IC_HS_SPKLEN_IC_HS_SPKLEN_Msk,
                    I2C_IC_HS_SPKLEN_IC_HS_SPKLEN_Pos, p4 - FILTER_CONDITION);

                /* (1/4 Hi-периода) */
                SET_VAL_MSK(base->IC_HS_SCL_HCNT,
                    I2C_IC_HS_SCL_HCNT_IC_HS_SCL_HCNT_Msk,
                    I2C_IC_HS_SCL_HCNT_IC_HS_SCL_HCNT_Pos, p4);
            }

            /* (пол Low-периода) - 1 */
            SET_VAL_MSK(base->IC_HS_SCL_LCNT,
                I2C_IC_HS_SCL_LCNT_IC_HS_SCL_LCNT_Msk,
                I2C_IC_HS_SCL_LCNT_IC_HS_SCL_LCNT_Pos, p2 - 1);
            break;

        case I2C_UndefinedSpeedMode:
            break;


    }

    return I2C_Enable(base, i2c_module_enable);
}



/*! 
 * @brief Разбор статусов регистра обрыва передачи
 *
 * @param base Базовый адрес модуля I2C

 * @retval #I2C_Status_Ok
 * @retval #I2C_Status_ArbitrationLost
 * @retval #I2C_Status_Timeout
 * @retval #I2C_Status_UserError
 * @retval #I2C_Status_AddrNack
 * @retval #I2C_Status_Nack
 * @retval #I2C_Status_BreakTransfer
 * @retval #I2C_Status_UnexpectedState
 * @retval #I2C_Status_SetStartError
 * @retval #I2C_Status_HsCodeError
 */
static i2c_status_t I2C_TxAbortSource_To_Status(I2C_Type *base)
{
    /* Произошел обрыв передачи. */

    uint32_t reg = 0;
    reg = base->IC_TX_ABRT_SOURCE;


    /* В TxFifo есть непереданные данные? */
    if ((reg & I2C_Abort_Tx_FlushCnt) != 0) {
        i2c_status_t ret;

        /* Сброс счетчика непереданных данных. */
        ret = I2C_Reset(base);
        if (ret != I2C_Status_Ok) {
            return ret;
        }
    }

    /* Master Rx попытка чтения с адреса General Call */
    if (((reg & I2C_Abort_HS_RStart_Dis) != 0)

        /* Master в HS режиме пытается отправить [RStart] условие, но возможность отключена. */
        || ((reg & I2C_Abort_GenCall_Read) != 0)

        /* Master пытается отправить [RStart] условие, но возможность отключена. */
        || ((reg & I2C_Abort_RStart_Dis) != 0)

        /*
         * Master пытается осуществить чтение режиме
         * 10 битной адресации, но возможность отключена.
         */
        || ((reg & I2C_Abort_Read_RStart_Dis) != 0)

        /*!< Попытка инициализировать Master-обмен при выключенном Master-режиме. */
        || ((reg & I2C_Abort_Master_Dis) != 0)

        /* У Slave есть запрос на передачу данных удаленному Master,   
         * но пользователь пытается произвести чтение в Master-режиме 
         * (пишет 1 в IC_DATA_CMD.CMD).
         */
        || ((reg & I2C_Abort_SlaveDataCmd_Error) != 0)
    ) {
        /* Некорректная работа с модулем I2C */
        return I2C_Status_UserError;
    }

    /* Master Tx нет ответа после передачи адреса Slave. */
    if (((reg & I2C_Abort_7B_Addr_Nack)   != 0)  ||
        ((reg & I2C_Abort_10B_Addr1_Nack) != 0)  ||
        ((reg & I2C_Abort_10B_Addr2_Nack) != 0)  ||
        ((reg & I2C_Abort_GenCall_Nack)   != 0)) {
        /* Slave устройство отправило NACK в ответ на адреса. */
        return I2C_Status_AddrNack;
    }

    /* Master Tx не получил ACK от Slave после отправки байта данных. */
    if ((reg & I2C_Abort_TxData_Nack) != 0) {
        /* Slave устройство отправило NACK в ответ на байт. */
        return I2C_Status_Nack;
    }

    /* Master/Slave-передатчик проигрывает арбитраж. */
    if (((reg & I2C_Abort_Arbitr_Lost) != 0)
        /* Slave теряет шину во время передачи данных. */
        || ((reg & I2C_Abort_SlaveArbitr_Lost) != 0)) {
        /* Потеря арбитража. */
        return I2C_Status_ArbitrationLost;
    }

    /* Master определил обрыв передачи. */
    if (((reg & I2C_Abort_MasterDetect) != 0)
        /* В TxFifo есть непереданные данные. */
        || ((reg & I2C_Abort_Tx_FlushCnt) != 0)) {
        /* Сброс флага: TxFifo есть непереданные данные. */
        i2c_status_t ret;
        ret = I2C_Reset(base);
        if (ret != I2C_Status_Ok) {
            return ret;
        }
        return I2C_Status_BreakTransfer;
    }

    /* Slave получил запрос на чтение, но в RxFifo уже есть данные. */
    if ((reg & I2C_Abort_RxFifo_NotEmpty) != 0) {
        return I2C_Status_UnexpectedState;
    }

    /* Master получил подтверждение на [Start] условие. */
    if ((reg & I2C_Abort_StartByte_Ack) != 0) {
        return I2C_Status_SetStartError;
    }

    /* Master в HS режиме получил подтверждение на HS code. */
    if ((reg & I2C_Abort_HsCode_Ack) != 0) {
        return I2C_Status_HsCodeError;
    }

    return I2C_Status_Ok;
}




/*!
 * @brief Ожидание выставления бита готовности и проверка ошибки шины
 *
 * @param base       Базовый адрес модуля I2C
 * @param flags_wait Флаг или флаги из @ref i2c_pending_status_flags установку которых
 *                   будет ожидать функция с возможность обьединения по ИЛИ:
 *
 *                   I2C_PendStat_Active           Ожидать пока на шине активность
 *                   I2C_PendStat_NotActive        Ожидать пока на шине НЕТ активности
 *
 *                   I2C_PendStat_TxFifo_Full      Ожидать пока TxFifo полон
 *                   I2C_PendStat_TxFifo_NotFull   Ожидать пока TxFifo НЕ полон
 *
 *                   I2C_PendStat_TxFifo_Empty     Ожидать пока TxFifo пуст
 *                   I2C_PendStat_TxFifo_NotEmpty  Oжидать пока TxFifo НЕ пуст
 *
 *                   I2C_PendStat_RxFifo_Empty     Ожидать пока RxFifo пуст
 *                   I2C_PendStat_RxFifo_NotEmpty  Ожидать пока RxFifo НЕ пуст
 *
 *                   I2C_PendStat_RxFifo_Full      Ожидать пока RxFifo полон
 *                   I2C_PendStat_RxFifo_NotFull   Ожидать пока RxFifo НЕ полон
 *
 *                   I2C_PendStat_Master_Active    Ожидать пока ЕСТЬ активность в Master-режиме
 *                   I2C_PendStat_Master_NotActive Ожидать пока НЕТ активность в Master-режиме
 *
 *                   I2C_PendStat_Slave_Active     Ожидать пока ЕСТЬ активность в Slave-режиме
 *                   I2C_PendStat_Slave_NotActive  Ожидать пока НЕТ активность в Slave-режиме
 *
 * @retval  #I2C_Status_Ok
 * @retval  #I2C_Status_ArbitrationLost
 * @retval  #I2C_Status_Timeout
 * @retval  #I2C_Status_UserError
 * @retval  #I2C_Status_Addr_Nak
 * @retval  #I2C_Status_Nak
 * @retval  #I2C_Status_BreakTransfer
 * @retval  #I2C_Status_UnexpectedState
 * @retval  #I2C_Status_SetStartError
 * @retval  #I2C_Status_HsCodeError
 */
static i2c_status_t I2C_PendingStatus(I2C_Type *base, uint32_t flags_wait)
{
    uint32_t volatile temp = 0;
    uint32_t status;
    uint32_t filter = 0xFF & (flags_wait >> 8);
    UNUSED(temp);
    flags_wait = (0xFF & flags_wait) | filter;


#if I2C_RETRY_TIMES != 0U
    uint32_t waitTimes = I2C_RETRY_TIMES;
#endif

    do { /* Ожидание выставления флага готовности статуса или таймаута. */

        status = base->IC_STATUS ^ filter;

        /*
         *  Аппаратно не реализовано
         *  if(status & тайм-ауту_на_шине){
         *      Очистка флага тайм-ауту_на_шине;
         *      result = I2C_Status_Timeout;
         *  }
         */

#if defined(I2C_TIMEOUT_RECOVERY) && I2C_TIMEOUT_RECOVERY
        /*
         *  Аппаратно не реализовано
         *  if( result != I2C_Status_Ok ){
         *      Восстановление работы модуля i2c после таймаута на шине;
         *      if (I2C_Reset(base)!=I2C_Status_Ok)
         *          return I2C_Status_HwError;
         *      break;
         *  }
         */
#endif

#if I2C_RETRY_TIMES != 0U
        waitTimes--;

        /* Ожидание выставления флага готовности статуса или таймаута. */
        /* I2C_Stat_Active Статус активности шины (Master или Slave) */
    } while ((flags_wait == (status & flags_wait)) && (waitTimes != 0U));

    if (waitTimes == 0U) {
#if defined(I2C_TIMEOUT_RECOVERY) && I2C_TIMEOUT_RECOVERY

        /* Восстановление работы модуля после таймаута на шине. */
        if (I2C_Reset(base) != I2C_Status_Ok)
            return I2C_Status_HwError;

#endif
        return I2C_Status_Timeout;
    }
#else
        /* Ожидание выставления флага готовности статуса.  */
    }
    while (flags_wait == (status & flags_wait));
#endif

    /* Получение значения флага: обрыв передачи. */
    if (GET_VAL_MSK(base->IC_RAW_INTR_STAT,
            I2C_IC_RAW_INTR_STAT_TX_ABRT_Msk,
            I2C_IC_RAW_INTR_STAT_TX_ABRT_Pos) == 1U)
    {
        /* Произошел обрыв передачи. */

        i2c_status_t ret;
        ret = I2C_TxAbortSource_To_Status(base);

        /* Сброс флага:  обрыв  передачи. */
        temp = GET_VAL_MSK(base->IC_CLR_TX_ABRT,
                I2C_IC_CLR_TX_ABRT_CLR_TX_ABRT_Msk,
                I2C_IC_CLR_TX_ABRT_CLR_TX_ABRT_Pos);

        return ret;
    }

    return I2C_Status_Ok;
}

/* Master: Блокирующий обмен. Уровень интерфейса: I */

i2c_status_t I2C_MasterAddrSet(I2C_Type *base, uint32_t address,
    i2c_addr_size_t addr_size)
{
    uint32_t reg = 0;
    i2c_status_t ret = I2C_Status_Ok;
    bool i2c_module_enable;

    /* В случае если настройки Slave-адреса и размер Slave-адреса не изменились,
     * то не переинициализировать модуль.
     */
    reg =  base->IC_TAR;
    if ((GET_VAL_MSK(reg, I2C_IC_TAR_IC_TAR_Msk, I2C_IC_TAR_IC_TAR_Pos)
            == address)
        && (GET_VAL_MSK(reg, I2C_IC_TAR_RSVD_IC_10BITADDR_MASTER_Msk,
                I2C_IC_TAR_RSVD_IC_10BITADDR_MASTER_Pos) == addr_size)) {
        return I2C_Status_Ok;
    }

    /* Ожидание завершения всех операций. */
    ret = I2C_PendingStatus(base, I2C_PendStat_Active);
    if (ret != I2C_Status_Ok) {
        return ret;
    }

    /* Текущее состояние модуля I2C. */
    i2c_module_enable = I2C_IsEnable(base);

    /* Отключение модуля I2C. */
    ret = I2C_Enable(base, false);
    if (ret != I2C_Status_Ok)
        return ret;

    /* Адрес Slave-устройства. */
    SET_VAL_MSK(reg, I2C_IC_TAR_IC_TAR_Msk, I2C_IC_TAR_IC_TAR_Pos, address);

    if (address == 0U) {
        /* Общий вызов (General Call). */
        SET_VAL_MSK(reg, I2C_IC_TAR_GC_OR_START_Msk,
            I2C_IC_TAR_GC_OR_START_Pos, 0U);

        SET_VAL_MSK(reg, I2C_IC_TAR_SPECIAL_Msk,
            I2C_IC_TAR_SPECIAL_Pos, 1U);
    } else {
        /* Обычная адресация. */
        SET_VAL_MSK(reg, I2C_IC_TAR_GC_OR_START_Msk,
            I2C_IC_TAR_GC_OR_START_Pos, 1U);

        SET_VAL_MSK(reg, I2C_IC_TAR_SPECIAL_Msk,
            I2C_IC_TAR_SPECIAL_Pos, 0U);
    }

    /* Настройка режима адресации: 7 бит или 10 бит. */
    SET_VAL_MSK(reg, I2C_IC_TAR_RSVD_IC_10BITADDR_MASTER_Msk,
        I2C_IC_TAR_RSVD_IC_10BITADDR_MASTER_Pos, addr_size);

    base->IC_TAR = reg;

    /* Возврат состояния модуля I2C к исходному. */
    return I2C_Enable(base, i2c_module_enable);
}


i2c_status_t I2C_MasterWriteBlocking(I2C_Type *base, const void *tx_buff,
    size_t tx_size, uint32_t flags)
{
    i2c_status_t ret = I2C_Status_Ok;
    bool first_byte = true;
    const uint8_t *buf = (const uint8_t *)tx_buff;

    assert(tx_buff != NULL);

    /* Если I2C_TransferStartFlag отработка Start-условия. */
    if (flags & I2C_TransferStartFlag) {
        /* Завершение выполнения предыдущей операций обмена. */
        ret = I2C_PendingStatus(base, I2C_PendStat_Active);

        if (ret != I2C_Status_Ok) {
            return ret;
        }
    }

    /* Цикл передачи данных. */
    while (tx_size != 0U) { /* Пока есть что передавать. */
        /* Ожидание освобождения места в TxFifo. */
        ret = I2C_PendingStatus(base, I2C_PendStat_TxFifo_Full);

        if (ret != I2C_Status_Ok) {
            return ret;
        }

        /* Работа с регистром: IC_DATA_CMD. */
        {
            uint32_t reg = 0;

            /* Передаваемые данные. */
            SET_VAL_MSK(reg, I2C_IC_DATA_CMD_DAT_Msk,
                I2C_IC_DATA_CMD_DAT_Pos, *buf++);

            /* Направление передачи: 0 - Запись. */
            SET_VAL_MSK(reg, I2C_IC_DATA_CMD_CMD_Msk,
                I2C_IC_DATA_CMD_CMD_Pos, I2C_Write);

            if ((tx_size == 1) && (flags & I2C_TransferStopFlag)) {
                /* Выдавать ли STOP после текущего байта: 1 - СТОП выдавать. */
                SET_VAL_MSK(reg, I2C_IC_DATA_CMD_RSVD_STOP_Msk,
                    I2C_IC_DATA_CMD_RSVD_STOP_Pos, 1);

            } else {
                SET_VAL_MSK(reg, I2C_IC_DATA_CMD_RSVD_STOP_Msk,
                    I2C_IC_DATA_CMD_RSVD_STOP_Pos, 0);
            }

            /* Выдается ли RESTART? */
            if ((first_byte) && (flags & I2C_TransferReStartFlag)) {
                first_byte = false;
                SET_VAL_MSK(reg, I2C_IC_DATA_CMD_RSVD_RESTART_Msk,
                    I2C_IC_DATA_CMD_RSVD_RESTART_Pos, 1);
            } else {
                SET_VAL_MSK(reg, I2C_IC_DATA_CMD_RSVD_RESTART_Msk,
                    I2C_IC_DATA_CMD_RSVD_RESTART_Pos, 0);
            }

            base->IC_DATA_CMD = reg;
            tx_size--;
        }
    }

    /* Если I2C_TransferStopFlag, то выполняется отработка Stop-условия. */
    if (flags & I2C_TransferStopFlag) {
        /* Завершение выполнения операций. */
        ret = I2C_PendingStatus(base, I2C_PendStat_Active);
    } else {
        ret = I2C_PendingStatus(base, I2C_PendStat_TxFifo_NotEmpty);
    }

    /* Игнорировать NACK на последний байт от Slave при записи данных. */
#if I2C_MASTER_TRANSMIT_IGNORE_LAST_NACK
    if (ret == I2C_Status_Nak) {
        ret = I2C_Status_Ok;
    }
#endif

    return ret;
}


i2c_status_t I2C_MasterReadBlocking(I2C_Type *base, void *rx_buff,
    size_t rx_size, uint32_t flags)
{
#if I2C_RETRY_TIMES != 0U
    uint32_t wait_times = I2C_RETRY_TIMES;
#endif
    i2c_status_t ret = I2C_Status_Ok;
    bool first_byte = true;
    uint8_t *buf = (uint8_t *)(rx_buff);
    size_t tx_size = rx_size;

    assert(rx_buff != NULL);

    /* Завершение выполнения предыдущей операции обмена. */
    if (0U != (flags & I2C_TransferStartFlag)) {
        ret = I2C_PendingStatus(base, I2C_PendStat_Active);
        /* Игнорировать NACK на последний байт от Slave */
#if I2C_MASTER_TRANSMIT_IGNORE_LAST_NACK
        if (ret == I2C_Status_Nak) {
            ret = I2C_Status_Ok;
        }
#endif

        if (ret != I2C_Status_Ok) {
            return ret;
        }
    }

    /* Очистка RxFifo. */
    {
        uint8_t depth_fifo = I2C_FIFO_DEPTH;
        /* Пока RxFifo не пуст, но не более размера Fifo. */
        while ((base->IC_STATUS & (uint32_t)I2C_Stat_RxFifo_NotEmpty)
            && (0U != depth_fifo)) {
            volatile uint32_t temp;
            temp = base->IC_DATA_CMD;
            depth_fifo--;
            UNUSED(temp);
        }

        if (0U == depth_fifo)
            return I2C_Status_HwError;
    }


    /* Цикл приема данных. */
    while (0U != rx_size) { /* Пока есть что принимать. */

        /* Ожидание пока TxFifo Full И RxFifo Empty. */
        ret = I2C_PendingStatus(base, (uint32_t)I2C_PendStat_TxFifo_Full | (uint32_t)I2C_PendStat_RxFifo_Empty);
        if (ret != I2C_Status_Ok) {
            return ret;
        }

        /* Запись в TxFifo, если есть место, и не все запросы на прием сделаны. */
        if (0U != (base->IC_STATUS & I2C_Stat_TxFifo_NotFull)
            && (0U != tx_size)) {

            uint32_t reg = 0;

            /* Направление передачи: 1 - чтение.*/
            SET_VAL_MSK(reg, I2C_IC_DATA_CMD_CMD_Msk, I2C_IC_DATA_CMD_CMD_Pos,
                I2C_Read);

            /* Выдавать ли STOP после текущего байта: 1 - СТОП выдавать. */
            if ((tx_size == 1) && (flags & I2C_TransferStopFlag)) {
                SET_VAL_MSK(reg, I2C_IC_DATA_CMD_RSVD_STOP_Msk,
                    I2C_IC_DATA_CMD_RSVD_STOP_Pos, 1);
            } else {
                SET_VAL_MSK(reg, I2C_IC_DATA_CMD_RSVD_STOP_Msk,
                    I2C_IC_DATA_CMD_RSVD_STOP_Pos, 0);
            }

            /* Выдается ли RESTART? */
            if ((first_byte) && (flags & I2C_TransferReStartFlag)) {
                first_byte = false;
                SET_VAL_MSK(reg, I2C_IC_DATA_CMD_RSVD_RESTART_Msk,
                    I2C_IC_DATA_CMD_RSVD_RESTART_Pos, 1);
            } else {
                SET_VAL_MSK(reg, I2C_IC_DATA_CMD_RSVD_RESTART_Msk,
                    I2C_IC_DATA_CMD_RSVD_RESTART_Pos, 0);
            }

            base->IC_DATA_CMD = reg;
            tx_size--;

#if I2C_RETRY_TIMES != 0U
            wait_times = I2C_RETRY_TIMES;
#endif
        }

        /* Получение данных из RxFIFO, если они имеются. */
        if (0U != (base->IC_STATUS & I2C_Stat_RxFifo_NotEmpty)
            && (0U != rx_size)) {
            *buf++ = (uint8_t)(base->IC_DATA_CMD);
            rx_size--;

#if I2C_RETRY_TIMES != 0U
            wait_times = I2C_RETRY_TIMES;
#endif
        }

#if I2C_RETRY_TIMES != 0U
        wait_times--;

        if (wait_times == 0U) {
#if defined(I2C_TIMEOUT_RECOVERY) && I2C_TIMEOUT_RECOVERY
            /* Восстановление работы модуля после таймаута на шине. */
            if (I2C_Reset(base) != I2C_Status_Ok)
                return I2C_Status_HwError;
#endif
            return I2C_Status_Timeout;
        }
#endif

    }

    if (flags & I2C_TransferStopFlag) {
        /* Завершение выполнения операций обмена. */
        ret = I2C_PendingStatus(base, I2C_PendStat_Active);

        if (ret != I2C_Status_Ok) {
            return ret;
        }
    }

    return I2C_Status_Ok;
}

/* Master: Блокирующий обмен. Уровень интерфейса: II */


i2c_status_t I2C_MasterTransferBlocking(I2C_Type *base, i2c_master_transfer_t *xfer)
{
    i2c_status_t result = I2C_Status_Ok;
    uint32_t subaddress;
    uint8_t subaddr_buf[4];

    assert(xfer != NULL);

    /* Защита от непредсказуемого поведения при передаче 0. */
    if (xfer->flags == 0U) {
        xfer->flags = I2C_TransferStartFlag | I2C_TransferStopFlag;
    }

    /* Если требуется начать передачу нового пакета данных. */
    if (0U != (xfer->flags & (uint32_t) I2C_TransferStartFlag)) {
        result = I2C_MasterAddrSet(base, xfer->slave_address, xfer->addr_size);
        if (result != I2C_Status_Ok) {
            return result;
        }

        /* Если требуется передать дополнительный адрес. */
        if (0U != xfer->subaddress_size) {
            /* Подготовка дополнительного адреса для передачи,
             * старший байт хранится по младшему адресу.
             */
            subaddress = xfer->subaddress;
            for (int32_t i = (int)xfer->subaddress_size - 1; i >= 0; i--) {
                subaddr_buf[i] = (uint8_t) subaddress & 0xffU;
                subaddress >>= 8;
            }

            /* Отправка дополнительного адреса. */
            result = I2C_MasterWriteBlocking(base, subaddr_buf,
                    xfer->subaddress_size, (uint32_t) I2C_TransferStartFlag);
            if (result != I2C_Status_Ok) {
                return result;
            }

            /* Замена флага I2C_TransferStartFlag на I2C_TransferReStartFlag. */
            if (xfer->direction == I2C_Read) {
                xfer->flags &= ~(uint32_t) I2C_TransferStartFlag;
                xfer->flags |= (uint32_t) I2C_TransferReStartFlag;
            }
        }
    }

    /* Если данные для передачи отсутствуют. */
    if (xfer->data_size == 0)
        return I2C_Status_Ok;

    /* Передача или прием данных. */
    switch (xfer->direction) {

        case I2C_Write:
            result = I2C_MasterWriteBlocking(base, xfer->data,
                    xfer->data_size, xfer->flags);
            break;

        case I2C_Read:
            result = I2C_MasterReadBlocking(base, xfer->data,
                    xfer->data_size, xfer->flags);
            break;
    }

    return result;
}

















/* Slave */

/*!
 * @brief Настройка Slave-адреса
 *
 * Запись нового значения в регистр адреса Slave-устройства.
 *
 * @param base          Базовый адрес модуля I2C.
 * @param address       Slave-адрес
 * @param ack_gen_call  Отвечать или нет General Call
 *
 * @retval #I2C_Status_Ok
 * @retval #I2C_Status_HwError
 */
i2c_status_t  I2C_SlaveSetAddress(I2C_Type *base,
    uint16_t address,
    bool ack_gen_call)
{
    i2c_status_t ret = I2C_Status_Ok;
    bool i2c_module_enable;

    /* Если Slave-адрес устройства и ACK_GEN_CALL НЕ изменились, 
     * то не переинициализировать модуль. 
     */
    if ((GET_VAL_MSK(base->IC_ACK_GENERAL_CALL,
                I2C_IC_ACK_GENERAL_CALL_ACK_GEN_CALL_Msk,
                I2C_IC_ACK_GENERAL_CALL_ACK_GEN_CALL_Pos) == ack_gen_call)
        &&
        (GET_VAL_MSK(base->IC_SAR,
                I2C_IC_SAR_IC_SAR_Msk,  I2C_IC_SAR_IC_SAR_Pos)  == address)
    ) {
        return I2C_Status_Ok;
    }

    /* Ожидание завершения всех операций. */
    ret = I2C_PendingStatus(base, I2C_PendStat_Active);
    if (ret != I2C_Status_Ok) {
        return ret;
    }

    /* Текущее состояние модуля I2C .*/
    i2c_module_enable = I2C_IsEnable(base);

    /* Отключение модуля I2C. */
    ret = I2C_Enable(base, false);
    if (ret != I2C_Status_Ok)
        return ret;

    /* Настройка Slave-адреса для модуля. */
    SET_VAL_MSK(base->IC_SAR, I2C_IC_SAR_IC_SAR_Msk,
        I2C_IC_SAR_IC_SAR_Pos, address);

    /* Реакция на общий вызов. */
    SET_VAL_MSK(base->IC_ACK_GENERAL_CALL,
        I2C_IC_ACK_GENERAL_CALL_ACK_GEN_CALL_Msk,
        I2C_IC_ACK_GENERAL_CALL_ACK_GEN_CALL_Pos, ack_gen_call);

    /* Возврат модуля I2C к исходному состоянию. */
    return I2C_Enable(base, i2c_module_enable);
}

/*!
 * @brief Получение конфигурации по умолчанию для Slave-устройства I2C.
 *
 * Эта функция обеспечивает следующую конфигурацию по умолчанию Slave-устройства I2C:
 * @code
 *   сonfig->address          = I2C_SLAVE_ADDR_DEFAULT;
 *   сonfig->ack_gen_call     = false;
 *   сonfig->i2c_addr_size    = I2C_Address7Bit;
 *   сonfig->enable_slave     = false;
 * @endcode
 *
 * После вызова этой функции возможно изменить настройки, если
 * требуется изменить конфигурацию, перед инициализацией с помощью
 * @ref I2C_SlaveInit(). Необходимо переопределить Slave-адрес желаемым адресом.

 * @param сonfig Предоставляемая пользователем структура конфигурации,
 *               для которой установлены значения по умолчанию.
 *               @ref #i2c_slave_config_t.
 */
void I2C_SlaveGetDefaultConfig(i2c_slave_config_t *config)
{
    assert(config != NULL);

    (void) memset(config, 0, sizeof(i2c_slave_config_t));

    config->address          = I2C_SLAVE_ADDR_DEFAULT; /*!< Slave-адрес                             */
    config->ack_gen_call     = false;                  /*!< Отмечать на General Call адрес          */
    config->i2c_addr_size    = I2C_Address7Bit;        /*!< Размерность адреса 7 или 10 бит         */
    config->enable_slave     = false;                  /*!< Включить модуль I2C после инициализации */
}

/*!
 * @brief Инициализация I2C Slave-модуля.
 *
 * Функция инциализирует ведомое периферийное устройство I2C, как описано
 * пользователем в переданной конфигурации.
 *
 * @param base   Базовый адрес модуля I2C
 * @param сonfig Пользовательская конфигурация устройства. Для получения
 *                    значений по умолчанию используется @ref I2C_SlaveGetDefaultConfig().
 *                    Значения по умолчанию можно переопределить до
 *                    вызова @ref I2C_SlaveInit().
 * @retval #I2C_Status_Ok
 * @retval #I2C_Status_HwError
 */
i2c_status_t I2C_SlaveInit(I2C_Type *base, const i2c_slave_config_t *config)
{
    i2c_status_t status;

    /* Выключение модуля I2C. */
    status = I2C_Enable(base, false);
    if (I2C_Status_Ok != status)
        return status;

    /* Установка адреса модуля. */
    I2C_SlaveSetAddress(base, config->address, config->ack_gen_call);

    /* Установка регистра конфигурации модуля I2C. */
    {
        uint32_t reg = 0;

        /* Выключение Master-режима. */
        SET_VAL_MSK(reg, I2C_IC_CON_MASTER_MODE_Msk,
            I2C_IC_CON_MASTER_MODE_Pos, 0U);

        /* Выбор разрядности адреса: 7 или 10 бит. */
        SET_VAL_MSK(reg, I2C_IC_CON_IC_10BITADDR_SLAVE_Msk,
            I2C_IC_CON_IC_10BITADDR_SLAVE_Pos,
            (uint32_t)config->i2c_addr_size);

        /* Включение Slave-режима. */
        SET_VAL_MSK(reg, I2C_IC_CON_IC_SLAVE_DISABLE_Msk,
            I2C_IC_CON_IC_SLAVE_DISABLE_Pos, 0U);

        base->IC_CON = reg;
    }

    /* Включение модуля I2C, если требуется. */
    status = I2C_Enable(base, config->enable_slave);
    if (I2C_Status_Ok != status)
        return status;

    return status;
}




/* Slave: Неблокирующий обмен */


/*!
 * @brief Вызов функции обратного вызова по событию из @ref I2C_SlaveTransferHandleIRQ().
 *
 * @note Задает тип события для структуры обмена и вызывает функцию обратного вызова
 *       обработки события, если это событие было разрешено в маске.
 *
 * @param base   Базовый адрес модуля I2C.
 * @param handle Slave-дескриптор для работы по прерыванию.
 * @param event  Вызываемое событие для Slave.
 */
static void I2C_SlaveInvokeEvent_ForIRQ(I2C_Type *base, i2c_slave_handle_t *handle, i2c_slave_event_transfer_t event)
{
    uint32_t event_mask     = handle->transfer.event_mask;
    handle->transfer.event = event;

    /* Если есть функция обратного вызова, и произошло событие заданное маской. */
    if (((handle->callback) != NULL) && ((event_mask & (uint32_t)event) != 0U)) {

        handle->callback(base, &handle->transfer, handle->user_data);

        /* Очистить счетчик обмена, потому-что есть новый буфер. */
        if ((event == I2C_SlaveEvent_Receive) ||
            (event == I2C_SlaveEvent_Transmit)) {

            handle->transfer.transferred_count = 0;
        }
    }
}



/*!
 * @brief Запуск работы модуля I2C в Slave-режиме.
 *
 * Вызов этой функции после @ref I2C_SlaveInit() и @ref I2C_SlaveTransferCreateHandle(),
 * запускает работу модуля I2C в режиме обмена по прeрываниям.
 * Slave отслеживает I2C шину и передает событие в функцию обратного вызова, 
 * которая была зарегистрирована при вызове @ref I2C_SlaveTransferCreateHandle().
 * Функция обратного вызова всегда вызывается из контекста прерывания.
 *
 * @param base    Базовый адрес модуля I2C.
 *
 * @param handle  Структура @ref i2c_slave_handle_t, в которой хранится состояние передачи.
 *
 * @param tx_data Данные, которые должны быть переданы в Master-устройство в ответ на запрос чтения.
 *                NULL, если Slave только принимает данные.
 *
 * @param tx_size Размер txData буфера в байтах.
 *
 * @param rx_data Данные, принятые Slave-устройством в ответ на запрос записи от Мaster-устройства.
 *                NULL если Slave только передает.
 *
 * @param rx_size Размер rxData буфера в байтах.
 *
 * @param event_mask Битовая маска, сформированная путем объединения enum по ИЛИ
 *                   #i2c_slave_event_transfer_t, чтобы указать, какие события
 *                   отправлять в функцию обратного вызова.
 *                   Другими допустимыми значениями являются 0, чтобы получать
 *                   события умолчанию только для передачи и приема,
 *                   или #I2C_SlaveEvents_All, чтобы включить все события,
 *                   или #I2C_SlaveEvents_Ordinary для работы в режиме без General Call.
 *
 * @retval #I2C_Status_Ok
 * @retval #I2C_Status_Busy Обмен уже запущен на для этого дескриптора .
 */
static i2c_status_t I2C_SlaveTransferNonBlockingInternal(I2C_Type *base,
    i2c_slave_handle_t *handle,
    const void *tx_data,
    size_t tx_size,
    void *rx_data,
    size_t rx_size,
    uint32_t event_mask)
{
    assert(handle != NULL);

    /* Возврат статуса Busy, если идет другой обмен. */
    if (I2C_SlaveFsm_Idle != handle->slave_fsm) {
        return I2C_Status_Busy;
    }

    /* Отключение всех прерываний во время настройки. */
    I2C_DisableInterrupts(base, (uint32_t)I2C_IRQ_All);

    /* Сохранение информации об обмене в дескрипторе. */
    handle->transfer.tx_data           = (const uint8_t *)tx_data;
    handle->transfer.tx_size           = tx_size;
    handle->transfer.rx_data           = (uint8_t *)rx_data;
    handle->transfer.rx_size           = rx_size;
    handle->transfer.transferred_count = 0;
    handle->transfer.event_mask        = event_mask |
        (uint32_t)I2C_SlaveEvent_Transmit |
        (uint32_t)I2C_SlaveEvent_Receive;

    /* Установка порога заполнения TxFIFO для вызова прерывания. */
    SET_VAL_MSK(base->IC_TX_TL, I2C_IC_TX_TL_TX_TL_Msk,
        I2C_IC_TX_TL_TX_TL_Pos, 1);

    /* Установка порога заполнения RxFIFO для вызова прерывания. */
    SET_VAL_MSK(base->IC_RX_TL, I2C_IC_RX_TL_RX_TL_Msk,
        I2C_IC_RX_TL_RX_TL_Pos, 0);

    /* Включение модуля I2C. */
    I2C_SlaveEnable(base, true);

    /* Включение прерываний. */
    I2C_EnableInterrupts(base, (uint32_t)I2C_IRQ_SlaveTst);

    return I2C_Status_Ok;
}

/*!
 * @brief Запуск передачи данных из Slave в Master по запросу.
 *
 * Функция может быть вызвана в ответ на событие #I2C_SlaveEvent_Transmit,
 * переданное в функцию обратного вызова, чтобы начать новую передачу
 * из Slave в Master.
 *
 * Набор событий, получаемых функцией обратного вызова, настраивается: 
 * для этого в параметр event_mask передается комбинация перечислителей
 * #i2c_slave_event_transfer_t для событий, которые требуется получать.
 * События #I2C_SlaveEvent_Transmit и #I2C_SlaveEvent_Receive всегда
 * включены и не требуют быть включенным в маску. В качестве альтернативы
 * возможно передать 0, чтобы получить набор по умолчанию только для событий
 * передачи и приема, которые всегда включены. Кроме того, есть константы
 * #I2C_SlaveEvents_All которая включает все события или
 * #I2C_SlaveEvents_Ordinary для работы в режиме без General Call.
 *
 * @param base       Базовый адрес модуля I2C.
 * @param transfer   Структура #i2c_slave_transfer_t.
 * @param tx_data    Данные для отправки в master.
 * @param tx_size    Колличество данных для отправки в байтах.
 * @param event_mask Битовая маска, сформированная путем объединения по
 *                   ИЛИ перечислителей #i2c_slave_event_transfer_t для
 *                   указания какие события отправлять в функцию обратного вызова.
 *                   Допустимыми значениями являются 0, чтобы получать только
 *                   события набор по умолчанию это события передачи и приема.
 *                   #I2C_SlaveEvents_All      Для включения всех событий используется
 *                   #I2C_SlaveEvents_Ordinary для работы в режиме без General Call.
 * @retval #I2C_Status_Ok
 * @retval #I2C_Status_Busy Slave передача уже запущенна для этого декскриптора.
 */
i2c_status_t I2C_SlaveSetSendBuffer(I2C_Type *base,
    volatile i2c_slave_transfer_t *transfer,
    const void                    *tx_data,
    size_t                         tx_size,
    uint32_t                       event_mask)
{
    return I2C_SlaveTransferNonBlockingInternal(base,
            transfer->handle,
            tx_data,
            tx_size,
            NULL,
            0u,
            event_mask);
}

/*!
 * @brief Инициирование Slave-устройством приема запросов от Master-устройства
 *
 * Функция может быть вызвана в ответ на вызов функции обратного вызова по
 * событию #I2C_SlaveEvent_Receive для старта нового приема данных.
 *
 * Набор событий, получаемых функцией обратного вызова, настраивается: 
 * для этого в параметр event_mask передается комбинация перечислителей
 * #i2c_slave_event_transfer_t для событий, которые требуется получать.
 * События #I2C_SlaveEvent_Transmit и #I2C_SlaveEvent_Receive всегда
 * включены и не требуют быть включенным в маску. В качестве альтернативы
 * возможно передать 0, чтобы получить набор по умолчанию только для событий
 * передачи и приема, которые всегда включены. Кроме того, константа
 * #I2C_SlaveEvents_All удобный способ включения всех событий.
 * #I2C_SlaveEvents_Ordinary для работы в режиме без General Call.
 *
 * @param base      Базовый адрес модуля I2C.
 * @param transfer  Структура #i2c_slave_transfer_t.
 * @param rxData    Буфер для хранения данных от Master-устройства.
 * @param rxSize    Размер буфера хранения данных в байтах.
 * @param eventMask Битовая маска, сформированная путем объединения
 *                  перечислителей #i2c_slave_event_transfer_t с помощью
 *                  операции ИЛИ для указания какие события отправлять
 *                  в функцию обратного вызова. 
 *                  Допустимыми значениями являются 0, чтобы получить набор
 *                  по умолчанию: только события передачи и приема.
 *                  #I2C_SlaveEvents_All      - для включения всех событий
 *                  #I2C_SlaveEvents_Ordinary - для работы в режиме без General Call
 *
 * @retval #I2C_Status_Ok
 * @retval #I2C_Status_Busy
 */
i2c_status_t I2C_SlaveSetReceiveBuffer(I2C_Type                     *base,
    volatile i2c_slave_transfer_t *transfer,
    void                          *rx_data,
    size_t                         rx_size,
    uint32_t                       event_mask)
{
    return I2C_SlaveTransferNonBlockingInternal(base,
            transfer->handle,
            NULL,
            0u,
            rx_data,
            rx_size,
            event_mask);
}

/*!
 * @brief Создание нового дескриптора для I2C Slave-режима.
 * 
 * @note Используется для работы в неблокирующих функциях (по прерыванию).
 *
 * Создание дескриптора для использования в неблокирующих функциях.
 * Однажды созданный дескриптор не требуется специально уничтожать.
 * Если требуется завершить обмен данными используется функция
 * @ref I2C_SlaveTransferAbort().
 *
 * @param base     Базовый адрес модуля I2C
 * @param handle   Дескриптор драйвера Slave-устройства
 * @param callback Пользовательская функция обратного вызова
 * @param userData Указатель на пользовательские данные передавамые
 *                 при вызове функции обратного вызова
 *
 * @retval #I2C_Status_Ok
 * @retval #I2C_Status_UnexpectedState
 */
i2c_status_t I2C_SlaveTransferCreateHandle(I2C_Type *base,
    i2c_slave_handle_t            *handle,
    i2c_slave_transfer_callback_t  callback,
    void                          *user_data)
{
    assert(handle != NULL);

    /* Очистка дескриптора. */
    (void) memset(handle, 0, sizeof(*handle));

    /* Сохранение параметров. */
    handle->callback = callback;
    handle->user_data = user_data;

    /* Инициализация FSM. */
    handle->slave_fsm = I2C_SlaveFsm_Idle;

    /* Сохранение указателя на дескриптор структуры передачи. */
    handle->transfer.handle = handle;

    /* Маскирование всех источников прерывания в модуле. */
    I2C_DisableInterrupts(base, (uint32_t)I2C_IRQ_All);

    /* Сохранение в дескрипторе номера прерывания. */
    handle->irq_num = IOIM_GetIRQNumber(base);

    /* Установка обработчика прерывания. */
    if (IOIM_SetIRQHandler(base, I2C_SlaveTransferHandleIRQ, handle)
        != IOIM_Status_Ok) {
        return I2C_Status_UnexpectedState;
    }

    return I2C_Status_Ok;
}


/*!
* @brief Инициализация приема для Slave-режима.
*
* Вызовите эту функцию после вызова @ref I2C_SlaveInit() и @ref I2C_SlaveTransferCreateHandle(),
* для начала обмена, управляемым Master-устройством.
* Slave отслеживает I2C шину и передает событие в функцию обратного вызова, 
* которая была зарегистрирована при вызове @ref I2C_SlaveTransferCreateHandle().
* Функция обратного вызова всегда вызывается из контекста прерывания.
*
* Если Slave-устройство не ожидает Tx обмена, Master получает от Slave запрос, 
* вызывает функцию обратного вызова с параметром #I2C_SlaveEvent_Transmit. 
*
* Если Slave-устройство не ожидает Rx обмена, Master передает в Slave запрос 
* вызывает функцию обратного вызова с параметром #I2C_SlaveEvent_Transmit.
*
* Набор событий, получаемых функцией обратного вызова, настраивается: 
* для этого в параметр event_mask передается комбинация перечислителей
* #i2c_slave_event_transfer_t для событий, которые требуется получать.
* События #I2C_SlaveEvent_Transmit и #I2C_SlaveEvent_Receive всегда
* включены и не требуют быть включенным в маску. В качестве альтернативы
* возможно передать 0, чтобы получить набор по умолчанию только для событий
* передачи и приема, которые всегда включены. Кроме того, константа
* #I2C_SlaveEvents_All удобный способ включения всех событий,
* #I2C_SlaveEvents_Ordinary для работы в режиме без General Call.
*
* @param base       Базовый адрес модуля I2C.
* @param handle     Структура @ref i2c_slave_handle_t, в которой
*                   хранится состояние передачи.
* @param eventMask  Битовая маска сформированная по ИЛИ элементами
*                   перечисления #i2c_slave_event_transfer_t для
*                   определения какие события приведут к вызову
*                   функции обратного вызова. 
*                   Другое принимаемое значение 0 устанавлиает события
*                   по умолчанию: это только прием и передача,
*                   так же можно использовать
*                   #I2C_SlaveEvents_All      - для реакции на все события,
*                   #I2C_SlaveEvents_Ordinary - для работы в режиме без General Call.
*
* @retval #I2C_Status_Ok
* @retval #I2C_Status_Busy На этом дескрипторе уже идет обмен данными.
*/
i2c_status_t I2C_SlaveTransferNonBlocking(I2C_Type *base,
    i2c_slave_handle_t *handle,
    uint32_t event_mask)
{
    return I2C_SlaveTransferNonBlockingInternal(base, handle,
            NULL, 0u, NULL, 0u, event_mask);
}


/*!
 * @brief Получение количества байтов, переданных во время неблокирующей
 *        транзакции (через прерывание) в Slave-режиме.
 *
 * @param base   Базовый адрес модуля I2C.
 * @param handle Cтруктура @ref i2c_slave_handle_t.
 * @param count  Количество байтов, переданных неблокирующей транзакцией на данный момент.
 *
 * @retval #I2C_Status_InvalidParameter
 * @retval #I2C_Status_Ok
 */
i2c_status_t I2C_SlaveTransferGetCount(I2C_Type *base,
    i2c_slave_handle_t *handle, size_t *count)
{
    assert(handle != NULL);

    UNUSED(base);

    if (NULL == count) {
        return I2C_Status_InvalidParameter;
    }

    /* Если активной передачи не происходит. */
    if (I2C_SlaveFsm_Idle == handle->slave_fsm) {
        *count = 0;
        return I2C_Status_NoTransferInProgress;
    }

    /* При активной предаче возвращение счетчика из дескриптора. */
    *count = handle->transfer.transferred_count;

    return I2C_Status_Ok;
}


/*!
 * @brief Остановка неблокирующих передач Slave-устройства, отключение модуля I2C.
 *
 * @note  Функцию можно вызвать в любое время, чтобы прекратить 
 *        обработку событий шины Slave-устройством.
 *
 * @param base   Базовый адрес модуля I2C.
 * @param handle Cтруктура @ref i2c_slave_handle_t, 
 *               в которой хранится состояние передачи.
 *
 * @retval #I2C_Status_Ok
 * @retval #I2C_Status_Idle
 */
i2c_status_t I2C_SlaveTransferAbort(I2C_Type *base, i2c_slave_handle_t *handle)
{
    i2c_status_t ret;

    /* Отключение всех прерываний. */
    I2C_DisableInterrupts(base, (uint32_t)I2C_IRQ_All);

    /* Отключение модуля I2C. */
    ret = I2C_Enable(base, false);
    if (ret != I2C_Status_Ok)
        return ret;

    handle->slave_fsm        = I2C_SlaveFsm_Idle;
    handle->transfer.tx_size = 0U;
    handle->transfer.rx_size = 0U;

    return ret;
}




/*!
* @brief Процедура для обработки прерываний в Slave-режиме.
*
* @note Эту функцию не нужно вызывать самостоятельно.
*
* @param base   Базовый адрес модуля I2C.
* @param handle Структура @ref i2c_slave_handle_t, 
*               в которой хранится состояние передачи.
*/
void I2C_SlaveTransferHandleIRQ(I2C_Type *base, i2c_slave_handle_t *handle)
{

    uint32_t irq_status = base->IC_INTR_STAT & I2C_IRQ_All; /* Статус прерывания с учетом маскирования в регистре IC_INTR_MASK */


    /* Отключение прерываний от модуля I2C. */
    NVIC_DisableIRQ(handle->irq_num);

    /* Обработка прерываний общая для всех режимов. */
    do {
        /*
         *  |  От большего
         *  |  приоритета
         *  |  к меньшему
         * \|/
         *  V
         */

        /************************/
        /*** ОБРАБОТКА ОШИБОК ***/

        /* [6] Передача прервана. */
        if ((irq_status & I2C_IRQ_TxAbrt) != 0) {
            I2C_ClearInterrupt(base, I2C_IRQ_TxAbrt);
            I2C_DisableInterrupts(base, I2C_IRQ_TxEmpty);

            /* Неожиданное состояние на шине - завершение обмена. */
            handle->slave_fsm = I2C_SlaveFsm_Idle;

            /* Установка причины обрыва в статус. */
            handle->transfer.completion_status =
                I2C_TxAbortSource_To_Status(base);
            I2C_SlaveInvokeEvent_ForIRQ(base, handle, I2C_SlaveEvent_Completion);

            break;
        }

        /* Прерывания, которые не должны присходить при нормальном режиме работы. */
        if ((irq_status & (I2C_IRQ_RxUnder | I2C_IRQ_RxOver |
                    I2C_IRQ_TxOver | I2C_IRQ_Activity))  != 0) {

            handle->slave_fsm = I2C_SlaveFsm_Idle;

            /* [0] Чтение из пустого RxFifo. */
            if ((irq_status & I2C_IRQ_RxUnder)  != 0) {
                handle->transfer.completion_status =
                    I2C_Status_UnexpectedState1;
            }

            /* [1] Переполнение RxFifo. */
            if ((irq_status & I2C_IRQ_RxOver)   != 0) {
                handle->transfer.completion_status =
                    I2C_Status_UnexpectedState2;
            }

            /* [3] Попытка записи в заполненный TxFifo. */
            if ((irq_status & I2C_IRQ_TxOver)   != 0) {
                handle->transfer.completion_status =
                    I2C_Status_UnexpectedState3;
            }

            /* [8] Активность на шине I2C (замаскирован). */
            if ((irq_status & I2C_IRQ_Activity) != 0) {
                handle->transfer.completion_status =
                    I2C_Status_UnexpectedState4;
            }

            I2C_SlaveInvokeEvent_ForIRQ(base, handle, I2C_SlaveEvent_Completion);
            assert(0);

            break;

        }

        /*** ОБРАБОТКА ОШИБОК ***/
        /************************/


        /* [10] START или RSTART условие на шине. */
        if ((irq_status & I2C_IRQ_StartDet) != 0) {
            I2C_ClearInterrupt(base, I2C_IRQ_StartDet);

            /* Если уже был обмен. */
            if (handle->slave_fsm != I2C_SlaveFsm_Idle) {       /* Завершение текущего обмена */

                handle->transfer.completion_status = I2C_Status_Ok;
                I2C_SlaveInvokeEvent_ForIRQ(base, handle,
                    I2C_SlaveEvent_Completion);
            }

            handle->slave_fsm = I2C_SlaveFsm_Init;
            I2C_SlaveInvokeEvent_ForIRQ(base, handle,
                I2C_SlaveEvent_MatchAddrSlave);
        }

        /* [11] Получен адрес General Call и отправлено подтверждение. */
        if ((irq_status & I2C_IRQ_GenCall) != 0) {
            I2C_ClearInterrupt(base, I2C_IRQ_GenCall);

            /* Если уже был обмен. */
            if (handle->slave_fsm != I2C_SlaveFsm_Idle) {       /* Пришел RStart - завершение текущего обмена */
                handle->slave_fsm = I2C_SlaveFsm_Init;          /* Сброс FSM в начало обмена*/

                handle->transfer.completion_status = I2C_Status_Ok;
                I2C_SlaveInvokeEvent_ForIRQ(base, handle,
                    I2C_SlaveEvent_Completion);
            }
            I2C_SlaveInvokeEvent_ForIRQ(base, handle,
                I2C_SlaveEvent_MatchAddrGen);
        }

        /* [5] Запрос данных удаленным Master. */
        if ((irq_status & I2C_IRQ_RdReq) != 0) {
            I2C_ClearInterrupt(base, I2C_IRQ_RdReq);

            if (handle->slave_fsm == I2C_SlaveFsm_Init) {
                handle->slave_fsm = I2C_SlaveFsm_Transmit;
                I2C_SlaveInvokeEvent_ForIRQ(base, handle,
                    I2C_SlaveEvent_Transmit);   /* Подготовка буфера для передачи. */

                I2C_EnableInterrupts(base, I2C_IRQ_TxEmpty);
                //irq_status |= I2C_IRQ_TxEmpty; /* Имитация выставления флага: TxFifo - Опустошение ниже уровня */
            } else {

                /* Установка состояния завершения передачи: неожиданное состояние. */
                handle->transfer.completion_status = I2C_Status_UnexpectedState6;
                I2C_SlaveInvokeEvent_ForIRQ(base, handle, I2C_SlaveEvent_Completion);

                assert(0);

                break;
            }
        }

        /* [2] RxFifo - Заполнен до уровня IC_RX_TL (IC_RX_TL == 1). */
        if ((irq_status & I2C_IRQ_RxFull) != 0) {
            uint32_t i = 0;

            if (handle->slave_fsm == I2C_SlaveFsm_Init) {

                handle->slave_fsm = I2C_SlaveFsm_Receive;
                I2C_SlaveInvokeEvent_ForIRQ(base, handle, I2C_SlaveEvent_Receive);   /* Подготовка буфера для приема. */
            }


            if (handle->slave_fsm == I2C_SlaveFsm_Receive) {

                /* Считывание всего RxFifo. */
                while ((GET_VAL_MSK(base->IC_STATUS, I2C_IC_STATUS_RFNE_Msk,
                            I2C_IC_STATUS_RFNE_Pos) != 0U)
                    && (i++ < I2C_FIFO_DEPTH)) {

                    volatile uint8_t temp;
                    temp = (uint8_t)GET_VAL_MSK(base->IC_DATA_CMD,
                            I2C_IC_DATA_CMD_DAT_Msk, I2C_IC_DATA_CMD_DAT_Pos);

                    if ((handle->transfer.rx_data != NULL) &&
                        (handle->transfer.rx_size != 0U)) {

                        *(handle->transfer.rx_data) = temp;
                        (handle->transfer.rx_size)--;
                        (handle->transfer.rx_data)++;
                        (handle->transfer.transferred_count)++;
                    } else {
                        /* Фиктивный прием данных. */
                    }
                }

            } else { /* Непредусмотренное состояние FSM. */

                /* Сброс FSM. */
                handle->slave_fsm = I2C_SlaveFsm_Idle;

                /* Установка состояния завершения передачи: неожиданное состояние. */
                handle->transfer.completion_status =
                    I2C_Status_UnexpectedState7;

                I2C_SlaveInvokeEvent_ForIRQ(base, handle,
                    I2C_SlaveEvent_Completion);

                assert(0);

                break;
            }


        }

        /* [4] TxFifo - Опустошение ниже уровня IC_TX_TL (IC_TX_TL == 1). */
        if ((irq_status & I2C_IRQ_TxEmpty) != 0) {
            uint32_t i = 0;

            /* Если есть данные на передачу. */
            if ((handle->transfer.tx_data != NULL) &&
                (handle->transfer.tx_size != 0U)) {

                /* Заполнение всего TxFifo. */
                while ((GET_VAL_MSK(base->IC_STATUS,
                            I2C_IC_STATUS_TFNF_Msk,
                            I2C_IC_STATUS_TFNF_Pos) != 0U)
                    && (i++ < I2C_FIFO_DEPTH)) {

                    /* Передаваемые данные. */
                    SET_VAL_MSK(base->IC_DATA_CMD,
                        I2C_IC_DATA_CMD_DAT_Msk,
                        I2C_IC_DATA_CMD_DAT_Pos, *(handle->transfer.tx_data));

                    (handle->transfer.tx_size)--;
                    (handle->transfer.tx_data)++;
                    (handle->transfer.transferred_count)++;

                    if (handle->transfer.tx_size == 0U) {
                        break;
                    }
                }
            } else {
                /* Данные на передачу отсутствуют. */
                I2C_DisableInterrupts(base, (uint32_t)I2C_IRQ_TxEmpty);
            }
        }



        /* [7] В Slave-режиме: 1 - Master не подтверждает передачу байта. */
        if ((irq_status & I2C_IRQ_RxDone) != 0) {
            I2C_ClearInterrupt(base, I2C_IRQ_RxDone);

            /* Отключение прерывания I2C_IRQ_TxEmpty. */
            I2C_DisableInterrupts(base, I2C_IRQ_TxEmpty);

            I2C_ClearInterrupt(base, I2C_IRQ_RxDone);
        }



        /* [9] На шине состояние STOP. */
        if ((irq_status & I2C_IRQ_StopDet) != 0) {
            I2C_ClearInterrupt(base, I2C_IRQ_StopDet);

            if (handle->slave_fsm != I2C_SlaveFsm_Idle) { /* Если драйвер I2C был в процессе обмена. */
                handle->slave_fsm = I2C_SlaveFsm_Idle;

                handle->transfer.completion_status = I2C_Status_Ok;
                I2C_SlaveInvokeEvent_ForIRQ(base, handle,
                    I2C_SlaveEvent_Completion);

                /* Отключение прерывания I2C_IRQ_TxEmpty*/
                I2C_DisableInterrupts(base, I2C_IRQ_TxEmpty);

            }
        }



    } while (0);

    /* Включение прерываний от модуля I2C */
    NVIC_EnableIRQ(handle->irq_num);



}
/* Slave: НЕ блокирующий обмен */

/*!
 * @}
 */
