/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @addtogroup gpio
 * @{
 */

/*! @file */

#include "hal_gpio.h"

/*!
 * @name Макросы для работы с полями битов. 
 * @{
 */
#define GET_MASK(len, pos)         (~(0xffffffff << (len)) << (pos)) /*!< Получить маску */
#define SET_VAL(BF, len, pos, val) ( (BF) = ((BF) & ~GET_MASK(len, pos)) | (((val) << (pos)) & GET_MASK(len, pos)) ) /*!< Установить значение поля */
#define GET_VAL(BF, len, pos)      ( ((BF) & GET_MASK(len, pos)) >> (pos) ) /*!< Получить значение поля */
#define GPIO_SECURE_MASK          0x10000000 /*!< Маска secure адреса */
/*! @}*/

/*!
 * @brief Получение Secure указателя на регистр PortMode из блока IOCTR.
 *
 * @param port       Номер порта GPIO.
 * 
 * @retval              Secure указатель на регистр PortMode.
 * @retval              0 - неверный указатель ( запрашиваемый порт не существует).
 */
static volatile uint32_t *GPIO_GetIOCTRModeRegNonSecure(uint32_t port);

/*!
 * @brief Получение Secure указателя на структуру регистров порта GPIO.
 *
 * @param port          Номер порта GPIO.
 * 
 * @retval              Secure указатель GPIO_Type.
 * @retval              0 - неверный указатель ( запрашиваемый порт не существует).
 */
static GPIO_Type *GPIO_GetPortRegsNonSecure(uint32_t port);

/*!
 * @brief Расширение однобитной маски до многобитной.
 *
 * @param one_bit_mask  Однобитная маска.
 * @param ext_bits      Количество бит, на которое расширяется один бит маски.
 * @param pattern       Шаблон заполнения расширенных бит маски.
 * 
 * @return              Расширенная маска битов.
 */
static uint32_t   GPIO_ExtendBitMask(uint32_t one_bit_mask, uint32_t ext_bits, uint32_t pattern);

/*!
 * @brief Установка режима работы выводов порта GPIO.
 *
 * @param port       Номер порта GPIO.
 * @param bit_mask   Битовая маска выбранных выводов.
 * @param mode       Режим работы.
 */
static void       GPIO_PortSetMode(uint32_t port, uint32_t bit_mask, uint32_t mode);

static volatile uint32_t *GPIO_GetIOCTRModeRegNonSecure(uint32_t port)
{
    switch (port) {
        case GPIO_PORTA:
            return &IOCTR->PA_MODE;
        case GPIO_PORTB:
            return &IOCTR->PB_MODE;
        case GPIO_PORTC:
            return &IOCTR->PC_MODE;
        case GPIO_PORTD:
            return &IOCTR->PD_MODE;
        default:
            return NULL;
    }
}

static GPIO_Type *GPIO_GetPortRegsNonSecure(uint32_t port)
{
    switch (port) {
        case GPIO_PORTA:
            return GPIO0;
        case GPIO_PORTB:
            return GPIO1;
        case GPIO_PORTC:
            return GPIO2;
        case GPIO_PORTD:
            return GPIO3;
        default:
            return NULL;
    }
}

static uint32_t GPIO_ExtendBitMask(uint32_t one_bit_mask, uint32_t ext_bits, uint32_t pattern)
{
    uint32_t extend_mask = ~(0xffffffff << (ext_bits));
    pattern &= extend_mask;
    uint32_t new_mask = 0;
    int i;

    for (i = 0; i < 32; i += ext_bits, one_bit_mask >>= 1) {
        if ((one_bit_mask & 0x1) != 0) {
            new_mask |= (pattern << i);
        }
    }

    return new_mask;
}

static void GPIO_PortSetMode(uint32_t port, uint32_t bit_mask, uint32_t mode)
{
    uint32_t reg_mask  = GPIO_ExtendBitMask(bit_mask, 2, 0xF);
    uint32_t reg_value = GPIO_ExtendBitMask(bit_mask, 2, mode) & reg_mask;
    volatile uint32_t *portMode = GPIO_GetIOCTRModeRegNonSecure(port);

    if (portMode == NULL) {
        return;
    }

    *portMode = (*portMode & ~reg_mask) | reg_value;
}
int32_t GPIO_GetInstance(GPIO_Type *port)
{
    uint32_t base = ((uint32_t) port) & ~GPIO_SECURE_MASK;

    switch(base) {
        case GPIO0_BASE:
            return GPIO_PORTA;
        case GPIO1_BASE:
            return GPIO_PORTB;
        case GPIO2_BASE:
            return GPIO_PORTC;
        case GPIO3_BASE:
            return GPIO_PORTD;
        default:
            return -1;
    }
}
gpio_mode_t GPIO_PinMode_Get(uint32_t pin)
{
    uint32_t port_num = GPIO_PORTPIN_GET_PORT_NUM(pin);
    uint32_t pin_num  = GPIO_PORTPIN_GET_PIN_NUM(pin);
    volatile uint32_t *portMode = GPIO_GetIOCTRModeRegNonSecure(port_num);

    if (portMode == NULL) {
        return GPIO_MODE_INVALID;
    }

    return GET_VAL(*portMode, 2, pin_num * 2);
}

void GPIO_PinMode_HiZ(uint32_t pin)
{
    uint32_t port_num = GPIO_PORTPIN_GET_PORT_NUM(pin);
    uint32_t pin_num  = GPIO_PORTPIN_GET_PIN_NUM(pin);
    volatile uint32_t *portMode = GPIO_GetIOCTRModeRegNonSecure(port_num);

    if (portMode == NULL) {
        return;
    }

    SET_VAL(*portMode, 2, pin_num * 2, GPIO_MODE_HI_Z);
}

void GPIO_PinMode_Function(uint32_t pin, gpio_pin_function_t func)
{
    uint32_t port_num = GPIO_PORTPIN_GET_PORT_NUM(pin);
    uint32_t pin_num  = GPIO_PORTPIN_GET_PIN_NUM(pin);
    volatile uint32_t *portMode = GPIO_GetIOCTRModeRegNonSecure(port_num);
    volatile uint32_t *reg_afl, *reg_afh;

    if (portMode == NULL || func > 0x7) {
        return;
    }

    SET_VAL(*portMode, 2, pin_num * 2, GPIO_MODE_AF);

    switch (port_num) {
        case GPIO_PORTA:
            reg_afl = &IOCTR->PA_AFL;
            reg_afh = &IOCTR->PA_AFH;
            break;
        case GPIO_PORTB:
            reg_afl = &IOCTR->PB_AFL;
            reg_afh = &IOCTR->PB_AFH;
            break;
        case GPIO_PORTC:
            reg_afl = &IOCTR->PC_AFL;
            reg_afh = &IOCTR->PC_AFH;
            break;
        case GPIO_PORTD:
            reg_afl = &IOCTR->PD_AFL;
            reg_afh = &IOCTR->PD_AFH;
            break;
        default:
            return;
    }

    if (pin_num < 8) {
        SET_VAL(*reg_afl, 3, pin_num * 4, func);
    } else {
        SET_VAL(*reg_afh, 3, pin_num % 8 * 4, func);
    }
}

void GPIO_PinMode_GPIO(uint32_t pin, gpio_pin_direction_t direction)
{
    uint32_t port_num = GPIO_PORTPIN_GET_PORT_NUM(pin);
    uint32_t pin_num  = GPIO_PORTPIN_GET_PIN_NUM(pin);
    volatile uint32_t *portMode = GPIO_GetIOCTRModeRegNonSecure(port_num);
    GPIO_Type *gpio_port = GPIO_GetPortRegsNonSecure(port_num);

    if (portMode == NULL || gpio_port == NULL) {
        return;
    }

    switch (direction) {
        case GPIO_DigitalInput:
            gpio_port->OUT_EN_CLR = GET_MASK(1, pin_num);
            break;
        case GPIO_DigitalOutput:
            gpio_port->OUT_EN_SET = GET_MASK(1, pin_num);
            break;
        default:
            return;
    }
    SET_VAL(*portMode, 2, pin_num * 2, GPIO_MODE_GPIO);
}

void GPIO_PinSet_PUPD(uint32_t pin, gpio_pin_pupd_t pupd)
{
    uint32_t port_num = GPIO_PORTPIN_GET_PORT_NUM(pin);
    uint32_t pin_num  = GPIO_PORTPIN_GET_PIN_NUM(pin);
    uint32_t pull = pupd & 0x3;
    uint32_t od = ((pupd & 0x4) != 0);
    volatile uint32_t *reg_pupd, *reg_otype;

    switch (port_num) {
        case GPIO_PORTA:
            reg_pupd  = &IOCTR->PA_PUPD;
            reg_otype = &IOCTR->PA_OTYPE;
            break;
        case GPIO_PORTB:
            reg_pupd  = &IOCTR->PB_PUPD;
            reg_otype = &IOCTR->PB_OTYPE;
            break;
        case GPIO_PORTC:
            reg_pupd  = &IOCTR->PC_PUPD;
            reg_otype = &IOCTR->PC_OTYPE;
            break;
        case GPIO_PORTD:
            reg_pupd  = &IOCTR->PD_PUPD;
            reg_otype = &IOCTR->PD_OTYPE;
            break;
        default:
            return;
    }

    SET_VAL(*reg_pupd,  2, pin_num * 2, pull);
    SET_VAL(*reg_otype, 1, pin_num, od);
}

void GPIO_PinSet_MaxCurrent(uint32_t pin, gpio_pin_max_current_t current)
{
    uint32_t port_num = GPIO_PORTPIN_GET_PORT_NUM(pin);
    uint32_t pin_num  = GPIO_PORTPIN_GET_PIN_NUM(pin);
    volatile uint32_t *reg_ds;

    if (current > 3) {
        return;
    }

    switch (port_num) {
        case GPIO_PORTA:
            reg_ds = &IOCTR->PA_DS;
            break;
        case GPIO_PORTB:
            reg_ds = &IOCTR->PB_DS;
            break;
        case GPIO_PORTC:
            reg_ds = &IOCTR->PC_DS;
            break;
        case GPIO_PORTD:
            reg_ds = &IOCTR->PD_DS;
            break;
        default:
            return;
    }

    SET_VAL(*reg_ds, 2, pin_num * 2, current);
}

void GPIO_PinSet_Schmitt(uint32_t pin, uint32_t value)
{
    uint32_t port_num = GPIO_PORTPIN_GET_PORT_NUM(pin);
    uint32_t pin_num  = GPIO_PORTPIN_GET_PIN_NUM(pin);
    volatile uint32_t *reg_itype;
    value &= 0x1;

    switch (port_num) {
        case GPIO_PORTA:
            reg_itype = &IOCTR->PA_ITYPE;
            break;
        case GPIO_PORTB:
            reg_itype = &IOCTR->PB_ITYPE;
            break;
        case GPIO_PORTC:
            reg_itype = &IOCTR->PC_ITYPE;
            break;
        case GPIO_PORTD:
            reg_itype = &IOCTR->PD_ITYPE;
            break;
        default:
            return;
    }

    SET_VAL(*reg_itype, 1, pin_num, value);
}

void GPIO_PinSet_SpeedRaise(uint32_t pin, uint32_t value)
{
    uint32_t port_num = GPIO_PORTPIN_GET_PORT_NUM(pin);
    uint32_t pin_num  = GPIO_PORTPIN_GET_PIN_NUM(pin);
    volatile uint32_t *reg_sr;
    value &= 0x1;

    switch (port_num) {
        case GPIO_PORTA:
            reg_sr = &IOCTR->PA_SR;
            break;
        case GPIO_PORTB:
            reg_sr = &IOCTR->PB_SR;
            break;
        case GPIO_PORTC:
            reg_sr = &IOCTR->PC_SR;
            break;
        case GPIO_PORTD:
            reg_sr = &IOCTR->PD_SR;
            break;
        default:
            return;
    }

    SET_VAL(*reg_sr, 1, pin_num, value);
}

void GPIO_PinWrite(uint32_t pin, uint32_t bit)
{
    uint32_t port_num = GPIO_PORTPIN_GET_PORT_NUM(pin);
    uint32_t pin_num  = GPIO_PORTPIN_GET_PIN_NUM(pin);
    GPIO_Type *gpio_port = GPIO_GetPortRegsNonSecure(port_num);
    bit &= 0x1;

    if (gpio_port == NULL) {
        return;
    }

    SET_VAL(gpio_port->DATA_OUT, 1, pin_num, bit);
}

void GPIO_PinToggle(uint32_t pin)
{
    uint32_t port_num = GPIO_PORTPIN_GET_PORT_NUM(pin);
    uint32_t pin_num  = GPIO_PORTPIN_GET_PIN_NUM(pin);
    GPIO_Type *gpio_port = GPIO_GetPortRegsNonSecure(port_num);

    if (gpio_port == NULL) {
        return;
    }

    gpio_port->DATA_OUT = gpio_port->DATA_OUT ^ (1 << pin_num);
}

uint32_t GPIO_PinRead(uint32_t pin)
{
    uint32_t port_num = GPIO_PORTPIN_GET_PORT_NUM(pin);
    uint32_t pin_num  = GPIO_PORTPIN_GET_PIN_NUM(pin);
    GPIO_Type *gpio_port = GPIO_GetPortRegsNonSecure(port_num);

    if (gpio_port == NULL) {
        return 0;
    }

    if (GET_VAL(gpio_port->OUT_EN_SET, 1, pin_num)  == GPIO_DigitalInput) {
        return GET_VAL(gpio_port->DATA_IN, 1, pin_num);
    } else {
        return GET_VAL(gpio_port->DATA_OUT, 1, pin_num);
    }
}

void GPIO_PinIRQ_Enable(uint32_t pin, gpio_int_type_t int_type)
{
    uint32_t port_num = GPIO_PORTPIN_GET_PORT_NUM(pin);
    uint32_t pin_num  = GPIO_PORTPIN_GET_PIN_NUM(pin);
    GPIO_Type *gpio_port = GPIO_GetPortRegsNonSecure(port_num);

    if (gpio_port == NULL || int_type > 3) {
        return;
    }

    if ((int_type & 0x1) != 0) {
        gpio_port->INT_POL_SET = (1 << pin_num);
    } else {
        gpio_port->INT_POL_CLR = (1 << pin_num);
    }

    if ((int_type & 0x2) != 0) {
        gpio_port->INT_TYPE_SET = (1 << pin_num);
    } else {
        gpio_port->INT_TYPE_CLR = (1 << pin_num);
    }

    gpio_port->INT_EN_SET = (1 << pin_num);
}

void GPIO_PinIRQ_Disable(uint32_t pin)
{
    uint32_t port_num = GPIO_PORTPIN_GET_PORT_NUM(pin);
    uint32_t pin_num  = GPIO_PORTPIN_GET_PIN_NUM(pin);
    GPIO_Type *gpio_port = GPIO_GetPortRegsNonSecure(port_num);

    if (gpio_port == NULL) {
        return;
    }

    gpio_port->INT_EN_CLR = (1 << pin_num);
}

uint32_t GPIO_PinIRQ_GetStatus(uint32_t pin)
{
    uint32_t port_num = GPIO_PORTPIN_GET_PORT_NUM(pin);
    uint32_t pin_num  = GPIO_PORTPIN_GET_PIN_NUM(pin);
    GPIO_Type *gpio_port = GPIO_GetPortRegsNonSecure(port_num);

    if (gpio_port == NULL) {
        return 0;
    }

    return GET_VAL(gpio_port->INT_STATUS, 1, pin_num);
}

void GPIO_PinIRQ_Clear(uint32_t pin)
{
    uint32_t port_num = GPIO_PORTPIN_GET_PORT_NUM(pin);
    uint32_t pin_num  = GPIO_PORTPIN_GET_PIN_NUM(pin);
    GPIO_Type *gpio_port = GPIO_GetPortRegsNonSecure(port_num);

    if (gpio_port == NULL) {
        return;
    }

    gpio_port->INT_STATUS = (1 << pin_num);
}

void GPIO_PortMode_HiZ(GPIO_Type *port, uint32_t bit_mask)
{
    if (port == NULL) {
        return;
    }
    int32_t port_num = GPIO_GetInstance(port);
    bit_mask &= 0xFFFF;
    GPIO_PortSetMode(port_num, bit_mask, GPIO_MODE_HI_Z);
}

void GPIO_PortMode_Function(GPIO_Type *port, uint32_t bit_mask, gpio_pin_function_t func)
{
    if (port == NULL || func > 7) {
        return;
    }
    bit_mask &= 0xFFFF;

    uint32_t af_value = GPIO_ExtendBitMask(0xFF, 4, func);
    uint32_t afl_mask = GPIO_ExtendBitMask(bit_mask & 0xFF, 4, 0xF);
    uint32_t afh_mask = GPIO_ExtendBitMask(bit_mask >> 8, 4, 0xF);
    volatile uint32_t *reg_afl, *reg_afh;
    int32_t port_num = GPIO_GetInstance(port);

    GPIO_PortSetMode(port_num, bit_mask, GPIO_MODE_AF);

    switch (port_num) {
        case GPIO_PORTA:
            reg_afl = &IOCTR->PA_AFL;
            reg_afh = &IOCTR->PA_AFH;
            break;
        case GPIO_PORTB:
            reg_afl = &IOCTR->PB_AFL;
            reg_afh = &IOCTR->PB_AFH;
            break;
        case GPIO_PORTC:
            reg_afl = &IOCTR->PC_AFL;
            reg_afh = &IOCTR->PC_AFH;
            break;
        case GPIO_PORTD:
            reg_afl = &IOCTR->PD_AFL;
            reg_afh = &IOCTR->PD_AFH;
            break;
        default:
            return;
    }

    if (afl_mask) {
        *reg_afl = (*reg_afl & ~afl_mask) | (af_value & afl_mask);
    }
    if (afh_mask) {
        *reg_afh = (*reg_afh & ~afh_mask) | (af_value & afh_mask);
    }
}

void GPIO_PortMode_GPIO(GPIO_Type *port, uint32_t bit_mask, gpio_pin_direction_t direction)
{
    if (port == NULL) {
        return;
    }
    int32_t port_num = GPIO_GetInstance(port);
    bit_mask &= 0xFFFF;

    switch (direction) {
        case GPIO_DigitalInput:
            port->OUT_EN_CLR = bit_mask;
            break;
        case GPIO_DigitalOutput:
            port->OUT_EN_SET = bit_mask;
            break;
        default:
            return;
    }
    GPIO_PortSetMode(port_num, bit_mask, GPIO_MODE_GPIO);
}

void GPIO_PortSet_PUPD(GPIO_Type *port, uint32_t bit_mask, gpio_pin_pupd_t pupd)
{
    uint32_t pull = pupd & 0x3;
    uint32_t od = ((pupd & 0x4) != 0)? 0xFFFF : 0x0000;
    bit_mask &= 0xFFFF;
    uint32_t pupd_value = GPIO_ExtendBitMask(0xFF, 2, pull);
    uint32_t pupd_mask  = GPIO_ExtendBitMask(bit_mask, 2, 0xF);
    volatile uint32_t *reg_pupd, *reg_otype;
    int32_t port_num = GPIO_GetInstance(port);

    switch (port_num) {
        case GPIO_PORTA:
            reg_pupd  = &IOCTR->PA_PUPD;
            reg_otype = &IOCTR->PA_OTYPE;
            break;
        case GPIO_PORTB:
            reg_pupd  = &IOCTR->PB_PUPD;
            reg_otype = &IOCTR->PB_OTYPE;
            break;
        case GPIO_PORTC:
            reg_pupd  = &IOCTR->PC_PUPD;
            reg_otype = &IOCTR->PC_OTYPE;
            break;
        case GPIO_PORTD:
            reg_pupd  = &IOCTR->PD_PUPD;
            reg_otype = &IOCTR->PD_OTYPE;
            break;
        default:
            return;
    }

    *reg_pupd  = (*reg_pupd & ~pupd_mask) | (pupd_value & pupd_mask);
    *reg_otype = (*reg_otype & ~bit_mask) | (od & bit_mask);
}

void GPIO_PortSet_MaxCurrent(GPIO_Type *port, uint32_t bit_mask, gpio_pin_max_current_t current)
{
    if (current > 3) {
        return;
    }

    bit_mask &= 0xFFFF;
    uint32_t cur_value = GPIO_ExtendBitMask(0xFF, 2, current);
    uint32_t cur_mask  = GPIO_ExtendBitMask(bit_mask, 2, 0xF);
    volatile uint32_t *reg_ds;
    int32_t port_num = GPIO_GetInstance(port);

    switch (port_num) {
        case GPIO_PORTA:
            reg_ds = &IOCTR->PA_DS;
            break;
        case GPIO_PORTB:
            reg_ds = &IOCTR->PB_DS;
            break;
        case GPIO_PORTC:
            reg_ds = &IOCTR->PC_DS;
            break;
        case GPIO_PORTD:
            reg_ds = &IOCTR->PD_DS;
            break;
        default:
            return;
    }

    *reg_ds = (*reg_ds & ~cur_mask) | (cur_value & cur_mask);
}

void GPIO_PortSet_Schmitt(GPIO_Type *port, uint32_t bit_mask, uint32_t bit_value)
{
    volatile uint32_t *reg_itype;
    int32_t port_num = GPIO_GetInstance(port);
    bit_mask &= 0xFFFF;

    switch (port_num) {
        case GPIO_PORTA:
            reg_itype = &IOCTR->PA_ITYPE;
            break;
        case GPIO_PORTB:
            reg_itype = &IOCTR->PB_ITYPE;
            break;
        case GPIO_PORTC:
            reg_itype = &IOCTR->PC_ITYPE;
            break;
        case GPIO_PORTD:
            reg_itype = &IOCTR->PD_ITYPE;
            break;
        default:
            return;
    }

    *reg_itype = (*reg_itype & ~bit_mask) | (bit_value & bit_mask);
}

void GPIO_PortSet_SpeedRaise(GPIO_Type *port, uint32_t bit_mask, uint32_t bit_value)
{
    volatile uint32_t *reg_sr;
    int32_t port_num = GPIO_GetInstance(port);
    bit_mask &= 0xFFFF;

    switch (port_num) {
        case GPIO_PORTA:
            reg_sr = &IOCTR->PA_SR;
            break;
        case GPIO_PORTB:
            reg_sr = &IOCTR->PB_SR;
            break;
        case GPIO_PORTC:
            reg_sr = &IOCTR->PC_SR;
            break;
        case GPIO_PORTD:
            reg_sr = &IOCTR->PD_SR;
            break;
        default:
            return;
    }

    *reg_sr = (*reg_sr & ~bit_mask) | (bit_value & bit_mask);
}

void GPIO_PortWrite(GPIO_Type *port, uint32_t bit_mask, uint32_t bit_value)
{
    if (port == NULL) {
        return;
    }

    bit_mask &= 0xFFFF;
    port->DATA_OUT = (port->DATA_OUT & ~bit_mask) | (bit_value & bit_mask);
}

void GPIO_PortToggle(GPIO_Type *port, uint32_t bit_mask)
{
    if (port == NULL) {
        return;
    }

    bit_mask &= 0xFFFF;
    port->DATA_OUT = port->DATA_OUT ^ bit_mask;
}

uint32_t GPIO_PortRead(GPIO_Type *port, uint32_t bit_mask)
{
    if (port == NULL) {
        return 0;
    }

    bit_mask &= 0xFFFF;
    return port->DATA_IN & bit_mask;
}

void GPIO_PortIRQ_Enable(GPIO_Type *port, uint32_t bit_mask, uint32_t int_type)
{
    if (port == NULL) {
        return;
    }

    bit_mask &= 0xFFFF;

    if ((int_type & 0x1) != 0) {
        port->INT_POL_SET = bit_mask;
    } else {
        port->INT_POL_CLR = bit_mask;
    }

    if ((int_type & 0x2) != 0) {
        port->INT_TYPE_SET = bit_mask;
    } else {
        port->INT_TYPE_CLR = bit_mask;
    }

    port->INT_EN_SET = bit_mask;
}

void GPIO_PortIRQ_Disable(GPIO_Type *port, uint32_t bit_mask)
{
    if (port == NULL) {
        return;
    }

    bit_mask &= 0xFFFF;

    port->INT_EN_CLR = bit_mask;
}

uint32_t GPIO_PortIRQ_GetStatus(GPIO_Type *port, uint32_t bit_mask)
{
    if (port == NULL) {
        return 0;
    }

    bit_mask &= 0xFFFF;

    return port->INT_STATUS & bit_mask;
}

void GPIO_PortIRQ_Clear(GPIO_Type *port, uint32_t bit_mask)
{
    if (port == NULL) {
        return;
    }

    bit_mask &= 0xFFFF;

    port->INT_STATUS = bit_mask;
}

uint32_t GPIO_PortSecureIRQ_GetStatus(GPIO_Type *port)
{
    if (port == NULL) {
        return 0;
    }

    return port->SEC_INT_STAT;
}

void GPIO_PortSecureIRQ_Clear(GPIO_Type *port)
{
    if (port == NULL) {
        return;
    }

    port->SEC_INT_CLR = 1;
}

void GPIO_PortSecureIRQ_SetMask(GPIO_Type *port, uint32_t mask)
{
    if (port == NULL) {
        return;
    }

    SET_VAL(port->SEC_INT_MASK, 2, 0, mask);
}

uint32_t GPIO_PortSecureIRQ_GetInfo1(GPIO_Type *port)
{
    if (port == NULL) {
        return 0;
    }

    return port->SEC_INT_INFO1;
}

uint32_t GPIO_PortSecureIRQ_GetInfo2(GPIO_Type *port)
{
    if (port == NULL) {
        return 0;
    }

    return port->SEC_INT_INFO2;
}

void GPIO_PortSet_NonsecureMask(GPIO_Type *port, uint32_t mask)
{
    if (port == NULL) {
        return;
    }

    port->PORT_NONSEC_MASK = mask;
}

/*!
 * @}
 */
