/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @defgroup dma_driver Драйвер модуля DMA
 *
 * @brief Драйвер прямого доступа к памяти
 *
 * Драйвер модуля DMA управляет каналами контроллеров DMA0 и DMA1.
 */

/*!
 * @addtogroup dma_driver
 * @{
 */

/*!
 * @file hal_dma.h
 *
 * @brief Интерфейс драйвера прямого доступа к памяти
 */

#ifndef HAL_DMA_H
#define HAL_DMA_H

#include "hal_common.h"

/*!
 * @brief Версия драйвера
 */
#define HAL_DMA_DRIVER_VERSION (MAKE_VERSION(1, 0, 0))

/*! @brief Максимальный размер транзакции на AHB шине */
#define DMA_AHB_MAX_BLOCK_SIZE (4095U)
/*! @brief Число каналов одного контроллера DMA */
#define HAL_FEATURE_DMA_NUMBER_OF_CHANNELS (8U)
/*! @brief Максимальное число аппаратных интерфейсов запросов DMA */
#define HAL_FEATURE_DMA_MAX_NUMBER_HW_HANDSHAKES (16U)
/*! @brief Общее число каналов DMA */
#define HAL_FEATURE_DMA_ALL_CHANNELS (HAL_FEATURE_DMA_NUMBER_OF_CHANNELS * 2U)
/*! @brief Размер выравнивания дескриптора */
#define HAL_FEATURE_DMA_LINK_DESCRIPTOR_ALIGN_SIZE (4U)
/*! @brief Объявление дескриптора и выравнивание адреса дескриптора */
#define DMA_ALLOCATE_HEAD_DESCRIPTORS(name, number) \
    dma_descriptor_t name[number] __attribute__((aligned(HAL_FEATURE_DMA_LINK_DESCRIPTOR_ALIGN_SIZE)))
/*! @brief Разница адресов идентичных регистров из соседних каналов */
#define REG_OFFSET (0x58)

/*! @brief Номер handshake запроса UART0 TX */
#define DMA_HANDSHAKE_UART0_TX (0U)
/*! @brief Номер handshake запроса UART0 RX */
#define DMA_HANDSHAKE_UART0_RX (1U)
/*! @brief Номер handshake запроса UART1 TX */
#define DMA_HANDSHAKE_UART1_TX (2U)
/*! @brief Номер handshake запроса UART1 RX */
#define DMA_HANDSHAKE_UART1_RX (3U)
/*! @brief Номер handshake запроса UART2 TX */
#define DMA_HANDSHAKE_UART2_TX (0U)
/*! @brief Номер handshake запроса UART2 RX */
#define DMA_HANDSHAKE_UART2_RX (1U)
/*! @brief Номер handshake запроса UART3 TX */
#define DMA_HANDSHAKE_UART3_TX (2U)
/*! @brief Номер handshake запроса UART3 RX */
#define DMA_HANDSHAKE_UART3_RX (3U)

/*! @brief Номер handshake запроса SPI0 TX */
#define DMA_HANDSHAKE_SPI0_TX  (4U)
/*! @brief Номер handshake запроса SPI0 RX */
#define DMA_HANDSHAKE_SPI0_RX  (5U)
/*! @brief Номер handshake запроса SPI1 TX */
#define DMA_HANDSHAKE_SPI1_TX  (6U)
/*! @brief Номер handshake запроса SPI1 RX */
#define DMA_HANDSHAKE_SPI1_RX  (7U)
/*! @brief Номер handshake запроса SPI2 TX */
#define DMA_HANDSHAKE_SPI2_TX  (8U)
/*! @brief Номер handshake запроса SPI2 RX */
#define DMA_HANDSHAKE_SPI2_RX  (9U)

/*! @brief Номер handshake запроса I2C0 TX */
#define DMA_HANDSHAKE_I2C0_TX  (10U)
/*! @brief Номер handshake запроса I2C0 RX */
#define DMA_HANDSHAKE_I2C0_RX  (11U)
/*! @brief Номер handshake запроса I2C1 TX */
#define DMA_HANDSHAKE_I2C1_TX  (12U)
/*! @brief Номер handshake запроса I2C1 RX */
#define DMA_HANDSHAKE_I2C1_RX  (13U)
/*! @brief Номер handshake запроса QSPI TX */
#define DMA_HANDSHAKE_QSPI_TX  (14U)
/*! @brief Номер handshake запроса QSPI RX */
#define DMA_HANDSHAKE_QSPI_RX  (15U)


/*! @name CHANNEL_CTL - Регистр управления каналом */
#define DMA_CHANNEL_CTL_BLOCKSIZE_MASK  (0xFFF00000000ULL) /*!< Маска поля 'Размер блока' в регистре CTL */
#define DMA_CHANNEL_CTL_BLOCKSIZE_SHIFT (32ULL)            /*!< Смещение поля 'Размер блока' в регистре CTL */
/*!
 *
 */
/*! @brief Размер блока */
#define DMA_CHANNEL_CTL_BLOCKSIZE(x) (((uint64_t)((uint64_t)(x)) << DMA_CHANNEL_CTL_BLOCKSIZE_SHIFT) & DMA_CHANNEL_CTL_BLOCKSIZE_MASK)

/*!
 * @brief Конфигурация регистра CTL
 *
 * @param blockSize        Размер блока
 * @param srcEnChain       Разрешение цепочки для источника
 * @param dstEnChain       Разрешение цепочки для источника
 * @param transferType     Тип передачи и управление размером блока
 * @param scatter          Разрешение разброса приемника
 * @param gather           Разрешение сбора источника
 * @param srcMSize         Размер пакета источника
 * @param dstMSize         Размер пакета приемника
 * @param sInc             Тип изменения адреса источника
 * @param dInc             Тип изменения адреса приемника
 * @param srcTransferWidth Разрядность обращения источника
 * @param dstTransferWidth Разрядность обращения приемника
 * @param int_en           Разрешение прерываний
 */
#define DMA_CHANNEL_CTL(blockSize, srcEnChain, dstEnChain, transferType, \
    scatter, gather, srcMSize, dstMSize, sInc, dInc, \
    srcTransferWidth, dstTransferWidth, int_en) \
0x0ULL | DMA_CHANNEL_CTL_BLOCKSIZE(blockSize) | \
DMA_CTL0_LO_LLP_SRC_EN_VAL(srcEnChain) | \
DMA_CTL0_LO_LLP_DST_EN_VAL(dstEnChain) | \
DMA_CTL0_LO_TT_FC_VAL(transferType) | \
DMA_CTL0_LO_DST_SCATTER_EN_VAL(scatter) | \
DMA_CTL0_LO_SRC_GATHER_EN_VAL(gather) | \
DMA_CTL0_LO_SRC_MSIZE_VAL(srcMSize) | \
DMA_CTL0_LO_DEST_MSIZE_VAL(dstMSize) | \
DMA_CTL0_LO_SINC_VAL(sInc) | \
DMA_CTL0_LO_DINC_VAL(dInc) | \
DMA_CTL0_LO_SRC_TR_WIDTH_VAL(srcTransferWidth) | \
DMA_CTL0_LO_DST_TR_WIDTH_VAL(dstTransferWidth) | \
DMA_CTL0_LO_INT_EN_VAL(int_en)

/*!
 * @brief Разрядность передачи канала DMA
 */
enum {
    DMA_Transfer8BitWidth   = 0U, /*!< Ширина одного слова 8 бит */
    DMA_Transfer16BitWidth  = 1U, /*!< Ширина одного слова 16 бит */
    DMA_Transfer32BitWidth  = 2U, /*!< Ширина одного слова 32 бит */
    DMA_Transfer64BitWidth  = 3U, /*!< Ширина одного слова 64 бит */
    DMA_Transfer128BitWidth = 4U, /*!< Ширина одного слова 128 бит */
    DMA_Transfer256BitWidth = 5U, /*!< Ширина одного слова 256 бит */
};

/*!
 * @brief Размер пакета передачи в словах данных разрядностью DMA_TransferBitWidth
 */
enum {
    DMA_BurstSize1   = 0U, /*!< Размер пакета 1 */
    DMA_BurstSize4   = 1U, /*!< Размер пакета 4 */
    DMA_BurstSize8   = 2U, /*!< Размер пакета 8 */
    DMA_BurstSize16  = 3U, /*!< Размер пакета 16 */
    DMA_BurstSize32  = 4U, /*!< Размер пакета 32 */
    DMA_BurstSize64  = 5U, /*!< Размер пакета 64 */
    DMA_BurstSize128 = 6U, /*!< Размер пакета 128 */
    DMA_BurstSize256 = 7U, /*!< Размер пакета 256 */
};

/*!
 * @brief Тип изменения адреса при пересылке
 */
enum {
    DMA_Incr     = 0U, /*!< Инкремент адреса */
    DMA_Decr     = 1U, /*!< Декремент адреса */
    DMA_NoChange = 2U, /*!< Адрес не изменяется */
};

/*!
 * @brief Статусы возврата из функций для драйвера DMA
 */
typedef enum _dma_status {
    DMA_Status_Success = 0U, /*!< Функция выполнена успешно */
    DMA_Status_Busy    = 1U, /*!< Выбранный канал DMA занят */
    DMA_Status_NoBase  = 3U, /*!< Указанного базового адреса DMA не существует */
    DMA_Status_Fail    = 4U, /*!< Ошибка выполнения функции */
} dma_status_t;

/*!
 * @brief Приоритеты каналов
 */
typedef enum _dma_priority {
    DMA_ChannelPriority0 = 0, /*!< Низший приоритет 0 */
    DMA_ChannelPriority1 = 1, /*!< Приоритет 1 */
    DMA_ChannelPriority2 = 2, /*!< Приоритет 2 */
    DMA_ChannelPriority3 = 3, /*!< Приоритет 3 */
    DMA_ChannelPriority4 = 4, /*!< Приоритет 4 */
    DMA_ChannelPriority5 = 5, /*!< Приоритет 5 */
    DMA_ChannelPriority6 = 6, /*!< Приоритет 6 */
    DMA_ChannelPriority7 = 7, /*!< Наивысший приоритет 7 */
} dma_priority_t;

/*!
 * @brief Источники прерываний DMA
 */
typedef enum _dma_int {
    DMA_IntTfr     = 1U,  /*!< Завершение DMA пересылки */
    DMA_IntBlock   = 2U,  /*!< Завершение передачи блока */
    DMA_IntDstTran = 4U,  /*!< Завершение передачи Приемнику */
    DMA_IntSrcTran = 8U,  /*!< Завершение передачи источнику */
    DMA_IntError   = 16U, /*!< Ошибка передачи AHB */
    DMA_AllIRQCount = 5U,
    DMA_AllIRQ     =       DMA_IntTfr     /*!< Маска всех прерываний. */
        | DMA_IntBlock   | DMA_IntDstTran
        | DMA_IntSrcTran | DMA_IntError,
} dma_irq_t;

/*!
 * @brief Тип пересылок DMA и управление размером блока
 */
typedef enum _dma_transfer_type {
    DMA_MemoryToMemory_DMA            = 0U, /*!< Память-Память, DMA управляет размером блока */
    DMA_MemoryToPeripheral_DMA        = 1U, /*!< Память-Периферия, DMA управляет размером блока */
    DMA_PeripheralToMemory_DMA        = 2U, /*!< Периферия-Память, DMA управляет размером блока */
    DMA_PeripheralToPeripheral_DMA    = 3U, /*!< Периферия-Периферия, DMA управляет размером блока */
    DMA_PeripheralToMemory_Peripheral = 4U, /*!< Периферия-Память, периферия управляет размером блока */
    DMA_PeripheralToPeripheral_SRC    = 5U, /*!< Периферия-Периферия, источник управляет размером блока */
    DMA_MemoryToPeripheral_Peripheral = 6U, /*!< Память-Периферия, периферия управляет размером блока */
    DMA_PeripheralToPeripheral_DST    = 7U, /*!< Периферия-Периферия, приемник управляет размером блока */
} dma_transfer_type_t;

/*! @brief Описатель следующего блока DMA (LLI) */
typedef struct _dma_descriptor {
    __IOM uint32_t SAR;    /*!< Адрес источника */
    __IOM uint32_t DAR;    /*!< Адрес приемника */
    __IOM uint32_t LLP;    /*!< Адрес следующего блока */
    __IOM uint32_t CTL_LO; /*!< Конфигурация передачи канала */
    __IOM uint32_t CTL_HI; /*!< Конфигурация передачи канала */
    __IOM uint32_t SSTAT;  /*!< Состояние источника */
    __IOM uint32_t DSTAT;  /*!< Состояние приемника */
} dma_descriptor_t;

/*! @brief Дескриптор на регистры канала DMA */
typedef struct _dma_channel_reg {
    __IOM uint32_t  SAR_LO;     /*!< Адрес Источника, младшие биты */
    __IOM uint32_t  SAR_HI;     /*!< Адрес Источника, старшие биты */
    __IOM uint32_t  DAR_LO;     /*!< Адрес Приемника, младшие биты */
    __IOM uint32_t  DAR_HI;     /*!< Адрес Приемника, старшие биты */
    __IOM uint32_t  LLP_LO;     /*!< Адрес Описателя следующего блока, младшие биты */
    __IOM uint32_t  LLP_HI;     /*!< Адрес Описателя следующего блока, старшие биты */
    __IOM uint32_t  CTL_LO;     /*!< Регистр управления, младшие биты */
    __IOM uint32_t  CTL_HI;     /*!< Регистр управления, старшие биты */
    __IOM uint32_t  SSTAT_LO;   /*!< Статус Источника, младшие биты */
    __IOM uint32_t  SSTAT_HI;   /*!< Статус Источника, старшие биты */
    __IOM uint32_t  DSTAT_LO;   /*!< Статус Приемника, младшие биты */
    __IOM uint32_t  DSTAT_HI;   /*!< Статус Приемника, старшие биты */
    __IOM uint32_t  SSTATAR_LO; /*!< Адрес Статуса Источника, младшие биты */
    __IOM uint32_t  SSTATAR_HI; /*!< Адрес Статуса Источника, старшие биты */
    __IOM uint32_t  DSTATAR_LO; /*!< Адрес Статуса Приемника, младшие биты */
    __IOM uint32_t  DSTATAR_HI; /*!< Адрес Статуса Приемника, старшие биты */
    __IOM uint32_t  CFG_LO;     /*!< Регистр конфигурации, младшие биты */
    __IOM uint32_t  CFG_HI;     /*!< Регистр конфигурации, старшие биты */
    __IOM uint32_t  SGR_LO;     /*!< Регистр сбора Источника, младшие биты */
    __IOM uint32_t  SGR_HI;     /*!< Регистр сбора Источника, старшие биты */
    __IOM uint32_t  DSR_LO;     /*!< Регистр разброса Приемника, младшие биты */
    __IOM uint32_t  DSR_HI;     /*!< Регистр разброса Приемника, старшие биты */
} dma_channel_regs;

/*! @brief Описание конфигурации пересылки */
struct dma_channel_ctl_cfg {
    uint16_t block_size;               /*!< Размер блока передачи */
    bool llp_src_en;                   /*!< Разрешение многоблочной передачи для Источника */
    bool llp_dst_en;                   /*!< Разрешение многоблочной передачи для Приемника */
    dma_transfer_type_t transfer_type; /*!< Тип пересылки */
    bool scatter_en;                   /*!< Разрешение режима Разброса */
    bool gather_en;                    /*!< Разрешение режима Сбора */
    uint8_t src_burst_size;            /*!< Размер пакета Источника */
    uint8_t dst_burst_size;            /*!< Размер пакета Приемника */
    uint8_t src_incr;                  /*!< Инкремент адреса Источника */
    uint8_t dst_incr;                  /*!< Инкремент адреса Приемника */
    uint8_t src_tr_width;              /*!< Ширина одного слова при передаче от Источника */
    uint8_t dst_tr_width;              /*!< Ширина одного слова при передаче в Приемник */
    bool int_en;                       /*!< Разрешение прерываний */
};

/*!
 * @brief Конфигурация канала DMA
 */
typedef struct _dma_channel_config {
    void *src_addr;     /*!< Адрес источника */
    void *dst_addr;     /*!< Адрес приемника */
    void *next_desc;    /*!< Указатель на дескриптор */
    uint64_t ctlx_cfg;  /*!< Конфигурация регистра CTLx */
    bool is_src_periph; /*!< Тип запроса источника */
    bool is_dst_periph; /*!< Тип запроса приемника */
} dma_channel_config_t;

struct _dma_handle;

/*! @brief Функция обратного вызова */
typedef void (*dma_callback)(struct _dma_handle *handle, void *user_data,
    dma_irq_t intmode);

/*!
 * @brief Управляющая структура передачи
 */
typedef struct _dma_handle {
    dma_callback callback;  /*!< Функция обратного вызова во время прерывания */
    void *user_data;        /*!< Параметр функции обратного вызова */
    DMA_Type *base;         /*!< Базовый адрес DMA */
    uint32_t channel;       /*!< Номер канала DMA */
} dma_handle_t;

/*!
 * @brief Конфигурация многоблочной передачи
 */
typedef struct _dma_multiblock_config {
    uint32_t count;                    /*!< Количество дескрипторов многоблочной передачи */
    uint32_t data_size;                /*!< Общее число данных */
    dma_transfer_type_t transfer_type; /*!< Тип передачи */
    bool scatter_en;                   /*!< Разрешение режима разброса Приемника */
    bool gather_en;                    /*!< Разрешение режима сбора Источника */
    uint32_t src_burst_size;           /*!< Размер пакета Источника */
    uint32_t dst_burst_size;           /*!< Размер пакета Приемника */
    uint32_t src_incr;                 /*!< Инкремент Источника */
    uint32_t dst_incr;                 /*!< Инкремент Приемника */
    uint32_t src_data_width;           /*!< Ширина передаваемых данных Источника */
    uint32_t dst_data_width;           /*!< Ширина передаваемых данных Приемника */
    void *src_addr;                    /*!< Адрес Источника */
    void *dst_addr;                    /*!< Адрес Приемника */
    bool int_en;                       /*!< Разрешение прерываний */
} dma_multiblock_config_t;

/*!
 * @name API HAL для драйвера модуля DMA
 * @{
 */

/*!
 * @brief Включение DMA
 *
 * @param base Базовый адрес DMA
 */
void DMA_Init(DMA_Type *base);

/*!
 * @brief Отключение DMA
 *
 * @param base Базовый адрес DMA
 */
void DMA_Deinit(DMA_Type *base);

/*!
 * @brief Инициализация структуры dma_handle_t
 *
 * @param handle  Управляющая структура передачи
 * @param base    Базовый адрес DMA
 * @param channel Номер канала
 *
 * @retval #DMA_Status_Success
 * @retval #DMA_Status_NoBase
 */
dma_status_t DMA_CreateHandle(dma_handle_t *handle, DMA_Type *base,
    uint32_t channel);

/*!
 * @brief Активация канала
 *
 * @param base    Базовый адрес DMA
 * @param channel Номер канала
 */
static inline void DMA_EnableChannel(DMA_Type *base, uint32_t channel)
{
    base->CHENREG_LO |= (uint32_t)((1 << (channel + 8)) | (1 << channel));
}

/*!
 * @brief Выключение канала
 *
 * @param base    Базовый адрес DMA
 * @param channel Номер канала
 */
void DMA_DisableChannel(DMA_Type *base, uint32_t channel);

/*!
 * @brief Установка запроса на передачу с периферией
 *
 * @param base    Базовый адрес DMA
 * @param channel Номер канала
 */
static inline void DMA_EnableChannelPeriphRq(DMA_Type *base,
    uint32_t channel)
{
    base->REQDSTREG_LO = ((1 << (channel + 8)) | (1 << channel));
    base->REQSRCREG_LO = ((1 << (channel + 8)) | (1 << channel));
}

/*!
 * @brief Установка аппаратного интерфейса запроса DMA
 *
 * @param handle  Управляющая структура передачи
 * @param enable  Разрешение аппаратного интерфейса
 * @param req_num Номер периферийного блока
 */
void DMA_HardwareHandshakeEnable(dma_handle_t *handle, bool enable,
    uint32_t req_num);

/*!
 * @brief Установка приоритета канала
 *
 * @param base     Базовый адрес DMA
 * @param channel  Номер канала
 * @param priority Приоритет
 */
void DMA_SetChannelPriority(DMA_Type *base, uint32_t channel,
    dma_priority_t priority
);

/*!
 * @brief Установка функции обратного вызова
 *
 * @param handle    Управляющая структура передачи
 * @param callback  Функция обратного вызова
 * @param user_data Данные пользователя
 */
void DMA_SetCallback(dma_handle_t *handle, dma_callback callback,
    void *user_data);

/*!
 * @brief Функция для проверки работы DMA канала
 *
 * @param base    Базовый адрес DMA
 * @param channel Номер канала
 * @retval true  - канал активен
 * @retval false - канал не активен
 */
static inline bool DMA_ChannelIsActive(DMA_Type *base, uint32_t channel)
{
    return ((base->CHENREG_LO >> channel) & 1);
}

/*!
 * @brief Получение информации о приоритете канала
 *
 * Эта функция считывает приоритет из регистра CFGx.
 *
 * @param base    Базовый адрес DMA
 * @param channel Номер канала
 * 
 * @retval #DMA_ChannelPriority0
 * @retval #DMA_ChannelPriority1
 * @retval #DMA_ChannelPriority2
 * @retval #DMA_ChannelPriority3
 * @retval #DMA_ChannelPriority4
 * @retval #DMA_ChannelPriority5
 * @retval #DMA_ChannelPriority6
 * @retval #DMA_ChannelPriority7
 */
dma_priority_t DMA_GetChannelPriority(DMA_Type *base, uint32_t channel);

/*!
 * @brief Получение маски для конфигурации пересылки
 *
 * @param ctlxcfg Описание конфигурации пересылки
 * 
 * @return Маска для конфигурации пересылки
 */
uint64_t DMA_GetCTLCfgMask(struct dma_channel_ctl_cfg *ctlxcfg);

/*!
 * @brief Подготовка канала к передаче
 *
 * Эта функция инициализирует структуру dma_channel_config_t
 *
 * @param config    Указатель на структуру конфигурации канала
 * @param src_addr  Начальный адрес Источника
 * @param dst_addr  Начальный адрес Приемника
 * @param xfer_cfg  Конфигурация регистра CTL
 */
void DMA_PrepareChannelTransfer(dma_channel_config_t *config,
    void *src_addr,
    void *dst_addr,
    uint64_t xfer_cfg
);

/*!
 * @brief Отправление конфигурации передачи
 *
 * Функция инициализирует регистры канала в соответствии с параметрами структуры dma_channel_config_t.
 * Используется для одноблочной передачи:
 * @code
    DMA_CreateHandle(handle, base, channel);
    uint64_t xfercfg = DMA_CHANNEL_CTL(
        LEN, true, true,
        DMA_MemoryToMemory_DMA,
        false, false,
        DMA_BurstSize4, DMA_BurstSize4,
        DMA_Incr, DMA_Incr,
        DMA_Transfer32BitWidth, DMA_Transfer32BitWidth,
        true
        );
    DMA_PrepareChannelTransfer(config, src_addr, dst_addr, xfercfg)
    DMA_SubmitChannelTransfer(handle, config);
    DMA_StartTransfer(handle)
 * @endcode
 *
 * @param handle Структура передачи
 * @param config Структура конфигурации передачи DMA канала
 *
 * @retval #DMA_Status_Success
 * @retval #DMA_Status_Busy
 * @retval #DMA_Status_NoBase
 */
dma_status_t DMA_SubmitChannelTransfer(dma_handle_t *handle,
    dma_channel_config_t *config
);

/*!
 * @brief Отправка дескриптора передачи
 *
 * Функция используется для конфигурации канала дескриптором, который начнет
 * Многоблочную передачу.
 * @code
    DMA_SetupDescriptor(next_desc0, xfercfg, src_addr, dst_addr, next_desc1);
    DMA_SetupDescriptor(next_desc1, xfercfg, src_addr, dst_addr, next_desc2);
    DMA_SetupDescriptor(next_desc2, xfercfg, src_addr, dst_addr, NULL);
    DMA_CreateHandle(handle, base, channel)
    uint64_t xfercfg = DMA_CHANNEL_CTL(
        LEN, true, true,
        DMA_MemoryToMemory_DMA,
        false, false,
        DMA_BurstSize4, DMA_BurstSize4,
        DMA_Incr, DMA_Incr,
        DMA_Transfer32BitWidth, DMA_Transfer32BitWidth,
        true
        );
    DMA_SubmitChannelDescriptor(handle, next_desc0)
    DMA_StartTransfer(handle)
 * @endcode
 *
 * @param handle     Управляющая структура передачи
 * @param descriptor Дескриптор передачи
 */
void DMA_SubmitChannelDescriptor(dma_handle_t *handle,
    dma_descriptor_t *descriptor
);

/*!
 * @brief Расчёт количества дескрипторов многоблочной передачи
 *
 * @param size_bytes Объем данных в байтах
 * @param transfer_width Ширина одного слова
 *
 * @return Количество дескрипторов многоблочной передачи
 */
uint32_t DMA_GetDescriptorCount(uint32_t size_bytes, uint8_t transfer_width);

/*!
 * @brief Прерывание передачи без потери данных
 *
 * @param handle Управляющая структура передачи
 */
void DMA_AbortTransfer(dma_handle_t *handle);

/*!
 * @brief Начать транзакцию
 *
 * Функция выгружает дескриптор из таблицы дескрипторов и
 * инициализирует регистр LLPx адресом выгруженного дескриптора.
 * После чего включается канал и начинается передача.
 *
 * @param handle Управляющая структура передачи
 *
 * @retval #DMA_Status_Success
 * @retval #DMA_Status_NoBase
 * @retval #DMA_Status_Busy
 */
dma_status_t DMA_StartTransfer(dma_handle_t *handle);

/*!
 * @brief Разрешение источника прерывания
 *
 * Функция разрешает формирование прерывания типа inttype. Также для
 * возникновения сигнала прерывания важно установить поле INT_EN в 1
 * в регистре CTLx.
 *
 * @param base    Базовый адрес DMA
 * @param channel Номер канала
 * @param inttype Тип прерывания
 */
void DMA_EnableChannelInterrupt(DMA_Type *base, uint32_t channel,
    uint8_t int_mask);

/*!
 * @brief Отключение источника прерывания
 *
 * @param base    Базовый адрес DMA
 * @param channel Номер канала
 * @param inttype Тип прерывания
 */
void DMA_DisableChannelInterrupt(DMA_Type *base, uint32_t channel,
    uint8_t int_mask);

/*!
 * @brief Инициализация дескриптора
 *
 * @param desc      Дескриптор DMA
 * @param xfercfg   Конфигурация передачи для дескриптора
 * @param src_addr  Адрес источника
 * @param dst_addr  Адрес приемника
 * @param next_desc Следующий дескриптор
 */
void DMA_SetupDescriptor(dma_descriptor_t *desc, uint64_t xfercfg,
    void *src_addr, void *dst_addr,
    void *next_desc
);

/*!
 * @brief Отправка конфигурации передачи в таблицу дескрипторов
 *
 * Использование:
 * Одноблочная передача
 * @code
    uint64_t xfercfg = DMA_CHANNEL_CTL(
        LEN, false, false,
        DMA_MemoryToMemory_DMA,
        false, false,
        DMA_BurstSize4, DMA_BurstSize4,
        DMA_Incr, DMA_Incr,
        DMA_Transfer32BitWidth, DMA_Transfer32BitWidth,
        true
        );
    DMA_CreateHandle(handle, base, channel)
    DMA_SubmitChannelTransferParameter(handle, xfercfg, src_addr, dst_addr);
    DMA_StartTransfer(handle)
 * @endcode
 *
 * @param handle    Управляющая структура передачи
 * @param xfer_cfg  Маска конфигурации пересылки
 * @param src_addr  Адрес данных передатчика
 * @param dst_addr  Адрес данных приемника
 *
 * @retval #DMA_Status_Success
 * @retval #DMA_Status_Busy
 * @retval #DMA_Status_NoBase
 */
dma_status_t DMA_SubmitChannelTransferParameter(dma_handle_t *handle,
    uint64_t xfer_cfg, void *src_addr, void *dst_addr
);

/*!
 * @brief Обработчик прерываний
 *
 * @param handle Дескриптор DMA
 */
void DMA_ChannelIRQHandle(dma_handle_t *handle);

/*!
 * @brief Разрешение режимов Разброса/Сбора
 *
 * @param ctlxcfg Конфигурация пересылки
 * @param scatter_en Разрешение режима Разброса
 * @param gather_en Разрешение режима Сбора
 */
void DMA_ScatterGatherEnable(struct dma_channel_ctl_cfg *ctlxcfg,
    bool scatter_en, bool gather_en);

/*!
 * @brief Настройка режима сбора источника
 *
 * @param handle   Управляющая структура передачи
 * @param count    Счетчик сбора
 * @param interval Интервал сбора
 */
void DMA_SetupSrcGather(dma_handle_t *handle, uint32_t count, uint32_t interval);

/*!
 * @brief Настройка режима разброса приемника
 *
 * @param handle   Управляющая структура передачи
 * @param count    Счетчик разброса
 * @param interval Интервал разброса
 */
void DMA_SetupDstScatter(dma_handle_t *handle, uint32_t count, uint32_t interval);

/*!
 * @brief Функция, инициализирующая DMA дескрипторы многоблочной передачи.
 *
 * @param desc   Указатель на начало массива дескрипторов многоблочной передачи
 * @param config Разрешение прерываний
 */
void DMA_InitMultiblockDescriptor(dma_descriptor_t *desc,
    dma_multiblock_config_t *config);

/*!
 * @}
 */

#endif /* HAL_DMA_H */

/*!
 * @}
 */
