/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




#include <eliot1_board.h>

static const board_pll_cfg_t board_xti_configs[] = {
    {
        .xti_freq = BOARD_XTI_0MHZ,
        .pll_main_clk = BOARD_HFI_MAINCLK_FREQ,
        .pll_sel = BOARD_HFI_PLL_SEL,
        .pll_man = BOARD_HFI_PLL_MAN_EN,
        .pll_od = BOARD_HFI_PLL_MAN_OD,
        .pll_nf = BOARD_HFI_PLL_MAN_NF,
        .pll_nr = BOARD_HFI_PLL_MAN_NR
    },
    {
        .xti_freq = BOARD_XTI_16MHZ,
        .pll_main_clk = 300000000,
        .pll_sel = 0,
        .pll_man = 1,
        .pll_od = 1,
        .pll_nf = 75,
        .pll_nr = 4
    },
    {
        .xti_freq = BOARD_XTI_24MHZ,
        .pll_main_clk = 300000000,
        .pll_sel = 0,
        .pll_man = 1,
        .pll_od = 1,
        .pll_nf = 25,
        .pll_nr = 2
    },
};

static const board_power_cfg_t board_power_configs[] = {
    /* BOARD_PowerConfig_0V9 */
    {
        .vlevel       = POWER_DcdcVLevel0,
        .fclk_freq    = 50000000,
        .sysclk_freq  = 25000000,
        .gnssclk_freq = 50000000,
        .qspiclk_freq = 30000000,
        .i2sclk_freq  = 16000000
    },
    /* BOARD_PowerConfig_1V0 */
    {
        .vlevel       = POWER_DcdcVLevel1,
        .fclk_freq    = 100000000,
        .sysclk_freq  = 50000000,
        .gnssclk_freq = 50000000,
        .qspiclk_freq = 60000000,
        .i2sclk_freq  = 20000000
    },
    /* BOARD_PowerConfig_1V1 */
    {
        .vlevel       = POWER_DcdcVLevel2,
        .fclk_freq    = 150000000,
        .sysclk_freq  = 50000000,
        .gnssclk_freq = 75000000,
        .qspiclk_freq = 75000000,
        .i2sclk_freq  = 25000000
    },
};

static const board_pll_cfg_t *selected_pll_config = NULL;
static const board_power_cfg_t *selected_pwr_config = NULL;

const board_pll_cfg_t *BOARD_FindAndSelectPLLConfig(uint32_t xti_hz)
{
    uint32_t cfg_idx;

    for (cfg_idx = 0; cfg_idx < DIM(board_xti_configs); cfg_idx++) {
        if (board_xti_configs[cfg_idx].xti_freq == xti_hz) {
            BOARD_SelectPLLConfig(&board_xti_configs[cfg_idx]);
            return &board_xti_configs[cfg_idx];
        }
    }
    BOARD_SelectPLLConfig(NULL);
    return NULL;
}

void BOARD_SelectPLLConfig(const board_pll_cfg_t *pll_cfg)
{
    selected_pll_config = pll_cfg;
}

board_cfg_status_t BOARD_ApplyPLLConfig(CLKCTR_Type *clkctr_base)
{
    const board_pll_cfg_t *cfg = selected_pll_config;
    if (cfg == NULL) {
        return BOARD_Status_UnknownConfig;
    }

    CLKCTR_SetXTI(clkctr_base, cfg->xti_freq);

    if (cfg->pll_man) {
        if (CLKCTR_SetPllMan(clkctr_base,
                cfg->xti_freq,
                cfg->pll_nr,
                cfg->pll_nf,
                cfg->pll_od) != CLKCTR_Status_Ok) {
            return BOARD_Status_HWError;
        }
    } else {
        CLKCTR_SetPll(clkctr_base, cfg->xti_freq, cfg->pll_sel);
    }
    return BOARD_Status_OK;
}

void BOARD_InitPWR(PWRCTR_Type *pwrctr_base, uint32_t vlvl, bool ext_dcdc)
{
    struct power_config pwr_cfg = {0};

    POWER_GetCurrentConfig(pwrctr_base, &pwr_cfg);
    pwr_cfg.dcdc_enable = !ext_dcdc;
    pwr_cfg.run_configuration.dcdc_level = vlvl;
    POWER_SetConfig(pwrctr_base, &pwr_cfg);
}

void BOARD_PrepareFlashToFreq(FCTR_Type *base, uint32_t freq)
{
    if (freq <= FLASH_NO_READ_LATENCY_MAX_FREQ) {
        SET_VAL_MSK(base->PSP_CTRL, FCTR_PSP_CTRL_READ_LATENCY_Msk,
            FCTR_PSP_CTRL_READ_LATENCY_Pos, 0);
    } else {
        /* Увеличение latency при больших частотах */
        SET_VAL_MSK(base->PSP_CTRL, FCTR_PSP_CTRL_READ_LATENCY_Msk,
            FCTR_PSP_CTRL_READ_LATENCY_Pos, 1);
    }
}

void BOARD_SelectPowerConfig(const board_power_cfg_t *pwr_cfg)
{
    selected_pwr_config = pwr_cfg;
}

const board_power_cfg_t *BOARD_FindAndSelectPowerConfig(uint32_t pwr_cfg_idx)
{
    if (pwr_cfg_idx < DIM(board_power_configs)) {
        BOARD_SelectPowerConfig(&board_power_configs[pwr_cfg_idx]);
        return &board_power_configs[pwr_cfg_idx];
    }
    BOARD_SelectPowerConfig(NULL);
    return NULL;
}

board_cfg_status_t BOARD_ApplyPowerConfig(CLKCTR_Type *clkctr_base,
    PWRCTR_Type *pwrctr_base, FCTR_Type *fctr_base, bool ext_dcdc)
{
    if (clkctr_base == NULL || pwrctr_base == NULL || fctr_base == NULL) {
        return BOARD_Status_InvalidArgs;
    }
    const board_power_cfg_t *pwr_cfg = selected_pwr_config;
    if (pwr_cfg == NULL) {
        return BOARD_Status_UnknownConfig;
    }
    const board_pll_cfg_t *pll_cfg = selected_pll_config;
    if (pll_cfg == NULL) {
        return BOARD_Status_UnknownConfig;
    }
    CLKCTR_SetPll(clkctr_base, 0, 0); /* Установка минимальной частоты */
    BOARD_InitPWR(pwrctr_base, pwr_cfg->vlevel, ext_dcdc); /* Установка напряжения ядра */
    BOARD_PrepareFlashToFreq(fctr_base, pwr_cfg->sysclk_freq);

    CLKCTR_SetSysDiv(clkctr_base,
        (pll_cfg->pll_main_clk / pwr_cfg->fclk_freq) - 1,  /* FCLK DIV */
        (pwr_cfg->fclk_freq / pwr_cfg->sysclk_freq) - 1,   /* SYSCLK DIV */
        (pll_cfg->pll_main_clk / pwr_cfg->gnssclk_freq) - 1,  /* GNSSCLK DIV */
        (pll_cfg->pll_main_clk / pwr_cfg->qspiclk_freq) - 1); /* QSPI DIV */

    board_cfg_status_t status = BOARD_ApplyPLLConfig(clkctr_base);
    if (status != BOARD_Status_OK) {
        return status;
    }
    if (FLASH_Init(fctr_base) != FLASH_Status_Ok) {
        return BOARD_Status_HWError;
    }
    return BOARD_Status_OK;
}

uint32_t BOARD_Config_GetFCLKFreq()
{
    if (selected_pwr_config == NULL) {
        return 0;
    }
    return selected_pwr_config->fclk_freq;
}

uint32_t BOARD_Config_GetSysCLKFreq()
{
    if (selected_pwr_config == NULL) {
        return 0;
    }
    return selected_pwr_config->sysclk_freq;
}

uint32_t BOARD_Config_GetGNSSCLKFreq()
{
    if (selected_pwr_config == NULL) {
        return 0;
    }
    return selected_pwr_config->gnssclk_freq;
}

uint32_t BOARD_Config_GetQSPICLKFreq()
{
    if (selected_pwr_config == NULL) {
        return 0;
    }
    return selected_pwr_config->qspiclk_freq;
}
