/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2024 Elvees (?)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 * This file is part of the TinyUSB stack.
 */

#include "eliot1_board.h"
#include "common/tusb_verify.h"
#include "bsp/board_api.h"

#define USB_REF_192MHZ 0
#define USB_REF_20MHZ  1
#define USB_REF_24MHZ  2
#define USB_REF_16MHZ  3
#define USB_REF_32MHZ  3
#define USB_REF_26MHZ  6
#define USB_REF_50MHZ  7

#define USB_DELAY_BASE    (100)
#define USB_DELAY_MIN     (50)
#define USB_DELAY_MAX     (1000)
#define USB_DELAY_MAX_MAX (10000)

static void usb_subsystem_get_out_of_reset(void);
static void eliot1_phy_ref_clk_setup(void);

//--------------------------------------------------------------------+
// Forward USB interrupt events to TinyUSB IRQ Handler
//--------------------------------------------------------------------+
void USB_Handler(void) {
  uint8_t rhport;
#if CFG_TUH_ENABLED
  rhport = BOARD_TUH_RHPORT;
#else
  rhport = BOARD_TUD_RHPORT;
#endif
  tusb_int_handler(rhport, true);
}

//--------------------------------------------------------------------+
// MACRO TYPEDEF CONSTANT ENUM
//--------------------------------------------------------------------+
void board_init(void) {
#ifndef ELIOT1M_TFC
  BOARD_InitAll();
#endif
  eliot1_phy_ref_clk_setup();
  usb_subsystem_get_out_of_reset();
}

//--------------------------------------------------------------------+
// Board porting API
//--------------------------------------------------------------------+

void board_led_write(bool state) {
  (void)state;
  // !TODO implement
}

uint32_t board_button_read(void) {
  // !TODO implement
  return 0;
}

static bool sym_ready = false;
static struct uart_handle huart;
static struct uart_transfer xfer;
static uint32_t size;
static uint32_t rx_buff;

void UART_Callback(UART_Type *base,
  struct uart_handle *handle, enum uart_status status, void *user_data)
{
  UNUSED(base);
  UNUSED(handle);
  UNUSED(status);
  UNUSED(user_data);

  sym_ready = true;
}

#ifdef BOARD_CONSOLE_UART
int board_getchar(void) {
  if (!sym_ready && huart.rx_state == 0) {
    UART_TransferCreateHandle(BOARD_CONSOLE_UART,
      &huart, UART_Callback, NULL);
    xfer.rx_data = (uint8_t *) &rx_buff;
    xfer.data_size = 1;
    size = 1;
    UART_TransferReceiveNonBlocking(BOARD_CONSOLE_UART, &huart,
      &xfer, (size_t *) &size);
    return 0;
  } else if (sym_ready) {
    huart.rx_state = 0;
    sym_ready = false;
    return (int)((char)rx_buff);
  }
  return 0;
}

int board_uart_read(uint8_t *buf, int len) {
  int cnt = 0;
  char ch = 0;
  for (; len > 0 && ch != '\r'; len--) {
    ch = UART_ReadByteWait(BOARD_CONSOLE_UART);
    UART_WriteByteWait(BOARD_CONSOLE_UART, ch);
    *buf++ = ch;
    cnt += 1;
  }

  *buf = '\n';
  UART_WriteByteWait(BOARD_CONSOLE_UART, '\n');
  cnt++;

  return cnt;
}

int board_uart_write(void const *buf, int len) {
  int i = 0;
  const uint8_t *ptr = buf;
  for (i = 0; i < len; i += 1) {
    UART_WriteByteWait(BOARD_CONSOLE_UART, *ptr++);
  }

  return len;
}
#else
int board_getchar(void) {
  (void) huart;
  (void) xfer;
  (void) size;
  (void) rx_buff;

  return 0;
}

int board_uart_read(uint8_t *buf, int len) {
  (void) buf;
  (void) len;

  return 0;
}

int board_uart_write(void const *buf, int len) {
  (void) buf;
  (void) len;

  return 0;
}
#endif

size_t board_get_unique_id(uint8_t id[], size_t max_len) {
  (void) id;
  (void) max_len;
  // !TODO implement
  return 0xbaadf00d;
}

#if CFG_TUSB_OS == OPT_OS_NONE
volatile uint32_t system_ticks = 0;

void SysTick_Handler(void) {
  // !TODO implement
  system_ticks++;
}

uint32_t board_millis(void) {
  return (uint32_t)(BOARD_GetTime() / 1000UL);
}

uint32_t tusb_time_millis_api(void) {
  return board_millis();
}

#endif

void HardFault_Handler(void) {
  TU_BREAKPOINT();
}

void usb_subsystem_get_out_of_reset(void)
{
  USB_URB_Type *urb_base = USB_URB_Secure;

  BOARD_Udelay(USB_DELAY_BASE);
  urb_base->USBSYSRESETN = 0x0;
  BOARD_Udelay(USB_DELAY_BASE);

  BOARD_Udelay(USB_DELAY_MAX_MAX);

  USB_URB_USBSYSRESETN_RESETN_PTYPE_SET(urb_base, 1);
  BOARD_Udelay(USB_DELAY_MIN);

  /* set utmi clk */
  USB_URB_USBSYSRESETN_PORTRESETN_PTYPE_SET(urb_base, 1);
  BOARD_Udelay(USB_DELAY_MIN);

  /* PRSTN released from 0 to 1 */
  USB_URB_USBSYSRESETN_PRSTN_PTYPE_SET(urb_base, 1);
  BOARD_Udelay(USB_DELAY_MIN);

  /* HRSTN released from 0 to 1 */
  USB_URB_USBSYSRESETN_HRESETN_PTYPE_SET(urb_base, 1);

  BOARD_Udelay(USB_DELAY_MIN);
}

void eliot1_phy_ref_clk_setup(void)
{
  USB_URB_Type *urb_base = USB_URB_Secure;
  enum clkctr_usbclk source_clk = CLKCTR_USBClkTypeXTIClk;

  int32_t pllbtune = -1;
  int32_t fsel = -1;

  uint32_t xti_freq_mhz = BOARD_XTI_FREQ;
  const uint32_t pllbtune_enable_values[] = {
    19200000UL,
    20000000UL,
    24000000UL,
    26000000UL,
    32000000UL
  };

  const uint32_t pllbtune_disable_values[] = {
    50000000UL
  };

  for (uint32_t i = 0UL; i < DIM(pllbtune_enable_values); i++) {
    if (xti_freq_mhz == pllbtune_enable_values[i]) {
      pllbtune = 1;
      switch (xti_freq_mhz) {
        case 19200000:
          fsel = USB_REF_192MHZ;
          break;
        case 20000000:
          fsel = USB_REF_20MHZ;
          break;
        case 24000000:
          fsel = USB_REF_24MHZ;
          break;
        case 26000000:
          fsel = USB_REF_26MHZ;
          break;
        case 32000000:
          fsel = USB_REF_32MHZ;
          break;
      }
    }
  }

  if (pllbtune < 0) {
    for (uint32_t i = 0UL; i < DIM(pllbtune_disable_values); i++) {
      if (xti_freq_mhz == pllbtune_disable_values[i]) {
        pllbtune = 0; 
        break;
      }
    }
  }

  if (pllbtune < 0) {
    pllbtune = 0;
    fsel = USB_REF_16MHZ;
    source_clk = CLKCTR_USBClkTypeHFIClk;
  }

  CLKCTR_SetSwitchUSBClk(CLKCTR, source_clk);

  USB_URB_USBPHYREFFREQSEL_PLLBTUNE_PTYPE_SET(urb_base, pllbtune);
  USB_URB_USBPHYREFFREQSEL_FSEL_PTYPE_SET(urb_base, fsel);
}
