/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




#include "eliot1_board.h"
#include "hal_jtm.h"
#include "hal_rwc.h"
#include "hal_wdt.h"

#define EMBEDDED_CLI_IMPL
#undef UNUSED
#include "embedded_cli.h"

#include "cli_time_support.h"

#define TEST_TCAL      129  /*!< Значение калибровочного параметра TCAL для партии микросхем MPW */
#define TEST_WCAL      1311 /*!< Значение калибровочного параметра WCAL для партии микросхем MPW */
#define TEST_WTCONF    0x3D /*!< Значение калибровочного параметра WTCONF для партии микросхем MPW */
#define TEST_WTCALCONF 0    /*!< Значение калибровочного параметра WTCALCONF для партии микросхем MPW */

#define RX_RING_BUFFER_SIZE 8192U /*!< Размер кольцевого буфера приема */
#define TX_BUFFER_SIZE      2U    /*!< Размер буфера передачи */

static EmbeddedCli *cli;

uint8_t rx_ring_buff[RX_RING_BUFFER_SIZE] = {0}; /*!< Кольцевой буфер для хранения принимаемых данных */
uint8_t send_buff[TX_BUFFER_SIZE] = {0};         /*!< Буфер для хранения передаваемых данных */

size_t received_bytes;                /*!< Количество полученных байт во время транзакции чтения */

volatile bool rx_on_going = false;    /*!< Состояние RX буфера: true - ожидает ввод */

volatile bool esc_pressed = false;    /*!< Флаг нажатия ESC */
volatile bool arrow_sequence = false; /*!< Флаг идентификации последовательности символов для стрелок */

jtm_status_t JTM_status;      /*!< Статус JTM модуля */
enum rwc_status RWC_status;   /*!< Статус RWC модуля */
enum uart_status UART_status; /*!< Статус UART модуля */

struct uart_transfer receive_xfer; /*!< Буфер приема символа */
struct uart_transfer send_xfer;    /*!< Буфер отправки символа */

struct uart_handle uart_handle;    /*!< Дескриптор состояния UART для неблокирующих функций обмена */

/* Объявления функций обратных вызовов терминальных команд. */

 /*!
  * @brief Функция печати символа для CLI.
  * 
  * @param cli Структура CLI.
  * @param c   Символ для отображения в CLI.
  */
void EmbeddedCliWriteChar(EmbeddedCli *cli, char c);

 /*!
  * @brief Функция-обработчик для неизвестной команды.
  * 
  * @param cli Структура CLI.
  * @param cmd Структура команды.
  */
void EmbeddedCliOnCommand(EmbeddedCli *cli, CliCommand *cmd);

/* Блок объявления пользовательских команд. */

/*!
 * @brief Функция-обработчик команды whoami.
 * 
 * @param cli     Структура CLI.
 * @param args    Параметры команды в виде строки.
 * @param context Не используется.
 */
void EmbeddedCliOnWhoAmI(EmbeddedCli *cli, char *args, void *context);

/*!
 * @brief Функция-обработчик команды cpuinfo.
 * 
 * @param cli     Структура CLI.
 * @param args    Параметры команды в виде строки.
 * @param context Не используется.
 */
void EmbeddedCliOnCPUinfo(EmbeddedCli *cli, char *args, void *context);

/*!
 * @brief Функция-обработчик команды timenow.
 * 
 * @param cli     Структура CLI.
 * @param args    Параметры команды в виде строки.
 * @param context Не используется.
 */
void EmbeddedCliOnTimenow(EmbeddedCli *cli, char *args, void *context);

/*!
 * @brief Функция-обработчик команды settime.
 * 
 * @param cli     Структура CLI.
 * @param args    Параметры команды в виде строки.
 * @param context Не используется.
 */
void EmbeddedCliOnSetTime(EmbeddedCli *cli, char *args, void *context);

/*!
 * @brief Функция-обработчик команды reboot.
 * 
 * @param cli     Структура CLI.
 * @param args    Параметры команды в виде строки.
 * @param context Не используется.
 */
void EmbeddedCliOnReboot(EmbeddedCli *cli, char *args, void *context);

/*!
 * @brief Функция обработки символа для CLI.
 *
 * Выполняет анализ переданного символа.
 * 
 * @param sym Обрабатываемый символ.
 */
void AnalyzeChar(char sym) 
{
    if (sym == 0) {
        return;
    }

    /* Если введен ненулевой символ. */
    if (sym != '\0') {
        /* Если ранее был введен ESC, проверяем следующий символ
         * на последовательность UpArrow/DownArrow.
         */
        if (esc_pressed && (sym == '[')) {
            arrow_sequence = true;
            return;
        } else if (arrow_sequence && (sym == 'A' || sym == 'B')) {
            /* Если введена полная последовательность UpArrow/DownArrow
             * передача символов в CLI.
             */
            embeddedCliReceiveChar(cli, '\x1B');
            embeddedCliReceiveChar(cli, '[');
            embeddedCliReceiveChar(cli, sym);
            embeddedCliProcess(cli);
            return;
        } else {
            /* Сброс всех флагов. */
            esc_pressed = false;
            arrow_sequence = false;
        }

        /* Общая проверка введенного символа: если введен ESC,
         * на следующем прерывании будет выполнена специальная проверка.
         */
        if (sym == 27U) {
            esc_pressed = true;
            return;
        } else {
            /* Передача введенного символа в CLI. */
            embeddedCliReceiveChar(cli, sym);
            embeddedCliProcess(cli);
        }
    }
}

/*!
 * @brief Функция обратного вызова по прерыванию UART.
 * 
 * @param base      Указатель на базовый адрес UART.
 * @param handle    Дескриптор UART.
 * @param status    Возвращаемый статус операций UART.
 * @param user_data Данные пользователя.
 */
void USARTUserCallBack(UART_Type *base, struct uart_handle *handle,
                       enum uart_status status, void *user_data)
{
    UNUSED(base);
    UNUSED(handle);
    UNUSED(user_data);
    UNUSED(status);

    rx_on_going = false;
}

int main()
{
    static jtm_config_t jtm_config;    /*!< Параметры JTM */
    struct rwc_config rwc_config;      /*!< Параметры RWC */
    
    uint16_t prev_sym_pos = 0; /*!< Позиция первого ненулевого символа при чтении из кольцевого буфера */

    BOARD_InitAll();

    /* Конфигурация UART. */
    UART_status = UART_TransferCreateHandle(BOARD_CONSOLE_UART, &uart_handle,
        USARTUserCallBack, NULL);
    if (UART_status != UART_Status_Ok) {
        printf("UART_TransferCreateHandle error : %d\r\n", UART_status);
    }

    UART_status = UART_TransferStartRingBuffer(BOARD_CONSOLE_UART, &uart_handle,
        rx_ring_buff, RX_RING_BUFFER_SIZE);

    if (UART_status != UART_Status_Ok) {
        printf("UART_TransferStartRingBuffer error : %d\r\n", UART_status);
    }          

    /* Инициализация буфера приема. */
    receive_xfer.rx_data   = rx_ring_buff;
    receive_xfer.data_size = RX_RING_BUFFER_SIZE;

    /* Инициализация буфера отправки. */
    send_xfer.tx_data   = send_buff;
    send_xfer.data_size = 2;

    /* Конфигурация JTM. */
    jtm_config.tcal = TEST_TCAL;
    jtm_config.wcal = TEST_WCAL;
    jtm_config.wtconf = TEST_WTCONF;
    jtm_config.wtcalconf = TEST_WTCALCONF;

    JTM_Init(JTM_Secure, &jtm_config);

    /* Конфигурация RWC. */

    /* Сброс и запрет прерываний. */
    NVIC_ClearPendingIRQ(RWC_ALARM_IRQn);
    NVIC_DisableIRQ(RWC_ALARM_IRQn);
    RWC_EnableAlarmTimerInterruptFromDPD(RWC_Secure, 0);
    RWC_InterruptClear(RWC_Secure);

    /* Получение конфигурации по умолчанию и инициализация таймера. */
    RWC_GetDefaultConfig(&rwc_config);
    RWC_Init(RWC_Secure, rwc_config);

    /* Конфигурация CLI. */
    cli = embeddedCliNewDefault();

    /* Проверка инициализации CLI. */
    if (cli == NULL) {
        printf("CLI was not created. Check sizes!\r\n");
        return -1;
    }

    cli->onCommand = &EmbeddedCliOnCommand;
    cli->writeChar = &EmbeddedCliWriteChar;

    /* Непосредственное добавление команд в CLI. */
    embeddedCliAddBinding(cli, (CliCommandBinding)
    {
        "whoami",
        "Return name of device",
        false,
        NULL,
        EmbeddedCliOnWhoAmI
    });

    embeddedCliAddBinding(cli, (CliCommandBinding)
    {
        "cpuinfo",
        "Print info about SOC",
        false,
        NULL,
        EmbeddedCliOnCPUinfo
    });

    embeddedCliAddBinding(cli, (CliCommandBinding)
    {
        "timenow",
        "Print current time of RWC timer",
        false,
        NULL,
        EmbeddedCliOnTimenow
    });

    embeddedCliAddBinding(cli, (CliCommandBinding)
    {
        "settime",
        "Set time/date of RWC timer\r\n\tSupported modes:\r\n\t1) settime date dd.mm.yyyy\r\n\t2) settime time hh:mm:ss\r\n\t3) settime dd.mm.yyyy hh:mm:ss",
        true,
        NULL,
        EmbeddedCliOnSetTime
    });

    embeddedCliAddBinding(cli, (CliCommandBinding)
    {
        "reboot",
        "Reboot the CPU",
        false,
        NULL,
        EmbeddedCliOnReboot
    });

    printf("%s UART Embedded CLI example\r\n", BOARD_NAME);
    printf("CLI was started. Enter your commands\r\nUse command help to see all commands are available\r\n");

    /* Инициирование вывода "пригласительного" символа '<'. */
    embeddedCliReceiveChar(cli, '\0');
    embeddedCliProcess(cli);

    while (1) {
        /* Считывание следующего символа из кольцевого буфера. */
        char next_sym = rx_ring_buff[prev_sym_pos];

        if (next_sym != 0) {
            AnalyzeChar(next_sym);

            rx_ring_buff[prev_sym_pos] = 0U;

            /* Если счетчик дошёл до конца, выполняется его сброс. */
            if (++prev_sym_pos == RX_RING_BUFFER_SIZE) {
                prev_sym_pos = 0;
            }
            next_sym = rx_ring_buff[prev_sym_pos];
        }
        
        if (!rx_on_going) {
            rx_on_going = true;
            UART_TransferReceiveNonBlocking(BOARD_CONSOLE_UART, &uart_handle,
                &receive_xfer, &received_bytes);

            if (received_bytes == RX_RING_BUFFER_SIZE) {
                rx_on_going = false;
            }
        }
    }

    BOARD_Udelay(1000000);

    asm volatile("bkpt");

    return 0;
}

/* Определения функций обратных вызовов исполняемых терминальных команд. */

void EmbeddedCliOnCommand(EmbeddedCli *cli, CliCommand *cmd)
{
    UNUSED(cli);
    printf("Unknown command \"%s\". Use help to see what commands are available\r\n",
        cmd->name);
}

void EmbeddedCliWriteChar(EmbeddedCli *cli, char c)
{
    UNUSED(cli);

    send_buff[0] = (int8_t) c;
    UART_TransferSendNonBlocking(BOARD_CONSOLE_UART, &uart_handle, &send_xfer);

    while (uart_handle.tx_state) {};
}

void EmbeddedCliOnWhoAmI(EmbeddedCli *cli, char *args, void *context)
{
    UNUSED(cli);
    UNUSED(args);
    UNUSED(context);

    printf("%s\r\n", BOARD_NAME);
}

void EmbeddedCliOnCPUinfo(EmbeddedCli *cli, char *args, void *context)
{
    UNUSED(cli);
    UNUSED(args);
    UNUSED(context);

    int32_t value;
    uint32_t sys_clk_mhz = CLKCTR_GetSysClk(CLOCK_BASE) / 1000000;
    uint32_t f_clk_mhz = CLKCTR_GetFClk(CLOCK_BASE) / 1000000;

    printf("%-15s: Armv8-M\r\n", "Architecture");
    printf("%-15s: Cortex-M33\r\n", "Model name");
    printf("%-15s: %ldMHz\r\n", "Core 0 (freq.)", sys_clk_mhz);
    printf("%-15s: %ldMHz\r\n", "Core 1 (freq.)", f_clk_mhz);

    JTM_status = JTM_GetParameterValue(JTM_Secure, JTM_Temperature, &value);
    printf("%-15s: %.03f\r\n", "Temperature", value / 1000.);

    JTM_status = JTM_GetParameterValue(JTM_Secure, JTM_Vcasn, &value);
    printf("%-15s: %.03f\r\n", "V CASN", value / 1000.);

    JTM_status = JTM_GetParameterValue(JTM_Secure, JTM_Vcore, &value);
    printf("%-15s: %.03f\r\n", "V CORE (VDDC)", value / 1000.);
}

void EmbeddedCliOnTimenow(EmbeddedCli *cli, char *args, void *context)
{
    UNUSED(cli);
    UNUSED(args);
    UNUSED(context);

    rtc_datetime_t time_now1, time_now2; /*!< Структура даты и времени для контроля чтения из RWC */
    uint32_t iter = 0;                   /*!< Количество попыток записи в RWC */

    /* Чтение данных из RWC. */
    do {
        RWC_status = RWC_GetDatetime(RWC_Secure, &time_now1);
        if (RWC_status != RWC_Status_Ok) {
            printf("RWC_GetDatetime error : %d\r\n", RWC_status);
            while (1)
                ;
        }

        RWC_status = RWC_GetDatetime(RWC_Secure, &time_now2);
        if (RWC_status != RWC_Status_Ok) {
            printf("RWC_GetDatetime error : %d\r\n", RWC_status);
            while (1)
                ;
        }

        iter++;
    } while (CompareTime(&time_now1, &time_now2) == 0 && iter < 10);

    /* Вывод ошибки, если за 10 попыток не удалось корректно считать время. */
    if (CompareTime(&time_now1, &time_now2) == 0) {
        printf("CompareTime error : read different values from single register\r\n");
        return;
    }

    printf ("Date: %02d.%02d.%02d\r\nTime: %02d:%02d:%02d\r\n",
        time_now1.day,
        time_now1.month,
        time_now1.year,
        time_now1.hour,
        time_now1.minute,
        time_now1.second);
}

void EmbeddedCliOnReboot(EmbeddedCli *cli, char *args, void *context)
{
    UNUSED(cli);
    UNUSED(args);
    UNUSED(context);

    struct wdt_config config; /* Конфигурация watch-dog таймера */

    uint32_t frequency = CLKCTR_GetSysClk(CLOCK_BASE);

    /* Установка таймера на 0.5 сек с разрешением прерывания и сброса. */
    WDT_GetDefaultConfig(&config);
    config.load = 0.5 * frequency;
    config.inten = WDT_IntenEnable;
    config.resen = WDT_ResenEnable;
    WDT_Init(SWDT_Secure, &config);

    printf("Rebooting...\r\n");
    while (1)
        ;
}

void EmbeddedCliOnSetTime(EmbeddedCli *cli, char *args, void *context)
{
    UNUSED(cli);
    UNUSED(context);

    rtc_datetime_t time_now1, time_now2;    /*!< Структура даты и времени для контроля чтения/записи RWC */
    uint32_t iter = 0;                      /*!< Количество попыток записи чтения и записи RWC */
    enum dtime_status_t status;             /*!< Возвращаемое значение функции установки времени/даты */

    /* Чтение данных из RWC. */
    do {
        RWC_status = RWC_GetDatetime(RWC_Secure, &time_now1);
        if (RWC_status != RWC_Status_Ok) {
            printf("RWC_GetDatetime error : %d\r\n", RWC_status);
            return;
        }

        RWC_status = RWC_GetDatetime(RWC_Secure, &time_now2);
        if (RWC_status != RWC_Status_Ok) {
            printf("RWC_GetDatetime error : %d\r\n", RWC_status);
            return;
        }
        iter++;
    } while (CompareTime(&time_now1, &time_now2) == 0 && iter < 10);

    /* Вывод ошибки, если за 10 попыток не удалось корректно считать время. */
    if (CompareTime(&time_now1, &time_now2) == 0) {
        printf("CompareTime error : read different values from single register\r\n");
        return;
    }

    /* Выполнение функции обработки введенных пользователем параметров
     * и проверка возвращаемого значения.
     */
    status = SetDatetime(&time_now1, args);
    if (status != Dtime_Status_SetDateOk && status != Dtime_Status_SetTimeOk) {
        printf("SetDatetime error : %s\r\n", InfoDtimeStatus(&status));
    }

    iter = 0;

    /* Запись новых значений даты/времени в RWC. */
    do {
        RWC_status = RWC_SetDatetime(RWC_Secure, &time_now1);
        if (RWC_status != RWC_Status_Ok) {
            printf("RWC_SetDatetime error : %d\r\n", RWC_status);
            return;
        }

        RWC_status = RWC_GetDatetime(RWC_Secure, &time_now2);
        if (RWC_status != RWC_Status_Ok) {
            printf("RWC_GetDatetime error : %d\r\n", RWC_status);
            return;
        }
        iter++;
    } while (CompareTime(&time_now1, &time_now2) == 0 && iter < 10);
}
