/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




#include "eliot1_board.h"
#include "hal_power.h"
#include "hal_ppu.h"

#define WAIT_TIME   (3U)

/* Структура конфигурации POWER */
struct power_config pwr_cfg;

/* Структура конфигурации RWC */
struct rwc_config rwc_cfg;

/* Стартовое время */
rtc_datetime_t start_time = {
    .year = 2024,
    .month = 7,
    .day = 12,
    .hour = 12,
    .minute = 5,
    .second = 5,
};

/* Текущее время */
rtc_datetime_t current_time;

/* Форматированный вывод даты/времени */
static void DisplayDatetime(rtc_datetime_t time)
{
    printf("Date: %02d.%02d.%02d\tTime: %02d:%02d:%02d\r\n",
        time.day, time.month, time.year,
        time.hour, time.minute, time.second);
}

/* Вывод текущего времени */
static void DisplayCurrentDateTime()
{
    RWC_GetDatetime(RWC_Secure, &current_time);
    DisplayDatetime(current_time);
}

/* Сравнение значений времени */
static uint32_t CompareTime(rtc_datetime_t t1, rtc_datetime_t t2)
{
    if (t1.year != t2.year)
        return 0;
    if (t1.month != t2.month)
        return 0;
    if (t1.day != t2.day)
        return 0;
    if (t1.hour != t2.hour)
        return 0;
    if (t1.minute != t2.minute)
        return 0;
    if (t1.second != t2.second)
        return 0;
    return 1;
}

/* Ожидание WAIT_TIME секунд и погружение в сон */
void standby_func()
{
    union rwc_union_reg ticks0, ticks1; /* Для хранения времени */
    ticks1.reg_value = 0;
    ticks0.reg_value = 0;
    uint32_t curr_seconds = 0;
    uint32_t alarm_seconds = 0;

#ifdef BOARD_DEBUG_LED0_PIN
    GPIO_PinToggle(BOARD_DEBUG_LED0_PIN);
#endif

    RWC_GetSecondsTimerCount(RWC_Secure, &ticks0.reg_value);
    alarm_seconds = ticks0.reg_value + WAIT_TIME;

    DisplayCurrentDateTime();

    printf("Switching to standby mode after\r\n");
    /* Ожидание 5 секунд */
    while (ticks1.reg_value < alarm_seconds) {
        /* Получение времени */
        RWC_GetSecondsTimerCount(RWC_Secure, &ticks1.reg_value);

        /*
         * Проверка обновления времени - выводится только
         * время отличное от предыдущего
        */
        if (ticks1.reg_value != ticks0.reg_value) {
            ticks0.reg_value = ticks1.reg_value;
            printf (" %lu\r\n", alarm_seconds - ticks1.reg_value + 1);
        }
    }

    /* Установка времени на WAIT_TIME секунд вперед */
    RWC_GetSecondsTimerCount(RWC_Secure, &curr_seconds);
    uint32_t standby_seconds = curr_seconds + WAIT_TIME;
    RWC_SetSecondsTimerMatch(RWC_Secure, standby_seconds);

    union rwc_union_reg reg;

    if (RWC_Status_Ok != RWC_GetInternalRegister(RWC_Secure, RWC_Config, &reg)){
        printf("RWC_GetInternalRegister error\r\n");
        while (1);
    }
    /* WKUP_STATE */
    reg.config.wake_in_en =  1;
    if (RWC_Status_Ok != RWC_SetInternalRegister(RWC_Secure, RWC_Config, reg)){
        printf("RWC_SetInternalRegister error\r\n");
        while (1);
    }

    /* Настройка прерываний */
    RWC_EnableAlarmTimerInterruptFromDPD(RWC_Secure, 1);
    NVIC_ClearPendingIRQ(RWC_ALARM_IRQn);
    NVIC_EnableIRQ(RWC_ALARM_IRQn);

    DisplayCurrentDateTime();

    printf("Switching to standby mode...\r\n");

    /*
     * Задержка для корректного вывода последнего сообщения
     * перед уходом в standby mode
     */
    BOARD_Udelay(200);

    /* Погружение процессора в сон */
    PPU_SetPDCMPPUSense(PPU_DomainSYS, PPU_SenseSYS, 0);

    if (POWER_Standby(PWRCTR_Secure) != POWER_Status_Ok) {
        printf("ERROR POWER_Standby()\r\n");
        while (1);
    }
}

/* Обработчик прерывания */
void RWC_ALARM_Handler(void)
{
    BOARD_InitAll();
    printf("RWC_ALARM Interrupt: woke up!\r\n");

    RWC_EnableAlarmTimerInterruptFromDPD(RWC_Secure, 0);
    RWC_InterruptClear(RWC_Secure);
    NVIC_ClearPendingIRQ(RWC_ALARM_IRQn);
    NVIC_DisableIRQ(RWC_ALARM_IRQn);

    __DSB();

    standby_func();
}

int main()
{
    /* Инициализация BSP части */
    BOARD_InitAll();

#ifdef BOARD_DEBUG_LED0_PIN
    GPIO_PinMode_GPIO (BOARD_DEBUG_LED0_PIN, GPIO_DigitalOutput);
    GPIO_PinWrite(BOARD_DEBUG_LED0_PIN, 0);
#endif

    /* Сброс и запрет прерываний */
    RWC_EnableAlarmTimerInterruptFromDPD(RWC_Secure, 0);
    RWC_InterruptClear(RWC_Secure);
    NVIC_ClearPendingIRQ(RWC_ALARM_IRQn);
    NVIC_DisableIRQ(RWC_ALARM_IRQn);

    /* Вывод информации о старте примера */
    printf("%s Standby example start\r\n", BOARD_NAME);
    printf("SysClk = %ld Hz\r\n", CLKCTR_GetSysClk(CLOCK_BASE));

    /* Получение конфигурации по умолчанию и инициализация таймера */
    RWC_GetDefaultConfig(&rwc_cfg);
    RWC_Init(RWC_Secure, rwc_cfg);

    /* Запись стартового времени */
    uint32_t recording_attempts = 0;
    while (1) {
        enum rwc_status rwc_status = RWC_Status_Ok;
        if (RWC_Status_Ok != (rwc_status = RWC_SetDatetime(RWC_Secure, &start_time))) {
            printf("RWC_SetDatetime error, returned %d\r\n", rwc_status);
            while (1);
        }
        if (RWC_Status_Ok != (rwc_status = RWC_GetDatetime(RWC_Secure, &current_time))) {
            printf("RWC_GetDatetime error, returned %d\r\n", rwc_status);
            while (1);
        }
        recording_attempts++;
        if (CompareTime(start_time, current_time))
            break;
    }
    printf("Set datetime successfully on %ld attempt\r\n", recording_attempts);

    standby_func();

    while (1);
}
