/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @defgroup vtu_driver Драйвер модуля VTU
 *
 * @brief Драйвер универсального блока таймеров
 *
 * Драйвер модуля универсального блока таймеров.
 */

/*!
 * @addtogroup vtu_driver
 * @{
 */

/*!
 * @file hal_vtu.h
 *
 * @brief Интерфейс драйвера универсального блока таймеров
 */

#ifndef HAL_VTU_H
#define HAL_VTU_H

#include "hal_common.h"

/*!
 * @brief Статусы драйвера универсального блока таймеров
 */
enum vtu_status {
    VTU_Status_Ok                 = 0, /*!< Нет ошибок */
    VTU_Status_InvalidArgument    = 1, /*!< Недопустимый аргумент */
    VTU_Status_TimerBusy          = 2, /*!< Таймер уже занят */
    VTU_Status_BadConfigure       = 3, /*!< Недопустимая конфигурация */
    VTU_Status_DriverError        = 4, /*!< Ошибка драйвера */
    VTU_Status_DualTimerNotCanRun = 5, /*!< Cдвоенный таймера неможет быть запущен,
                                        * так как второй таймер уже работает */
    VTU_Status_TimerNotInit       = 6, /*!< Таймер не инициализирован */
};

/*!
 * @brief Режимы работы тамеров универсального блока таймеров
 */
enum vtu_mode {
    VTU_LowPower    = 0, /*!< Режим остановки таймера */
    VTU_PWMDual8Bit = 1, /*!< Режим сдвоенного 8-битного таймера */
    VTU_PWM16Bit    = 2, /*!< Режим 16-битного таймера */
    VTU_Capture     = 3, /*!< Режим захвата */
};

/*!
 * @brief Управление фронтами при режиме захвата
 */
enum vtu_capture_edge_control {
    VTU_CaptureRisingEdgeResetNo        = 0, /*!< Захват по возрастающему фронту, сброса счетчика нет */
    VTU_CaptureFallingEdgeResetNo       = 1, /*!< Захват по ниспадающему фронту, сброса счетчика нет */
    VTU_CaptureRisingEdgeResetYes       = 2, /*!< Захват по возрастающему фронту, сброс счетчика есть */
    VTU_CaptureFallingEdgeResetYes      = 3, /*!< Захват по ниспадающему фронту, сброс счетчика есть */
    VTU_CaptureBothEdgeResetNo          = 4, /*!< Захват по возрастающему фронту, сброса счетчика нет */
    VTU_CaptureBothEdgeResetRisingEdge  = 5, /*!< Захват по возрастающему фронту, сброс счетчика по возрастающему фронту */
    VTU_CaptureBothEdgeResetFallingEdge = 6, /*!< Захват по возрастающему фронту, сброс счетчика ниспадающему фронту */
    VTU_CaptureBothEdgeResetBothEdge    = 7, /*!< Захват по возрастающему фронту, сброс счетчика обоим фронтам */
};

/*!
 * @brief Управление полярностью ШИМ
 */
enum vtu_pwm_polarity {
    VTU_One  = 0, /*!< Высокий уровень импульса */
    VTU_Zero = 1, /*!< Низкий уровень импульса */
};

/*!
 * @brief Управление прерываниями
 */
enum vtu_interrupt_control {
    VTU_NoInterrupt            = 0, /*!< Отключение всех прерываний */
    VTU_LowByteDutyCycleMatch  = 1, /*!< По совпадению цикла для первого сдвоенного таймера */
    VTU_LowBytePeriodMatch     = 2, /*!< По совпадению периода для первого сдвоенного таймера */
    VTU_HighByteDutyCycleMatch = 4, /*!< По совпадению цикла для второго сдвоенного таймера */
    VTU_HighBytePeriodMatch    = 8, /*!< По совпадению периода для второго сдвоенного таймера */
    VTU_DutyCycleMatch         = 1, /*!< По совпадению цикла для таймера */
    VTU_PeriodMatch            = 2, /*!< По совпадению периода для таймера */
    VTU_CaptureToPERCAPx       = 1, /*!< Захват по началу импульса */
    VTU_CaptureToDTYCAPx       = 2, /*!< Захват по концу импульса */
    VTU_CounterOverflow        = 4, /*!< По переполнению счетчика */
};

/*! @brief Функция обратного вызова */
typedef void (*vtu_callback)(VTU_Type *base, uint32_t timer,
    enum vtu_interrupt_control value);

/*!
 * @brief Структура для конфигурации VTU
 */
struct vtu_config {
    enum vtu_mode                 mode;                  /*!< Режимы работы тамера, кроме VTU_LowPower */
    enum vtu_capture_edge_control capture_edge_control1; /*!< Управление фронтами захвата для режима захвата для TIO1 */
    enum vtu_capture_edge_control capture_edge_control2; /*!< Управление фронтами захвата для режима захвата для TIO2 */
    enum vtu_pwm_polarity         pwm_polarity;          /*!< Полярность ШИМ */
    enum vtu_pwm_polarity         pwm_polarity2;         /*!< Полярность второго вывода ШИМ для 16-битного режима*/
    enum vtu_interrupt_control    interrupt_control;     /*!< Разрешение прерываний */
    uint8_t                       prescaler;             /*!< Значение предделителя */
    uint16_t                      counter;               /*!< Начальное значение счетчика */
    uint16_t                      period;                /*!< Период ШИМ */
    uint16_t                      duty_cycle_capture;    /*!< Ширина импульса */
};

/*!
 * @brief Номер таймера и его режим работы для прерываний
 */
enum timer_num_mode {
    VTU_Timer0Mode8bit = 5,        /*!< Таймер 0, режим 8 бит */ 
    VTU_Timer0Mode16bit = 9,       /*!< Таймер 0, режим 16 бит */
    VTU_Timer1Mode8bit = 96,       /*!< Таймер 1, режим 8 бит */
    VTU_Timer2Mode8bit = 1280,     /*!< Таймер 2, режим 8 бит */
    VTU_Timer2Mode16bit = 2304,    /*!< Таймер 2, режим 16 бит */
    VTU_Timer3Mode8bit = 24576,    /*!< Таймер 3, режим 8 бит */
};

/*!
 * @name Интерфейс драйвера
 * @{
 */

/*!
 * @name Инициализация и деинициализации таймера
 * @{
 */

/*!
 * @brief Создание конфигурации по умолчанию
 *
 * Функция инициализации структуры с настройками таймера "по умолчанию"
 *
 * @param config Конфигурация таймера
 *
 * @retval #VTU_Status_Ok
 * @retval #VTU_Status_InvalidArgument
 */
enum vtu_status VTU_GetDefaultConfig(struct vtu_config *config);

/*!
 * @brief Инициализация таймера
 *
 * Функция инициализации таймера с указанными настройками
 *
 * @param base        Система VTU
 * @param timer       Таймер в системе VTU
 * @param config      Конфигурация таймера
 *
 * @retval #VTU_Status_Ok
 * @retval #VTU_Status_InvalidArgument
 */
enum vtu_status VTU_Init(VTU_Type *base, uint32_t timer, struct vtu_config *config);

/*!
 * @brief Деинициализация таймера
 *
 * Функция деинициализации таймера
 *
 * @param base  Подсистема VTU
 * @param timer Таймер
 *
 * @retval #VTU_Status_Ok
 * @retval #VTU_Status_InvalidArgument
 */
enum vtu_status VTU_Deinit(VTU_Type *base, uint32_t timer);

/*!
 * @}
 */

/*!
 * @name Функции управления VTU
 * @{
 */

/*!
 * @brief Разрешение работы таймера
 *
 * @param base   Подсистема VTU
 * @param timer  Таймер
 * @param enable Разрешение работы (1) или запрещение работы (0)
 *
 * @return Статус
 */
enum vtu_status VTU_EnableTimer(VTU_Type *base, uint32_t timer, bool enable);

/*!
 * @brief Установка значения счетчика
 *
 * @param base     Подсистема VTU
 * @param timer    Таймер
 * @param value    Значение
 * @param extended Использовать (1) или не использовать (0) двухбайтное значение value
 *
 * @return Статус
 */
enum vtu_status VTU_SetCounter(VTU_Type *base, uint32_t timer,
    uint16_t value, bool extended);

/*!
 * @brief Получение значения счетчика
 *
 * @param base     Подсистема VTU
 * @param timer    Таймер
 * @param extended Вернуть двухбайтовое (1) или однобайтное (0) значение value
 *
 * @return Значение счетчика
 */
uint16_t VTU_GetCounter(VTU_Type *base, uint32_t timer, bool extended);

/*!
 * @brief Установка значения предделителя
 *
 * @param base  Подсистема VTU
 * @param timer Таймер
 * @param value Значение
 *
 * @return Статус
 */
enum vtu_status VTU_SetPrescaler(VTU_Type *base, uint32_t timer,
    uint8_t value);

/*!
 * @brief Получение значения предделителя
 *
 * @param base  Подсистема VTU
 * @param timer Таймер
 *
 * @return Значение предделителя
 */
uint8_t VTU_GetPrescaler(VTU_Type *base, uint32_t timer);

/*!
 * @brief Установка значения периода генерации шим
 * без учета предделителя
 *
 * @param base     Подсистема VTU
 * @param timer    Таймер
 * @param value    Значение
 * @param extended Использовать (1) или не использовать (0) двухбайтное значение value
 *
 * @return Статус
 */
enum vtu_status VTU_SetPeriodCapture(VTU_Type *base, uint32_t timer,
    uint16_t value, bool extended);

/*!
 * @brief Получение значения периода генерации шим
 * без учета предделителя 
 *
 * @param base     Подсистема VTU
 * @param timer    Таймер
 * @param extended Вернуть двухбайтное (1) или однобайтное (0) значение value
 *
 * @return Значение счетчика
 */
uint16_t VTU_GetPeriodCapture(VTU_Type *base, uint32_t timer,
    bool extended);

/*!
 * @brief Установка периода импульса шим без учета предделителя
 *
 * @param base     Подсистема VTU
 * @param timer    Таймер
 * @param value    Значение
 * @param extended Использовать (1) или не использовать (0) двухбайтное значение value
 *
 * @return Статус
 */
enum vtu_status VTU_SetDutyCycleCapture(VTU_Type *base, uint32_t timer,
    uint16_t value, bool extended);

/*!
 * @brief Получение периода импульса шим без учета предделителя
 *
 * @param base     Подсистема VTU
 * @param timer    Таймер
 * @param extended Вернуть двухбайтное (1) или однобайтное (0) значение value
 *
 * @return Значение счетчика
 */
uint16_t VTU_GetDutyCycleCapture(VTU_Type *base, uint32_t timer,
    bool extended);

/*!
 * @brief Установка функции обратного вызова
 * 
 * @param callback Указатель на пользовательскую функцию обратного вызова
 */
void VTU_SetCallback(uint32_t timer, vtu_callback callback);

/*!
 * @brief Разрешение работы прерывания
 *
 * @param base   Подсистема VTU
 * @param timer  Таймер
 * @param value  Прерывания
 * @param enable Разрешены прерывания по маске (1) или запрещены прерывания по маске (0)
 * @param mode   Режим таймера для интерпретации прерываний
 *
 * @return Статус
 */
enum vtu_status VTU_EnableTimerIRQ(VTU_Type *base, uint32_t timer,
    enum vtu_interrupt_control value, bool enable, enum vtu_mode mode);

/*!
 * @brief Получение прерываний
 *
 * @param base  Подсистема VTU
 * @param timer Таймер
 * @param mode  Режим таймера для интерпретации прерываний
 *
 * @return Прерывания
 */
enum vtu_interrupt_control VTU_GetTimerIRQ(VTU_Type *base, uint32_t timer,
    enum vtu_mode mode);

/*!
 * @brief Очистка прерываний
 *
 * @param base   Подсистема VTU
 * @param timer  Таймер
 * @param values Прерывания
 * @param mode   Режим таймера для интерпретации прерываний
 *
 * @return Прерывания
 */    
enum vtu_status VTU_ClearTimerIRQ(VTU_Type *base, uint32_t timer,
    enum vtu_interrupt_control values, enum vtu_mode mode);

/*!
 * @brief Установка полярности ШИМ
 *
 * @param base       Подсистема VTU
 * @param timer      Таймер
 * @param value1     Полярность первого сигнала ШИМ
 * @param value2     Полярность второго сигнала ШИМ
 * @param use_value2 Использовать (1) или не использовать (0) второй сигнал ШИМ
 *
 * @return Статус
 */
enum vtu_status VTU_SetPWMPolarity(VTU_Type *base, uint32_t timer,
    enum vtu_pwm_polarity value1, enum vtu_pwm_polarity value2,
    bool use_value2);


/*!
 * @brief Получение полярности ШИМ
 *
 * @param base       Подсистема VTU
 * @param timer      Таймер
 * @param value1     Полярность первого сигнала ШИМ
 * @param value2     Полярность второго сигнала ШИМ
 * @param use_value2 Использовать (1) или не использовать (0) второй сигнал ШИМ
 *
 * @return Статус 
 */
enum vtu_status VTU_GetPWMPolarity(VTU_Type *base, uint32_t timer,
    enum vtu_pwm_polarity *value1, enum vtu_pwm_polarity *value2,
    bool use_value2);

/*!
 * @brief Установка типа захвата
 *
 * @param base       Подсистема VTU
 * @param timer      Таймер
 * @param value1     Полярность первого сигнала ШИМ
 * @param value2     Полярность второго сигнала ШИМ
 * @param use_value2 Использовать (1) или не использовать (0) второй сигнал ШИМ
 *
 * @return Статус
 */
enum vtu_status VTU_SetCaptureEdgeCtrl(VTU_Type *base, uint32_t timer,
    enum vtu_capture_edge_control value1,
    enum vtu_capture_edge_control value2,
    bool use_value2);

/*!
 * @brief Получение типа захвата
 *
 * @param base   Подсистема VTU
 * @param timer  Таймер
 * @param value1     Полярность первого сигнала ШИМ
 * @param value2     Полярность второго сигнала ШИМ
 * @param use_value2 Использовать (1) или не использовать (0) второй сигнал ШИМ
 *
 * @return Статус 
 */
enum vtu_status VTU_GetCaptureEdgeCtrl(VTU_Type *base, uint32_t timer,
    enum vtu_capture_edge_control *value1,
    enum vtu_capture_edge_control *value2,
    bool use_value2);

/*!
 * @}
 */

/*!
 * @brief Получение статуса выполнения функции,
 * тип результата которой отличен от enum vtu_status
 *
 * @return Статус
 */
enum vtu_status VTU_GetLastAPIStatus(void); 

/*!
 * @}
 */

#endif /* HAL_VTU_H */

/*!
 * @}
 */
