/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @addtogroup uart_driver
 * @{
 */

/*!
 * @file hal_uart_dma.h
 *
 * @brief Дополнение драйвера UART с пересылкой данных через DMA
 */

#ifndef HAL_SPI_DMA_H
#define HAL_SPI_DMA_H

#include "hal_dma.h"
#include "hal_uart.h"

/*! @brief Версия драйвера */
#define HAL_UART_DMA_DRIVER_VERSION (MAKE_VERSION(1, 0, 0))

/*!
 * @brief Дескриптор UART-DMA передачи
 */
typedef struct _uart_dma_handle uart_dma_handle_t;

/*! @brief Функция обратного вызова */
typedef void (*uart_dma_transfer_callback_t)(UART_Type *base,
    uart_dma_handle_t *handle,
    enum uart_status status,
    void *user_data);

/*!
 * @brief Дескриптор UART-DMA передачи
 */
struct _uart_dma_handle {
    UART_Type *base;                       /*!< Базовый адрес UART */

    uart_dma_transfer_callback_t callback; /*!< Функция обратного вызова */
    void *user_data;                       /*!< Пользовательские данные */
    uint32_t rx_data_size_all;             /*!< Число данных на прием */
    uint32_t tx_data_size_all;             /*!< Число данных на передачу */

    dma_handle_t *tx_dma_handle;           /*!< Дескриптор DMA на передачу */
    dma_handle_t *rx_dma_handle;           /*!< Дескриптор DMA на прием */

    dma_descriptor_t *tx_desc;
    dma_descriptor_t *rx_desc;

    volatile uint8_t tx_state;             /*!< Состояние шины TX */
    volatile uint8_t rx_state;             /*!< Состояние шины RX */
};

#if defined(__cplusplus)
extern "C" {
#endif

/*!
 * @brief Функция для инициализации полей дескриптора UART-DMA
 *
 * @param base          Базовый адрес UART
 * @param handle        Дескриптор UART-DMA передачи
 * @param callback      Функция обратного вызова
 * @param user_data     Пользовательские данные
 * @param tx_dma_handle Дескриптор DMA для передачи данных
 * @param rx_dma_handle Дескриптор DMA для приема данных
 */
void UART_TransferCreateHandleDMA(UART_Type *base,
    uart_dma_handle_t *handle,
    uart_dma_transfer_callback_t callback,
    void *user_data, dma_handle_t *tx_dma_handle,
    dma_handle_t *rx_dma_handle);

/*!
 * @brief Функция осуществляющая передачу данных по UART,
          данные в буфер попадают через DMA канал
 *
 * @param base   Базовый адрес UART
 * @param handle Дескриптор UART-DMA
 * @param xfer   Структура, содержащая указатель на буфер передачи и
 *               размер передаваемых данных
 *
 * @retval #UART_DMA_Status_InvalidArgument
 * @retval #UART_DMA_Status_TxBusy
 * @retval #UART_DMA_Status_DMA_Busy
 * @retval #UART_DMA_Status_Fail
 * @retval #UART_DMA_Status_Success
 */
enum uart_status UART_TransferSendDMA(UART_Type *base,
    uart_dma_handle_t *handle, struct uart_transfer *xfer);

/*!
 * @brief Функция осуществляющая передачу данных по UART,
          данные в буфер попадают через DMA канал
 *
 * @param base   Базовый адрес UART
 * @param handle Дескриптор UART-DMA
 * @param xfer   Структура, содержащая указатель на буфер приема и
 *               размер принимаемых данных
 *
 * @retval #UART_DMA_Status_InvalidArgument
 * @retval #UART_DMA_Status_RxBusy
 * @retval #UART_DMA_Status_DMA_Busy
 * @retval #UART_DMA_Status_Fail
 * @retval #UART_DMA_Status_Success
 */
enum uart_status UART_TransferReceiveDMA(UART_Type *base,
    uart_dma_handle_t *handle, struct uart_transfer *xfer);

/*!
 * @brief Функция прерывающая передачу данных между UART(TX) и DMA
 *
 * @param base   Базовый адрес UART
 * @param handle Дескриптор UART-DMA
 */
void UART_TransferAbortSendDMA(UART_Type *base, uart_dma_handle_t *handle);

/*!
 * @brief Функция прерывающая передачу данных между UART(RX) и DMA
 *
 * @param base   Базовый адрес UART
 * @param handle Дескриптор UART-DMA
 */
void UART_TransferAbortReceiveDMA(UART_Type *base, uart_dma_handle_t *handle);

/*!
 * @brief Ожидание завершения передачи.
          Выход из функции будет осуществлен по окончании UART транзакций
 *
 * @param base Базовый адрес UART
 */
static inline void UART_WaitWhileActive(UART_Type *base)
{
    /* Ожидание завершения передачи */
    while (base->USR & UART_USR_BUSY_Msk);
}

/*!
 * @brief Инициализация дескрипторов DMA для многоблочной передачи TX
 *
 * @param base       Базовый адрес SPI
 * @param desc       Указатель на массив дескрипторов
 * @param count      Количество DMA дескрипторов
 * @param data_size  Общее число передаваемых данных
 * @param data_width Ширина одного передаваемого слова
 * @param src_addr   Адрес Источника
 */
static inline void UART_DMADescriptorInitTX(UART_Type *base,
    dma_descriptor_t *desc, uint32_t count, uint32_t data_size,
    uint32_t data_width, void *src_addr)
{
    dma_multiblock_config_t config = {
        .count          = count,
        .data_size      = data_size,
        .transfer_type  = DMA_MemoryToPeripheral_DMA,
        .scatter_en     = false,
        .gather_en      = false,
        .src_burst_size = DMA_BurstSize32,
        .dst_burst_size = DMA_BurstSize1,
        .src_incr       = DMA_Incr,
        .dst_incr       = DMA_NoChange,
        .src_data_width = data_width,
        .dst_data_width = data_width,
        .src_addr       = src_addr,
        .dst_addr       = (void *) &base->THR,
        .int_en         = true
    };

    DMA_InitMultiblockDescriptor(desc, &config);
}

/*!
 * @brief Инициализация дескрипторов DMA для многоблочной передачи TX
 *
 * @param base       Базовый адрес SPI
 * @param desc       Указатель на массив дескрипторов
 * @param count      Количество DMA дескрипторов
 * @param data_size  Общее число передаваемых данных
 * @param data_width Ширина одного передаваемого слова
 * @param dst_addr   Адрес Приемника
 */
static inline void UART_DMADescriptorInitRX(UART_Type *base,
    dma_descriptor_t *desc, uint32_t count, uint32_t data_size,
    uint32_t data_width, void *dst_addr)
{
    dma_multiblock_config_t config = {
        .count          = count,
        .data_size      = data_size,
        .transfer_type  = DMA_PeripheralToMemory_DMA,
        .scatter_en     = false,
        .gather_en      = false,
        .src_burst_size = DMA_BurstSize1,
        .dst_burst_size = DMA_BurstSize32,
        .src_incr       = DMA_NoChange,
        .dst_incr       = DMA_Incr,
        .src_data_width = data_width,
        .dst_data_width = data_width,
        .src_addr       = (void *) &base->RBR,
        .dst_addr       = dst_addr,
        .int_en         = true
    };

    DMA_InitMultiblockDescriptor(desc, &config);
}

#if defined(__cplusplus)
}
#endif

#endif  /* HAL_UART_DMA_H */

/*!
 * @}
 */
