/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @addtogroup uart_driver
 * @{
 */

/*!
 * @file hal_uart.c
 *
 * @brief Имплементация драйвера UART
 */

#include "hal_uart.h"
#include "hal_ioim.h"

/*!
 * @brief Состояние процесса приема/передачи
 *
 * Используется при обмене данными по прерыванию.
 */
enum {
    UART_TxIdle = 0U, /*!< TX свободен */
    UART_TxBusy = 1U, /*!< TX занят */
    UART_RxIdle = 2U, /*!< RX свободен */
    UART_RxBusy = 3U, /*!< RX занят */
};

size_t UART_TransferGetRxRingBufferLength(struct uart_handle *handle)
{
    size_t size;
    uint16_t rx_ring_buffer_head = handle->rx_ring_buffer_head;
    uint16_t rx_ring_buffer_tail = handle->rx_ring_buffer_tail;

    /* Проверка аргументов. */
    assert(handle != NULL);

    if (rx_ring_buffer_tail > rx_ring_buffer_head) {
        size = (size_t) rx_ring_buffer_head + handle->rx_ring_buffer_size - (size_t) rx_ring_buffer_tail;
    } else {
        size = (size_t) rx_ring_buffer_head - (size_t) rx_ring_buffer_tail;
    }

    return size;
}

size_t UART_TransferGetTxRingBufferLength(struct uart_handle *handle)
{
    size_t size;
    uint16_t tx_ring_buffer_head = handle->tx_ring_buffer_head;
    uint16_t tx_ring_buffer_tail = handle->tx_ring_buffer_tail;

    /* Проверка аргументов. */
    assert(handle != NULL);

    if (tx_ring_buffer_tail > tx_ring_buffer_head) {
        size = (size_t) tx_ring_buffer_head + handle->tx_ring_buffer_size - (size_t) tx_ring_buffer_tail;
    } else {
        size = (size_t) tx_ring_buffer_head - (size_t) tx_ring_buffer_tail;
    }

    return size;
}

/*!
 * @brief Заполнен ли кольцевой приемный буфер
 *
 * @param handle Указатель на дескриптор
 *
 * @return true  - полностью заполен
 * @return false - есть место
 */
static bool UART_TransferIsRxRingBufferFull(struct uart_handle *handle)
{
    return (UART_TransferGetRxRingBufferLength(handle) == (handle->rx_ring_buffer_size - 1U));
}

/*!
 * @brief Заполнен ли кольцевой буфер отправки
 *
 * @param handle Указатель на дескриптор
 *
 * @return true  - полностью заполен
 * @return false - есть место
 */
static bool UART_TransferIsTxRingBufferFull(struct uart_handle *handle)
{
    return (UART_TransferGetTxRingBufferLength(handle) == (handle->tx_ring_buffer_size - 1U));
}

enum uart_status UART_TransferStartRingBuffer(UART_Type *base, struct uart_handle *handle,
    uint8_t *ring_buffer, size_t ring_buffer_size)
{
    /* Проверка аргументов. */
    if ((base == NULL) || (handle == NULL) || (ring_buffer == NULL)) {
        return UART_Status_InvalidArgument;
    }

    /* Настройка кольцевого буфера. */
    handle->rx_ring_buffer      = ring_buffer;
    handle->rx_ring_buffer_size = ring_buffer_size;
    handle->rx_ring_buffer_head = 0U;
    handle->rx_ring_buffer_tail = 0U;

    /* IER.ERBFI - прерывание по доступности полученных данных. */
    SET_VAL_MSK(base->IER, UART_IER_ERBFI_Msk, UART_IER_ERBFI_Pos, 1U);

    return UART_Status_Ok;
}

enum uart_status UART_TransferStartTxRingBuffer(UART_Type *base,
    struct uart_handle *handle,
    uint8_t *tx_ring_buffer, size_t ring_buffer_size)
{
    /* Проверка аргументов. */
    if ((base == NULL) || (handle == NULL) || (tx_ring_buffer == NULL)
            || ring_buffer_size == 0) {
        return UART_Status_InvalidArgument;
    }

    /*
     * Отключить IRQ при настройке дескриптора передачи; если
     * прерывание происходит во время процесса настройки - портится
     * значение дескриптора.
    */
    uint32_t interrupt_mask = UART_GetEnabledInterrupts(base);
    UART_DisableInterrupts(base, interrupt_mask);

    /* Настройка кольцевого буфера. */
    handle->tx_ring_buffer      = tx_ring_buffer;
    handle->tx_ring_buffer_size = ring_buffer_size;
    handle->tx_ring_buffer_head = 0U;
    handle->tx_ring_buffer_tail = 0U;

    UART_EnableInterrupts(base, interrupt_mask);

    return UART_Status_Ok;
}

enum uart_status UART_TransferStopRingBuffer(UART_Type *base, struct uart_handle *handle)
{
    /* Проверка аргументов. */
    if ((base == NULL) || (handle == NULL)) {
        return UART_Status_InvalidArgument;
    }

    /*
     * Если приемник в режиме ожидания, то прерывания блокируются. В
     * случае, если идет прием, прерывания запрещаются в обработчике
     * прерывания.
     *
     * IER.ERBFI - прерывание по доступности полученных данных.
     */
    if (handle->rx_state == (uint8_t) UART_RxIdle) {
        SET_VAL_MSK(base->IER, UART_IER_ERBFI_Msk, UART_IER_ERBFI_Pos, 0U);
    }

    handle->rx_ring_buffer      = NULL;
    handle->rx_ring_buffer_size = 0U;
    handle->rx_ring_buffer_head = 0U;
    handle->rx_ring_buffer_tail = 0U;

    return UART_Status_Ok;
}

enum uart_status UART_TransferStopTxRingBuffer(UART_Type *base, struct uart_handle *handle)
{
    /* Проверка аргументов. */
    if ((base == NULL) || (handle == NULL)) {
        return UART_Status_InvalidArgument;
    }

    if (handle->tx_ring_buffer == NULL) {
        return UART_Status_Ok;
    }
    /*
     * Отключить IRQ при настройке дескриптора передачи; если
     * прерывание происходит во время процесса настройки - портится
     * значение дескриптора.
    */
    uint32_t interrupt_mask = UART_GetEnabledInterrupts(base);
    UART_DisableInterrupts(base, interrupt_mask);

    handle->tx_ring_buffer      = NULL;
    handle->tx_ring_buffer_size = 0U;
    handle->tx_ring_buffer_head = 0U;
    handle->tx_ring_buffer_tail = 0U;

    UART_EnableInterrupts(base, interrupt_mask);

    return UART_Status_Ok;
}

enum uart_status UART_Init(UART_Type *base, const struct uart_config *config,
    uint32_t src_clock_hz)
{
    int result;

    /* Проверка аргументов. */
    if ((base == NULL) || (config == NULL) || (src_clock_hz == 0U)) {
        return UART_Status_InvalidArgument;
    }

    /* Программный сброс UART. */
    SET_VAL_MSK(base->SRR, UART_SRR_UR_Msk, UART_SRR_UR_Pos, 1U);

    /* Включить FIFO. */
    if ((config->enable_txfifo) || (config->enable_rxfifo)) {
        SET_VAL_MSK(base->FCR, UART_FCR_FIFOE_Msk, UART_FCR_FIFOE_Pos, 1U);
    }

    if (config->enable_txfifo) {
        /* Сброс TxFIFO. */
        SET_VAL_MSK(base->FCR, UART_FCR_XFIFOR_Msk, UART_FCR_XFIFOR_Pos, 1U);
        /* Триггер прерывания уровня заполненности TxFIFO. */
        /* SET_VAL_MSK(base->FCR, UART_FCR_TET_Msk, UART_FCR_TET_Pos, config->tx_watermark); */

    }

    if (config->enable_rxfifo) {
        /* Сброс RxFIFO. */
        SET_VAL_MSK(base->FCR, UART_FCR_RFIFOR_Msk, UART_FCR_RFIFOR_Pos, 1U);
        /* Триггер прерывания уровня заполненности RxFIFO. */
        /* SET_VAL_MSK(base->FCR, UART_FCR_RT_Msk, UART_FCR_RT_Pos, config->rx_watermark); */
    }

    /* Настройка конфигурации UART. */
    /* Бит доступа к делителю частоты (DLL/LPDLL и LPDLH), 1 - доступен. */
    SET_VAL_MSK(base->LCR, UART_LCR_DLAB_Msk, UART_LCR_DLAB_Pos, 0U);
    /* Бит обрыва линии. */
    SET_VAL_MSK(base->LCR, UART_LCR_BC_Msk, UART_LCR_BC_Pos, config->break_line);
    /* Ручное управление битом четности. */
    SET_VAL_MSK(base->LCR, UART_LCR_SP_Msk, UART_LCR_SP_Pos, config->parity_manual);
    /* Тип четности (0 - нечет, 1 - чет). */
    SET_VAL_MSK(base->LCR, UART_LCR_EPS_Msk, UART_LCR_EPS_Pos, config->parity_mode);
    /* Передавать/принимать бит четности. */
    SET_VAL_MSK(base->LCR, UART_LCR_PEN_Msk, UART_LCR_PEN_Pos, config->enable_parity);
    /* Количество стоп-битов. */
    SET_VAL_MSK(base->LCR, UART_LCR_STOP_Msk, UART_LCR_STOP_Pos, config->stop_bit_count);
    /* Количество бит данных в передаваемом символе. */
    SET_VAL_MSK(base->LCR, UART_LCR_DLS_Msk, UART_LCR_DLS_Pos, config->bit_count_per_char);
    /* Активация инфракрасного режима. */
    SET_VAL_MSK(base->MCR, UART_MCR_SIRE_Msk, UART_MCR_SIRE_Pos, config->enable_infrared);
    /* Включение автоматического контроля управления потоком возможно только при включенных FIFO (FCR[0] = 1). */
    SET_VAL_MSK(base->MCR, UART_MCR_AFCE_Msk, UART_MCR_AFCE_Pos, config->enable_hardware_flow_control);
    /* Активация петлевого режима. */
    SET_VAL_MSK(base->MCR, UART_MCR_LOOPBACK_Msk, UART_MCR_LOOPBACK_Pos, config->enable_loopback);

    /* Установка скорости. */
    result = UART_SetBaudRate(base, config->baudrate_bps, src_clock_hz);
    if (result != UART_Status_Ok) {
        return result;
    }

    {
        /* Сброс регистра состояния линии. */
        volatile uint32_t temp = base->LSR;
        UNUSED(temp);
    }

    return UART_Status_Ok;
}

enum uart_status UART_Deinit(UART_Type *base)
{
    /* Проверка аргументов. */
    if (base == NULL) {
        return UART_Status_InvalidArgument;
    }

    /*
     * LSR.TEMT - бит отсутствия передаваемых данных в FIFO буфере и
     * сдвиговом регистре передатчика.
     */
    while (GET_VAL_MSK(base->LSR, UART_LSR_TEMT_Msk, UART_LSR_TEMT_Pos) == 0U)
        ;

    /* Программный сброс UART. */
    SET_VAL_MSK(base->SRR, UART_SRR_UR_Msk, UART_SRR_UR_Pos, 1U);

    /* Отключить все прерывания. */
    base->IER = 0U;

    /* ij отключить запросы dma */

    return UART_Status_Ok;
}

enum uart_status UART_GetDefaultConfig(struct uart_config *config)
{
    /* Проверка аргументов. */
    if (config == NULL) {
        return UART_Status_InvalidArgument;
    }

    /* Инициализируем структуру нулевым значением. */
    (void) memset(config, 0, sizeof(*config));

    /* Инициализируем все поля. */
    config->baudrate_bps                 = 115200U;
    config->enable_parity                = false;
    config->parity_mode                  = UART_ParityOdd;
    config->parity_manual                = false;
    config->stop_bit_count               = UART_OneStopBit;
    config->bit_count_per_char           = UART_8BitsPerChar;
    config->enable_rxfifo                = true;
    config->enable_txfifo                = true;
    config->enable_loopback              = false;
    config->enable_infrared              = false;
    config->enable_hardware_flow_control = false;
    config->break_line                   = false;

    return UART_Status_Ok;
}

enum uart_status UART_SetBaudRate(UART_Type *base, uint32_t baudrate_bps,
    uint32_t src_clock_hz)
{
    float ftemp;
    /* Проверка аргументов. */
    if ((base == NULL) || (baudrate_bps == 0U) || (src_clock_hz == 0U)) {
        return UART_Status_InvalidArgument;
    }

    float fdivisor     =  ((float)src_clock_hz) / ((float)baudrate_bps * 16);
    uint32_t divisor_fraction = (uint32_t)(modff(fdivisor, &ftemp) * 16);
    uint32_t divisor_integer  = (uint32_t)fdivisor;


    /* Бит доступа к делителю частоты (DLL/LPDLL и LPDLH), 1 - доступен. */
    SET_VAL_MSK(base->LCR, UART_LCR_DLAB_Msk, UART_LCR_DLAB_Pos, 1U);

    SET_VAL_MSK(base->DLH, UART_DLH_DLH_Msk, UART_DLH_DLH_Pos, divisor_integer >> 8);
    SET_VAL_MSK(base->DLL, UART_DLL_DLL_Msk, UART_DLL_DLL_Pos, divisor_integer);

    SET_VAL_MSK(base->DLF, UART_DLF_DLF_Msk, UART_DLF_DLF_Pos, divisor_fraction);

    /* Бит доступа к делителю частоты (DLL/LPDLL и LPDLH), 1 - доступен. */
    SET_VAL_MSK(base->LCR, UART_LCR_DLAB_Msk, UART_LCR_DLAB_Pos, 0U);

    return UART_Status_Ok;
}

enum uart_status UART_WriteBlocking(UART_Type *base, const uint8_t *data,
    size_t length)
{
#if UART_RETRY_TIMES
    uint32_t wait_times;
#endif /* UART_RETRY_TIMES */

    /* Проверка аргументов. */
    if ((base == NULL) || (data == NULL)) {
        return UART_Status_InvalidArgument;
    }

    /* Включен ли TxFIFO? */
    if (GET_VAL_MSK(base->FCR, UART_FCR_FIFOE_Msk, UART_FCR_FIFOE_Pos) == 0U) {
        return UART_Status_InvalidArgument;
    }

    /* ----------------------------------------------------------------- *
     * ------------- Цикл по элементам массива на передачу ------------- */
    for (; length > 0U; length--) {

        /* ----------------------------------------------------------------- *
         * --- Ожидание пока TxFIFO не представит место для новых данных --- */
#if UART_RETRY_TIMES
        wait_times = UART_RETRY_TIMES;
        /* Если USR.TFNF == 0, то TxFIFO полон. */
        while ((GET_VAL_MSK(base->USR, UART_USR_TFNF_Msk, UART_USR_TFNF_Pos) == 0U) && (--wait_times != 0U))
#else
        while (GET_VAL_MSK(base->USR, UART_USR_TFNF_Msk, UART_USR_TFNF_Pos) == 0U)
#endif /* UART_RETRY_TIMES */
        {
            ;
        }
        /* --- Ожидание пока TxFIFO не представит место для новых данных --- *
         * ----------------------------------------------------------------- */

#if UART_RETRY_TIMES
        /* Не дождались освобождения буфера TxFIFO? */
        if (wait_times == 0U) {
            return UART_Status_Timeout;
        }
#endif /* UART_RETRY_TIMES */

        /* Записать байт данных в TxFIFO; доступен если LCR[7](DLAB) == 0x0. */
        SET_VAL_MSK(base->THR, UART_THR_THR_Msk, UART_THR_THR_Pos, *data);
        data++;
    }
    /* ------------- Цикл по элементам массива на передачу ------------- *
     * ----------------------------------------------------------------- */

    /* ----------------------------------------------------------------- *
     * ----------------- Ожидание, завершения передачи ----------------- */

#if UART_RETRY_TIMES
    wait_times = UART_RETRY_TIMES;

    /* LSR.TEMT == 1 - отсутствие передаваемых данных в FIFO буфере и сдвиговом регистре передатчика. */
    while ((GET_VAL_MSK(base->LSR, UART_LSR_TEMT_Msk, UART_LSR_TEMT_Pos) == 0U) && (--wait_times != 0U))
#else
    while (GET_VAL_MSK(base->LSR, UART_LSR_TEMT_Msk, UART_LSR_TEMT_Pos) == 0U)
#endif /* UART_RETRY_TIMES */
    {
        ;
    }

#if UART_RETRY_TIMES
    if (wait_times == 0U) {
        return UART_Status_Timeout;
    }
#endif /* UART_RETRY_TIMES */
    /* ----------------- Ожидание, завершения передачи ----------------- *
     * ----------------------------------------------------------------- */

    return UART_Status_Ok;
}

enum uart_status UART_ReadBlocking(UART_Type *base, uint8_t *data, size_t length)
{
    enum uart_status status = UART_Status_Ok;

#if UART_RETRY_TIMES
    uint32_t wait_times;
#endif

    /* Проверка аргументов. */
    if ((base == NULL) || (data == NULL)) {
        return UART_Status_InvalidArgument;
    }

    /* Включен ли FIFO? */
    if (GET_VAL_MSK(base->FCR, UART_FCR_FIFOE_Msk, UART_FCR_FIFOE_Pos) == 0U) {
        return UART_Status_InvalidArgument;
    }

    {
        /* Сброс ошибок, чтение LSR. */
        volatile uint32_t status_flag;
        status_flag = base->LSR;
        UNUSED(status_flag);
    }

    /* Цикл по элементам буфера на заполнение. */
    for (; length > 0U; length--) {

        /* ----------------------------------------------------------------- *
         * ------------ Ожидание новых данных в RxFIFO --------------------- */
#if UART_RETRY_TIMES
        wait_times = UART_RETRY_TIMES;
        while ((GET_VAL_MSK(base->LSR, UART_LSR_DR_Msk, UART_LSR_DR_Pos) == 0U) && (--wait_times != 0U))
#else
        while (GET_VAL_MSK(base->LSR, UART_LSR_DR_Msk, UART_LSR_DR_Pos) == 0U)
#endif
        {
            ;
        }
#if UART_RETRY_TIMES
        if (wait_times == 0U) {
            status = UART_Status_Timeout;
            break;
        }
#endif
        /* ------------ Ожидание новых данных в RxFIFO --------------------- *
         * ----------------------------------------------------------------- */

        /* ------------------------------------------------------- *
         * ------------- Обработка ошибок приема ----------------- */

        /* Проверить наличие ошибок; флаги сбрасываются при чтении LSR. */
        uint32_t status_flag = base->LSR;

        /* LSR.OE - флаг переполнения RxFIFO. */
        if (GET_VAL_MSK(status_flag, UART_LSR_OE_Msk, UART_LSR_OE_Pos) != 0U) {
            /* Cброс RxFIFO. */
            SET_VAL_MSK(base->FCR, UART_FCR_RFIFOR_Msk, UART_FCR_RFIFOR_Pos, 1U);
            status = UART_Status_RxError;
            break;
        }
        /* LSR.PE == 1 - ошибка четности. */
        if (GET_VAL_MSK(status_flag, UART_LSR_PE_Msk, UART_LSR_PE_Pos) != 0U) {
            status = UART_Status_ParityError;
        }
        /* LSR.FE == 1 - ошибка кадрирования в случае равенства нулю стоп-бита. */
        if (GET_VAL_MSK(status_flag, UART_LSR_FE_Msk, UART_LSR_FE_Pos) != 0U) {
            status = UART_Status_FramingError;
        }
        /* LSR.BI == 1 - обрыв линии. */
        if (GET_VAL_MSK(status_flag, UART_LSR_BI_Msk, UART_LSR_BI_Pos) != 0U) {
            status = UART_Status_BreakLineError;
        }

        /* ------------------ Обработка ошибок приема ---------------------- *
         * ----------------------------------------------------------------- */

        if (status != UART_Status_Ok)
            break;

        /* Прием байта данных. */
        *data = (uint8_t) GET_VAL_MSK(base->RBR, UART_RBR_RBR_Msk, UART_RBR_RBR_Pos);
        data++;
    }

    return status;
}

enum uart_status UART_TransferCreateHandle(UART_Type *base,
    struct uart_handle *handle, uart_transfer_callback_t callback,
    void *user_data)
{
    /* Проверка аргументов. */
    if ((base == NULL) || (handle == NULL)) {
        return UART_Status_InvalidArgument;
    }

    (void) memset(handle, 0, sizeof(*handle));

    /* Установить состояние приема и передачи по прерыванию в дескрипторе. */
    handle->rx_state = (uint8_t) UART_RxIdle;
    handle->tx_state = (uint8_t) UART_TxIdle;

    /* Установить обратный вызов и пользовательские данные. */
    handle->callback  = callback;
    handle->user_data = user_data;

    /* Установить обработчик прерывания */
    if (IOIM_SetIRQHandler(base, UART_TransferHandleIRQ, handle)
        != IOIM_Status_Ok)
    {
        return UART_Status_Fail;
    }

    return UART_Status_Ok;
}

enum uart_status UART_TransferSendNonBlocking(UART_Type *base,
    struct uart_handle *handle, struct uart_transfer *xfer)
{
    /* Проверка аргументов. */
    if ((base == NULL) || (handle == NULL) || (xfer == NULL)) {
        return UART_Status_InvalidArgument;
    }

    /* Проверка структуры передачи UART. */
    if ((xfer->data_size == 0U) || (xfer->tx_data == NULL)) {
        return UART_Status_InvalidArgument;
    }

    /* Вернуть ошибку, если уже в процессе передачи. */
    if (handle->tx_state == (uint8_t) UART_TxBusy) {
        return UART_Status_TxBusy;
    } else {
        /*
         * Отключить IRQ при настройке дескриптора передачи; если
         * прерывание происходит во время процесса настройки - портится
         * значение дескриптора.
         */
        uint32_t interrupt_mask = UART_GetEnabledInterrupts(base);
        UART_DisableInterrupts(base, interrupt_mask);

        /* Включен режим кольцевого Tx буфера. */
        if (handle->tx_ring_buffer != NULL) {
            /* Данные не все передаются, если кольцевой буфер переполнен */
            for ( uint32_t i = 0; i < xfer->data_size 
			        && !UART_TransferIsTxRingBufferFull(handle); i++) {
                handle->tx_ring_buffer[handle->tx_ring_buffer_head++] = xfer->tx_data[i];
                if (handle->tx_ring_buffer_head == handle->tx_ring_buffer_size) {
                    handle->tx_ring_buffer_head = 0;
                }
            }
        } else {
            handle->tx_data          = xfer->tx_data;
            handle->tx_data_size     = xfer->data_size;
            handle->tx_data_size_all = xfer->data_size;
            handle->tx_state         = (uint8_t) UART_TxBusy;
        }

        UART_EnableInterrupts(base, interrupt_mask);

        /* IER.ETBEI - прерывание по опустошении регистра Tx. */
        SET_VAL_MSK(base->IER, UART_IER_ETBEI_Msk, UART_IER_ETBEI_Pos, 1U);

        /*ij Здесь должно сработает прерывание по опустошению регистра данных Tx */
    }

    return UART_Status_Ok;
}

enum uart_status UART_WriteTxRing(UART_Type *base, struct uart_handle *handle,
    const uint8_t *data, size_t *length)
{
    /* Проверка аргументов. */
    if ((base == NULL) || (handle == NULL) || (data == NULL)
        || (length == NULL) || (*length == 0)) {
        return UART_Status_InvalidArgument;
    }

    if (handle->tx_ring_buffer == NULL) {
        return UART_Status_TxRingBufferNull;
    }

    /*
     * Отключить IRQ при настройке дескриптора передачи; если
     * прерывание происходит во время процесса настройки - портится
     * значение дескриптора.
    */
    uint32_t interrupt_mask = UART_GetEnabledInterrupts(base);
    UART_DisableInterrupts(base, interrupt_mask);

    /* Данные не все передаются, если кольцевой буфер переполнен */
    for ( ; *length > 0 && !UART_TransferIsTxRingBufferFull(handle); (*length)--) {
        handle->tx_ring_buffer[handle->tx_ring_buffer_head++]
                = *data++;
        if (handle->tx_ring_buffer_head == handle->tx_ring_buffer_size) {
            handle->tx_ring_buffer_head = 0;
        }
    }

    UART_EnableInterrupts(base, interrupt_mask);

    /* IER.ETBEI - прерывание по опустошении регистра Tx. */
    SET_VAL_MSK(base->IER, UART_IER_ETBEI_Msk, UART_IER_ETBEI_Pos, 1U);

    return UART_Status_Ok;
}
enum uart_status UART_TransferAbortSend(UART_Type *base, struct uart_handle *handle)
{
    /* Проверка аргументов. */
    if (handle == NULL) {
        return UART_Status_InvalidArgument;
    }

    /* IER.ETBEI - прерывание по опустошении регистра Tx. */
    SET_VAL_MSK(base->IER, UART_IER_ETBEI_Msk, UART_IER_ETBEI_Pos, 0U);
    /* Очистка TxFIFO. */
    SET_VAL_MSK(base->FCR, UART_FCR_XFIFOR_Msk, UART_FCR_XFIFOR_Pos, 1U);

    handle->tx_data_size = 0U;
    handle->tx_state     = (uint8_t) UART_TxIdle;

    return UART_Status_Ok;
}

enum uart_status UART_TransferGetSendCount(UART_Type *base,
    struct uart_handle *handle, uint32_t *count)
{
    /* Проверка аргументов. */
    if ((handle == NULL) || (count == NULL)) {
        return UART_Status_InvalidArgument;
    }

    if (handle->tx_state == (uint8_t) UART_TxIdle) {
        return UART_Status_NoTransferInProgress;
    }

    *count = handle->tx_data_size_all - handle->tx_data_size -
        GET_VAL_MSK(base->TFL, UART_TFL_TFL_Msk, UART_TFL_TFL_Pos);

    return UART_Status_Ok;
}

enum uart_status UART_TransferReceiveNonBlocking(UART_Type *base,
    struct uart_handle *handle, struct uart_transfer *xfer, size_t *receivedBytes)
{
    uint32_t i;
    size_t bytes_to_copy = 0U;     /* Сколько байтов скопировать из кольцевого буфера в пользовательскую память */
    size_t bytes_to_receive;       /* Сколько байтов получить */
    size_t bytes_current_received; /* Сколько байтов получено на данный момент */
    uint32_t interrupt_mask = 0U;

    /* Проверка аргументов. */
    if ((base == NULL) || (handle == NULL) || (xfer == NULL)) {
        return UART_Status_InvalidArgument;
    }

    /* Проверка структуры передачи UART. */
    if ((xfer->data_size == 0U) || (xfer->rx_data == NULL)) {
        return UART_Status_InvalidArgument;
    }

    /* Уже в процессе получения данных? */
    if (handle->rx_state == (uint8_t) UART_RxBusy) {
        return UART_Status_RxBusy;
    }

    /*
     * Как получить данные:
     *  1. Если кольцевой буфер RX не включен, то сохраняем xfer->data и
     *     xfer->dataSize в дескрипторе UART, разрешаем прерывание для
     *     сохранения полученных данных в xfer->data. Когда все данные
     *     получены, происходит обратный вызов функции, установленной в
     *     UART_TransferCreateHandle().
     *
     *  2. Если кольцевой буфер RX включен и не пуст, сначала получаем
     *     данные из кольцевого буфера.
     *     a) Если в кольцевом буфере достаточно данных, копируем их в
     *        xfer->data и завершаем работу.
     *     б) Если в кольцевом буфере недостаточно данных, копируем их все
     *        в xfer->data, сохраняем xfer->data оставшееся пустое местом
     *        в дескриптор UART, данные будут получены в это пустое
     *        пространство и по завершении произойдет обратный вызов.
     */
    bytes_to_receive       = xfer->data_size;
    bytes_current_received = 0U;

    /* Если используется кольцевой буфер RX. */
    if (handle->rx_ring_buffer != NULL) {

        /* Отключить прерывание, защитить кольцевой буфер. */
        interrupt_mask = UART_GetEnabledInterrupts(base);
        UART_DisableInterrupts(base, interrupt_mask);

        /* Количество байтов в кольцевом буфере RX на данный момент. */
        bytes_to_copy = UART_TransferGetRxRingBufferLength(handle);
        if (bytes_to_copy != 0U) {
            bytes_to_copy = MIN(bytes_to_receive, bytes_to_copy);
            bytes_to_receive -= bytes_to_copy;

            /* Копировать данные из кольцевого буфера в пользовательскую память. */
            for (i = 0U; i < bytes_to_copy; i++) {
                xfer->rx_data[bytes_current_received++] = handle->rx_ring_buffer[handle->rx_ring_buffer_tail];

                /*
                 * Обернуть до 0.
                 * Не использовать деление по модулю, %, потому что он
                 * может быть большим и медленным.
                 */
                if ((size_t) handle->rx_ring_buffer_tail + 1U == handle->rx_ring_buffer_size) {
                    handle->rx_ring_buffer_tail = 0U;
                } else {
                    handle->rx_ring_buffer_tail++;
                }
            }
        }

        /*
         * Если в кольцевом буфере недостаточно данных, то необходимо
         * прочитать больше данных. Если в кольцевом буфере нет данных,
         * то сохраняем запрос в дескриптор UART.
         */
        if (bytes_to_receive != 0U) {
            handle->rx_data          = xfer->rx_data + bytes_current_received;
            handle->rx_data_size     = bytes_to_receive;
            handle->rx_data_size_all = xfer->data_size;
            handle->rx_state         = (uint8_t) UART_RxBusy;
        }
        /* Повторно включить прерывание. */
        UART_EnableInterrupts(base, interrupt_mask);

        /* Вызов callback-функции, если все принято. */
        if (bytes_to_receive == 0U) {
            if (handle->callback != NULL) {
                handle->callback(base, handle, UART_Status_RxIdle, handle->user_data);
            }
        }
    } else {
        /* Кольцевой буфер не используется. */
        interrupt_mask = UART_GetEnabledInterrupts(base);

        /* Отключить прерывание при настройке дескриптора. */
        UART_DisableInterrupts(base, interrupt_mask);

        handle->rx_data          = xfer->rx_data + bytes_current_received;
        handle->rx_data_size     = bytes_to_receive;
        handle->rx_data_size_all = bytes_to_receive;
        handle->rx_state         = (uint8_t) UART_RxBusy;

        /* Повторно включить прерывание. */
        UART_EnableInterrupts(base, interrupt_mask);

        /* IER.ERBFI - прерывание по доступности полученных данных. */
        SET_VAL_MSK(base->IER, UART_IER_ERBFI_Msk, UART_IER_ERBFI_Pos, 1U);
    }

    /* Возвращает количество прочитанных байтов. */
    if (receivedBytes != NULL) {
        *receivedBytes = bytes_current_received;
    }

    return UART_Status_Ok;
}

enum uart_status UART_TransferAbortReceive(UART_Type *base, struct uart_handle *handle)
{
    /* Проверка аргументов. */
    if (handle == NULL) {
        return UART_Status_InvalidArgument;
    }

    /*
     * Стоп, только если не используется кольцевой буфер, функция никак не
     * влияет на прием данных через кольцевой буфер (кольцевой буфер может
     * все еще принимaть данные).
     */
    if (handle->rx_ring_buffer == NULL) {
        /* IER.ERBFI прерывание по доступности полученных данных. */
        SET_VAL_MSK(base->IER, UART_IER_ERBFI_Msk, UART_IER_ERBFI_Pos, 0U);
        /* Очистка RxFifo. */
        SET_VAL_MSK(base->FCR, UART_FCR_RFIFOR_Msk, UART_FCR_RFIFOR_Pos, 1U);
    }

    handle->rx_data_size = 0U;
    handle->rx_state     = (uint8_t) UART_RxIdle;

    return UART_Status_Ok;
}

enum uart_status UART_TransferGetReceiveCount(UART_Type *base,
    struct uart_handle *handle, uint32_t *count)
{
    UNUSED(base);

    /* Проверка аргументов. */
    if ((handle == NULL) || (count == NULL)) {
        return UART_Status_InvalidArgument;
    }

    if (handle->rx_state == (uint8_t) UART_RxIdle) {
        return UART_Status_NoTransferInProgress;
    }

    *count = handle->rx_data_size_all - handle->rx_data_size;

    return UART_Status_Ok;
}

void UART_TransferHandleIRQ(UART_Type *base, struct uart_handle *handle)
{
    bool receive_enabled; /* Если true - принимаем данные */
    bool send_enabled;    /* Если true - передаем данные */

    /*
     * rx_data_size - размер оставшихся данных для получения
     * tx_data_size - размер оставшихся данных для отправки
     */
    receive_enabled = ((handle->rx_data_size != 0U) || (handle->rx_ring_buffer != NULL));
    send_enabled    = (handle->tx_ring_buffer != NULL
                            && UART_TransferGetTxRingBufferLength(handle) > 0)
                        || (handle->tx_data_size != 0U);


    /*-------------------------------------------------*/
    /*------------ Обработка ошибок приема ------------*/

    uint32_t state = base->LSR; /* Биты ошибок сбрасываются при чтении регистра LSR */

    /* LSR.RFE - суммарный бит ошибки приемника. */
    if (GET_VAL_MSK(state, UART_LSR_RFE_Msk, UART_LSR_RFE_Pos) != 0U) {
        /* Сброс RxFIFO. */
        SET_VAL_MSK(base->FCR, UART_FCR_RFIFOR_Msk, UART_FCR_RFIFOR_Pos, 1U);

        /* Если подцеплен callback. */
        if (handle->callback != NULL) {

            enum uart_status status = UART_Status_Ok;
            /* LSR.BI - обрыв линии. */
            if (GET_VAL_MSK(state, UART_LSR_BI_Msk, UART_LSR_BI_Pos) != 0U) {
                status = UART_Status_BreakLineError;
            }

            /* LSR.FE - ошибка кадрирования. */
            if (GET_VAL_MSK(state, UART_LSR_FE_Msk, UART_LSR_FE_Pos) != 0U) {
                status = UART_Status_FramingError;
            }

            /* LSR.PE - ошибка четности (контрольного бита). */
            if (GET_VAL_MSK(state, UART_LSR_PE_Msk, UART_LSR_PE_Pos) != 0U) {
                status = UART_Status_ParityError;
            }

            /* LSR.OE - ошибка переполнения. */
            if (GET_VAL_MSK(state, UART_LSR_OE_Msk, UART_LSR_OE_Pos) != 0U) {
                status = UART_Status_RxFifoBufferOverrun;
            }

            handle->callback(base, handle, status, handle->user_data);
        }
    }
    /*------------ Обработка ошибок приема ------------*/
    /*-------------------------------------------------*/

    /*-------------------------------------------------*/
    /*----------------- Работа с FIFO -----------------*/

    /* USR.RFNE == 0 - RxFIFO пуст. */
    /* USR.TFNF == 0 - TxFIFO полон. */

    /* Читаем из RxFIFO пока что-то есть или передаем в TxFIFO пока неполон. */
    while ((receive_enabled && (GET_VAL_MSK(base->USR, UART_USR_RFNE_Msk, UART_USR_RFNE_Pos) != 0U))
        || (send_enabled && (GET_VAL_MSK(base->USR, UART_USR_TFNF_Msk, UART_USR_TFNF_Pos) != 0U)))
    {
        /*-------------------------------------------------------------------------------------------------------*/
        /*-------------------------------------------------- RX -------------------------------------------------*/

        /*
         * Если идет процесс приема и в RxFIFO что-то есть (если бит RFNE
         * регистра USR равен 1 значит FIFO приемника не пуст).
         */
        if (receive_enabled && (GET_VAL_MSK(base->USR, UART_USR_RFNE_Msk, UART_USR_RFNE_Pos) != 0U)) {

            /*
             * Прием в буфер приложения, если такой есть;
             * rx_data_size - размер ожидаемых данных.
             */
            if (handle->rx_data_size != 0U) {
                {
                    uint8_t rxdata;
                    /* RBR.RBR - регистр приема. */
                    rxdata           = (uint8_t) GET_VAL_MSK(base->RBR, UART_RBR_RBR_Msk, UART_RBR_RBR_Pos);
                    *handle->rx_data = rxdata;
                }
                handle->rx_data_size--;
                handle->rx_data++;

                /* Истина, если не все данные приняли или есть кольцевой буфер Rx. */
                receive_enabled = ((handle->rx_data_size != 0U) || (handle->rx_ring_buffer != NULL));

                /* Если все приняли. */
                if (handle->rx_data_size == 0U) {

                    /*Если нет кольцевого буфера Rx. */
                    if (handle->rx_ring_buffer == NULL) {
                        /* IER.ERBFI - запрет прерывания по доступности Rx данных. */
                        SET_VAL_MSK(base->IER, UART_IER_ERBFI_Msk, UART_IER_ERBFI_Pos, 0U);
                    }

                    /* Перевод приемника в состояние простоя. */
                    handle->rx_state = (uint8_t) UART_RxIdle;

                    /* Если вызов callback подцеплен, то сообщаем на верхний уровень, что  все принято. */
                    if (handle->callback != NULL) {
                        handle->callback(base, handle, UART_Status_RxIdle, handle->user_data);
                    }
                }
            }
            /* Прием в кольцевой буфер. */
            else {
                /* Есть ли кольцевой буфер Rx? */
                if (handle->rx_ring_buffer != NULL) {
                    /* Кольцевой буфер RX заполнен? */
                    if (UART_TransferIsRxRingBufferFull(handle)) {
                        /* Если callback подцеплен, то уведомить о переполнении буфера. */
                        if (handle->callback != NULL) {
                            handle->callback(base, handle, UART_Status_RxRingBufferOverrun, handle->user_data);
                        }
                    }

                    /* Если после callback вызова кольцевой буфер все еще заполнен, то затираем самые старые данные. */
                    if (UART_TransferIsRxRingBufferFull(handle)) {
                        /* Увеличиваем индекс первого элемента для записи в кольцевой буфер. */
                        if ((size_t) handle->rx_ring_buffer_tail + 1U == handle->rx_ring_buffer_size) {
                            handle->rx_ring_buffer_tail = 0U;
                        } else {
                            handle->rx_ring_buffer_tail++;
                        }
                    }

                    {
                        /* RBR.RBR - регистр приема. */
                        uint8_t temp;
                        temp = (uint8_t) GET_VAL_MSK(base->RBR, UART_RBR_RBR_Msk, UART_RBR_RBR_Pos);
                        handle->rx_ring_buffer[handle->rx_ring_buffer_head] = temp;
                    }

                    /* Увеличиваем индекс первого элемента для чтения из кольцевого буфера. */
                    if ((size_t) handle->rx_ring_buffer_head + 1U == handle->rx_ring_buffer_size) {
                        handle->rx_ring_buffer_head = 0U;
                    } else {
                        handle->rx_ring_buffer_head++;
                    }
                }
            }
        }
        /*-------------------------------------------------- RX -------------------------------------------------*/
        /*-------------------------------------------------------------------------------------------------------*/

        /*-------------------------------------------------------------------------------------------------------*/
        /*-------------------------------------------------- TX -------------------------------------------------*/

        /*
         * Если передача данных разрешена и TxFIFO не полон, то докинуть в
         * него байт. TxFIFO полон, когда USR.TFNF == 0.
         */
        if (send_enabled && (GET_VAL_MSK(base->USR, UART_USR_TFNF_Msk, UART_USR_TFNF_Pos) != 0U)) {
            /* THR.THR - регистр передачи. */

            /* Кольцевой буфер включен */
            if (handle->tx_ring_buffer != NULL) {
                SET_VAL_MSK(base->THR, UART_THR_THR_Msk, UART_THR_THR_Pos,
                        handle->tx_ring_buffer[handle->tx_ring_buffer_tail++]);
                if (handle->tx_ring_buffer_tail == handle->tx_ring_buffer_size) {
                    handle->tx_ring_buffer_tail = 0;
                }
                send_enabled = (UART_TransferGetTxRingBufferLength(handle) > 0);
            } else {
                SET_VAL_MSK(base->THR, UART_THR_THR_Msk, UART_THR_THR_Pos, *handle->tx_data);

                /*
                 * Если после этого еще останутся данные для передачи, то
                 * передача останется разрешенной.
                 */
                handle->tx_data_size--;
                handle->tx_data++;
                send_enabled = handle->tx_data_size != 0U;
            }
        }

    } /* while() */

    /*----------------- Работа с FIFO -----------------*/
    /*-------------------------------------------------*/

    /* Если нет данных для передачи и произошло прерывание по опустошению передатчика. */
    if ((!send_enabled) && (GET_VAL_MSK(base->IIR, UART_IIR_IID_Msk, UART_IIR_IID_Pos) == 2UL)) {

        /* Установка TxState в idle только когда все данные были переданы в шину. */
        handle->tx_state = (uint8_t) UART_TxIdle;

        /*
         * Отключение прерывания при простое Tx.
         * IER.ETBEI - прерывание по опустошении регистра Tx.
         */
        SET_VAL_MSK(base->IER, UART_IER_ETBEI_Msk, UART_IER_ETBEI_Pos, 0U);

        /* Если подцеплен callback обработчик, то сообщаем на верний уровень что все передано. */
        if (handle->callback != NULL) {
            handle->callback(base, handle, UART_Status_TxIdle, handle->user_data);
        }
    }
    /*-------------------------------------------------- TX -------------------------------------------------*/
    /*-------------------------------------------------------------------------------------------------------*/
}

/*!
 * @}
 */
