/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @defgroup timer_driver Драйвер модуля TIM
 *
 * @brief Драйвер таймеров общего назначения
 *
 * Драйвер модуля таймеров общего назначения управляет таймерами TIM и TIM1
 * общего назначения и таймера LPTIM, для работы в режимах энергосбережения.
 */

/*!
 * @addtogroup timer_driver
 * @{
 */

/*!
 * @file hal_timer.h
 *
 * @brief Интерфейс драйвера модуля таймеров общего назначения
 */

#ifndef HAL_TIMER_H
#define HAL_TIMER_H

#if defined(__cplusplus)
extern "C" {
#endif /* __cplusplus */

#include "hal_common.h"

#define TIMER_COUNT                      3                           /*!< Количество таймеров общего назначения */
#define TIMER_HARDWARE_FIELD_MAX         (0xFFFFFFFFUL)              /*!< Максимально возможное аппаратное значение поля счетчика */
#define TIMER_SOFTWARE_FIELD_MAX         (0xFFFFFFFFFFFFFFFFULL)     /*!< Максимально возможное программное значение поля счетчика */
#define TIMER_SOFTWARE_FIELD_HIGH_OFFSET (32)                        /*!< Смещение старшей части программного значения поля счетчика */

/*!
 * @brief Статусы драйвера таймеров общего назначения
 */
enum timer_status {
    TIMER_Status_Ok              = 0, /*!< Нет ошибок */
    TIMER_Status_InvalidArgument = 1, /*!< Недопустимый аргумент */
    TIMER_Status_TimerBusy       = 2, /*!< Таймер уже занят */
    TIMER_Status_BadConfigure    = 3, /*!< Недопустимая конфигурация */
    TIMER_Status_NotIni          = 4, /*!< Работа с неинициализированным таймером */
    TIMER_Status_NotSupport      = 5, /*!< Функция не поддерживается */
};

/*!
 * @brief Режимы счета импульсов таймером
 */
enum timer_type_of_counting {
    TIMER_Work  = 0, /*!< Стандартный счет частоты */
    TIMER_Debug = 1, /*!< Счет частоты с учетом отладки (CTI) */
};

/*!
 * @brief Режим работы таймера общего назначения
 */
enum timer_work_mode {
    TIMER_Hardware = 0, /*!< Работа в 32-битном режиме по аппаратным настройкам */
    TIMER_Software = 1, /*!< Работа в режиме эмуляции 64-битного таймера */
};

/*!
 * @brief Конфигурация аппаратной части таймера общего назначения
 */
struct timer_hardware_config {
    uint32_t                    start_value;      /*!< Стартовое значение счетчика */
    uint32_t                    reload_value;     /*!< Загружаемое значение счетчика */
    uint32_t                    interrupt_enable; /*!< Разрешение прерывания */
    enum timer_type_of_counting work_type;        /*!< Тип работы */
    uint32_t                    start_enable;     /*!< Разрешение работы */
};

/*!
 * @brief Функция обратного вызова
 */
typedef void (*callback_t)(TIM_Type *base);

/*!
 * @name Интерфейс драйвера
 * @{
 */

/*!
 * @brief Инициализация таймера общего назначения
 *
 * @note Конфигурация аппаратуры таймера используется полностью при режиме
 *       работы #TIMER_Hardware, а при #TIMER_Software - только поля work_type
 *       и start_enable.
 *
 * @param base     Таймер
 * @param config   Конфигурация аппаратуры таймера
 * @param mode     Режим работы
 * @param callback Функция обратного вызова
 * @param ticks_h  Начальное значение для старшей части счетчика обратного счета
 *                 при режиме работы #TIMER_Software
 *
 * @retval #TIMER_Status_Ok
 * @retval #TIMER_Status_InvalidArgument
 * @retval #TIMER_Status_TimerBusy
 */
enum timer_status TIMER_Init(TIM_Type *base,
    struct timer_hardware_config config, enum timer_work_mode mode,
    callback_t callback, uint32_t ticks_h);

/*!
 * @brief Деинициализация таймера общего назначения
 *
 * @param base Таймер
 *
 * @retval #TIMER_Status_Ok
 * @retval #TIMER_Status_InvalidArgument
 */
enum timer_status TIMER_Deinit(TIM_Type *base);

/*!
 * @brief Запуск таймера общего назначения
 *
 * @param base Таймер
 *
 * @retval #TIMER_Status_Ok
 * @retval #TIMER_Status_InvalidArgument
 */
enum timer_status TIMER_Run(TIM_Type *base);

/*!
 * @brief Остановка таймера общего назначения
 *
 * @param base Таймер
 *
 * @retval #TIMER_Status_Ok
 * @retval #TIMER_Status_InvalidArgument
 */
enum timer_status TIMER_Stop(TIM_Type *base);

/*!
 * @brief Сброс таймера общего назначения
 *
 * @param base Таймер
 *
 * @retval #TIMER_Status_NotSupport
 * @retval #TIMER_Status_InvalidArgument
 */
enum timer_status TIMER_Reset(TIM_Type *base);

/*!
 * @brief Получение количества тиков
 *
 * @param base Таймер
 *
 * @return Количество подсчитанных тактов
 */
uint64_t TIMER_GetTicks(TIM_Type *base);

/*!
 * @brief Установка количества тиков
 *
 * @param base  Таймер
 * @param ticks Количество тиков
 *
 * @retval #TIMER_Status_NotSupport
 */
enum timer_status TIMER_SetTick(TIM_Type *base, uint64_t ticks);

/*!
 * @brief Получение результата выполнения последней функции
 *
 * Возвращает статус выполнения последней функции, у которой тип возвращаемого
 * значения отличен от #timer_status.
 *
 * @retval #TIMER_Status_Ok
 * @retval #TIMER_Status_InvalidArgument
 * @retval #TIMER_Status_TimerBusy
 * @retval #TIMER_Status_BadConfigure
 * @retval #TIMER_Status_NotIni
 * @retval #TIMER_Status_NotSupport
 */
enum timer_status TIMER_GetAPIStatus();

/*!
 * @brief Получение значения регистра счетчика таймера
 *
 * @param base Таймер
 *
 * @return Количество подсчитанных тактов
 */
static inline uint32_t TIMER_GetTimerHardwareValue(TIM_Type *base)
{
    return base->VALUE;
}

/*!
 * @brief Инициализация структуры таймера общего назначения
 *
 * @note Конфигурация аппаратуры таймера используется полностью при режиме
 *       работы #TIMER_Hardware, а при #TIMER_Software - только поля work_type
 *       и start_enable.
 *
 * @param base     Таймер
 * @param config   Конфигурация аппаратуры таймера
 * @param mode     Режим работы
 * @param callback Функция обратного вызова
 * @param ticks_h  Начальное значение для старшей части счетчика обратного счета
 *                 при режиме работы #TIMER_Software
 *
 * @retval #TIMER_Status_Ok
 * @retval #TIMER_Status_InvalidArgument
 * @retval #TIMER_Status_TimerBusy
 */
enum timer_status TIMER_SetConfig(TIM_Type *base,
    struct timer_hardware_config config, enum timer_work_mode mode,
    callback_t callback, uint32_t ticks_h);

/*!
 * @brief Включение прерывания
 *
 * Включает прерывание в таймере; NVIC не конфигурирует.
 *
 * @param base Базовый адрес
 *
 * @retval #TIMER_Status_Ok
 * @retval #TIMER_Status_InvalidArgument
 */
enum timer_status TIMER_IRQEnable(TIM_Type *base);

/*!
 * @brief Отключение прерывания
 *
 * Выключает прерывание в таймере; NVIC не конфигурирует.
 *
 * @param base Базовый адрес
 *
 * @retval #TIMER_Status_Ok
 * @retval #TIMER_Status_InvalidArgument
 */
enum timer_status TIMER_IRQDisable(TIM_Type *base);

/*!
 * @brief Чтение статуса прерывания
 *
 * @param base Базовый адрес
 *
 * @retval 0 - прерывание отсутствует
 * @retval 1 - прерывание присутствует
 */
uint32_t TIMER_IRQGetStatus(TIM_Type *base);

/*!
 * @brief Сброс прерывания
 *
 * @param base Базовый адрес
 *
 * @retval #TIMER_Status_Ok
 * @retval #TIMER_Status_InvalidArgument
 */
enum timer_status TIMER_IRQClear(TIM_Type *base);

/*!
 * @}
 */

#if defined(__cplusplus)
}
#endif /* __cplusplus */

#endif /* HAL_TIMER_H */

/*!
 * @}
 */
