/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @addtogroup spi_driver
 * @{
 */

/*!
 * @file hal_spi_dma.c
 *
 * @brief Имплементация дополнения драйвера SPI с пересылкой данных через DMA
 */


#include "hal_spi_dma.h"

#define SPI_UNIT_COUNT (3U) /*!< Количество модулей SPI */

/*!
 * @brief Дескриптор для SPI-DMA передач
 */
typedef struct _spi_dma_private_handle {
    SPI_Type *base;           /*!< Базовый адрес SPI */
    spi_dma_handle_t *handle; /*!< Дескриптор SPI-DMA */
} spi_dma_private_handle_t;

/*! @brief Таблица дескрипторов для SPI-DMA передач */
static spi_dma_private_handle_t s_dma_private_handles[SPI_UNIT_COUNT];

/*!
 * @brief Функция обратного вызова tx по прерыванию DMA
 *
 * @param handle        Дескриптор DMA
 * @param user_data     Пользовательские данные
 * @param intmode       Источник прерывания
 * @param is_ctrl       Флаг прерывания контроллера
 */
static void SPI_TxDMACallback(dma_handle_t *handle, void *user_data,
    dma_irq_t inttype);
/*!
 * @brief Функция обратного вызова rx по прерыванию DMA
 *
 * @param handle        Дескриптор DMA
 * @param user_data     Пользовательские данные
 * @param intmode       Источник прерывания
 * @param is_ctrl       Флаг прерывания контроллера
 */
static void SPI_RxDMACallback(dma_handle_t *handle, void *user_data,
    dma_irq_t inttype);

static void SPI_TxDMACallback(dma_handle_t *handle, void *user_data,
    dma_irq_t inttype)
{
    UNUSED(handle);

    spi_dma_private_handle_t *priv_handle =
        (spi_dma_private_handle_t *) user_data;
    spi_dma_handle_t *spi_handle = priv_handle->handle;
    if ((spi_handle->tx_in_progress) & (inttype == DMA_IntTfr)) {
        if (!spi_handle->rx_in_progress) {
            spi_handle->state = (uint32_t) SPI_TransStatus_Idle;
        }
        spi_handle->tx_in_progress = false;
    }
    if (spi_handle->callback != NULL) {
        (spi_handle->callback)(priv_handle->base, spi_handle,
            spi_handle->user_data, inttype);
    }
}

static void SPI_RxDMACallback(dma_handle_t *handle, void *user_data,
    dma_irq_t inttype)
{
    UNUSED(handle);

    spi_dma_private_handle_t *priv_handle =
        (spi_dma_private_handle_t *) user_data;
    spi_dma_handle_t *spi_handle = priv_handle->handle;
    if ((spi_handle->rx_in_progress) & (inttype == DMA_IntTfr)) {
        if (!spi_handle->tx_in_progress) {
            spi_handle->state = (uint32_t) SPI_TransStatus_Idle;
        }
        spi_handle->rx_in_progress = false;
    }
    if (spi_handle->callback != NULL) {
        (spi_handle->callback)(priv_handle->base, spi_handle,
            spi_handle->user_data, inttype);
    }
}

enum spi_status SPI_MasterTransferCreateHandleDMA(SPI_Type *base,
    spi_dma_callback_t callback, void *user_data,
    spi_dma_handle_t *handle, dma_handle_t *tx_handle,
    dma_handle_t *rx_handle)
{
    assert(!(NULL == base));
    if (NULL == base) {
        return SPI_Status_InvalidArgument;
    }

    assert(!(NULL == handle));
    if (NULL == handle) {
        return SPI_Status_InvalidArgument;
    }

    /* Получение номера используемого SPI. */
    uint32_t instance = SPI_GetInstance(base);
    /* Заполнение структуры нулями. */
    (void)memset(handle, 0, sizeof(spi_dma_handle_t));

    /* Инициализация полей структуры. */
    handle->tx_handle = tx_handle;
    handle->rx_handle = rx_handle;

    /* Установка IDLE состояния. */
    handle->state = SPI_TransStatus_Idle;

    /* Загрузка данных в таблицу дескрипторов. */
    s_dma_private_handles[instance].base   = base;
    s_dma_private_handles[instance].handle = handle;

    /* Установка функций обратного вызова. */
    DMA_SetCallback(handle->tx_handle, SPI_TxDMACallback,
        &s_dma_private_handles[instance]);
    DMA_SetCallback(handle->rx_handle, SPI_RxDMACallback,
        &s_dma_private_handles[instance]);

    s_dma_private_handles[instance].handle->callback = callback;
    s_dma_private_handles[instance].handle->user_data = user_data;

    return SPI_Status_Ok;
}

enum spi_status SPI_MasterTransferDMA(SPI_Type *base,
    spi_dma_handle_t *handle, spi_transfer_t *xfer)
{
    assert(handle != NULL);
    assert(xfer != NULL);

    if ((handle == NULL) || (xfer == NULL)) {
        return SPI_Status_InvalidArgument;
    }

    if (xfer->data_size == 0U) {
        return SPI_Status_InvalidArgument;
    }

    if (handle->state == SPI_TransStatus_Busy) {
        return SPI_Status_Busy;
    }

    /* Получение номера SPI контроллера. */
    uint32_t instance = SPI_GetInstance(base);
    if (!(instance < 4)) {
        return SPI_Status_InvalidArgument;
    }
    handle->transfer_size = xfer->data_size;
    spi_config_internal_t *spi_config = SPI_GetConfig(base);
    dma_channel_config_t ch_cfg;

    uint32_t tx_channel = handle->tx_handle->channel;
    uint32_t rx_channel = handle->rx_handle->channel;
    /* Размер передачи */
    uint32_t data_size = handle->transfer_size = xfer->data_size;

    /* Адрес Rx и Tx буфера */
    uint32_t address = (uint32_t) &base->DR0;

    /* Прием */
    uint32_t transfer_width;
    if (spi_config->frame_width_bits - 1 > SPI_Data16Bits) {
        transfer_width = DMA_Transfer32BitWidth;
    } else if (spi_config->frame_width_bits - 1 > SPI_Data8Bits) {
        transfer_width = DMA_Transfer16BitWidth;
    } else {
        transfer_width = DMA_Transfer8BitWidth;
    }

    /* Включение режима полнодуплексного обмена. */
    SET_VAL_MSK(base->CTRLR0, SPI_CTRLR0_TMOD_Msk, SPI_CTRLR0_TMOD_Pos, SPI_ModeDuplex);

    /* Конфигурация канала на передачу */
    uint64_t xfercfg;
    /* Адреса Источника и Приемника */
    void *src_addr = NULL, *dst_addr = NULL;
    /* Инкремент адреса при DMA пересылке */
    uint32_t desc_count;

    dma_descriptor_t *desc = NULL;

    /* Проверка на наличие буфера приема. */
    if (xfer->rx_data != NULL) {
        if (data_size > DMA_AHB_MAX_BLOCK_SIZE) {
            /* Дескриптор многоблочной передачи */
            desc = handle->rx_desc;
            if (desc == NULL) {
                /*
                * Возврат статуса неудачи в случае отсутствия
                * дескриптора многоблочной передачи.
                */
                return SPI_Status_Fail;
            }
            desc_count = DMA_GetDescriptorCount(data_size, transfer_width);
            SPI_DMADescriptorInitRX(base, desc, desc_count, data_size,
                DMA_Incr, transfer_width, (void *) xfer->rx_data);
            /* Отправка значений для инициализации многоблочной передачи. */
            DMA_SubmitChannelDescriptor(handle->rx_handle, desc);
        } else {
            /* Подготовка канала к передаче. */
            src_addr = (void *) address;
            dst_addr = (void *) xfer->rx_data;
            xfercfg = DMA_CHANNEL_CTL(data_size,
                    false, false,
                    DMA_PeripheralToMemory_DMA,
                    false, false,
                    DMA_BurstSize1, DMA_BurstSize1,
                    DMA_NoChange, DMA_Incr,
                    transfer_width, transfer_width,
                    true
                );
            DMA_PrepareChannelTransfer(&ch_cfg, src_addr,
                dst_addr, xfercfg);
            /* Отправка значений для инициализации регистров канала. */
            DMA_SubmitChannelTransfer(handle->rx_handle, &ch_cfg);
        }
        /* Разрешение считывания данных из Rx буфера для DMA. */
        SPI_EnableRxDMA(base, true);
        /* Выдача сигнала прерывания DMA по завершении пересылки для канала отправки. */
        DMA_EnableChannelInterrupt(handle->rx_handle->base,
            rx_channel, DMA_IntTfr | DMA_IntError | DMA_IntBlock);
        /* Включение аппаратного запроса DMA. */
        DMA_HardwareHandshakeEnable(handle->rx_handle, true,
            DMA_HANDSHAKE_SPI0_RX + instance * 2);
        /* Rx активирован. */
        handle->rx_in_progress = true;
        /* SPI занят. */
        handle->state = SPI_TransStatus_Busy;
        /* Включение канала и начало передачи. */
        DMA_StartTransfer(handle->rx_handle);
    }

    /* Передача */
    uint8_t increment = DMA_Incr;
    if (xfer->tx_data != NULL || handle->dummy_data != NULL) {
        if (data_size > DMA_AHB_MAX_BLOCK_SIZE) {
            desc = handle->tx_desc;
            if (desc == NULL) {
                return SPI_Status_Fail;
            }
            if (xfer->tx_data != NULL) {
                src_addr = (void *) xfer->tx_data;
                increment = DMA_Incr;
            } else if (handle->dummy_data != NULL) {
                src_addr = handle->dummy_data;
                increment = DMA_NoChange;
            }
            desc_count = DMA_GetDescriptorCount(data_size, transfer_width);
            SPI_DMADescriptorInitTX(base, desc, desc_count, data_size,
                increment, transfer_width, src_addr);
            /* Отправка значений для инициализации многоблочной передачи. */
            DMA_SubmitChannelDescriptor(handle->tx_handle, desc);
        } else {
            if (xfer->tx_data != NULL) {
                src_addr = (void *) xfer->tx_data;
                increment = DMA_Incr;
            } else if (handle->dummy_data != NULL) {
                src_addr = handle->dummy_data;
                increment = DMA_NoChange;
            }

            dst_addr = (void *) address;
            xfercfg = DMA_CHANNEL_CTL(data_size,
                    false, false,
                    DMA_MemoryToPeripheral_DMA,
                    false, false,
                    DMA_BurstSize1, DMA_BurstSize1,
                    increment, DMA_NoChange,
                    transfer_width, transfer_width,
                    true
                );
            DMA_PrepareChannelTransfer(&ch_cfg, src_addr,
                dst_addr, xfercfg);
            DMA_SubmitChannelTransfer(handle->tx_handle, &ch_cfg);
        }
        /* Включение аппаратного запроса DMA. */
        DMA_HardwareHandshakeEnable(handle->tx_handle, true,
            DMA_HANDSHAKE_SPI0_TX + instance * 2);
        /* Разрешение считывания данных из Tx буффера для DMA. */
        SPI_EnableTxDMA(base, true);
        /* Выдача сигнала прерывания DMA по завершении пересылки для канала отправки. */
        DMA_EnableChannelInterrupt(handle->tx_handle->base,
            tx_channel, DMA_IntTfr | DMA_IntBlock | DMA_IntError);
        /* Tx активирован. */
        handle->tx_in_progress = true;
        /* SPI занят. */
        handle->state = SPI_TransStatus_Busy;
        /* Включение канала и начало передачи. */
        DMA_StartTransfer(handle->tx_handle);
    }

    return SPI_Status_Ok;
}

enum spi_status SPI_MasterHalfDuplexTransferDMA(SPI_Type *base,
    spi_dma_handle_t *handle, spi_half_duplex_transfer_t *xfer)
{
    enum spi_status status = SPI_Status_Ok;

    if (base == NULL) {
        return SPI_Status_InvalidArgument;
    }

    if (handle == NULL) {
        return SPI_Status_InvalidArgument;
    }

    if (xfer == NULL) {
        return SPI_Status_InvalidArgument;
    }

    if (handle->state == SPI_TransStatus_Busy) {
        return SPI_Status_Busy;
    }

    if (handle->rx_handle == NULL) {
        return SPI_Status_InvalidArgument;
    }

    if (handle->tx_handle == NULL) {
        return SPI_Status_InvalidArgument;
    }

    if (xfer->rx_data == NULL || xfer->rx_data_size == 0UL) {
        return SPI_Status_InvalidArgument;
    }

    if (xfer->tx_data == NULL || xfer->tx_data_size == 0UL) {
        return SPI_Status_InvalidArgument;
    }

    /* Получение номера SPI контроллера. */
    uint32_t instance = SPI_GetInstance(base);
    if (!(instance < 4)) {
        return SPI_Status_InvalidArgument;
    }

    /* Прием */
    uint32_t transfer_width;
    spi_config_internal_t *spi_config = SPI_GetConfig(base);
    if (spi_config->frame_width_bits - 1 > SPI_Data16Bits) {
        transfer_width = DMA_Transfer32BitWidth;
    } else if (spi_config->frame_width_bits - 1 > SPI_Data8Bits) {
        transfer_width = DMA_Transfer16BitWidth;
    } else {
        transfer_width = DMA_Transfer8BitWidth;
    }

    SPI_Enable(base, 0U);

    /* Включение режима полудуплексного обмена. */
    SET_VAL_MSK(base->CTRLR0, SPI_CTRLR0_TMOD_Msk,
        SPI_CTRLR0_TMOD_Pos, SPI_ModeHalfDuplex);
    uint32_t rx_remaining_frame = (xfer->rx_data_size >> transfer_width);
    SET_VAL_MSK(base->CTRLR1, SPI_CTRLR1_NDF_Msk, SPI_CTRLR1_NDF_Pos,
        rx_remaining_frame - 1);
    
    SET_VAL_MSK(base->SSIENR, SPI_SSIENR_SSI_EN_Msk, SPI_SSIENR_SSI_EN_Pos,
        1U);

    /* Адрес Rx и Tx буфера */
    uint32_t address = (uint32_t) &base->DR0;

    uint32_t data_size = (xfer->rx_data_size >> transfer_width);
    dma_descriptor_t *desc = NULL;
    dma_channel_config_t ch_cfg;
    uint32_t tx_channel = handle->tx_handle->channel;
    uint32_t rx_channel = handle->rx_handle->channel;
    /* Конфигурация канала на передачу */
    uint64_t xfercfg;
    /* Адреса Источника и Приемника */
    void *src_addr = NULL, *dst_addr = NULL;
    /* Инкремент адреса при DMA пересылке */
    uint32_t desc_count;
    if (data_size > DMA_AHB_MAX_BLOCK_SIZE) {
        /* Дескриптор многоблочной передачи */
        desc = handle->rx_desc;
        if (desc == NULL) {
            /*
            * Возврат статуса неудачи в случае отсутствия
            * дескриптора многоблочной передачи.
            */
            return SPI_Status_Fail;
        }
        desc_count = DMA_GetDescriptorCount(data_size, transfer_width);
        SPI_DMADescriptorInitRX(base, desc, desc_count, data_size,
            DMA_Incr, transfer_width, (void *) xfer->rx_data);
        /* Отправка значений для инициализации многоблочной передачи. */
        DMA_SubmitChannelDescriptor(handle->rx_handle, desc);
    } else {
        /* Подготовка канала к передаче. */
        src_addr = (void *) address;
        dst_addr = (void *) xfer->rx_data;
        xfercfg = DMA_CHANNEL_CTL(data_size,
                false, false,
                DMA_PeripheralToMemory_DMA,
                false, false,
                DMA_BurstSize1, DMA_BurstSize1,
                DMA_NoChange, DMA_Incr,
                transfer_width, transfer_width,
                true
            );
        DMA_PrepareChannelTransfer(&ch_cfg, src_addr,
            dst_addr, xfercfg);
        /* Отправка значений для инициализации регистров канала. */
        DMA_SubmitChannelTransfer(handle->rx_handle, &ch_cfg);
    }
    /* Разрешение считывания данных из Rx буфера для DMA. */
    SPI_EnableRxDMA(base, true);
    /* Выдача сигнала прерывания DMA по завершении пересылки для канала отправки. */
    DMA_EnableChannelInterrupt(handle->rx_handle->base,
        rx_channel, DMA_IntTfr | DMA_IntError | DMA_IntBlock);
    /* Включение аппаратного запроса DMA. */
    DMA_HardwareHandshakeEnable(handle->rx_handle, true,
        DMA_HANDSHAKE_SPI0_RX + instance * 2);
    /* Rx активирован. */
    handle->rx_in_progress = true;
    /* Включение канала и начало передачи. */
    DMA_StartTransfer(handle->rx_handle);

    /* Передача */
    uint8_t increment = DMA_Incr;
    data_size = (xfer->tx_data_size >> transfer_width);
    if (xfer->tx_data != NULL || handle->dummy_data != NULL) {
        if (data_size > DMA_AHB_MAX_BLOCK_SIZE) {
            desc = handle->tx_desc;
            if (desc == NULL) {
                return SPI_Status_Fail;
            }
            if (xfer->tx_data != NULL) {
                src_addr = (void *) xfer->tx_data;
                increment = DMA_Incr;
            } else if (handle->dummy_data != NULL) {
                src_addr = handle->dummy_data;
                increment = DMA_NoChange;
            }
            desc_count = DMA_GetDescriptorCount(data_size, transfer_width);
            SPI_DMADescriptorInitTX(base, desc, desc_count, data_size,
                increment, transfer_width, src_addr);
            /* Отправка значений для инициализации многоблочной передачи. */
            DMA_SubmitChannelDescriptor(handle->tx_handle, desc);
        } else {
            if (xfer->tx_data != NULL) {
                src_addr = (void *) xfer->tx_data;
                increment = DMA_Incr;
            } else if (handle->dummy_data != NULL) {
                src_addr = handle->dummy_data;
                increment = DMA_NoChange;
            }

            dst_addr = (void *) address;
            xfercfg = DMA_CHANNEL_CTL(data_size,
                    false, false,
                    DMA_MemoryToPeripheral_DMA,
                    false, false,
                    DMA_BurstSize1, DMA_BurstSize1,
                    increment, DMA_NoChange,
                    transfer_width, transfer_width,
                    true
                );
            DMA_PrepareChannelTransfer(&ch_cfg, src_addr,
                dst_addr, xfercfg);
            DMA_SubmitChannelTransfer(handle->tx_handle, &ch_cfg);
        }
        /* Включение аппаратного запроса DMA. */
        DMA_HardwareHandshakeEnable(handle->tx_handle, true,
            DMA_HANDSHAKE_SPI0_TX + instance * 2);
        /* Разрешение считывания данных из Tx буффера для DMA. */
        SPI_EnableTxDMA(base, true);
        /* Выдача сигнала прерывания DMA по завершении пересылки для канала отправки. */
        DMA_EnableChannelInterrupt(handle->tx_handle->base,
            tx_channel, DMA_IntTfr | DMA_IntBlock | DMA_IntError);
        /* Tx активирован. */
        handle->tx_in_progress = true;
        /* Включение канала и начало передачи. */
        DMA_StartTransfer(handle->tx_handle);
    }

    /* SPI занят. */
    handle->state = SPI_TransStatus_Busy;

    return status;
}

void SPI_MasterTransferAbortDMA(spi_dma_handle_t *handle)
{
    assert(NULL != handle);

    /* Остановка tx канала. */
    DMA_AbortTransfer(handle->tx_handle);
    /* Остановка rx канала. */
    DMA_AbortTransfer(handle->rx_handle);

    handle->tx_in_progress = false;
    handle->rx_in_progress = false;
    handle->state          = SPI_TransStatus_Idle;
}

/*!
 * @}
 */
