/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @defgroup sdmmc_driver Драйвер модуля SDMMC
 *
 * @brief Драйвер SDMMC контроллера SD и MMC карт
 *
 * Драйвер содержит функции инциализации карты, подсчета размера пространства
 * памяти карты, синхронные и асинхронные операции чтения и записи карты.
 */

/*!
 * @addtogroup sdmmc_driver
 * @{
 */

/*!
 * @file hal_sdmmc.h
 *
 * @brief Интерфейс драйвера модуля SDMMC
 */

#ifndef HAL_SDMMC_H
#define HAL_SDMMC_H

#if defined(__cplusplus)
extern "C" {
#endif /* __cplusplus */

#include "hal_common.h"

/*!
 * @name Количество слотов под карты и их типы
 * @{
 */
enum {
    SDMMC_TypeMMC = 0, /*!< Тип карты MMC */
    SDMMC_TypeSD  = 1  /*!< Тип карты SD */
};

#define SDMMC_IS_MMC(x) ((x)->type == SDMMC_TypeMMC) /*!< Проверка на тип MMC */
#define SDMMC_IS_SD(x)  ((x)->type == SDMMC_TypeSD)  /*!< Проверка на тип SD */
/*!
 * @}
 */

/*!
 * @brief Формула подсчета делителя выходной частоты контроллера SDMMC
 *
 * @param input_freq_hz    Входная частота контроллера SDMMC
 * @param required_freq_hz Необходимая выходная частота контроллера SDMMC
 *
 * @return Делитель частоты
 */
#define SDMMC_CALC_DIVIDER(input_freq_hz, required_freq_hz) (((input_freq_hz) / (required_freq_hz)) / 2)


/*!
 * @name Константы контроллера SDMMC
 * @{
 */
#define SDMMC_SDHC_SECTOR_SIZE         512        /*!< Размер сектора High Capacity карты */
#define SDMMC_SD_SEND_IF_COND_PATTERN  0x1AA      /*!< Начальный паттерн инициализации SD карты */
#define SDMMC_SD_OCR_INIT_VALUE        0xFF80     /*!< Значение OCR регистра при инициализации */
#define SDMMC_MMC_RCA_ADDR             0x00010000 /*!< Относительный адрес MMC карты */
#define SDMMC_TIMEOUTCONTROL_MAX_VALUE 0xE        /*!< Максимальное значение таймера ожидания сигнала на линиях DAT */
/*!
 * @}
 */

/*!
 * @name Рабочие напряжения контроллера SDMMC
 * @{
 */
enum {
    SDMMC_HostPWR_3V3 = 0x7, /*!< 3,3 вольта */
    SDMMC_HostPWR_3V0 = 0x6, /*!< 3,0 вольта */
    SDMMC_HostPWR_1V8 = 0x5  /*!< 1,8 вольта */
};
/*!
 * @}
 */

/*!
 * @name Типы слота карты контроллера SDMMC
 * @{
 */
#define SDMMC_CORECFG0_SLOTTYPE_REMOVABLE  0 /*!< Извлекаемая карта */
#define SDMMC_CORECFG0_SLOTTYPE_EMBEDDED   1 /*!< Встроеная карта */
#define SDMMC_CORECFG0_SLOTTYPE_SHARED_BUS 2 /*!< Разделяемая шина */
/*!
 * @}
 */

/*!
 * @name Направления передачи SDMA канала контроллера SDMMC
 * @{
 */
enum {
    SDMMC_SDMA_TransferWrite = 0, /*!< Запись */
    SDMMC_SDMA_TransferRead  = 1  /*!< Чтение */
};
/*!
 * @}
 */

/*!
 * @name Типы и размеры ответов карты
 * @{
 */
enum {
    SDMMC_NoResponce             = 0, /*!< Без ответа */
    SDMMC_ResponceLength136      = 1, /*!< Длина ответа - 136 бит */
    SDMMC_ResponceLength48       = 2, /*!< Длина ответа - 48 бит */
    SDMMC_ResponceLength48_Check = 3  /*!< Длина ответа - 48 бит с проверкой */
};
/*!
 * @}
 */

/*!
 * @name Ширина шины данных карты в битах
 * @{
 */
enum {
    SDMMC_DataBusTransferWidth_1Bit    = 0, /*!< 1 бит */
    SDMMC_DataBusTransferWidth_4bit    = 1, /*!< 4 бит */
    SDMMC_ExtDataBusTransferWidth_8bit = 1  /*!< 8 бит */
};
/*!
 * @}
 */

/*!
 * @name Режимы UHS-I карты SD
 * @{
 */
#define SDMMC_SD_UHS_MODE_DEFAULT_SDR12   0 /*!< Default/SDR12 */
#define SDMMC_SD_UHS_MODE_HIGHSPEED_SDR25 1 /*!< HighSpeed/SDR25 */
#define SDMMC_SD_UHS_MODE_SDR50           2 /*!< SDR50 */
#define SDMMC_SD_UHS_MODE_SDR104          3 /*!< SDR104 */
#define SDMMC_SD_UHS_MODE_DDR50           4 /*!< DDR50 */
/*!
 * @}
 */

/*!
 * @name Выравнивание адреса буфера данных SDMA
 *
 * Возможные варианты значений:
 *  - 0 -   4KB (Detects A11 Carry out)
 *  - 1 -   8KB (Detects A12 Carry out)
 *  - 2 -  16KB (Detects A13 Carry out)
 *  - 3 -  32KB (Detects A14 Carry out)
 *  - 4 -  64KB (Detects A15 Carry out)
 *  - 5 - 128KB (Detects A16 Carry out)
 *  - 6 - 256KB (Detects A17 Carry out)
 *  - 7 - 512KB (Detects A18 Carry out)
 *
 * @{
 */
#define SDMMC_SDMA_ALIGN                  0                                                     /*!< Выравнивание адреса */
#define SDMMC_SDMA_BLOCK_SIZE             (4096 << SDMMC_SDMA_ALIGN)                            /*!< Размер блока данных SDMA в байтах */
#define SDMMC_SDMA_BLOCK_ALIGN            __attribute__((aligned(SDMMC_SDMA_BLOCK_SIZE)))       /*!< Атрибут GCC выравнивания по размеру блока SDMA */
#define SDMMC_SDMA_IS_BLOCK_ALIGN_ADDR(x) (((uint32_t) (x) & (SDMMC_SDMA_BLOCK_SIZE - 1)) == 0) /*!< Проверка выравнивания адреса по размеру блока данных SDMA */
/*!
 * @}
 */

/*!
 * @brief Статусы драйвера SDMMC
 */
typedef enum {
    SDMMC_Status_Ok  = 0, /*!< Ошибок нет */
    SDMMC_Status_Err = 1  /*!< Ошибка исполнения */
} sdmmc_status_t;

/*!
 * @brief Рабочие напряжения питания карты
 */
typedef enum {
    SDMMC_3v3      = 0, /*!< 3,3 В (для карт MMC и SD в режиме Default Speed или High Speed) */
    SDMMC_1v8      = 1, /*!< 1,8 В (для карт MMC) */
    SDMMC_3v3To1v8 = 2  /*!< 3,3 В с переключением на 1,8 В (для карт SD в режимах UHS-I) */
} sdmmc_voltage_t;

/*!
 * @brief Контекст драйвера контроллера SDMMC
 */
typedef struct {
    int32_t                dev_num;          /*!< Номер порта SDMMC контроллера */
    const sdmmc_port_cfg_t *cfg;             /*!< Конфигурация контроллера SDMMC */
    SDMMC_Type             *regs;            /*!< Указатель на структуру регистров контроллера SDMMC */
    uint32_t               rca;              /*!< Относительный адрес карты */
    int32_t                type;             /*!< Тип карты: 0 - MMC, 1 - SD */
    int32_t                sdhc_mode;        /*!< Доступность режима SDHC: 0 - SDSC, 1 - SDHC/SDXC */
    int32_t                hs_mode;          /*!< Доступность режима High Speed: 0 - Default, 1 - High */
    int32_t                ddr_mode;         /*!< Доступность режима Double Data Rate: 0 - SDR, 1 - DDR */
    int32_t                version;          /*!< Версия SD карты: 1 - версия до 1.1, 2 - версия 2.2 и выше */
    sdmmc_voltage_t        gpio_vol;         /*!< Напряжение внешних выводов GPIO и карты SD или MMC */
    int32_t                need_1V8en;       /*!< Необходимость переключения напряжения питания с 3,3 В на 1,8 В у SD карты */
    uint32_t               freq_input;       /*!< Входная тактовая частота контроллера SDMMC  */
    int32_t                freq_divider;     /*!< Делитель выходной тактовой частоты контроллера SDMMC */
    int32_t                lock;             /*!< Флаг блокировки контроллера SDMMC */
    uint64_t               total_size;       /*!< Общий размер пространства памяти карты */
    uint32_t               cid[4];           /*!< Значение регистра CID */
    uint32_t               csd[4];           /*!< Значение регистра CSD */
} sdmmc_card_t;

/*!
 * @brief Инициализация SDMMC контроллера и вставленной карты SD или MMC
 *
 * @note Буфер памяти для SDMA должен быть выровнен по границе блока SDMA и
 * размером 512 байт.
 *
 * @param sd       Контекст драйвера SDMMC
 * @param num      Номер контроллера SDMMC
 * @param vol      Выбор напряжения питания карты
 * @param init_buf Буфер памяти для SDMA
 *
 * @retval #SDMMC_Status_Ok
 * @retval #SDMMC_Status_Err
 */
sdmmc_status_t SDMMC_InitCard(sdmmc_card_t *sd, uint32_t num,
    sdmmc_voltage_t vol, void *init_buf);

/*!
 * @brief Остановка SDMMC контроллера и выключение питания вставленной карты
 *
 * @param sd Контекст драйвера SDMMC
 */
void SDMMC_DisableCard(sdmmc_card_t *sd);

/*!
 * @brief Подсчет размера пространства памяти карты
 *
 * Подсчет общего пространства памяти карты через регистры параметров, если
 * подсчет по параметрам не удался, то может применяться итеративный метод
 * определения размера, но с повреждением информации на карте. Подсчитанный
 * размер записыватся в поле total_size контекста sd в байтах.
 *
 * @note Буфер памяти для SDMA должен быть выровнен по границе блока SDMA и
 * размером 512 байт.
 *
 * @param sd         Контекст драйвера SDMMC
 * @param sector_buf Буфер памяти для SDMA
 * @param unsafe     Использовать (1) или не использовать (0) небезопасные
 *                   методы определения емкости карты
 *
 * @retval #SDMMC_Status_Ok
 * @retval #SDMMC_Status_Err
 */
sdmmc_status_t SDMMC_CalcMemorySpace(sdmmc_card_t *sd, void *sector_buf,
    bool unsafe);

/*!
 * @brief Чтение карты блоками размером 512 байт
 *
 * @note Буфер памяти для SDMA должен быть выровнен по границе блока SDMA.
 *
 * @param sd          Контекст драйвера SDMMC
 * @param start_block Номер первого блока памяти карты
 * @param data        Буфер памяти для SDMA
 * @param nblocks     Количество считываемых блоков памяти
 *
 * @retval #SDMMC_Status_Ok
 * @retval #SDMMC_Status_Err
 */
sdmmc_status_t SDMMC_Read(sdmmc_card_t *sd, uint32_t start_block, void *data,
    uint32_t nblocks);

/*!
 * @brief Асинхронное чтение карты блоками размером 512 байт
 *
 * @note Буфер памяти для SDMA должен быть выровнен по границе блока SDMA.
 *
 * @param sd          Контекст драйвера SDMMC
 * @param start_block Номер первого блока памяти карты
 * @param data        Буфер памяти для SDMA
 * @param nblocks     Количество считываемых блоков памяти
 *
 * @retval #SDMMC_Status_Ok
 * @retval #SDMMC_Status_Err
 */
sdmmc_status_t SDMMC_ReadAsync(sdmmc_card_t *sd, uint32_t start_block,
    void *data, uint32_t nblocks);

/*!
 * @brief Ожидание завершения операции асинхронного чтения памяти карты
 *
 * @param sd Контекст драйвера SDMMC
 *
 * @retval #SDMMC_Status_Ok
 * @retval #SDMMC_Status_Err
 */
sdmmc_status_t SDMMC_ReadWait(sdmmc_card_t *sd);

/*!
 * @brief Запись карты блоками размером 512 байт
 *
 * @note Буфер памяти для SDMA должен быть выровнен по границе блока SDMA.
 *
 * @param sd          Контекст драйвера SDMMC
 * @param start_block Номер первого блока памяти карты
 * @param data        Буфер памяти для SDMA
 * @param nblocks     Количество записываемых блоков памяти
 *
 * @retval #SDMMC_Status_Ok
 * @retval #SDMMC_Status_Err
 */
sdmmc_status_t SDMMC_Write(sdmmc_card_t *sd, uint32_t start_block,
    const void *data, uint32_t nblocks);

/*!
 * @brief Асинхронная запись карты блоками размером 512 байт
 *
 * @note Буфер памяти для SDMA должен быть выровнен по границе блока SDMA.
 *
 * @param sd          Контекст драйвера SDMMC
 * @param start_block Номер первого блока памяти карты
 * @param data        Буфер памяти для SDMA
 * @param nblocks     Количество записываемых блоков памяти
 *
 * @retval #SDMMC_Status_Ok
 * @retval #SDMMC_Status_Err
 */
sdmmc_status_t SDMMC_WriteAsync(sdmmc_card_t *sd, uint32_t start_block,
    const void *data, uint32_t nblocks);

/*!
 * @brief Ожидание завершения операции асинхронной записи памяти карты
 *
 * @param sd Контекст драйвера SDMMC
 *
 * @retval #SDMMC_Status_Ok
 * @retval #SDMMC_Status_Err
 */
sdmmc_status_t SDMMC_WriteWait(sdmmc_card_t *sd);

#if defined(__cplusplus)
}
#endif /* __cplusplus */

#endif /* HAL_SDMMC_H */

/*!
 * @}
 */
