/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @addtogroup qspi_driver
 * @{
 */

/*!
 * @file hal_qspi_dma.h
 *
 * @brief Дополнение драйвера QSPI для обмена данными с помощью DMA
 */

#ifndef HAL_QSPI_DMA_H
#define HAL_QSPI_DMA_H

#include "hal_qspi.h"
#include "hal_dma.h"

#ifdef __cplusplus
extern "C" {
#endif

/*!
 * @brief Дескриптор QSPI-DMA передачи 
 */
typedef struct {
    QSPI_Type *base;
    dma_handle_t *tx_handle;
    dma_handle_t *rx_handle;
    void *dummy_data;
    dma_descriptor_t *tx_desc;
    dma_descriptor_t *rx_desc;
} qspi_dma_handle_t;

/*!
 * @brief Статусы выполнения функций
 */
typedef enum {
    QSPI_DMA_Status_Success         = 0U, /*!< Успешное выполнение функции */
    QSPI_DMA_Status_Fail            = 1U, /*!< Неудачное выполнение функции */
    QSPI_DMA_Status_InvalidArgument = 2U, /*!< Неверно переданные аргументы в функцию */
} qspi_dma_status_t;

/*!
 * @brief Инициализация контекста передачи QSPI-DMA
 *
 * @param base Базовый адрес QSPI
 * @param handle Указатель на контекст
 * @param tx_handle Указатель на контекст драйвера DMA для передачи данных
 * @param rx_handle Указатель на контекст драйвера DMA для приема данных
 */
void QSPI_TransferCreateHandleDMA(QSPI_Type *base, qspi_dma_handle_t *handle,
    dma_handle_t *tx_handle, dma_handle_t *rx_handle);

/*!
 * @brief Запись данных в QSPI TX
 *
 * @param handle Дескриптор DMA-QSPI передачи
 * @param addr Адрес Источника
 * @param incr Инкремент адреса
 * @param transfer_width Ширина одного слова
 * @param size Общий размер данных (в байтах)
 * 
 * @retval #QSPI_DMA_Status_Success
 * @retval #QSPI_DMA_Status_Fail
 * @retval #QSPI_DMA_Status_InvalidArgument
 */
qspi_dma_status_t QSPI_WriteDataDMA(qspi_dma_handle_t *handle, void *addr,
    uint8_t incr, uint8_t transfer_width, uint32_t size);

/*!
 * @brief Чтение данных из QSPI RX
 *
 * @param handle Дескриптор DMA-QSPI передачи
 * @param addr Адрес Приемника
 * @param incr Инкремент адреса
 * @param transfer_width Ширина одного слова
 * @param size Общий размер данных (в байтах)
 * 
 * @retval #QSPI_DMA_Status_Success
 * @retval #QSPI_DMA_Status_Fail
 * @retval #QSPI_DMA_Status_InvalidArgument
 */
qspi_dma_status_t QSPI_ReadDataDMA(qspi_dma_handle_t *handle, void *addr,
    uint8_t incr, uint8_t transfer_width, uint32_t size);

/*!
 * @brief Расчет количества дескрипторов многоблочной передачи,
 * нужных для считывания данных из NOR Flash памяти
 *
 * @param size_in_bytes Число данных в байтах
 * 
 * @return Количество дескрипторов многоблочной передачи
 */
uint32_t QSPI_GetReadDMADescriptorsCount(uint32_t size_in_bytes);

/*!
 * @brief Инициализация группы дескрипторов DMA (многоблочная передача) для приема данных в RX буфер.
 *
 * @param base          Базовый адрес DMA
 * @param desc          Указатель на массив дескрипторов DMA
 * @param data_size     Общий размер передаваемых данных
 * @param dst_addr      Адрес Приемника
 * @param src_addr_incr Инкремент адреса
 */
void QSPI_DMAReadDescriptorInitRX(QSPI_Type *base, dma_descriptor_t *desc,
    uint32_t data_size, void *dst_addr, uint8_t dst_addr_incr);

/*!
 * @brief Расчет количества дескрипторов многоблочной передачи,
 * нужных для считывания данных из NOR Flash памяти при посылке
 * фиктивных данных
 *
 * @param size_in_bytes Число данных в байтах
 * 
 * @return Количество дескрипторов многоблочной передачи
 */
static inline uint32_t QSPI_GetDummyDMADescriptorsCount(uint32_t size_in_bytes)
{
    return DMA_GetDescriptorCount(size_in_bytes, DMA_Transfer8BitWidth);
}

/*!
 * @brief Инициализация группы дескрипторов DMA (многоблочная передача) для отправки данных в TX буфер.
 *
 * @param base       Базовый адрес DMA
 * @param desc       Указатель на массив дескрипторов DMA
 * @param count      Число дескрипторов многоблочной передачи
 * @param data_size  Общий размер передаваемых данных
 * @param data_width Размер одного слова данных
 * @param src_addr   Адрес Источника
 */
static inline void QSPI_DMADescriptorInitTX(QSPI_Type *base, dma_descriptor_t *desc,
    uint32_t count, uint32_t data_size, uint32_t data_width, void *src_addr, uint8_t src_addr_incr)
{
    dma_multiblock_config_t config = {
        .count          = count,
        .data_size      = data_size,
        .transfer_type  = DMA_MemoryToPeripheral_DMA,
        .scatter_en     = false,
        .gather_en      = false,
        .src_burst_size = DMA_BurstSize32,
        .dst_burst_size = DMA_BurstSize1,
        .src_incr       = src_addr_incr,
        .dst_incr       = DMA_NoChange,
        .src_data_width = data_width,
        .dst_data_width = data_width,
        .src_addr       = src_addr,
        .dst_addr       = (void *) &base->TX_DATA,
        .int_en         = true
    };

    DMA_InitMultiblockDescriptor(desc, &config);
}

/*!
 * @brief Инициализация группы дескрипторов DMA (многоблочная передача) для приема данных в RX буфер.
 * 
 * @param base          Базовый адрес DMA
 * @param desc          Указатель на массив дескрипторов DMA
 * @param data_size     Общий размер передаваемых данных
 * @param dst_addr      Адрес Приемника
 * @param src_addr_incr Инкремент адреса
 */
static inline void QSPI_DMADescriptorInitRX(QSPI_Type *base, dma_descriptor_t *desc,
    uint32_t count, uint32_t data_size, uint32_t data_width, void *dst_addr)
{
    dma_multiblock_config_t config = {
        .count          = count,
        .data_size      = data_size,
        .transfer_type  = DMA_PeripheralToMemory_DMA,
        .scatter_en     = false,
        .gather_en      = false,
        .src_burst_size = DMA_BurstSize1,
        .dst_burst_size = DMA_BurstSize32,
        .src_incr       = DMA_NoChange,
        .dst_incr       = DMA_Incr,
        .src_data_width = data_width,
        .dst_data_width = data_width,
        .src_addr       = (void *) &base->RX_DATA,
        .dst_addr       = dst_addr,
        .int_en         = true
    };

    DMA_InitMultiblockDescriptor(desc, &config);
}

#ifdef __cplusplus
}
#endif

#endif /* HAL_QSPI_DMA_H */

/*!
 * @}
 */
