/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */





/*
 * @addtogroup pwm_driver
 * @{
 */

/*!
 * @file hal_pwm.c
 *
 * @brief Имплементация драйвера широтно-импульсного модулятора
 */

#include "hal_pwm_newgen.h"
#include "hal_ioim.h"

/*!
 * @brief Действия по возникновению события ШИМ
 */
enum pwm_outx_cmd {
    PWM_OutXCmdNo     = 0, /*!< Не выполнять действий */
    PWM_OutXCmdClear  = 1, /*!< Сбросить OUTx в 0 */
    PWM_OutXCmdSet    = 2, /*!< Установить OUTx в 1 */
    PWM_OutXCmdToggle = 3, /*!< Инвертировать OUTx */
};

/*!
 * @brief Поведение канала ШИМ
 */
enum pwm_run_command {
    PWM_RunCmdStop      = 0, /*!< Остановка после следующего переключения счетчика CTRCNT */
    PWM_RunCmdStopEvent = 1, /*!< Остановка при совершении следующих событий:
                                  up-count режим: остановка при CTRCNT==CTRPRD
                                  down-count режим: остановка при CTRCNT==0
                                  up-down-count режим: остановка при CTRCNT==0 */
    PWM_RunCmdRun       = 2, /*!< Запуск */
};

/*!
 * @brief Предделитель частоты ШИМ
 */
enum pwm_prescaler_divmux {
    PWM_PrescalerDivMux1      = 0, /*!< Частота от предделителя делится на 1 */
    PWM_PrescalerDivMux2      = 1, /*!< Частота от предделителя делится на 2 */
    PWM_PrescalerDivMux4      = 2, /*!< Частота от предделителя делится на 4 */
    PWM_PrescalerDivMux8      = 3, /*!< Частота от предделителя делится на 8 */
    PWM_PrescalerDivMux16     = 4, /*!< Частота от предделителя делится на 16 */
    PWM_PrescalerDivMuxPWMClk = 5, /*!< Используется внешняя частота PWM_CLK */
};

/*!
 * @brief Механизм загрузки активного регистра из теневого регистра для регистра программного управления выходами
 */
enum pwm_ldcswrf {
    PWM_LdcswrfCtrcntZero          = 0, /*!< Загрузка при CTRCNT=0 */
    PWM_LdcswrfCtrcntEquCtrprd     = 1, /*!< Загрузка при CTRCNT=CTRPRD */
    PWM_LdcswrfCtrcntZeroEquCtrprd = 2, /*!< Загрузка при CTRCNT=0 или CTRCNT=CTRPRD */
    PWM_LdcswrfDirect              = 3, /*!< Загружать напрямую при обращении CPU без использования теневого регистра */
};

/*!
 * @brief Работа блока trip unit
 */
enum pwm_trip_unit_signal {
    PWM_TripUnitSignalNotUsed = 0, /*!< Вход не используется */
    PWM_TripUnitSignalUsed    = 1, /*!< Вход используется */
};

/*!
 * @brief Разрешения сброса предделителя при возникновения событий SYNCI или SWFSYNC
 */
enum pwm_prescaler_syncrst {
    PWM_PrescalerSyncRstDis   = 0, /*!< Сброс запрещен */
    PWM_PrescalerSyncRstEn    = 1, /*!< Сброс разрешен */
};

/*!
 * @brief Управление режимом работы предделителя канала
 */
enum pwm_prescaler_mode {
    PWM_PrescModeTimerIsRun = 0, /*!< Предделитель формирует частоту только при включенном таймере */
    PWM_PrescModeAlways     = 1, /*!< Предделитель формирует частоту не зависимо от включенности таймера */
};

/*!
 * @brief Управление состоянием предделителя канала
 */
enum pwm_prescaler_cmd {
    PWM_PrescCmdReset = 0, /*!< Счетчик предделителя сбрасывается в «0» */
    PWM_PrescCmdSave  = 1, /*!< Счетчик предделителя сохраняет состояние на момент останова */
};


/*!
 * @brief Направление счета после синхронизации
 * @note Этот бит используется, только когда счетчик работает в up-down режиме.
 */
enum pwm_dirsync {
    PWM_DirSyncDown = 0, /*!< После синхронизации счетчик декрементируется */
    PWM_DirSyncUp   = 1, /*!< После синхронизации счетчик инкрементируется */
};

/*!
 * @brief Выбор источника выходного сигнала SYNCO
 */
enum pwm_syncosel {
    PWM_SyncoSelSynci         = 0, /*!< SYNCI */
    PWM_SyncoSelCtrcntZero    = 1, /*!< Счетчик равен нулю (CTRCNT==0) */
    PWM_SyncoSelCtrcntEquCmpb = 2, /*!< Счетчик равен значению регистра сравнения B (CTRCNT==CMPB) */
    PWM_SyncoSelOff           = 3, /*!< SYNCO отключен */
};

/*!
 * @brief Управление моментом переписи данных из теневого регистра периода в активный
 */
enum pwm_loadprd {
    PWM_LoadPrdRegister = 0, /*!< Регистр периода (CTRPRD) загружается из
                                  теневого регистра, когда счетчик (CTRCNT)
                                  равен нулю (запись или чтение CTRPRD
                                  осуществляется через теневой регистр) */
    PWM_LoadPrdDirect   = 1, /*!< CTRPRD загружается напрямую без использования
                                  теневого регистра (запись или чтение CTRPRD
                                  осуществляется напрямую) */
};

/*!
 * @brief Сигнал разрешения загрузки счетчика из регистра фазы
 */
enum pwm_syncphsen {
    PWM_SyncPhsEnDis = 0, /*!< Загрузка CTRCNT из регистра фазы CTRPHS запрещена */
    PWM_SyncPhsEnEn  = 1, /*!< Загрузка CTRCNT из регистра фазы CTRPHS во время синхронизации разрешена */
};

/*!
 * @brief Режим работы счетчика CRTCNT
 */
enum pwm_cntmode {
    PWM_CntModeUp     = 0, /*!< Up-count режим */
    PWM_CntModeDown   = 1, /*!< Down-count режим */
    PWM_CntModeUpDown = 2, /*!< Up-down-count режим */
    PWM_CntModeOff    = 3, /*!< Счет не осуществляется */
};

/*!
 * @brief Режим работы регистра CMPx
 */
enum pwm_scmpxmode {
    PWM_SCmpxModeReg    = 0, /*!< Работа с теневым регистром, все запросы
                                  CPU проходят через теневой регистр */
    PWM_SCmpxModeDirect = 1, /*!< Прямой режим, используется только
                                  активный регистр CMPx */
};

/*!
 * @brief Выбор режима загрузки данных из теневого регистра в активный CMPx
 */
enum pwm_ldxmode {
    PWM_LdxModeCtrcntZero            = 0, /*!< Загрузка при CTRCNT=0 */
    PWM_LdxModeCtrcntEquCtrprd       = 1, /*!< Загрузка при CTRCNT=CTRPRD */
    PWM_LdxModeCtrcntZeroOrEquCtrprd = 2, /*!< Загрузка при CTRCNT=0 или CTRCNT=CTRPRD */
    PWM_LdxModeNoLoad                = 3, /*!< Загрузка не осуществляется */
};

/*!
 * @brief Конфигурация канала широтно-импульсного модулятора
 */
struct pwm_channel_config {
    uint32_t channel;                                       /*!< Конфигурируемый канал */
    /* Группа 1 */
    /* Предделитель */
    enum pwm_prescaler_mode      prescaler_mode;            /*!< Режим управления  */
    enum pwm_prescaler_cmd       prescaler_cmd;             /*!< Режим сброса счетчика предделителя */
    enum pwm_prescaler_syncrst   prescaler_syncrst;         /*!< Режим сброса при событиях */
    uint8_t                      prescaler;                 /*!< Предделитель (PRESPRD) */
    enum pwm_prescaler_divmux    prescaler_divmux;          /*!< Мультиплексор чатоты */
    /* Основной счетчик */
    enum pwm_cntmode             cntmode;                   /*!< Режим работы */
    uint32_t                     counter;                   /*!< Значение счетчика */
    uint32_t                     period;                    /*!< Период */
    enum pwm_loadprd             loadprd;                   /*!< Моментом переписи данных */
    uint32_t                     ctrphs;                    /*!< Фазы синхронизации блока */
    enum pwm_syncphsen           syncphsen;                 /*!< Загрузки счетчика из регистра фазы */
    /* Блок сравнения */
    uint32_t                     cmpa;                      /*!< Регистр сравнения CMPA */
    uint32_t                     cmpb;                      /*!< Регистр сравнения CMPB */
    enum pwm_scmpxmode           scmpamode;                 /*!< Режим работы регистра CMPA */
    enum pwm_scmpxmode           scmpbmode;                 /*!< Режим работы регистра CMPB */
    enum pwm_ldxmode             ldamode;                   /*!< режима загрузки данных из теневого регистра в активный CMPA */
    enum pwm_ldxmode             ldbmode;                   /*!< режима загрузки данных из теневого регистра в активный CMPB */
    /* Группа 2 */
    /* Блок реакции на событие */
    enum pwm_outx_cmd            cnt_eq_prd_outa;           /*!< Событие канала OUTA по совпадению счетчика и периода */
    enum pwm_outx_cmd            cnt_eq_prd_outb;           /*!< Событие канала OUTB по совпадению счетчика и периода */
    enum pwm_outx_cmd            cnt_eq_cmpa_dec_outa;      /*!< Событие канала OUTA по совпадению счетчика и регистра CMPA при декременте счетчика */
    enum pwm_outx_cmd            cnt_eq_cmpa_inc_outa;      /*!< Событие канала OUTA по совпадению счетчика и регистра CMPA при инкременте счетчика */
    enum pwm_outx_cmd            cnt_eq_cmpa_dec_outb;      /*!< Событие канала OUTB по совпадению счетчика и регистра CMPA при декременте счетчика */
    enum pwm_outx_cmd            cnt_eq_cmpa_inc_outb;      /*!< Событие канала OUTB по совпадению счетчика и регистра CMPA при инкременте счетчика */
    enum pwm_outx_cmd            cnt_eq_cmpb_dec_outa;      /*!< Событие канала OUTA по совпадению счетчика и регистра CMPB при декременте счетчика */
    enum pwm_outx_cmd            cnt_eq_cmpb_inc_outa;      /*!< Событие канала OUTA по совпадению счетчика и регистра CMPB при инкременте счетчика */
    enum pwm_outx_cmd            cnt_eq_cmpb_dec_outb;      /*!< Событие канала OUTB по совпадению счетчика и регистра CMPB при декременте счетчика */
    enum pwm_outx_cmd            cnt_eq_cmpb_inc_outb;      /*!< Событие канала OUTB по совпадению счетчика и регистра CMPB при инкременте счетчика */
    enum pwm_outx_cmd            cnt_eq_zero_outa;          /*!< Событие канала OUTA по достижению счетчиком 0 */
    enum pwm_outx_cmd            cnt_eq_zero_outb;          /*!< Событие канала OUTB по достижению счетчиком 0 */
    enum pwm_outx_cmd            sw_forced_outa;            /*!< Событие канала OUTA при кратковременном программном событии */
    enum pwm_outx_cmd            sw_forced_outb;            /*!< Событие канала OUTB при кратковременном программном событии */
    enum pwm_outx_cmd            sw_forced_long_outa;       /*!< Событие канала OUTA при долговременном программном событии */
    enum pwm_outx_cmd            sw_forced_long_outb;       /*!< Событие канала OUTB при долговременном программном событии */
    enum pwm_ldcswrf             ldcswrf;                   /*!< Загрузка активного регистра из теневого регистра для регистра программного управления выходами */
    /* Блок прерываний */
    uint32_t              pwm_int_enable;            /*!< Разрешение прерывания */
    enum pwm_int_source          pwm_int_source;            /*!< Источник прерывания */
    enum pwm_eventprd            eventprd;                  /*!< Период прерывания */
    /* Группа 3 */
    /* Генератор запретной зоны */
    uint16_t                     dz_rising_edge_delay_clk;  /*!< Запратная зона в тактах около возрастающего фронта */
    uint16_t                     dz_falling_edge_delay_clk; /*!< Запратная зона в тактах около ниспадающего фронта */
    uint32_t           dz_rising_edge_source;     /*!< Источник сигнала для генерации запрещенной зоны около возрастающего фронта */
    uint32_t           dz_falling_edge_source;    /*!< Источник сигнала для генерации запрещенной зоны около ниспадающего фронта */
    enum pwm_dz_outx_inv         dz_rising_edge_outa_inv;   /*!< Полярность OUTA после генерации запрещенной зоны */
    enum pwm_dz_outx_inv         dz_falling_edge_outb_inv;  /*!< Полярность OUTB после генерации запрещенной зоны */
    enum pwm_dz_mode             dz_outa_enable;            /*!< Выбор режима работы блока запрещенной зоны при формирования OUTA */
    enum pwm_dz_mode             dz_outb_enable;            /*!< Выбор режима работы блока запрещенной зоны при формирования OUTB */
    /* Блок дробления выходного сигнала */
    enum pwm_chopper_duty        chopper_duty;              /*!< Скважность дробящего сигнала */
    enum pwm_chopper_freq        chopper_freq;              /*!< Частота дробящего сигнала */
    enum pwm_chopper_first_width chopper_first_width;       /*!< Ширина первого импульса */
    uint32_t        chopper_work;              /*!< Работа блока дробления */
    /* Блок реакции на внешнее воздействие */
    uint8_t                      inputs_mask_one;           /*!< Выбор используемых TU[7:0] сигналов для канала PWM работающих в режиме однократного срабатывания */
    uint8_t                      inputs_mask_mult;          /*!< Выбор используемых TU[7:0] сигналов для канала PWM работающих в режиме многократного срабатывания */
    enum pwm_trip_unit_action    trip_unit_action_outa;     /*!< Реакции на событие блока trip unit для OUTA */
    enum pwm_trip_unit_action    trip_unit_action_outb;     /*!< Реакции на событие блока trip unit для OUTB */
    uint32_t              pwmtu_int_one;             /*!< Разрешение прерывания блока */
    uint32_t              pwmtu_int_mult;            /*!< Разрешение прерывания блока */

    /* Запуск/останов канала*/
    enum pwm_run_command cmd;                               /*!< Разрешение работы */
};

/*!
 * @brief Базовые адреса ШИМ
 */
static PWM_Type *pwm_bases[] = PWM_UNITS;
static pwm_handle_t *pwm_handle[PWM_CHANNEL_COUNT];

static void PWM_CommonIRQHandler(void *unused);

/*!
 * @brief Получение конфигурации ШИМ по умолчанию
 *
 * @param cfg Конфигурация контроллера ШИМ
 */
static void PWM_GetChannelDefaultConfig(struct pwm_channel_config *cfg)
{
    cfg->channel = 0;
    /* Группа 1 */
    /* Предделитель */
    cfg->prescaler_mode = PWM_PrescModeTimerIsRun;
    cfg->prescaler_cmd = PWM_PrescCmdReset;
    cfg->prescaler_syncrst = PWM_PrescalerSyncRstDis;
    cfg->prescaler = 0;
    cfg->prescaler_divmux = PWM_PrescalerDivMux1;
    /* Основной счетчик */
    cfg->cntmode = PWM_CntModeUpDown;
    cfg->counter = 0;
    cfg->period = 0;
    cfg->loadprd = PWM_LoadPrdDirect;
    cfg->ctrphs = 0;
    cfg->syncphsen = PWM_SyncPhsEnDis;
    /* Блок сравнения */
    cfg->cmpa = 0;
    cfg->cmpb = 0;
    cfg->scmpamode = PWM_SCmpxModeReg;
    cfg->scmpbmode = PWM_SCmpxModeReg;
    cfg->ldamode = 0;
    cfg->ldbmode = 0;
    /* Группа 2 */
    /* Блок реакции на событие */
    cfg->cnt_eq_prd_outa = PWM_OutXCmdNo;
    cfg->cnt_eq_prd_outb = PWM_OutXCmdNo;
    cfg->cnt_eq_cmpa_dec_outa = PWM_OutXCmdNo;
    cfg->cnt_eq_cmpa_inc_outa = PWM_OutXCmdNo;
    cfg->cnt_eq_cmpa_dec_outb = PWM_OutXCmdNo;
    cfg->cnt_eq_cmpa_inc_outb = PWM_OutXCmdNo;
    cfg->cnt_eq_cmpb_dec_outa = PWM_OutXCmdNo;
    cfg->cnt_eq_cmpb_inc_outa = PWM_OutXCmdNo;
    cfg->cnt_eq_cmpb_dec_outb = PWM_OutXCmdNo;
    cfg->cnt_eq_cmpb_inc_outb = PWM_OutXCmdNo;
    cfg->cnt_eq_zero_outa = PWM_OutXCmdNo;
    cfg->cnt_eq_zero_outb = PWM_OutXCmdNo;
    cfg->sw_forced_outa = PWM_OutXCmdNo;
    cfg->sw_forced_outb = PWM_OutXCmdNo;
    cfg->sw_forced_long_outa = PWM_OutXCmdNo;
    cfg->sw_forced_long_outb = PWM_OutXCmdNo;
    cfg->ldcswrf = PWM_LdcswrfCtrcntZero;
    /* Блок прерываний */
    cfg->pwm_int_enable = 0;
    cfg->pwm_int_source = 0;
    cfg->eventprd = PWM_EventPrdNo;
    //cfg->pwm_soft_int_enable = 0;
    /* Группа 3 */
    /* Генератор запретной зоны */
    cfg->dz_rising_edge_delay_clk = 0;
    cfg->dz_falling_edge_delay_clk = 0;
    cfg->dz_rising_edge_source = PWM_OutA;
    cfg->dz_falling_edge_source = PWM_OutA;
    cfg->dz_rising_edge_outa_inv = PWM_DzSignalOutxInvOff;
    cfg->dz_falling_edge_outb_inv = PWM_DzSignalOutxInvOff;
    cfg->dz_outa_enable = PWM_DzModeOff;
    cfg->dz_outb_enable = PWM_DzModeOff;
    /* Блок дробления выходного сигнала */
    cfg->chopper_duty = PWM_ChopperDuty_1_8;
    cfg->chopper_freq = PWM_ChopperFreqClk_8;
    cfg->chopper_first_width = PWM_ChopperFirstWidth_0_8;
    cfg->chopper_work = 0;
    /* Блок реакции на внешнее воздействие */
    cfg->inputs_mask_one = 0;
    cfg->inputs_mask_mult = 0;
    cfg->trip_unit_action_outa = PWM_TripUnitActionHigh;
    cfg->trip_unit_action_outb = PWM_TripUnitActionHigh;
    cfg->pwmtu_int_one = 0;
    cfg->pwmtu_int_mult = 0;
    //cfg->pwmtu_soft_int_one = 0;
    //cfg->pwmtu_soft_int_mult = 0;

    /* Запуск/останов канала*/
    cfg->cmd = PWM_RunCmdStop;
}

/*!
 * @brief Первичная инициализация канала ШИМ
 *
 * @param hpwm Контекст драйвера ШИМ
 * @param cfg Конфигурация канала ШИМ
 */
static void PWM_InitChannel(pwm_handle_t *hpwm, struct pwm_channel_config cfg)
{
    PWM_Type *channel_offset;
    uint32_t run_pos, run_mask;
    uint32_t pres_mode_pos, pres_mode_mask;
    uint32_t pres_cmd_pos, pres_cmd_mask;

    channel_offset = pwm_bases[hpwm->channel];
    run_pos = (PWM_CTRRUN_RUN1_Pos - PWM_CTRRUN_RUN0_Pos) * cfg.channel + PWM_CTRRUN_RUN0_Pos;
    run_mask = PWM_CTRRUN_RUN0_Msk << run_pos;
    pres_mode_pos = (PWM_CTRRUN_PRESMODE1_Pos - PWM_CTRRUN_PRESMODE0_Pos) * cfg.channel + PWM_CTRRUN_PRESMODE0_Pos;
    pres_mode_mask = PWM_CTRRUN_PRESMODE0_Msk << pres_mode_pos;
    pres_cmd_pos = (PWM_CTRRUN_PRESRST1_Pos - PWM_CTRRUN_PRESRST0_Pos) * cfg.channel + PWM_CTRRUN_PRESRST0_Pos;
    pres_cmd_mask = PWM_CTRRUN_PRESRST1_Msk << pres_cmd_pos;

    /* Останов канала */
    SET_VAL_MSK(channel_offset->CTRRUN, run_mask, run_pos, PWM_RunCmdStop);
    /* Нужно исследовать while (PWM_GetCntStat(base, cfg.channel)); */

    /* Предделитель */
    SET_VAL_MSK(channel_offset->CTRRUN, pres_mode_mask, pres_mode_pos, cfg.prescaler_mode);
    SET_VAL_MSK(channel_offset->CTRRUN, pres_cmd_mask, pres_cmd_pos, cfg.prescaler_cmd);
    PWM_CLKCTL_SYNCRST_PTYPE_SET(channel_offset, cfg.prescaler_syncrst);
    PWM_CLKCTL_PRESPRD_PTYPE_SET(channel_offset, cfg.prescaler);
    PWM_CLKCTL_DEVMUX_PTYPE_SET(channel_offset, cfg.prescaler_divmux);
    /* Основной счетчик */
    PWM_CLKCTL_CNTMODE_PTYPE_SET(channel_offset, cfg.cntmode);
    PWM_CTRCNT_CTRCNT_PTYPE_SET(channel_offset, cfg.counter);
    PWM_CLKCTL_LOADPRD_PTYPE_SET(channel_offset, cfg.loadprd);
    PWM_CTRPRD_CTRPRD_PTYPE_SET(channel_offset, cfg.period);
    PWM_CTRPHS_CTRPHS_PTYPE_SET(channel_offset, cfg.ctrphs);
    PWM_CLKCTL_SYNCPHSEN_PTYPE_SET(channel_offset, cfg.syncphsen);
    /* Блок сравнения */
    PWM_CMPCTL_SCMPAMODE_PTYPE_SET(channel_offset, cfg.scmpamode);
    PWM_CMPCTL_SCMPBMODE_PTYPE_SET(channel_offset, cfg.scmpbmode);
    PWM_CMPCTL_LDAMODE_PTYPE_SET(channel_offset, cfg.ldamode);
    PWM_CMPCTL_LDBMODE_PTYPE_SET(channel_offset, cfg.ldbmode);
    PWM_CMPA_CMPA_PTYPE_SET(channel_offset, cfg.cmpa);
    PWM_CMPB_CMPB_PTYPE_SET(channel_offset, cfg.cmpb);
    /* Блок реакции на событие */
    PWM_EMCTLA_EPRD_PTYPE_SET(channel_offset, cfg.cnt_eq_prd_outa);
    PWM_EMCTLB_EPRD_PTYPE_SET(channel_offset, cfg.cnt_eq_prd_outb);
    PWM_EMCTLA_ECMPAD_PTYPE_SET(channel_offset, cfg.cnt_eq_cmpa_dec_outa);
    PWM_EMCTLA_ECMPAI_PTYPE_SET(channel_offset, cfg.cnt_eq_cmpa_inc_outa);
    PWM_EMCTLB_ECMPAD_PTYPE_SET(channel_offset, cfg.cnt_eq_cmpa_dec_outb);
    PWM_EMCTLB_ECMPAI_PTYPE_SET(channel_offset, cfg.cnt_eq_cmpa_inc_outb);
    PWM_EMCTLA_ECMPBD_PTYPE_SET(channel_offset, cfg.cnt_eq_cmpb_dec_outa);
    PWM_EMCTLA_ECMPBI_PTYPE_SET(channel_offset, cfg.cnt_eq_cmpb_inc_outa);
    PWM_EMCTLB_ECMPBD_PTYPE_SET(channel_offset, cfg.cnt_eq_cmpb_dec_outb);
    PWM_EMCTLB_ECMPBI_PTYPE_SET(channel_offset, cfg.cnt_eq_cmpb_inc_outb);
    PWM_EMCTLA_EZRO_PTYPE_SET(channel_offset, cfg.cnt_eq_zero_outa);
    PWM_EMCTLB_EZRO_PTYPE_SET(channel_offset, cfg.cnt_eq_zero_outb);
    PWM_EMSWFR_ACTSFA_PTYPE_SET(channel_offset, cfg.sw_forced_outa);
    PWM_EMSWFR_ACTSFB_PTYPE_SET(channel_offset, cfg.sw_forced_outb);
    PWM_EMCSWFR_LONGSFA_PTYPE_SET(channel_offset, cfg.sw_forced_long_outa);
    PWM_EMCSWFR_LONGSFB_PTYPE_SET(channel_offset, cfg.sw_forced_long_outb);
    PWM_EMSWFR_LDCSWRF_PTYPE_SET(channel_offset, cfg.ldcswrf);
    /* Блок прерываний */
    PWM_ICSEL_INTEN_PTYPE_SET(channel_offset, cfg.pwm_int_enable);
    PWM_ICSEL_INTSEL_PTYPE_SET(channel_offset, cfg.pwm_int_source);
    PWM_ICCTL_EVENTPRD_PTYPE_SET(channel_offset, cfg.eventprd);
    /* Генератор запретной зоны */
    PWM_DZRPER_PER_PTYPE_SET(channel_offset, cfg.dz_rising_edge_delay_clk);
    PWM_DZFPER_PER_PTYPE_SET(channel_offset, cfg.dz_falling_edge_delay_clk);
    PWM_DZCTL_INMUXR_PTYPE_SET(channel_offset, cfg.dz_rising_edge_source);
    PWM_DZCTL_INMUXF_PTYPE_SET(channel_offset, cfg.dz_falling_edge_source);
    PWM_DZCTL_INVMUXA_PTYPE_SET(channel_offset, cfg.dz_rising_edge_outa_inv);
    PWM_DZCTL_INVMUXB_PTYPE_SET(channel_offset, cfg.dz_falling_edge_outb_inv);
    PWM_DZCTL_OUTMUXA_PTYPE_SET(channel_offset, cfg.dz_outa_enable);
    PWM_DZCTL_OUTMUXB_PTYPE_SET(channel_offset, cfg.dz_outb_enable);
    /* Блок дробления выходного сигнала */
    PWM_CHCTL_CHDUTY_PTYPE_SET(channel_offset, cfg.chopper_duty);
    PWM_CHCTL_CHCLKDEV_PTYPE_SET(channel_offset, cfg.chopper_freq);
    PWM_CHCTL_FIRSTWTH_PTYPE_SET(channel_offset, cfg.chopper_first_width);
    PWM_CHCTL_CHEN_PTYPE_SET(channel_offset, cfg.chopper_work);
    /* Блок реакции на внешнее воздействие */
    PWM_TUSEL_ONE_PTYPE_SET(channel_offset, cfg.inputs_mask_one);
    PWM_TUSEL_MULT_PTYPE_SET(channel_offset, cfg.inputs_mask_mult);
    PWM_TUCTL_TUA_PTYPE_SET(channel_offset, cfg.trip_unit_action_outa);
    PWM_TUCTL_TUB_PTYPE_SET(channel_offset, cfg.trip_unit_action_outb);
    PWM_TUINTM_ONE_PTYPE_SET(channel_offset, cfg.pwmtu_int_one);
    PWM_TUINTM_MULT_PTYPE_SET(channel_offset, cfg.pwmtu_int_mult);

    SET_VAL_MSK(channel_offset->CTRRUN, run_mask, run_pos, cfg.cmd);
}

/*!
 * @brief Установка скважности вывода ШИМ
 *
 * @param base Базовый адрес канала ШИМ
 */
static void PWM_SetupCmp(PWM_Type *base)
{
    uint32_t cmpa = base->CMPA;
    uint32_t cmpb = base->CMPB;

    if (cmpa == cmpb) {
        // Need to bugfix PWM
        base->CMPB += 1;
    }
    PWM_EMCTLB_ECMPAD_PTYPE_SET(base, PWM_OutXCmdNo);
    PWM_EMCTLB_ECMPAI_PTYPE_SET(base, PWM_OutXCmdNo);

    PWM_EMCTLB_ECMPBD_PTYPE_SET(base, PWM_OutXCmdSet);
    PWM_EMCTLB_ECMPBI_PTYPE_SET(base, PWM_OutXCmdClear);
}

/*!
 * @brief Включение/выключение работы канала ШИМ
 *
 * @param hpwm Контекст драйвера ШИМ
 * @param enable Включение/выключение
 */
static void PWM_Run(pwm_handle_t *hpwm, bool enable)
{
    PWM_Type *base = pwm_bases[hpwm->channel];
    uint32_t run_pos = (PWM_CTRRUN_RUN1_Pos - PWM_CTRRUN_RUN0_Pos) *
        hpwm->channel + PWM_CTRRUN_RUN0_Pos;
    uint32_t run_mask = PWM_CTRRUN_RUN0_Msk << run_pos;
    uint32_t run_cmd = enable ? PWM_RunCmdRun : PWM_RunCmdStop;

    if (!enable) {
        if (hpwm->out) {
            base->CMPB = 0UL;
        } else {
            base->CMPA = 0UL;
        }
    }

    SET_VAL_MSK(base->CTRRUN, run_mask, run_pos, run_cmd);
}

/*!
 * @brief Установка скважности вывода ШИМ
 *
 * @param hpwm Контекст драйвера ШИМ
 */
static void PWM_SetDutyCycle(pwm_handle_t *hpwm)
{
    float percent = hpwm->duty_cycle;
    if (percent < 0.0f) {
        percent = 0.0f;
    } else if (percent > 1.0f) {
        percent = 1.0f;
    }

    PWM_Type *base = pwm_bases[hpwm->channel];

    uint32_t cmp = (uint32_t)(percent * GET_VAL_MSK(base->CTRPRD,
                PWM_CMPB_CMPB_Msk, PWM_CMPB_CMPB_Pos));
    if (hpwm->out) {
        PWM_CMPB_CMPB_PTYPE_SET(base, cmp);
    } else {
        PWM_CMPA_CMPA_PTYPE_SET(base, cmp);
    }

    PWM_SetupCmp(base);
}

/*!
 * @brief Установка периода канала ШИМ
 *
 * @param hpwm Контекст драйвера ШИМ
 */
static void PWM_SetPeriodUs(pwm_handle_t *hpwm)
{
    PWM_Type *base = pwm_bases[hpwm->channel];
    uint32_t divmux = GET_VAL_MSK(base->CLKCTL,
            PWM_CLKCTL_DEVMUX_Msk, PWM_CLKCTL_DEVMUX_Pos);
    uint32_t prescaler = GET_VAL_MSK(base->CLKCTL,
            PWM_CLKCTL_PRESPRD_Msk, PWM_CLKCTL_PRESPRD_Pos);

    float pwmclk_mhz = (float) hpwm->ref_clk / (1 << divmux) / (prescaler + 1U) / 1e6;
    uint32_t period = ((uint32_t)(pwmclk_mhz * hpwm->period_us) >> 1);
    bool period_setup_done = false;
    if (period < 100) {
        for (uint32_t i = 0UL; i < 6UL; i++) {
            divmux = i;
            for (uint32_t j = 0UL; j < 0xFF; j++) {
                prescaler = j;
                pwmclk_mhz = (float) hpwm->ref_clk / (1UL << divmux) / (prescaler + 1UL) / 1e6;
                period = ((uint32_t)(pwmclk_mhz * hpwm->period_us) >> 1);
                if (period > 100) {
                    period_setup_done = true;
                    break;
                }
            }
            if (period_setup_done) {
                break;
            }
        }
        if (!period_setup_done) {
            return;
        } else {
            SET_VAL_MSK(base->CLKCTL, PWM_CLKCTL_PRESPRD_Msk, PWM_CLKCTL_PRESPRD_Pos, 0UL);
            SET_VAL_MSK(base->CLKCTL, PWM_CLKCTL_DEVMUX_Msk, PWM_CLKCTL_DEVMUX_Pos, divmux);
            SET_VAL_MSK(base->CLKCTL, PWM_CLKCTL_PRESPRD_Msk, PWM_CLKCTL_PRESPRD_Pos, prescaler);
        }
    }
    PWM_CTRPRD_CTRPRD_PTYPE_SET(base, period);

    PWM_SetDutyCycle(hpwm);
}

__used
static void PWM_SetupIRQ (pwm_handle_t *hpwm)
{
    PWM_Type *base = pwm_bases[hpwm->channel];
    if (!hpwm->int_en) {
        base->ICSEL &= ~PWM_ICSEL_INTEN_Msk;
    } else {
        pwm_handle[hpwm->channel] = hpwm;
        IOIM_SetIRQHandler(PWM, PWM_CommonIRQHandler, NULL);
        SET_VAL_MSK(base->ICSEL, PWM_ICSEL_INTSEL_Msk, PWM_ICSEL_INTSEL_Pos,
            hpwm->int_source);
        base->ICSEL |= PWM_ICSEL_INTEN_Msk;
    }
}

void PWM_Init(pwm_handle_t *hpwm)
{
    struct pwm_channel_config config;
    PWM_GetChannelDefaultConfig(&config);
    config.channel              = hpwm->channel;
    config.cntmode              = PWM_CntModeUpDown;
    config.prescaler_divmux     = PWM_PrescalerDivMux8;
    config.prescaler            = 3UL;
    config.prescaler_mode       = PWM_PrescModeAlways;
    config.prescaler_cmd        = PWM_PrescCmdSave;
    config.scmpamode            = PWM_SCmpxModeDirect;
    config.scmpbmode            = PWM_SCmpxModeDirect;
    config.ldamode              = PWM_LdxModeNoLoad;
    config.ldbmode              = PWM_LdxModeNoLoad;
    config.cnt_eq_cmpa_inc_outa = PWM_OutXCmdClear;
    config.cnt_eq_cmpa_dec_outa = PWM_OutXCmdSet;
    config.cnt_eq_cmpb_inc_outb = PWM_OutXCmdClear;
    config.cnt_eq_cmpb_dec_outb = PWM_OutXCmdSet;
    config.cmd                  = PWM_RunCmdRun;
    config.period               = 1000000UL;
    config.cmpa                 = config.period / 2;
    config.cmpa                 = config.period / 2;
    PWM_InitChannel(hpwm, config);
}

void PWM_Enable(pwm_handle_t *hpwm, bool enable)
{
    if (enable) {
        PWM_SetPeriodUs(hpwm);
    }
    PWM_Run(hpwm, enable);
}

void PWM_ChannelIRQHandler(pwm_handle_t *hpwm)
{
    PWM_Type *base = pwm_bases[hpwm->channel];
    PWM_IntCallback(hpwm);
    base->ICCLR |= PWM_ICCLR_INT_Msk;
}

static void PWM_CommonIRQHandler(void *unused)
{
    UNUSED(unused);

    for (uint32_t i = 0UL; i < DIM(pwm_handle); i++) {
        if (pwm_bases[i]->ICSTS & PWM_ICSTS_INT_Msk) {
            PWM_ChannelIRQHandler(pwm_handle[i]);
        }
    }
}

__attribute__((weak)) void PWM_IntCallback(pwm_handle_t *hpwm)
{
    UNUSED(hpwm);
}
