/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




#include "hal_ppu.h"

/*SCB */
#define SCB_CPACR_FPU           (0xF << 20)   /*!< cp10 and cp11 set in CPACR  */
#define CPU0CPUID 0
#define CPU1CPUID 1

/* Блок определений функций и типов функций*/
typedef enum ppu_status (*func)(void); /* Тип функции для выполнения специальных
                                        * действий перед установкой состояния */

/* Функции для выполнения специальных действий перед установкой состояния */
#define F_NO ((func)0) /* Макрос указывающий, что запрашиваемый режим PPU 
                        * недоступен. Действий никаких нет */
static enum ppu_status F_EM(void); /* Действий никаких нет  */
static enum ppu_status CROF(void); /* Действия для блока CRYPTO при переводе
                                    * в Off */
static enum ppu_status GMOF(void); /* Действия для блока GMS при переводе
                                    * в Off */
static enum ppu_status C0OF(void); /* Действия для блока CPU0 при переводе
                                    * в Off */
static enum ppu_status C1OF(void); /* Действия для блока CPU1 при переводе
                                    * в Off */

/*!
 * @brief Разрешение WIC и EWC
 */
static void PPU_WICAndEWCEnable(int index_cpu);

/*!
 * @brief Получение ID CPU
 */
static inline uint32_t get_CPUID();

/*!
 * @brief Установка режима сна
 * 
 * @param sleepdeep 1 - DeepSleep, 0 - Sleep
 */
static void SetSleepMode (uint32_t sleepdeep);

/*!
 * @brief Получение индекса блока PPU по базовому адресу
 * 
 * @param base Базовый адрес блока
 * 
 * @return -1 - Блок не представлен, >=0 - индекс блока
 */
static int32_t IndexFromBase(PPU_Type * base);

/*!
 * @brief Общая для всех установка состояния PPU
 * 
 * @param base Базовый адрес блока
 * @param mode Состояние
 */
static void PPU_SetStateX(PPU_Type * base, enum ppu_power_mode mode);

/*!
 * @brief Общая для всех установка динамического состояния PPU
 * 
 * @param base Базовый адрес блока
 * @param mode Состояние
 */
static void PPU_SetStateXDynamic(PPU_Type * base, enum ppu_power_mode mode);

/*!
 * @brief Разрешение работы WIC+EVC
 * 
 * @param index_cpu Индекс CPU
 */
static void PPU_WICAndEWCEnable(int index_cpu);

/* Блок глобальных переменных */

static enum ppu_status last_error = PPU_Status_Ok; /* Признак корректности 
                                                    * выполнения функции */

/* Функции запросов отключения CPUy от CPUx */
ReqOffForCPU CPU0ToCPU1 = NULL;
ReqOffForCPU CPU1ToCPU0 = NULL;

/* Массив функций определяющих возможность статической конфигурации:
   1) F_NO - статическая конфигураци невозможна
   2) F_EM - статическая конфигураци возможна, дополнительных действий нет
   3) другое - статическая конфигураци возможна, дополнительные действия есть
*/
static func SpecFuncStatic[PPU_DomainMax + 1][PPU_PowerModeMax + 1] = {
    /*Off         MemRet      LogicRet    MemOff      On          DbgRecov
            OffEmu      MemRetEmu   FullRet     FuncRet     WarmRst       */
    { F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_EM, F_NO, F_NO}, /* PPU_DomainCPU0 */
    { F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_EM, F_NO, F_NO}, /* PPU_DomainCPU1 */
    { F_EM, F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_EM, F_NO, F_NO}, /* PPU_DomainDEBUG */
    { CROF, F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_EM, F_NO, F_NO}, /* PPU_DomainCRYPTO */
    { GMOF, F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_EM, F_NO, F_NO}, /* PPU_DomainGMS */
    { F_EM, F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_EM, F_NO, F_NO}, /* PPU_DomainGNSS */
    { F_EM, F_EM, F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_EM, F_NO, F_NO}, /* PPU_DomainSRAM0 */
    { F_EM, F_EM, F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_EM, F_NO, F_NO}, /* PPU_DomainSRAM1 */
    { F_EM, F_EM, F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_EM, F_NO, F_NO}, /* PPU_DomainSRAM2 */
    { F_EM, F_EM, F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_EM, F_NO, F_NO}, /* PPU_DomainSRAM3 */
    { F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_EM, F_NO, F_NO}, /* PPU_DomainSYS */
};

/* Массив функций определяющих возможность статической конфигурации:
   1) F_NO - статическая конфигураци невозможна
   2) F_EM - статическая конфигураци возможна, дополнительных действий нет
   3) другое - статическая конфигураци возможна, дополнительные действия есть
*/
static func SpecFuncDynamic[PPU_DomainMax + 1][PPU_PowerModeMax + 1] = {
    /*Off         MemRet      LogicRet    MemOff      On          DbgRecov
            OffEmu      MemRetEmu   FullRet     FuncRet     WarmRst       */
    { C0OF, F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_EM, F_NO, F_NO}, /* PPU_DomainCPU0 */
    { C1OF, F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_EM, F_NO, F_NO}, /* PPU_DomainCPU1 */
    { F_EM, F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_EM, F_NO, F_NO}, /* PPU_DomainDEBUG */
    { F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_NO}, /* PPU_DomainCRYPTO */
    { F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_NO}, /* PPU_DomainGMS */
    { F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_NO}, /* PPU_DomainGNSS */
    { F_EM, F_NO, F_EM, F_NO, F_NO, F_NO, F_NO, F_NO, F_EM, F_NO, F_NO}, /* PPU_DomainSRAM0 */
    { F_EM, F_NO, F_EM, F_NO, F_NO, F_NO, F_NO, F_NO, F_EM, F_NO, F_NO}, /* PPU_DomainSRAM1 */
    { F_EM, F_NO, F_EM, F_NO, F_NO, F_NO, F_NO, F_NO, F_EM, F_NO, F_NO}, /* PPU_DomainSRAM2 */
    { F_EM, F_NO, F_EM, F_NO, F_NO, F_NO, F_NO, F_NO, F_EM, F_NO, F_NO}, /* PPU_DomainSRAM3 */
    { F_EM, F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_NO, F_EM, F_NO, F_NO}, /* PPU_DomainSYS */
};

/* Допустимые адреса блоков PPU. Должны соответствовать расположению в
 * SpecFuncDynamic и SpecFuncStatic
 */
PPU_Type * AllPPU[PPU_DomainMax + 1] = {
    CPU0_PPU_Secure,   CPU1_PPU_Secure,  DEBUG_PPU_Secure,
    CRYPTO_PPU_Secure, GMS_PPU_Secure,   GNSS_PPU_Secure,
    SRAM0_PPU_Secure,  SRAM1_PPU_Secure, SRAM2_PPU_Secure,
    SRAM3_PPU_Secure,  SYS_PPU_Secure,
};

/* Блок реализации функций */

static inline uint32_t get_CPUID()
{
   return CPU_IDENTITY_Secure->CPUID;
}

static void SetSleepMode (uint32_t sleepdeep) {
    if (sleepdeep)
    {
        SET_VAL_MSK(SCB->SCR, SCB_SCR_SLEEPDEEP_Msk, SCB_SCR_SLEEPDEEP_Pos, 1U);
        SET_VAL_MSK(SCB->SCR, SCB_SCR_SLEEPONEXIT_Msk, SCB_SCR_SLEEPONEXIT_Pos,
            1U);
    }
    else
    {
        SET_VAL_MSK(SCB->SCR, SCB_SCR_SLEEPDEEP_Msk, SCB_SCR_SLEEPDEEP_Pos, 0U);
        SET_VAL_MSK(SCB->SCR, SCB_SCR_SLEEPONEXIT_Msk, SCB_SCR_SLEEPONEXIT_Pos,
            0U);
    }
}

static enum ppu_status F_EM(void)
{
    return PPU_Status_Ok;
}

static enum ppu_status CROF(void)
{
    if (GET_VAL_MSK(CRYPTO_PPU_Secure->PWSR, PPU_PWSR_PWR_STATUS_Msk,
        PPU_PWSR_PWR_STATUS_Pos) == PPU_PowerModeOff)
        return PPU_Status_Ok;
    *((volatile unsigned int *)0x50088A80) = 0x00000001;
    return PPU_Status_Ok;
}

static enum ppu_status GMOF(void)
{
    if (GMS_PPU_Secure->PWSR != 0) {
        *((volatile unsigned int *)0x5009000C) |= (0x1 << 11);
    }
    return PPU_Status_Ok;
}

static enum ppu_status C0OF(void)
{
    if (get_CPUID() == CPU0CPUID)
    { /* Отключение питания CPU0 из CPU0 */
        SetSleepMode(1);
    }
    else
    { /* Отключение питания CPU0 из CPU1 */
        if (GET_VAL_MSK(CPU0_PPU_Secure->PWSR, PPU_PWSR_PWR_STATUS_Msk,
            PPU_PWSR_PWR_STATUS_Pos) != PPU_PowerModeOff)
        {
            /* Вызов механизма отключения CPU0 из CPU1 */
            if (CPU1ToCPU0 != NULL)
                CPU1ToCPU0();
            else 
                return PPU_Status_ConfigError;
            /* Ожидание состояния */
            while (GET_VAL_MSK(CPU0_PPU_Secure->PWSR, PPU_PWSR_PWR_STATUS_Msk,
                PPU_PWSR_PWR_STATUS_Pos) != PPU_PowerModeOff)
                ;
        }
    }
    return PPU_Status_Ok;
}

static enum ppu_status C1OF(void){
    /* Для выключения PD_CPU ядро нужно перевести в deepsleep,
     * настроить WIC/EWC, настроить FPU.*/
    if (get_CPUID() == CPU1CPUID)
    { /* Отключение питания CPU1 из CPU1 */
        SetSleepMode(1); /* Режим глубокого сна */
        SCB->CPACR &= ~SCB_CPACR_FPU;
        SCnSCB->CPPWR |= 1<<20;
        *((unsigned int *)0xE000E00C) |= 1<<20; /* Разрешить выключение FPU */
    }
    else
    { /* Отключение питания CPU1 из CPU0 */
        if (GET_VAL_MSK(CPU1_PPU_Secure->PWSR, PPU_PWSR_PWR_STATUS_Msk,
            PPU_PWSR_PWR_STATUS_Pos) != PPU_PowerModeOff)
        {
            /* Вызов механизма отключения CPU1 из CPU0 */
            if (CPU0ToCPU1 != NULL)
                CPU0ToCPU1();
            else
                return PPU_Status_ConfigError;
            /* Ожидание состояния */
            while (GET_VAL_MSK(CPU1_PPU_Secure->PWSR, PPU_PWSR_PWR_STATUS_Msk,
                PPU_PWSR_PWR_STATUS_Pos) != PPU_PowerModeOff)
                ;
        }
    }
    return PPU_Status_Ok;
}

static int32_t IndexFromBase(PPU_Type * base)
{
    int i;
    for (i = PPU_DomainMax; i >= 0; i--)
        if (AllPPU[i] == base)
            break;
    return i;
}

static void PPU_SetStateX(PPU_Type * base, enum ppu_power_mode mode)
{
    do {
        base->PWPR = mode;
        while(base->STSR == mode && base->PWSR != mode) {
            __NOP();
        }
    } while(base->PWSR != mode);
}

static void PPU_SetStateXDynamic(PPU_Type * base, enum ppu_power_mode mode)
{
    uint32_t pwpr_reg;
    pwpr_reg = GET_VAL_MSK(base->PWPR, 0xffffffff, 0);
    SET_VAL_MSK(pwpr_reg, PPU_PWPR_PWR_DYN_EN_Msk,
        PPU_PWPR_PWR_DYN_EN_Pos, 1);
    SET_VAL_MSK(pwpr_reg, PPU_PWPR_PWR_POLICY_Msk,
        PPU_PWPR_PWR_POLICY_Pos, mode);
    SET_VAL_MSK(base->PWPR, 0xffffffff, 0, pwpr_reg);
}

void PPU_StateOffRequestHandler(void)
{
    PPU_Type *base;
    uint32_t cpu_id = get_CPUID();
    base = (cpu_id == CPU0CPUID) ? CPU0_PPU_Secure : CPU1_PPU_Secure;
    /* Режим глубокого сна */
    SetSleepMode(1); 
    /* WIC/EWC */
    PPU_WICAndEWCEnable(cpu_id);
    if (cpu_id == 1) {
        /* Отключаем FPU */
        SCB->CPACR &= ~SCB_CPACR_FPU;
        SCnSCB->CPPWR |= 1<<20;
        *((volatile unsigned int *)0xE000E00C) |= 1<<20; /* Разрешить 
                                                          * выключение FPU */
    }
    PPU_SetStateDynamic(base, PPU_PowerModeOff);
    __WFI();
}

enum ppu_status PPU_Init(PPU_Type * base, struct ppu_config * cfg)
{
    if ((NULL == cfg) || (IndexFromBase(base) < 0))
        return PPU_Status_InvalidArgument;
    if (base == CPU1_PPU_Secure)
        CPU0ToCPU1 = cfg->reqForCPU;
    if (base == CPU0_PPU_Secure)
        CPU1ToCPU0 = cfg->reqForCPU;
    return PPU_Status_Ok;
}

enum ppu_status PPU_SetState(PPU_Type * base, enum ppu_power_mode mode)
{
    int32_t index;
    enum ppu_status status;
    if (mode > PPU_PowerModeMax)
        return PPU_Status_InvalidArgument;

    index = IndexFromBase(base);
    if (index < 0)
        return PPU_Status_InvalidArgument;
    
    if (SpecFuncStatic[index][mode] == F_NO)
        return PPU_Status_FeatureNotSupport;
    status = SpecFuncStatic[index][mode]();
    if (status != PPU_Status_Ok)
        return status;
    PPU_SetStateX(base, mode);

    return PPU_Status_Ok;
}

enum ppu_status PPU_SetStateDynamic(PPU_Type * base, enum ppu_power_mode mode)
{
    int32_t index;
    enum ppu_status status;

    if (mode > PPU_PowerModeMax)
        return PPU_Status_InvalidArgument;

    index = IndexFromBase(base);
    if (index < 0)
        return PPU_Status_InvalidArgument;

    if (SpecFuncDynamic[index][mode] == F_NO)
        return PPU_Status_FeatureNotSupport;

    status = SpecFuncDynamic[index][mode]();
    if (status != PPU_Status_Ok)
        return status;

    PPU_SetStateXDynamic(base, mode);

    return PPU_Status_Ok;
}

enum ppu_power_mode PPU_GetPowerState(PPU_Type * base)
{
    last_error = PPU_Status_Ok;
    if (IndexFromBase(base) < 0) {
        last_error = PPU_Status_InvalidArgument;
        return PPU_PowerModeOff;
    }
    return GET_VAL_MSK(base->PWSR, PPU_PWSR_PWR_STATUS_Msk,
        PPU_PWSR_PWR_STATUS_Pos);
}

enum ppu_status PPU_GetLastAPIStatus(void)
{
    return last_error;
}

static void PPU_WICAndEWCEnable(int index_cpu)
{
    switch (index_cpu) {
        case 0:
            SET_VAL_MSK(SYSCTR_Secure->WICCTRL,
                SYSCTR_WICCTRL_CPU0WICEN_SET_Msk,
                SYSCTR_WICCTRL_CPU0WICEN_SET_Pos, 1U);
            SET_VAL_MSK(SYSCTR_Secure->EWCCTRL,
                SYSCTR_EWCCTRL_EWC0EN_SET_Msk,
                SYSCTR_EWCCTRL_EWC0EN_SET_Pos, 1U);
            break;
        case 1:
            SET_VAL_MSK(SYSCTR_Secure->WICCTRL,
                SYSCTR_WICCTRL_CPU1WICEN_SET_Msk,
                SYSCTR_WICCTRL_CPU1WICEN_SET_Pos, 1U);
            SET_VAL_MSK(SYSCTR_Secure->EWCCTRL,
                SYSCTR_EWCCTRL_EWC1EN_SET_Msk,
                SYSCTR_EWCCTRL_EWC1EN_SET_Pos, 1U);
            break;
        default:
            return;
    }
}

uint32_t PPU_GetPDxSenseFromPDy(enum ppu_domain_index pd_dst,
    enum ppu_sense_index pd_src)
{
    uint32_t res = 0;
    last_error = PPU_Status_Ok;

    switch (pd_dst) {
        case PPU_DomainSYS:
            res = GET_VAL_MSK(SYSCTR_Secure->PDCM_SYS_SENSE, 1, pd_src);
            break;
        case PPU_DomainSRAM0:
            if ((pd_src <= PPU_SenseCPU1) || (pd_src = PPU_SenseSRAM0)) {
                res = GET_VAL_MSK(SYSCTR_Secure->PDCM_SRAM0_SENSE, 1, pd_src);
            }else {
                last_error = PPU_Status_FeatureNotSupport;
                res = 0;
            }
            break;
        case PPU_DomainSRAM1:
            if ((pd_src <= PPU_SenseCPU1) || (pd_src = PPU_SenseSRAM1)) {
                res = GET_VAL_MSK(SYSCTR_Secure->PDCM_SRAM1_SENSE, 1, pd_src);
            }else {
                last_error = PPU_Status_FeatureNotSupport;
                res = 0;
            }
            break;
        case PPU_DomainSRAM2:
            if ((pd_src <= PPU_SenseCPU1) || (pd_src = PPU_SenseSRAM2)) {
                res = GET_VAL_MSK(SYSCTR_Secure->PDCM_SRAM2_SENSE, 1, pd_src);
            }else {
                last_error = PPU_Status_FeatureNotSupport;
                res = 0;
            }
            break;
        case PPU_DomainSRAM3:
            if ((pd_src <= PPU_SenseCPU1) || (pd_src = PPU_SenseSRAM3)) {
                res = GET_VAL_MSK(SYSCTR_Secure->PDCM_SRAM3_SENSE, 1, pd_src);
            }else {
                last_error = PPU_Status_FeatureNotSupport;
                res = 0;
            }
            break;
        default:
            last_error = PPU_Status_FeatureNotSupport;
            res = 0;
    }
    return res;
}

enum ppu_status PPU_SetPDCMPPUSense(enum ppu_domain_index pd_dst,
    enum ppu_sense_index pd_src, uint32_t sense)
{
    enum ppu_status res = PPU_Status_Ok;

    switch (pd_dst) {
        case PPU_DomainSYS:
            SET_VAL_MSK(SYSCTR_Secure->PDCM_SYS_SENSE, 1, pd_src, sense);
            break;
        case PPU_DomainSRAM0:
            if ((pd_src <= PPU_SenseCPU1) || (pd_src = PPU_SenseSRAM0))
                SET_VAL_MSK(SYSCTR_Secure->PDCM_SRAM0_SENSE, 1, pd_src, sense);
            else 
                res = PPU_Status_FeatureNotSupport;
            break;
        case PPU_DomainSRAM1:
            if ((pd_src <= PPU_SenseCPU1) || (pd_src = PPU_SenseSRAM1))
                SET_VAL_MSK(SYSCTR_Secure->PDCM_SRAM1_SENSE, 1, pd_src, sense);
            else
                res = PPU_Status_FeatureNotSupport;
            break;
        case PPU_DomainSRAM2:
            if ((pd_src <= PPU_SenseCPU1) || (pd_src = PPU_SenseSRAM2))
                SET_VAL_MSK(SYSCTR_Secure->PDCM_SRAM2_SENSE, 1, pd_src, sense);
            else
                res = PPU_Status_FeatureNotSupport;
            break;
        case PPU_DomainSRAM3:
            if ((pd_src <= PPU_SenseCPU1) || (pd_src = PPU_SenseSRAM3))
                SET_VAL_MSK(SYSCTR_Secure->PDCM_SRAM3_SENSE, 1, pd_src, sense);
            else
                res = PPU_Status_FeatureNotSupport;
            break;
        default:
            res = PPU_Status_FeatureNotSupport;
    }
    return res;
}

enum ppu_status PPU_GetIRQStatus(PPU_Type * base, enum ppu_event_name *irq,
    enum ppu_add_event_name *add_irq)
{
    int32_t index;
    index = IndexFromBase(base);
    if ((index < 0) || (irq == NULL) || (add_irq == NULL))
        return PPU_Status_InvalidArgument;
    *irq = base->ISR;
    *add_irq = base->AISR;
    return PPU_Status_Ok;
}

enum ppu_status PPU_GetIRQMask(PPU_Type * base, enum ppu_event_name *irq,
    enum ppu_add_event_name *add_irq)
{
    int32_t index;
    index = IndexFromBase(base);
    if ((index < 0) || (irq == NULL) || (add_irq == NULL))
        return PPU_Status_InvalidArgument;
    *irq = base->IMR;
    *add_irq = base->AIMR;
    return PPU_Status_Ok;
}

enum ppu_status PPU_SetIRQStatus(PPU_Type *base, enum ppu_event_name irq,
    enum ppu_add_event_name add_irq)
{
    int32_t index;
    index = IndexFromBase(base);
    if (index < 0)
        return PPU_Status_InvalidArgument;
    base->ISR |= irq;
    base->AISR |= add_irq;
    return PPU_Status_Ok;
}

enum ppu_status PPU_SetIRQMask(PPU_Type * base, enum ppu_event_name irq,
    enum ppu_add_event_name add_irq)
{
    int32_t index;
    index = IndexFromBase(base);
    if (index < 0)
        return PPU_Status_InvalidArgument;
    base->IMR |= irq;
    base->AIMR |= add_irq;
    return PPU_Status_Ok;
}

enum ppu_status PPU_ClrIRQStatus(PPU_Type *base, enum ppu_event_name irq,
    enum ppu_add_event_name add_irq)
{
    int32_t index;
    index = IndexFromBase(base);
    if (index < 0)
        return PPU_Status_InvalidArgument;
    base->ISR &= ~irq;
    base->AISR &= ~add_irq;
    return PPU_Status_Ok;
}

enum ppu_status PPU_ClrIRQMask(PPU_Type * base, enum ppu_event_name irq,
    enum ppu_add_event_name add_irq)
{
    int32_t index;
    index = IndexFromBase(base);
    if (index < 0)
        return PPU_Status_InvalidArgument;
    base->IMR &= ~irq;
    base->AIMR &= ~add_irq;
    return PPU_Status_Ok;
}
