/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @addtogroup qspi_driver
 * @{
 */

/*!
 * @file hal_nor_flash.h
 *
 * @brief Интерфейс драйвера флеш-памяти NOR
 */

#ifndef HAL_NOR_FLASH_H
#define HAL_NOR_FLASH_H

#ifdef __cplusplus
extern "C" {
#endif

#include "hal_common.h"
#include "hal_qspi.h"

#if QSPI_USE_DMAC
#include "hal_qspi_dma.h"
#endif

#define DUMMY_BYTE 0x00 /*!< Фиктивные данные */

/*!
 * @name Поля внутреннего статусного регистра
 * @{
 */

#define FLASH_STAT_BUSY (1 << 0) /*!< Производится очистка/запись */
#define FLASH_STAT_WEL  (1 << 1) /*!< Бит Write Enable Latch */

/*!
 * @}
 */

/*!
 * @brief Основной набор команд для флеш-памяти NOR
 */
typedef struct _nor_command_set {
    uint8_t write_status_cmd;      /*!< Запись в статусный регистр */
    uint8_t page_write_memory_cmd; /*!< Программирование страницы */
    uint8_t read_memory_command;   /*!< Чтение памяти */
    uint8_t write_disable_cmd;     /*!< Запрет записи */
    uint8_t read_status_cmd;       /*!< Включение записи */
    uint8_t write_enable_cmd;      /*!< Разрешение записи */
    uint8_t erase_sector_cmd;      /*!< Очистка сектора */
    uint8_t erase_chip_cmd;        /*!< Очистка всей памяти */
} nor_command_set_t;

/*!
 * @brief Статусы драйвера флеш-памяти NOR
 */
typedef enum {
    NOR_Status_Success         = 0U, /*!< Успешное завершение операции */
    NOR_Status_Fail            = 1U, /*!< Ошибка при выполнении операции */
    NOR_Status_InvalidArgument = 2U, /*!< Неверный аргумент */
    NOR_Status_Timeout         = 3U, /*!< Превышено время ожидания */
} nor_status_t;

/*!
 * @brief Структура первичной конфигурации флеш-памяти NOR
 */
typedef struct _nor_config {
    void *mem_control_config;    /*!< Конфигурация контроллера памяти, должен быть присвоен указатель на заполненную структуру @ref qspi_nor_init_config_t */
    void *quad_control_config;   /*!< Конфигурация режима QUAD, должен быть присвоен указатель на заполненную структуру @ref qspi_nor_config_t */
    QSPI_Type *driver_base_addr; /*!< Базовый адрес контроллера */
#if defined(QSPI_DRIVER_VERSION_1_6_0)
    QSPI_XIP_Type *driver_xip_base_addr;    /*!< Базовый адрес контроллера QSPI XIP */
#endif
} nor_config_t;

/*!
 * @brief Контекст драйвера флеш-памяти NOR
 */
typedef struct _nor_handle {
    QSPI_Type *driver_base_addr;    /*!< Базовый адрес контроллера */
#if defined(QSPI_DRIVER_VERSION_1_6_0)
    QSPI_XIP_Type *driver_xip_base_addr;    /*!< Базовый адрес контроллера QSPI XIP */
#endif
    uint32_t page_size_bytes;       /*!< Размер страницы в байтах */
    uint32_t sector_size_bytes;     /*!< Размер сектора в байтах */
    uint32_t memory_size_bytes;     /*!< Общий размер памяти микросхемы флеш-памяти */
    uint32_t max_sector_erase_time; /*!< Максимальное время очистки сектора в мс */
    uint32_t max_page_program_time; /*!< Максимальное время записи страницы в мкс */
    uint32_t max_chip_erase_time;   /*!< Максимальное время очистки всей флеш-памяти в мс */
    void *device_specific;          /*!< Указатель на конкретное устройство, должен быть присвоен указатель на структуру @ref qspi_nor_handle_t */
    qspi_xip_config_t xip_config;   /*!< Структура с настройками режима XIP */
    qspi_config_t qspi_config;      /*!< Конфигурация контроллера QSPI */
} nor_handle_t;

/*!
 * @brief Получение конфигурации XIP QSPI по умолчанию
 *
 * @param qspi_xip_config Конфигурационная структура режима XIP QSPI
 */
void QSPI_GetDefaultConfigXIP(qspi_xip_config_t *qspi_xip_config);

/*!
 * @brief Получение стандартного набора комманд SPI Flash
 *
 * @param command_set Набор команд SPI Flash
 */
void QSPI_GetDefaultCommandSet(nor_command_set_t *command_set);

/*!
 * @brief Заполнение регистра XIPCFG контроллера QSPI
 *
 * @param base            Базовый адрес контроллера QSPI
 * @param qspi_xip_config Конфигурационная структура режима XIP QSPI
 */
#if defined(QSPI_DRIVER_VERSION_1_6_0)
nor_status_t QSPI_ConfigureXIP(QSPI_XIP_Type *base,
    qspi_xip_config_t *qspi_xip_config);
#elif defined(QSPI_DRIVER_VERSION_2_0_0)
nor_status_t QSPI_ConfigureXIP(QSPI_Type *base,
    qspi_xip_config_t *qspi_xip_config);
#else
    #error "Please define QSPI controller version"
#endif

/*!
 * @brief Включение и конфигурация режима QSPI XIP
 *
 * @param handle Контекст драйвера флеш-памяти NOR
 */
nor_status_t QSPI_EnableXIP(nor_handle_t *handle);

/*!
 * @brief Выключение режима QSPI XIP
 *
 * @param handle Контекст драйвера флеш-памяти NOR
 */
void QSPI_DisableXIP(nor_handle_t *handle);

/*!
 * @brief Инициализация устройства флеш-памяти NOR
 *
 * Функция инициализирует контроллер QSPI и флеш-память NOR.
 *
 * @param config          Конфигурация флеш-памяти NOR
 * @param handle          Контекст драйвера флеш-памяти NOR
 *
 * @retval #NOR_Status_Success
 * @retval #NOR_Status_Fail
 * @retval #NOR_Status_InvalidArgument
 */
nor_status_t QSPI_NorFlashInit(nor_config_t *config, nor_handle_t *handle);

/*!
 * @brief Чтение данных с флеш-памяти NOR
 *
 * @param handle  Контекст драйвера флеш-памяти NOR
 * @param address Начальный адрес памяти
 * @param buffer  Буфер памяти
 * @param length  Размер данных в байтах
 *
 * @retval #NOR_Status_Success
 * @retval #NOR_Status_Fail
 * @retval #NOR_Status_InvalidArgument
 */
nor_status_t NOR_FlashRead(nor_handle_t *handle, uint32_t address,
    uint8_t *buffer, uint32_t length);

/*!
 * @brief Программирование страницы флеш-памяти NOR
 *
 * @param handle  Контекст драйвера флеш-памяти NOR
 * @param address Адрес программируемой страницы
 * @param buffer  Буфер памяти
 *
 * @retval #NOR_Status_Success
 * @retval #NOR_Status_Fail
 * @retval #NOR_Status_InvalidArgument
 * @retval #NOR_Status_Timeout
 */
nor_status_t NOR_FlashPageProgram(nor_handle_t *handle, uint32_t address,
    uint8_t *buffer, uint32_t length);

/*!
 * @brief Очистка блока памяти
 *
 * @param handle  Контекст драйвера флеш-памяти NOR
 * @param address Начальный адрес для очистки
 * @param size    Размер блока для очистки
 *
 * @retval #NOR_Status_Success
 * @retval #NOR_Status_Fail
 * @retval #NOR_Status_InvalidArgument
 * @retval #NOR_Status_Timeout
 */
nor_status_t NOR_FlashEraseBlock(nor_handle_t *handle, uint32_t address,
    uint32_t size);

/*!
 * @brief Очистка чипа флеш-памяти NOR
 *
 * @param handle Контекст драйвера флеш-памяти NOR
 *
 * @retval #NOR_Status_Success
 * @retval #NOR_Status_InvalidArgument
 * @retval #NOR_Status_Timeout
 */
nor_status_t NOR_FlashEraseChip(nor_handle_t *handle);

/*!
 * @brief Чтение в режиме XIP
 *
 * @param address  Адрес чтения XIP
 * @param buffer Буфер записи прочитанных данных
 * @param length Размер буфера для чтения
 */
void NOR_FlashReadXIP(uint32_t address, uint8_t *buffer, uint32_t length);

/*!
 * @brief Заполнение блока флеш-памяти NOR
 *
 * @param handle  Контекст драйвера NOR Flash
 * @param address Адрес блока для записи
 * @param page_pointer Указатель на буфер для записи
 * @param length Размер буфера для записи
 *
 * @retval #NOR_Status_Success
 * @retval #NOR_Status_Fail
 * @retval #NOR_Status_InvalidArgument
 */
nor_status_t NOR_FlashProgramBlock(nor_handle_t *handle,
    uint32_t address, uint8_t *page_pointer, uint32_t length);

#if QSPI_USE_DMAC
void QSPI_SetDMAHandle(qspi_dma_handle_t *handle);
#endif

#ifdef __cplusplus
}
#endif

#endif /* HAL_NOR_FLASH_H */

/*!
 * @}
 */
