/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @defgroup ioim Драйвер менеджера прерываний IO устройств и DMA каналов
 *
 * @brief Менеджер прерываний IO устройств и каналов DMA
 *
 * Менеджер прерываний выполняет регистрацию векторов прерываний
 * устройств IO (UART, I2C, I2S, SPI) и каналов DMA. При срабатывании прерывания
 * вызывает обработчик из драйвера устройства с передачей указателей
 * на базовый адрес и контекст. Обработчик драйвера должен перед
 * этим быть зарегистрирован соответствующей функцией.
 */

/*!
 * @addtogroup ioim
 * @{
 */

/*!
 * @file hal_ioim.h
 *
 * @brief Интерфейс менеджера прерываний IO устройств
 */

#ifndef HAL_IOIM_H
#define HAL_IOIM_H

#if defined(__cplusplus)
extern "C" {
#endif /* __cplusplus */

#include "hal_common.h"

#define IOIM_NA_IRQ_NUM (-16) /*!< Несуществующий номер прерывания */

/*!
 * @brief Возвращаемые статусы IOIM
 */
typedef enum {
    IOIM_Status_Ok          = 0, /*!< Ошибок нет */
    IOIM_Status_UnknownBase = 1, /*!< Неизвестный базовый адрес устройства */
    IOIM_Status_NullHandler = 2, /*!< Адрес обработчика прерывания равен 0 */
} ioim_status_t;

/*!
 * @brief Получение номера прерывания в системе
 *
 * @param base Базовый адрес устройства
 *
 * @return Номер прерывания устройства
 */
int32_t IOIM_GetIRQNumber(void *base);

/*!
 * @brief Установка обработчика прерывания для устройства IO
 *
 * Функция вносит в свою таблицу прерываний обработчик handler и
 * включает вектор прерывания в системе. При срабатывании прерывания
 * в обработчик будут переданы аргументы base и handle.
 *
 * @param base    Базовый адрес устройства
 * @param handler Указатель на функцию обработчик прерывания
 * @param handle  Контекст драйвера устройства
 *
 * @retval #IOIM_Status_Ok
 * @retval #IOIM_Status_UnknownBase
 * @retval #IOIM_Status_NullHandler
 */
ioim_status_t IOIM_SetIRQHandler(void *base, void *handler, void *handle);

/*!
 * @brief Сброс обработчика прерывания для устройства IO
 *
 * Обработчик прерывания устройства удаляется из таблицы,
 * отключается вектор прерывания для данного устройства.
 *
 * @param base Базовый адрес устройства
 *
 * @retval #IOIM_Status_Ok
 * @retval #IOIM_Status_UnknownBase
 */
ioim_status_t IOIM_ClearIRQHandler(void *base);

/*!
 * @brief Установка обработчика прерывания для DMA
 * 
 * @param base Базовый адрес DMA
 * @param channel Номер канала DMA
 * @param handler Указатель на обработчик прерывания
 * @param handle Контекст драйвера DMA
 * 
 * @retval #IOIM_Status_Ok 
 * @retval #IOIM_Status_UnknownBase
 * @retval #IOIM_Status_NullHandler
 */
ioim_status_t IOIM_SetIRQHandler_DMA(void *base, uint32_t channel,
    void *handler, void *handle);

/*!
 * @brief Сброс обработчика прерывания для DMA
 * 
 * Обработчик прерывания DMA удаляется из таблицы,
 * отключается вектор прерывания для данного DMA.
 * 
 * @param base Базовый адрес DMA
 * @param channel Номер канала DMA
 * 
 * @retval #IOIM_Status_Ok
 * @retval #IOIM_Status_UnknownBase
 */
ioim_status_t IOIM_ClearIRQHandler_DMA(void *base, uint32_t channel);

#if defined(__cplusplus)
}
#endif /* __cplusplus */

#endif /* HAL_IOIM_H */

/*!
 * @}
 */
