/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @addtogroup ioim
 * @{
 */

/*!
 * @file hal_ioim.c
 *
 * @brief Имплементация менеджера прерываний IO устройств
 */

#include "hal_ioim.h"

#define IOIM_ADDR_SECURE_MSK 0x10000000 /*!< Маска определения адреса Secure */

/*!
 * @brief Пара соответствия базового адреса устройства и номера прерывания
 */
typedef struct {
    uint32_t base; /*!< Базовый адрес устройства IO */
    uint32_t irqn; /*!< Номер прерывания в системе */
} ioim_base_irq_t;

/*!
 * @brief Пара соответствия базового адреса DMA и номера канала
 */
typedef struct {
    uint32_t base;    /*!< Базовый адрес DMA */
    uint32_t channel; /*!< Номер канала DMA */
    uint32_t irqn;    /*!< Номер прерывания в системе */
} ioim_dma_ch_irq_t;

/*!
 * @brief Таблица трансляции базового адреса устройства в номер прерывания
 */
static const ioim_base_irq_t base_to_irqn[] = {
    { UART0_BASE,   UART0_IRQn },
    { UART1_BASE,   UART1_IRQn },
    { UART2_BASE,   UART2_IRQn },
    { UART3_BASE,   UART3_IRQn },
    { SPI0_BASE,    SPI0_IRQn },
    { SPI1_BASE,    SPI1_IRQn },
    { SPI2_BASE,    SPI2_IRQn },
    { I2C0_BASE,    I2C0_IRQn },
    { I2C1_BASE,    I2C1_IRQn },
    { I2S_BASE,     I2S_IRQn },
    { GPIO0_BASE,   GPIO_0_IRQn },
    { GPIO1_BASE,   GPIO_1_IRQn },
    { GPIO2_BASE,   GPIO_2_IRQn },
    { GPIO3_BASE,   GPIO_3_IRQn },
    { CAN_BASE,     CAN_IRQn },
    { PWRCTR_BASE,  VMON_IRQn },
    { DMA0_BASE,    DMA0_IRQn },
    { DMA1_BASE,    DMA1_IRQn },
    { JTM_BASE,     JTM_IRQn },
    { VTU0_BASE,    VTU0_IRQn },
    { VTU1_BASE,    VTU1_IRQn },
};

/*!
 * @brief Таблица трансляции номера канала DMA в номер прерывания
 */
static const ioim_dma_ch_irq_t dma_ch_irqn[] = {
    {DMA0_BASE, 0, DMA0_CH0_IRQn},
    {DMA0_BASE, 1, DMA0_CH1_IRQn},
    {DMA0_BASE, 2, DMA0_CH2_IRQn},
    {DMA0_BASE, 3, DMA0_CH3_IRQn},
    {DMA0_BASE, 4, DMA0_CH4_IRQn},
    {DMA0_BASE, 5, DMA0_CH5_IRQn},
    {DMA0_BASE, 6, DMA0_CH6_IRQn},
    {DMA0_BASE, 7, DMA0_CH7_IRQn},

    {DMA1_BASE, 0, DMA1_CH0_IRQn},
    {DMA1_BASE, 1, DMA1_CH1_IRQn},
    {DMA1_BASE, 2, DMA1_CH2_IRQn},
    {DMA1_BASE, 3, DMA1_CH3_IRQn},
    {DMA1_BASE, 4, DMA1_CH4_IRQn},
    {DMA1_BASE, 5, DMA1_CH5_IRQn},
    {DMA1_BASE, 6, DMA1_CH6_IRQn},
    {DMA1_BASE, 7, DMA1_CH7_IRQn},
};

#define IOIM_HANDLERS_COUNT (sizeof(base_to_irqn) / sizeof(ioim_base_irq_t)) /*!< Количество обработчиков прерываний */
#define IOIM_DMA_HANDLERS_COUNT (sizeof(dma_ch_irqn) / sizeof(ioim_dma_ch_irq_t)) /*!< Количество обработчиков прерываний DMA */

/*!
 * @brief Контекст вызова обработчика прерывания
 */
typedef struct {
    void *handler; /*!< Указатель на обработчик прерывания */
    void *base;    /*!< Базовый адрес устройства IO */
    void *handle;  /*!< Аргумент обработчика прерывания */
} ioim_handler_ctx_t;

/*!
 * @brief Контекст вызова обработчика прерывания DMA канала
 */
typedef struct {
    void *handler;    /*!< Указатель на обработчик прерывания */
    void *base;       /*!< Базовый адрес устройства DMA */
    void *handle;     /*!< Аргумент обработчика прерывания */
    uint32_t channel; /*!< Номер канала DMA */
} ioim_dma_ch_handler_ctx_t;

/*!
 * @brief Таблица обработчиков прерываний
 */
static ioim_handler_ctx_t ioim_handlers[IOIM_HANDLERS_COUNT] = {0};

/*!
 * @brief Таблица обработчиков прерываний для каналов DMA
 */
static ioim_dma_ch_handler_ctx_t ioim_dma_handlers[IOIM_DMA_HANDLERS_COUNT] = {0};

/*!
 * @brief Получение номера прерывания в таблице
 *
 * @param base Базовый адрес устройства
 *
 * @return Номер прерывания устройства в таблице
 */
static int32_t IOIM_GetHandlerNumber(void *base)
{
    uint32_t base_addr = (uint32_t) base & ~IOIM_ADDR_SECURE_MSK;

    for (uint32_t i = 0; i < IOIM_HANDLERS_COUNT; i++) {
        if (base_addr == base_to_irqn[i].base) {
            return i;
        }
    }
    return IOIM_NA_IRQ_NUM;
}

/*!
 * @brief Получение номера прерывания из таблицы обработчиков DMA
 * 
 * @param base Базовый адрес DMA
 * @param channel Номер канала DMA
 * 
 * @return Номер прерывания канала в таблице
 */
static int32_t IOIM_GetHandlerNumber_DMA(void *base, uint32_t channel)
{
    uint32_t base_addr = (uint32_t) base & ~IOIM_ADDR_SECURE_MSK;

    for (uint32_t i = 0; i < IOIM_DMA_HANDLERS_COUNT; i++) {
        if (base_addr == dma_ch_irqn[i].base && channel == dma_ch_irqn[i].channel) {
            return i;
        }
    }
    return IOIM_NA_IRQ_NUM;
}

int32_t IOIM_GetIRQNumber(void *base)
{
    int32_t handler_num = IOIM_GetHandlerNumber(base);

    if (handler_num >= 0) {
        return base_to_irqn[handler_num].irqn;
    }

    return IOIM_NA_IRQ_NUM;
}

ioim_status_t IOIM_SetIRQHandler(void *base, void *handler, void *handle)
{
    int32_t handler_num = IOIM_GetHandlerNumber(base);

    if (handler_num  == IOIM_NA_IRQ_NUM) {
        return IOIM_Status_UnknownBase;
    }

    if (handler == NULL) {
        return IOIM_Status_NullHandler;
    }

    ioim_handlers[handler_num].handler = handler;
    ioim_handlers[handler_num].base = base;
    ioim_handlers[handler_num].handle = handle;

    int32_t irq_num = base_to_irqn[handler_num].irqn;
    NVIC_EnableIRQ(irq_num);

    return IOIM_Status_Ok;
}

ioim_status_t IOIM_ClearIRQHandler(void *base)
{
    int32_t handler_num = IOIM_GetHandlerNumber(base);

    if (handler_num  == IOIM_NA_IRQ_NUM) {
        return IOIM_Status_UnknownBase;
    }

    ioim_handlers[handler_num].handler = 0;
    ioim_handlers[handler_num].base = 0;
    ioim_handlers[handler_num].handle = 0;

    int32_t irq_num = base_to_irqn[handler_num].irqn;
    NVIC_DisableIRQ(irq_num);

    return IOIM_Status_Ok;
}

ioim_status_t IOIM_SetIRQHandler_DMA(void *base, uint32_t channel,
    void *handler, void *handle)
{
    int32_t handler_num = IOIM_GetHandlerNumber_DMA(base, channel);

    if (handler_num  == IOIM_NA_IRQ_NUM) {
        return IOIM_Status_UnknownBase;
    }

    if (handler == NULL) {
        return IOIM_Status_NullHandler;
    }

    ioim_dma_handlers[handler_num].handler = handler;
    ioim_dma_handlers[handler_num].base    = base;
    ioim_dma_handlers[handler_num].handle  = handle;
    ioim_dma_handlers[handler_num].channel = channel;

    int32_t irq_num = dma_ch_irqn[handler_num].irqn;
    NVIC_EnableIRQ(irq_num);

    return IOIM_Status_Ok;
}

ioim_status_t IOIM_ClearIRQHandler_DMA(void *base, uint32_t channel)
{
    int32_t handler_num = IOIM_GetHandlerNumber_DMA(base, channel);

    if (handler_num  == IOIM_NA_IRQ_NUM) {
        return IOIM_Status_UnknownBase;
    }

    ioim_dma_handlers[handler_num].handler = 0;
    ioim_dma_handlers[handler_num].base    = 0;
    ioim_dma_handlers[handler_num].handle  = 0;
    ioim_dma_handlers[handler_num].channel = 0;

    int32_t irq_num = dma_ch_irqn[handler_num].irqn;
    NVIC_DisableIRQ(irq_num);

    return IOIM_Status_Ok;
}

/*!
 * @brief Вектора прерываний IOIM
 */
#define IOIM_DRIVER_HANDLER(n) void IOIM ## n ## _DriverIRQHandler() \
    { \
        assert(ioim_handlers[n].handler); \
        ((void(*)(void*, void*))ioim_handlers[n].handler)( \
            ioim_handlers[n].base, ioim_handlers[n].handle); \
        __DSB(); \
        __ISB(); \
    }

/*!
 * @brief Вектора прерываний IOIM DMA 
 */
#define IOIM_DMA_DRIVER_HANDLER(n) void IOIM ## n ## _DMADriverIRQHandler() \
    { \
        assert(ioim_dma_handlers[n].handler); \
        ((void(*)(void*))ioim_dma_handlers[n].handler)( \
            ioim_dma_handlers[n].handle); \
        __DSB(); \
        __ISB(); \
    }

/*!
 * @brief Вектор прерывания IOIM номер 0
 */
IOIM_DRIVER_HANDLER(0);
/*!
 * @brief Вектор прерывания IOIM номер 1
 */
IOIM_DRIVER_HANDLER(1);
/*!
 * @brief Вектор прерывания IOIM номер 2
 */
IOIM_DRIVER_HANDLER(2);
/*!
 * @brief Вектор прерывания IOIM номер 3
 */
IOIM_DRIVER_HANDLER(3);
/*!
 * @brief Вектор прерывания IOIM номер 4
 */
IOIM_DRIVER_HANDLER(4);
/*!
 * @brief Вектор прерывания IOIM номер 5
 */
IOIM_DRIVER_HANDLER(5);
/*!
 * @brief Вектор прерывания IOIM номер 6
 */
IOIM_DRIVER_HANDLER(6);
/*!
 * @brief Вектор прерывания IOIM номер 7
 */
IOIM_DRIVER_HANDLER(7);
/*!
 * @brief Вектор прерывания IOIM номер 8
 */
IOIM_DRIVER_HANDLER(8);
/*!
 * @brief Вектор прерывания IOIM номер 9
 */
IOIM_DRIVER_HANDLER(9);
/*!
 * @brief Вектор прерывания IOIM номер 10
 */
IOIM_DRIVER_HANDLER(10);
/*!
 * @brief Вектор прерывания IOIM номер 11
 */
IOIM_DRIVER_HANDLER(11);
/*!
 * @brief Вектор прерывания IOIM номер 12
 */
IOIM_DRIVER_HANDLER(12);
/*!
 * @brief Вектор прерывания IOIM номер 13
 */
IOIM_DRIVER_HANDLER(13);
/*!
 * @brief Вектор прерывания IOIM номер 14
 */
IOIM_DRIVER_HANDLER(14);
/*!
 * @brief Вектор прерывания IOIM номер 15
 */
IOIM_DRIVER_HANDLER(15);
/*!
 * @brief Вектор прерывания IOIM номер 16
 */
IOIM_DRIVER_HANDLER(16);
/*!
 * @brief Вектор прерывания IOIM номер 17
 */
IOIM_DRIVER_HANDLER(17);
/*!
 * @brief Вектор прерывания IOIM номер 18
 */
IOIM_DRIVER_HANDLER(18);
/*!
 * @brief Вектор прерывания IOIM номер 19
 */
IOIM_DRIVER_HANDLER(19);
/*!
 * @brief Вектор прерывания IOIM номер 20
 */
IOIM_DRIVER_HANDLER(20);

/*!
 * @brief Вектор прерывания IOIM для DMA0_CH0
 */
IOIM_DMA_DRIVER_HANDLER(0);
/*!
 * @brief Вектор прерывания IOIM для DMA0_CH1
 */
IOIM_DMA_DRIVER_HANDLER(1);
/*!
 * @brief Вектор прерывания IOIM для DMA0_CH2
 */
IOIM_DMA_DRIVER_HANDLER(2);
/*!
 * @brief Вектор прерывания IOIM для DMA0_CH3
 */
IOIM_DMA_DRIVER_HANDLER(3);
/*!
 * @brief Вектор прерывания IOIM для DMA0_CH4
 */
IOIM_DMA_DRIVER_HANDLER(4);
/*!
 * @brief Вектор прерывания IOIM для DMA0_CH5
 */
IOIM_DMA_DRIVER_HANDLER(5);
/*!
 * @brief Вектор прерывания IOIM для DMA0_CH6
 */
IOIM_DMA_DRIVER_HANDLER(6);
/*!
 * @brief Вектор прерывания IOIM для DMA0_CH7
 */
IOIM_DMA_DRIVER_HANDLER(7);
/*!
 * @brief Вектор прерывания IOIM для DMA1_CH0
 */
IOIM_DMA_DRIVER_HANDLER(8);
/*!
 * @brief Вектор прерывания IOIM для DMA1_CH1
 */
IOIM_DMA_DRIVER_HANDLER(9);
/*!
 * @brief Вектор прерывания IOIM для DMA1_CH2
 */
IOIM_DMA_DRIVER_HANDLER(10);
/*!
 * @brief Вектор прерывания IOIM для DMA1_CH3
 */
IOIM_DMA_DRIVER_HANDLER(11);
/*!
 * @brief Вектор прерывания IOIM для DMA1_CH4
 */
IOIM_DMA_DRIVER_HANDLER(12);
/*!
 * @brief Вектор прерывания IOIM для DMA1_CH5
 */
IOIM_DMA_DRIVER_HANDLER(13);
/*!
 * @brief Вектор прерывания IOIM для DMA1_CH6
 */
IOIM_DMA_DRIVER_HANDLER(14);
/*!
 * @brief Вектор прерывания IOIM для DMA1_CH7
 */
IOIM_DMA_DRIVER_HANDLER(15);

/*!
 * @}
 */
