/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @addtogroup i2c_driver
 * @{
 */

/*!
 * @file hal_i2c_dma.h
 *
 * @brief Дополнение драйвера I2C с пересылкой данных через DMA
 */

#ifndef HAL_I2C_DMA_H
#define HAL_I2C_DMA_H

#include "hal_dma.h"
#include "hal_i2c.h"

/*! @brief Версия драйвера */
#define HAL_SPI_DMA_DRIVER_VERSION (MAKE_VERSION(0, 1, 0))
/*! @brief Глубина буфера I2C контроллера */
#define I2C_MAX_BUFFER_SIZE (8U)

/*!
 * @brief Дескриптор I2C-DMA
 */
typedef struct _i2c_master_dma_handle i2c_master_dma_handle_t;

/*! @brief Функция обратного вызова */
typedef void (*i2c_master_dma_transfer_callback_t)
(I2C_Type *base, i2c_master_dma_handle_t *handle, void *user_data);

/*!
 * @brief Контекст данных прерывания I2C-DMA
 */
struct _i2c_master_dma_handle {
    i2c_master_transfer_states state;                       /*!< Состояние передачи */
    uint32_t remaining_bytes_DMA;                           /*!< Число байт, участвующее в передаче между I2C и DMA */
    uint8_t *buf;                                           /*!< Указатель на буфер приема/передачи данных */
    uint32_t *dummy_data;                                   /*!< Указатель на фиктивные данные для приема */
    dma_handle_t *tx_dma;                                   /*!< Контекст драйвера DMA на передачу */
    dma_handle_t *rx_dma;                                   /*!< Контекст драйвера DMA на прием */
    dma_descriptor_t *tx_desc;                              /*!< Дескриптор многоблочной передачи DMA */
    dma_descriptor_t *rx_desc;                              /*!< Дескриптор многоблочной передачи DMA */
    i2c_master_dma_transfer_callback_t completion_callback; /*!< Функция обратного вызова */
    void *user_data;                                        /*!< Пользовательские данные */
};

#if defined(__cplusplus)
extern "C" {
#endif

/*!
 * @brief Функция инициализации дескриптора I2C-DMA
 *
 * @param base       Базовый адрес I2C master
 * @param handle     Дескриптор I2C-DMA
 * @param callback   Функция обратного вызова
 * @param user_data  Пользовательские данные
 * @param tx_dma     Дескриптор DMA для передачи
 * @param rx_dma     Дескриптор DMA для приема
 */
void I2C_MasterTransferCreateHandleDMA(I2C_Type *base,
    i2c_master_dma_handle_t *handle,
    i2c_master_dma_transfer_callback_t callback,
    void *user_data,
    dma_handle_t *tx_dma, dma_handle_t *rx_dma);

/*!
 * @brief Функция, запускающая I2C транзакцию.
          Данные в буфер/из буфера I2C передаются с помощью I2C.
 *
 * @param base   Базовый адрес I2C master
 * @param handle Дескриптор I2C-DMA
 * @param xfer   Структура конфигурации для I2C транзакции
 *
 * @retval #I2C_DMA_Status_Success
 * @retval #I2C_DMA_Status_DMA_Busy
 * @retval #I2C_DMA_Status_InvalidArgument
 * @retval #I2C_DMA_Status_Addr_Nak
 * @retval #I2C_DMA_Status_Subaddr_Nak
 * @retval #I2C_DMA_Status_Fail
 */
i2c_status_t I2C_MasterTransferDMA(I2C_Type *base,
    i2c_master_dma_handle_t *handle, i2c_master_transfer_t *xfer);

/*!
 * @brief Прекращение передачи I2C
 *
 * @param base   Базовый адрес I2C master
 * @param handle Дескриптор SPI-DMA
 */
void I2C_MasterTransferAbortDMA(I2C_Type *base,
    i2c_master_dma_handle_t *handle);

/*!
 * @brief Инициализация дескрипторов DMA для многоблочной передачи TX
 *
 * @param base       Базовый адрес SPI
 * @param desc       Указатель на массив дескрипторов
 * @param count      Количество DMA дескрипторов
 * @param data_size  Общее число передаваемых данных
 * @param data_width Ширина 1ого передаваемого слова
 * @param src_incr   Инкремент адреса Передатчика
 * @param src_addr   Адрес Передатчика
 */
static inline void I2C_DMADescriptorInitTX(I2C_Type *base,
    dma_descriptor_t *desc, uint32_t count, uint32_t data_size,
    uint8_t src_incr, void *src_addr)
{
    dma_multiblock_config_t config = {
        .count          = count,
        .data_size      = data_size,
        .transfer_type  = DMA_MemoryToPeripheral_DMA,
        .scatter_en     = false,
        .gather_en      = false,
        .src_burst_size = DMA_BurstSize32,
        .dst_burst_size = DMA_BurstSize1,
        .src_incr       = src_incr,
        .dst_incr       = DMA_NoChange,
        .src_data_width = DMA_Transfer8BitWidth,
        .dst_data_width = DMA_Transfer8BitWidth,
        .src_addr       = src_addr,
        .dst_addr       = (void *) &base->IC_DATA_CMD,
        .int_en         = true
    };

    DMA_InitMultiblockDescriptor(desc, &config);
}

/*!
 * @brief Инициализация дескрипторов DMA для многоблочной передачи RX
 *
 * @param base       Базовый адрес SPI
 * @param desc       Указатель на массив дескрипторов
 * @param count      Количество DMA дескрипторов
 * @param data_size  Общее число передаваемых данных
 * @param data_width Ширина 1ого передаваемого слова
 * @param dst_incr   Инкремент адреса Приемника
 * @param dst_addr   Адрес Приемника
 */
static inline void I2C_DMADescriptorInitRX(I2C_Type *base,
    dma_descriptor_t *desc, uint32_t count, uint32_t data_size,
    uint8_t dst_incr, void *dst_addr)
{
    dma_multiblock_config_t config = {
        .count          = count,
        .data_size      = data_size,
        .transfer_type  = DMA_PeripheralToMemory_DMA,
        .scatter_en     = false,
        .gather_en      = false,
        .src_burst_size = DMA_BurstSize1,
        .dst_burst_size = DMA_BurstSize32,
        .src_incr       = DMA_NoChange,
        .dst_incr       = dst_incr,
        .src_data_width = DMA_Transfer8BitWidth,
        .dst_data_width = DMA_Transfer8BitWidth,
        .src_addr       = (void *) &base->IC_DATA_CMD,
        .dst_addr       = dst_addr,
        .int_en         = true
    };

    DMA_InitMultiblockDescriptor(desc, &config);
}

#if defined(__cplusplus)
}
#endif

#endif /* HAL_I2C_DMA_H */

/*!
 * @}
 */
