/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @defgroup gpio Драйвер модуля GPIO
 * @brief Драйвер для управления внешними выводами
 *
 * Драйвер содержит функции управления выводами микросхемы Eliot01 в режимах
 * программного управления состоянием вывода (GPIO модуль), а также установкой
 * альтернативной функции драйвера для работы с устройствами (IOCTR модуль).
 */

/*!
 * @addtogroup gpio
 * @{
 */

/*! @file */

#ifndef HAL_GPIO_H
#define HAL_GPIO_H

#ifdef  __cplusplus
extern "C" {
#endif

#include "hal_common.h"

#ifndef GPIO_PORTA
#define GPIO_PORTA                         0U /*!< Номер порта A */
#endif
#ifndef GPIO_PORTB
#define GPIO_PORTB                         1U /*!< Номер порта B */
#endif
#ifndef GPIO_PORTC
#define GPIO_PORTC                         2U /*!< Номер порта C */
#endif
#ifndef GPIO_PORTD
#define GPIO_PORTD                         3U /*!< Номер порта D */
#endif

#ifndef GPIO_PORTPIN
#define GPIO_PORTPIN(port,pin)             \
    ((((port) & 0xF) << 4) | ((pin) & 0xF)) /*!< Создать соответствие порт-вывод */
#endif
#ifndef GPIO_PORTPIN_GET_PIN_NUM
#define GPIO_PORTPIN_GET_PIN_NUM(portpin)  ((portpin) & 0xF) /*!< Получить номер вывода из соответствия порт-вывод */
#endif
#ifndef GPIO_PORTPIN_GET_MASK
#define GPIO_PORTPIN_GET_MASK(portpin)     \
    (1 << GPIO_PORTPIN_GET_PIN_NUM(portpin)) /*!< Получить маску вывода из соответствия порт-вывод */
#endif
#ifndef GPIO_PORTPIN_GET_PORT_NUM
#define GPIO_PORTPIN_GET_PORT_NUM(portpin) (((portpin) >> 4) & 0xF) /*!< Получить номер порта из соответствия порт-вывод */
#endif

/*!
 * @name Режимы работы выводов GPIO 
 * @{
 */
/*!
 * @brief Список режимов работы вывода GPIO
 * @{
 */
typedef enum {
    GPIO_MODE_HI_Z      = 0x0,  /*!< Режим высокоимпендансного состояния вывода */
    GPIO_MODE_GPIO      = 0x1,  /*!< Режим программируемого вывода GPIO */
    GPIO_MODE_AF        = 0x2,  /*!< Режим альтернативной функции вывода для работы с устройствами */
    GPIO_MODE_INVALID   = 0x3,  /*!< Несуществующий, неверный режим вывода */
} gpio_mode_t;
/*! @}*/

/*!
 * @brief Список альтернативных функций IOCTR выводов устройств
 * @{
 */
typedef enum {
    GPIO_ALT_FUNC_TRACE_JTAG_FBIST  = 0,    /*!< Альтернативная функция вывода для работы JTRACE, JTAG и FBIST */
    GPIO_ALT_FUNC_PWM_VTU           = 1,    /*!< Альтернативная функция вывода для работы PWM и VTU */
    GPIO_ALT_FUNC_I2C_I2S           = 2,    /*!< Альтернативная функция вывода для работы I2C и I2S */
    GPIO_ALT_FUNC_SPI0_SPI1         = 3,    /*!< Альтернативная функция вывода для работы SPIO и SPI1 */
    GPIO_ALT_FUNC_UART              = 4,    /*!< Альтернативная функция вывода для работы UART0, UART1, UART2 и UART3 */
    GPIO_ALT_FUNC_CAN_GNSS_USB      = 5,    /*!< Альтернативная функция вывода для работы CAN, GNSS и USB */
    GPIO_ALT_FUNC_QSPI_SPI2         = 6,    /*!< Альтернативная функция вывода для работы ASPI и SPI2 */
    GPIO_ALT_FUNC_SDMMC_SMC         = 7,    /*!< Альтернативная функция вывода для работы SDMMC и SMC */
} gpio_pin_function_t;
/*! @}*/

/*! @}*/

/*!
 * @name Характеристики внешних выводов GPIO
 * @{
 */

/*!
 * @brief Направление вывода GPIO (вход или выход)
 * @{
 */
typedef enum {
    GPIO_DigitalInput  = 0U,                /*!< Направление работы вывода - вход */
    GPIO_DigitalOutput = 1U,                /*!< Направление работы вывода - выход */
} gpio_pin_direction_t;
/*! @}*/

/*!
 * @brief Настройки внутренних резистивных подтяжек на выводах для режимов Push-Pull и Open-Drain
 * @{
 */
typedef enum {
    GPIO_PULL_NONE     = 0,                 /*!< Без резистивных подтяжек на выводе */
    GPIO_PULL_DOWN     = 1,                 /*!< Резистивная подтяжка на минус питания на выводе */
    GPIO_PULL_UP       = 3,                 /*!< Резистивная подтяжка на плюс питания на выводе */
    GPIO_PULL_NONE_OD  = 4,                 /*!< Без резистивных подтяжек и режим Open-Drain на выводе */
    GPIO_PULL_DOWN_OD  = 5,                 /*!< Резистивная подтяжка на минус питания и режим Open-Drain на выводе */
    GPIO_PULL_UP_OD    = 7,                 /*!< Резистивная подтяжка на плюс питания и режим Open-Drain на выводе */
} gpio_pin_pupd_t;
/*! @}*/

/*!
 * @brief Настройки максимального выходного тока вывода
 * @{
 */
typedef enum {
    GPIO_MAX_CURRENT_2mA  = 0,              /*!< Максимальный выходной ток вывода 2мА */
    GPIO_MAX_CURRENT_4mA  = 1,              /*!< Максимальный выходной ток вывода 4мА */
    GPIO_MAX_CURRENT_8mA  = 2,              /*!< Максимальный выходной ток вывода 8мА */
    GPIO_MAX_CURRENT_12mA = 3,              /*!< Максимальный выходной ток вывода 12мА */
} gpio_pin_max_current_t;
/*! @}*/

/*!
 * @brief Список типов прерываний входных выводов GPIO
 * @{
 */
typedef enum {
    GPIO_INT_LVL_LO       = 0,              /*!< Прерывание по низкому логическому уровню на входном выводе */
    GPIO_INT_LVL_HI       = 1,              /*!< Прерывание по высокому логическому уровню на входном выводе */
    GPIO_INT_EDGE_FALL    = 2,              /*!< Прерывание по спадающему фронту логического уровня на входном выводе */
    GPIO_INT_EDGE_RISE    = 3,              /*!< Прерывание по восходящему фронту логического уровня на входном выводе */
} gpio_int_type_t;
/*! @}*/

/*!
 * @name Secure прерывания порта GPIO
 * @{
 */
#define GPIO_SEC_INT_SEC_ACC_MASK               0x1     /*!< Маска прерывания SEC_ACC */
#define GPIO_SEC_INT_PORT_NONSEC_MASK           0x2     /*!< Маска прерывания PORT_NONSEC */
/*! @}*/

/*! @}*/

/*!
 * @name HAL функции для работы с конкретным выводом GPIO по Secure адресам
 * @{
 */

/*!
 * @brief Чтение режима работы вывода GPIO
 *
 * @param pin   Номер вывода из карты выводов GPIO
 * 
 * @retval      #GPIO_MODE_HI_Z
 * @retval      #GPIO_MODE_GPIO
 * @retval      #GPIO_MODE_AF
 * @retval      #GPIO_MODE_INVALID
 */
gpio_mode_t GPIO_PinMode_Get(uint32_t pin);

/*!
 * @brief Установка вывода в состояние Hi-Z
 *
 * @param pin        Номер вывода из карты выводов GPIO
 */
void GPIO_PinMode_HiZ(uint32_t pin);

/*!
 * @brief Установка вывода в режим альтернативной функции IOCTR
 *
 * @param pin        Номер вывода из карты выводов GPIO
 * @param func       Номер альтернативной функции вывода
 */
void GPIO_PinMode_Function(uint32_t pin, gpio_pin_function_t func);

/*!
 * @brief Установка вывода в режим GPIO
 *
 * @param pin        Номер вывода из карты выводов GPIO
 * @param direction  Направление вывода
 */
void GPIO_PinMode_GPIO(uint32_t pin, gpio_pin_direction_t direction);

/*!
 * @brief Настройка резистивной подтяжки вывода GPIO
 *
 * @param pin        Номер вывода из карты выводов GPIO
 * @param pupd       Тип внешней подтяжки вывода
 */
void GPIO_PinSet_PUPD(uint32_t pin, gpio_pin_pupd_t pupd);

/*!
 * @brief Настройка максимального выходного тока вывода GPIO
 *
 * @param pin        Номер вывода из карты выводов GPIO
 * @param current    Номер максимального тока из списка доступных значений
 */
void GPIO_PinSet_MaxCurrent(uint32_t pin, gpio_pin_max_current_t current);

/*!
 * @brief Включение триггера Шмидта на входном выводе GPIO
 *
 * @param pin        Номер вывода из карты выводов GPIO
 * @param value      Значения: 0 - отключен, 1 - включен
 */
void GPIO_PinSet_Schmitt(uint32_t pin, uint32_t value);

/*!
 * @brief Настройка скорости нарастания сигнала на выводе GPIO
 *
 * @param pin        Номер вывода из карты выводов GPIO
 * @param value      Значения: 0 - fast, 1 - slow
 */
void GPIO_PinSet_SpeedRaise(uint32_t pin, uint32_t value);

/*!
 * @brief Установка логического уровня вывода GPIO
 *
 * @param pin        Номер вывода из карты выводов GPIO
 * @param bit        Значения: 0 - низкий логический уровень, 1 - высокий логический уровень
 */
void GPIO_PinWrite(uint32_t pin, uint32_t bit);

/*!
 * @brief Инвертирование логического уровня вывода GPIO
 *
 * @param pin        Номер вывода из карты выводов GPIO
 */
void GPIO_PinToggle(uint32_t pin);

/*!
 * @brief Чтение логического уровня вывода GPIO
 *
 * @param pin        Номер вывода из карты выводов GPIO
 * 
 * @retval           0 - низкий логический уровень
 * @retval           1 - высокий логический уровень
 */
uint32_t GPIO_PinRead(uint32_t pin);

/*!
 * @brief Включение прерывания на входном выводе GPIO
 *
 * @param pin        Номер вывода из карты выводов GPIO
 * @param int_type   Тип прерывания
 */
void GPIO_PinIRQ_Enable(uint32_t pin, gpio_int_type_t int_type);

/*!
 * @brief Отключение прерывания на входном выводе GPIO
 *
 * @param pin        Номер вывода из карты выводов GPIO
 */
void GPIO_PinIRQ_Disable(uint32_t pin);

/*!
 * @brief Чтение статуса прерывания на входном выводе GPIO
 *
 * @param pin        Номер вывода из карты выводов GPIO
 * 
 * @retval           0 - прерывание отсутствует
 * @retval           1 - прерывание присутствует
 */
uint32_t GPIO_PinIRQ_GetStatus(uint32_t pin);

/*!
 * @brief Сброс прерывания на входном выводе GPIO
 *
 * @param pin        Номер вывода из карты выводов GPIO
 */
void GPIO_PinIRQ_Clear(uint32_t pin);
/*! @}*/

/*!
 * @name HAL функции для работы с конкретным портом GPIO по маске выводов
 * @{
 */

/*!
 * @brief Определение номера порта GPIO
 *
 * @param port       Базовый адрес порта GPIO
 * 
 * @retval -1        Порт GPIO не определен
 * @retval 0         Порт GPIOA
 * @retval 1         Порт GPIOB
 * @retval 2         Порт GPIOC
 * @retval 3         Порт GPIOD
 */
int32_t GPIO_GetInstance(GPIO_Type *port);

/*!
 * @brief Установка выводов порта в состояние Hi-Z
 *
 * @param port       Базовый адрес порта GPIO
 * @param bit_mask   Битовая маска выбранных выводов
 */
void GPIO_PortMode_HiZ(GPIO_Type *port, uint32_t bit_mask);

/*!
 * @brief Установка выводов порта в режим альтернативной функции IOCTR
 *
 * @param port       Базовый адрес порта GPIO
 * @param bit_mask   Битовая маска выбранных выводов
 * @param func       Номер альтернативной функции вывода
 */
void GPIO_PortMode_Function(GPIO_Type *port, uint32_t bit_mask, gpio_pin_function_t func);

/*!
 * @brief Установка выводов порта в режим GPIO
 *
 * @param port       Базовый адрес порта GPIO
 * @param bit_mask   Битовая маска выбранных выводов
 * @param direction  Направление вывода
 */
void GPIO_PortMode_GPIO(GPIO_Type *port, uint32_t bit_mask, gpio_pin_direction_t direction);

/*!
 * @brief Настройка резистивной подтяжки выводов порта GPIO
 *
 * @param port       Базовый адрес порта GPIO
 * @param bit_mask   Битовая маска выбранных выводов
 * @param pupd       Тип внешней подтяжки вывода 
 */
void GPIO_PortSet_PUPD(GPIO_Type *port, uint32_t bit_mask, gpio_pin_pupd_t pupd);

/*!
 * @brief Настройка максимального выходного тока выводов порта GPIO
 *
 * @param port       Базовый адрес порта GPIO
 * @param bit_mask   Битовая маска выбранных выводов
 * @param current    Номер максимального тока из списка доступных значений
 */
void GPIO_PortSet_MaxCurrent(GPIO_Type *port, uint32_t bit_mask, gpio_pin_max_current_t current);

/*!
 * @brief Включение триггера Шмидта на входных выводах порта GPIO
 *
 * @param port       Базовый адрес порта GPIO
 * @param bit_mask   Битовая маска выбранных выводов
 * @param bit_value  Значения по маске битов: 0 - отключен, 1 - включен
 */
void GPIO_PortSet_Schmitt(GPIO_Type *port, uint32_t bit_mask, uint32_t bit_value);

/*!
 * @brief Настройка скорости нарастания сигнала на выводах порта GPIO
 *
 * @param port       Базовый адрес порта GPIO
 * @param bit_mask   Битовая маска выбранных выводов
 * @param bit_value  Значения по маске битов: 0 - fast, 1 - slow
 */
void GPIO_PortSet_SpeedRaise(GPIO_Type *port, uint32_t bit_mask, uint32_t bit_value);

/*!
 * @brief Установка логического уровня выводов порта GPIO
 *
 * @param port       Базовый адрес порта GPIO
 * @param bit_mask   Битовая маска выбранных выводов
 * @param bit_value  Значения по маске битов: 0 - низкий логический уровень, 1 - высокий логический уровень
 */
void GPIO_PortWrite(GPIO_Type *port, uint32_t bit_mask, uint32_t bit_value);

/*!
 * @brief Инвертирование логического уровня выводов порта GPIO
 *
 * @param port       Базовый адрес порта GPIO
 * @param bit_mask   Битовая маска выбранных выводов
 */
void GPIO_PortToggle(GPIO_Type *port, uint32_t bit_mask);
/*!
 * @brief Чтение логического уровня выводов порта GPIO
 *
 * @param port       Базовый адрес порта GPIO
 * @param bit_mask   Битовая маска выбранных выводов
 * 
 * @return 32-битная маска, каждый бит которой означает либо низкий логический уровень вывода (0), либо высокий логический уровень (1)
 */
uint32_t GPIO_PortRead(GPIO_Type *port, uint32_t bit_mask);

/*!
 * @brief Включение прерываний на входных выводах порта GPIO
 *
 * @param port       Базовый адрес порта GPIO
 * @param bit_mask   Битовая маска выбранных выводов
 * @param int_type   Тип прерывания из списка
 */
void GPIO_PortIRQ_Enable(GPIO_Type *port, uint32_t bit_mask, uint32_t int_type);

/*!
 * @brief Отключение прерываний на входных выводах порта GPIO
 *
 * @param port       Базовый адрес порта GPIO
 * @param bit_mask   Битовая маска выбранных выводов
 */
void GPIO_PortIRQ_Disable(GPIO_Type *port, uint32_t bit_mask);

/*!
 * @brief Чтение статусов прерываний на входных выводах порта GPIO
 *
 * @param port       Базовый адрес порта GPIO
 * @param bit_mask   Битовая маска выбранных выводов
 * 
 * @return 32-битная маска, каждый бит которой означает либо прерывание отсутствует (0), либо прерывание присутствует (1)
 */
uint32_t GPIO_PortIRQ_GetStatus(GPIO_Type *port, uint32_t bit_mask);

/*!
 * @brief Сброс прерываний на входных выводах порта GPIO
 *
 * @param port       Базовый адрес порта GPIO
 * @param bit_mask   Битовая маска выбранных выводов
 */
void GPIO_PortIRQ_Clear(GPIO_Type *port, uint32_t bit_mask);

/*!
 * @brief Чтение статуса прерывания по нарушению безопасности порта GPIO
 *
 * @param port       Базовый адрес порта GPIO
 * 
 * @retval           0 - прерывание отсутствует
 * @retval           1 - прерывание присутствует
 */
uint32_t GPIO_PortSecureIRQ_GetStatus(GPIO_Type *port);

/*!
 * @brief Сброс прерывания по нарушению безопасности порта GPIO
 *
 * @param port       Базовый адрес порта GPIO
 */
void GPIO_PortSecureIRQ_Clear(GPIO_Type *port);

/*!
 * @brief Установка разрешений прерываний по нарушению безопасности порта GPIO
 *
 * @param port       Базовый адрес порта GPIO
 * @param mask       Маска разрешения прерываний (SEC_ACC_MASK и PORT_NONSEC_MASK)
 */
void GPIO_PortSecureIRQ_SetMask(GPIO_Type *port, uint32_t mask);

/*!
 * @brief Чтение информации Info1 о прерывании по нарушению безопасности порта GPIO
 *
 * @param port       Базовый адрес порта GPIO
 * 
 * @return           Данные информации размером 32 бита
 */
uint32_t GPIO_PortSecureIRQ_GetInfo1(GPIO_Type *port);

/*!
 * @brief Чтение информации Info2 о прерывании по нарушению безопасности порта GPIO
 *
 * @param port       Базовый адрес порта GPIO
 * 
 * @return           Данные информации размером 32 бита
 */
uint32_t GPIO_PortSecureIRQ_GetInfo2(GPIO_Type *port);

/*!
 * @brief Установка разрешения для Non-secure обращений к порту GPIO
 *
 * @param port       Базовый адрес порта GPIO
 * @param mask       Маска битов разрешения доступа (0 - secure, 1 - non-secure)
 */
void GPIO_PortSet_NonsecureMask(GPIO_Type *port, uint32_t mask);
/*! @}*/

#ifdef  __cplusplus
}
#endif

#endif  /* HAL_GPIO_H */

/*!
 * @}
 */
