/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @addtogroup dualtimer_driver
 * @{
 */

/*!
 * @file hal_dualtimer.c
 *
 * @brief Имплементация драйвера таймеров общего назначения
 */

#include "hal_dualtimer.h"

#define DUALTIMER_ADDRESS_SEC_MSK      (0x10000000) /*!< Маска для доверенного/
                                                     * недоверенного сдвоенного
                                                     * таймера */

#define DUALTIMER_CTRL_REG_DEINIT      (0x20UL)     /*!< Значение регистра CTRL
                                                     * при деинициализации */

/*!
 * @brief Регистр управления 
 */
union dualtimer_control_reg {
    uint32_t value;                                  /*!< Значение регистра */
    struct  {
        uint32_t one_shot_count   : FIELD_BIT(0,0);  /*!< Выбор однократного/повторяющегося запуска */
        uint32_t timer_size       : FIELD_BIT(1,1);  /*!< Выбор 16/32-битной разрядности счетчика */
        uint32_t timer_prescale   : FIELD_BIT(3,2);  /*!< Выбор предделителя */
        uint32_t                  : FIELD_BIT(4,4);  /*!< Резерв */
        uint32_t interrupt_enable : FIELD_BIT(5,5);  /*!< Разрешение прерывания */
        uint32_t timer_mode       : FIELD_BIT(6,6);  /*!< Режим таймера */
        uint32_t timer_enable     : FIELD_BIT(7,7);  /*!< Разрешение работы таймера */
        uint32_t                  : FIELD_BIT(31,8); /*!< Резерв */
    };
};

/* Статус выполнения функций, которые возвращают тип результата
 * отличный от enum dualtimer_status */
static enum dualtimer_status last_status = DUALTIMER_Status_Ok;

/* Базовые адреса таймеров общего назначения */
static DTIM_Type *const dualtimer_bases[DUALTIMER_NUMBER_OF_DUALTIMERS] = {DTIM};

/*!
 * @brief Проверка корректности базового адреса сдвоенного таймера \
 *        и индекса таймера
 *
 * @param base Базовый адрес сдвоенного таймера
 *
 * @retval #DUALTIMER_Status_Ok
 * @retval #DUALTIMER_Status_InvalidArgument
 */
static enum dualtimer_status DUALTIMER_TestBase(DTIM_Type *base, uint32_t index)
{
    uint32_t instance;
    
    uint32_t base_addr = ((uint32_t) base) & ~DUALTIMER_ADDRESS_SEC_MSK;

    for (instance = 0; instance < DUALTIMER_NUMBER_OF_DUALTIMERS; instance++) {
        if (dualtimer_bases[instance] == (DTIM_Type *)base_addr) {
            break;
        }
    }

    if ((instance == DUALTIMER_NUMBER_OF_DUALTIMERS) ||
        (index > DUALTIMER_MAX_INDEX)) 
    {
        return DUALTIMER_Status_InvalidArgument;
    }


    return DUALTIMER_Status_Ok;
}

enum dualtimer_status DUALTIMER_GetDefaultConfig(
    struct dualtimer_hardware_config *config)
{
    if (NULL == config) {
        return DUALTIMER_Status_InvalidArgument;
    }

    config->load          = 0;
    config->bg_load       = 0;
    
    config->enable        = DUALTIMER_Disable; 
    config->mode          = DUALTIMER_FreeRunning;
    config->int_ctrl      = DUALTIMER_InterruptEnable;
    config->prescale      = DUALTIMER_Prescale1;
    config->size          = DUALTIMER_TimerSize16;
    config->cyclicity     = DUALTIMER_WrappingMode;

    return DUALTIMER_Status_Ok;
}

enum dualtimer_status DUALTIMER_Init(DTIM_Type *base, uint32_t index,
    struct dualtimer_hardware_config config)
{
    union dualtimer_control_reg reg;
    
    if (DUALTIMER_Status_Ok != DUALTIMER_TestBase(base, index))
        return DUALTIMER_Status_InvalidArgument;

    reg.value = 0;
    reg.one_shot_count    = config.cyclicity;
    reg.timer_size        = config.size;
    reg.timer_prescale    = config.prescale;
    reg.interrupt_enable  = config.int_ctrl;
    reg.timer_mode        = config.mode;
    reg.timer_enable      = config.enable;

    if (0 == index) {
        if (((base->TIMER1CONTROL) & DTIM_TIMER1CONTROL_TIMERENABLE_Msk) ==
            DTIM_TIMER1CONTROL_TIMERENABLE_Msk)
            {
            return DUALTIMER_Status_TimerBusy;
            }
        base->TIMER1INTCLR  = 0;
        base->TIMER1BGLOAD  = config.bg_load;
        base->TIMER1LOAD    = config.load;
        base->TIMER1CONTROL = reg.value;
    } else {
        if (((base->TIMER2CONTROL) & DTIM_TIMER2CONTROL_TIMERENABLE_Pos) ==
            DTIM_TIMER2CONTROL_TIMERENABLE_Pos)
            {
            return DUALTIMER_Status_TimerBusy;
            }
        base->TIMER2INTCLR  = 0;
        base->TIMER2BGLOAD  = config.bg_load;
        base->TIMER2LOAD    = config.load;
        base->TIMER2CONTROL = reg.value;
    }
    
    return DUALTIMER_Status_Ok;
}

enum dualtimer_status DUALTIMER_Deinit(DTIM_Type *base, uint32_t index)
{
    if (DUALTIMER_Status_Ok != DUALTIMER_TestBase(base, index))
        return DUALTIMER_Status_InvalidArgument;
    
    if (0 == index) {
        base->TIMER1CONTROL = DUALTIMER_CTRL_REG_DEINIT;
        base->TIMER1INTCLR  = 0;
    } else {
        base->TIMER2CONTROL = DUALTIMER_CTRL_REG_DEINIT;
        base->TIMER2INTCLR  = 0;
    }
    return DUALTIMER_Status_Ok;
}

enum dualtimer_status DUALTIMER_Run(DTIM_Type *base, uint32_t index)
{
    if (DUALTIMER_Status_Ok != DUALTIMER_TestBase(base, index))
        return DUALTIMER_Status_InvalidArgument;

    if (0 == index) {
        base->TIMER1CONTROL |= DTIM_TIMER1CONTROL_TIMERENABLE_Msk;
    } else {
        base->TIMER2CONTROL |= DTIM_TIMER2CONTROL_TIMERENABLE_Msk;
    }

    return DUALTIMER_Status_Ok;
}

enum dualtimer_status DUALTIMER_Stop(DTIM_Type *base, uint32_t index)
{
    if (DUALTIMER_Status_Ok != DUALTIMER_TestBase(base, index))
        return DUALTIMER_Status_InvalidArgument;

    if (0 == index) {
        base->TIMER1CONTROL &= ~DTIM_TIMER1CONTROL_TIMERENABLE_Msk;
    } else {
        base->TIMER2CONTROL &= ~DTIM_TIMER2CONTROL_TIMERENABLE_Msk;
    }

    return DUALTIMER_Status_Ok;
}

uint32_t DUALTIMER_GetTick(DTIM_Type *base, uint32_t index)
{
    last_status = DUALTIMER_Status_Ok;
    if (DUALTIMER_Status_Ok == DUALTIMER_TestBase(base, index)) {
        if (0 == index) {
            return base->TIMER1VALUE;
        } else {
            return base->TIMER2VALUE;
        }        
    }

    last_status = DUALTIMER_Status_InvalidArgument;
    return 0;
}

enum dualtimer_status DUALTIMER_GetAPIStatus()
{
    return last_status;
}

uint32_t DUALTIMER_GetRawStatus(DTIM_Type *base, uint32_t index)
{
    last_status = DUALTIMER_Status_Ok;
    if (DUALTIMER_Status_Ok == DUALTIMER_TestBase(base, index)) {
        if (0 == index) {
            return base->TIMER1RIS;
        } else {
            return base->TIMER2RIS;
        }        
    }

    last_status = DUALTIMER_Status_InvalidArgument;
    return 0;
}

uint32_t DUALTIMER_GetStatus(DTIM_Type *base, uint32_t index)
{
    last_status = DUALTIMER_Status_Ok;
    if (DUALTIMER_Status_Ok == DUALTIMER_TestBase(base, index)) {
        if (0 == index) {
            return base->TIMER1MIS;
        } else {
            return base->TIMER2MIS;
        }        
    }

    last_status = DUALTIMER_Status_InvalidArgument;
    return 0;
}

enum dualtimer_status DUALTIMER_Reload(DTIM_Type* base,
    uint32_t index, uint32_t value)
{
    if (DUALTIMER_Status_Ok != DUALTIMER_TestBase(base, index))
        return DUALTIMER_Status_InvalidArgument;

    if (0 == index) {
        base->TIMER1LOAD = value;
    } else {
        base->TIMER2LOAD = value;
    }
    
    return DUALTIMER_Status_Ok;
}

enum dualtimer_status DUALTIMER_IrqClr(DTIM_Type* base, uint32_t index)
{
    if (DUALTIMER_Status_Ok != DUALTIMER_TestBase(base, index))
        return DUALTIMER_Status_InvalidArgument;

    if (0 == index) {
        base->TIMER1INTCLR = 0;
    } else {
        base->TIMER2INTCLR = 0;
    }
    return DUALTIMER_Status_Ok;
}

/*!
 * @}
 */
