/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




/*!
 * @addtogroup clkctr_driver
 * @{
 */

/*!
 * @file hal_clkctr.c
 *
 * @brief Имплементация драйвера модуля CLKCTR
 */

#include "hal_clkctr.h"
#include "hal_rwc.h"

#define CLKCTR_FORCE_CLK_DEV_EN 1 /*!< Устройство тактируется принудительно */

/*!
 * @name Макросы для формирования полей регистра Config PLL 
 * @{
 */
#define CLKCTR_PLL_CFG_NR_MAN(x) (((x) << CLKCTR_PLLCFG_NR_MAN_Pos) \
    & CLKCTR_PLLCFG_NR_MAN_Msk) /*!< Поле NR_MAN */
#define CLKCTR_PLL_CFG_NF_MAN(x) (((x) << CLKCTR_PLLCFG_NF_MAN_Pos) \
    & CLKCTR_PLLCFG_NF_MAN_Msk) /*!< Поле NF_MAN */
#define CLKCTR_PLL_CFG_OD_MAN(x) (((x) << CLKCTR_PLLCFG_OD_MAN_Pos) \
            & CLKCTR_PLLCFG_OD_MAN_Msk) /*!< Поле OD_MAN */
#define CLKCTR_PLL_CFG_MAN(x) (((x)<< CLKCTR_PLLCFG_MAN_Pos) \
            & CLKCTR_PLLCFG_MAN_Msk) /*!< Поле MAN */
#define CLKCTR_PLL_CFG_SEL(x) (((x) << CLKCTR_PLLCFG_SEL_Pos) \
            & CLKCTR_PLLCFG_SEL_Msk) /*!< Поле SEL */
/*!
 * @}
 */

/* Переменные частот тактирования */
static uint32_t xti        = 0; /* По умолчанию 0 - не задано, отличное от 0 - частота на входе */
static uint32_t xti32      = 0; /* По умолчанию 0 - не задано, отличное от 0 - частота на входах XTI32 и XTO32 */
static uint32_t hfi        = 0; /* По умолчанию 0 - не задано, отличное от 0 - частота на входе; значение можно хранить в OTP (flash?), т.к. hfi - уникально для чипа */
static uint32_t lfi        = 0; /* По умолчанию 0 - не задано, отличное от 0 - частота на входе; значение можно хранить в OTP (flash?), т.к. hfi - уникально для чипа */
static uint32_t i2s_extclk = 0; /* По умолчанию 0 - не задано, отличное от 0 - частота на входе */
static uint32_t pmudis     = 0; /* По умолчанию подтяжка к 0 */

/*
 * Массив величин подстройки частоты в процентах от номинальной частоты
 * генератора HFI.
 */
static const float delta_HFITRIM[] = {
      0.0,   3.5,   7.0,  10.0,  13.5,  17.0,  20.5,  24.0,
     27.5,  30.5,  34.0,  37.5,  41.0,  44.5,  47.5,  51.0,
     -3.0,  -9.0, -13.5, -18.5, -23.0, -27.5, -32.0, -36.5,
    -41.0, -46.0, -50.5, -55.0, -59.5, -64.0, -68.5, -73.5,
};

/* Значение частоты для функции CLKCTR_SetPll */
static double MAINCLK_FREQUENCY = HFI_FREQUENCY;

void CLKCTR_SetPll(CLKCTR_Type * base, uint32_t xti_hz, uint32_t pll_mul)
{
    struct clkctr_pll_cfg config; /* Настройки PLL */

    /* Отключить PLL и установить источник частоты для MAINCLK. */
    CLKCTR_SetSwitchMainClk(base, CLKCTR_MainClkTypeHFIClk);
    CLKCTR_GetPLLConfig(base, &config);
    config.man = 0;
    config.sel = 0;
    CLKCTR_SetPLLConfig(base, config);

    /* Установить источник частоты для PLLCLK (HFI или XTI). */
    if (xti_hz == 0) {
        CLKCTR_SetSwitchPLLRef(base, CLKCTR_PLLRefTypeHFIClk);
        MAINCLK_FREQUENCY = hfi;
    } else {
        CLKCTR_SetSwitchPLLRef(base, CLKCTR_PLLRefTypeXTIClk);
        MAINCLK_FREQUENCY = xti_hz;
    }

    /* Ограничить множитель PLL в соответствии с документацией. */
    if (pll_mul > PLL_MAX_MULTIPLIER) {
        pll_mul = PLL_MAX_MULTIPLIER;
    }

    /*
     * Ограничить множитель PLL в соответствии с максимальной частотой
     * MAINCLK.
     */
    if ((MAINCLK_FREQUENCY * (pll_mul + 1)) > CLKCTR_PLLCLK_MAX) {
        pll_mul = (CLKCTR_PLLCLK_MAX / MAINCLK_FREQUENCY) - 1;
    }

    /*
     * Отключить использование PLL, если MAINCLK получается меньше
     * допустимого значения.
     */
    if ((MAINCLK_FREQUENCY * (pll_mul + 1)) < CLKCTR_PLLCLK_OD_MIN) {
        pll_mul = PLL_MIN_MULTIPLIER;
    }

    /*
     * Установить множитель PLL и переключить источник частоты MAINCLK на
     * PLL.
     */
    if (pll_mul > PLL_MIN_MULTIPLIER) {
        CLKCTR_GetPLLConfig(base, &config);
        config.man = 0;
        config.sel = pll_mul;
        CLKCTR_SetPLLConfig(base, config);
        CLKCTR_SetSwitchMainClk(base, CLKCTR_MainClkTypePLLClk);
        MAINCLK_FREQUENCY = MAINCLK_FREQUENCY * (pll_mul + 1);
    }
}

enum clkctr_status CLKCTR_SetPllMan(CLKCTR_Type * base, uint32_t xti_hz,
    uint32_t nr, uint32_t nf, uint32_t od)
{
    uint32_t in_freq;
    double vco;
    struct clkctr_pll_cfg config; /* Настройки PLL */

    if (base == NULL) {
        return CLKCTR_Status_InvalidArgument;
    }

    /* Проверка входных параметров. */
    if (nr == 0 || nr > CLKCTR_NR_MAN_MAX
        || nf == 0 || nf > CLKCTR_NF_MAN_MAX
        || od == 0 || od > CLKCTR_OD_MAN_MAX) {
        return CLKCTR_Status_ConfigureError;
    }
    if (!(od == 1 || (od & 0x1) == 0)) {
        return CLKCTR_Status_ConfigureError;
    }

    if (xti_hz == 0) {
        in_freq = CLKCTR_GetHFIClk(base);
    } else {
        in_freq = xti_hz;
    }

    /* Проверка ограничений для опорной частоты. */
    if (in_freq < CLKCTR_PLLREF_MIN || in_freq > CLKCTR_PLLREF_MAX) {
        return CLKCTR_Status_ConfigureError;
    }

    /*
     * Ограничение множителя PLL в соответствии с диапозоном частот Vco.
     */
    vco = ((double) in_freq / nr) * nf;
    if (vco < CLKCTR_PLLCLK_OD_MIN || vco > CLKCTR_PLLCLK_OD_MAX) {
        return CLKCTR_Status_ConfigureError;
    }

    /* Отключить PLL и установить источник частоты для MAINCLK. */
    CLKCTR_SetSwitchMainClk(base, CLKCTR_MainClkTypeHFIClk);
    CLKCTR_GetPLLConfig(base, &config);
    config.man = 0;
    config.sel = 0;
    CLKCTR_SetPLLConfig(base, config);

    /* Установить источник частоты для PLLCLK (HFI или XTI). */
    if (xti_hz == 0) {
        CLKCTR_SetSwitchPLLRef(base, CLKCTR_PLLRefTypeHFIClk);
    } else {
        CLKCTR_SetSwitchPLLRef(base, CLKCTR_PLLRefTypeXTIClk);
    }

    CLKCTR_GetPLLConfig(base, &config);
    config.man = 1;
    config.nr_man = nr - 1;
    config.nf_man = nf - 1;
    config.od_man = od - 1;
    config.sel = 1;
    CLKCTR_SetPLLConfig(base, config);
    CLKCTR_SetSwitchMainClk(base, CLKCTR_MainClkTypePLLClk);
    MAINCLK_FREQUENCY = vco / od;

    return CLKCTR_Status_Ok;
}

void CLKCTR_SetSysDiv(CLKCTR_Type * base, uint16_t fclk_div,
    uint16_t sysclk_div, uint16_t gnssclk_div, uint16_t qspiclk_div)
{
    if (sysclk_div > CLKCTR_MAX_SYSCLK_DIV) {
        sysclk_div = CLKCTR_MAX_SYSCLK_DIV;
    }
    CLKCTR_SetSysClkDiv(base, sysclk_div + 1);

    if (fclk_div > CLKCTR_MAX_FCLK_DIV) {
        fclk_div = CLKCTR_MAX_FCLK_DIV;
    }
    CLKCTR_SetFClkDiv(base, fclk_div + 1);

    if (gnssclk_div > CLKCTR_MAX_GNSSCLK_DIV) {
        gnssclk_div = CLKCTR_MAX_GNSSCLK_DIV;
    }
    CLKCTR_SetGNSSClkDiv(base, gnssclk_div + 1);

    if (qspiclk_div > CLKCTR_MAX_QSPICLK_DIV) {
        qspiclk_div = CLKCTR_MAX_QSPICLK_DIV;
    }
    CLKCTR_SetQSPIClkDiv(base, qspiclk_div + 1);
}

enum clkctr_status CLKCTR_SetXTI(CLKCTR_Type * base, uint32_t frequency)
{
    UNUSED(base);
    if ((frequency < CLKCTR_XTI_MIN) || (frequency > CLKCTR_XTI_MAX))
        return CLKCTR_Status_ConfigureError;

    xti = frequency;

    return CLKCTR_Status_Ok;
}

uint32_t CLKCTR_GetXTI(CLKCTR_Type * base)
{
    UNUSED(base);
    return xti;
}

enum clkctr_status CLKCTR_SetXTI32(CLKCTR_Type * base, uint32_t frequency)
{
    UNUSED(base);
    if ((frequency < CLKCTR_XTI32_MIN) || (frequency > CLKCTR_XTI32_MAX))
        return CLKCTR_Status_ConfigureError;

    xti32 = frequency;

    return CLKCTR_Status_Ok;
}

uint32_t CLKCTR_GetXTI32(CLKCTR_Type * base)
{
    UNUSED(base);
    return xti32;
}

enum clkctr_status CLKCTR_SetHFI(CLKCTR_Type * base, uint32_t frequency)
{
    UNUSED(base);
    if ((frequency < CLKCTR_HFI_MIN) || (frequency > CLKCTR_HFI_MAX))
        return CLKCTR_Status_ConfigureError;

    hfi = frequency;

    return CLKCTR_Status_Ok;
}

uint32_t CLKCTR_GetHFI(CLKCTR_Type * base)
{
    UNUSED(base);
    return hfi;
}

enum clkctr_status CLKCTR_SetLFI(CLKCTR_Type * base, uint32_t frequency)
{
    UNUSED(base);
    if ((frequency < CLKCTR_LFI_MIN) || (frequency > CLKCTR_LFI_MAX))
        return CLKCTR_Status_ConfigureError;

    lfi = frequency;

    return CLKCTR_Status_Ok;
}

uint32_t CLKCTR_GetLFI(CLKCTR_Type * base)
{
    UNUSED(base);
    return lfi;
}

enum clkctr_status CLKCTR_SetI2SExtClk(CLKCTR_Type * base, uint32_t frequency)
{
    UNUSED(base);
    if ((frequency < CLKCTR_I2S_EXTCLK_MIN)
        || (frequency > CLKCTR_I2S_EXTCLK_MAX))
    {
        return CLKCTR_Status_ConfigureError;
    }

    i2s_extclk = frequency;

    return CLKCTR_Status_Ok;
}

uint32_t CLKCTR_GetI2SExtClk(CLKCTR_Type * base)
{
    UNUSED(base);
    return i2s_extclk;
}

enum clkctr_status CLKCTR_GetAllDiv(CLKCTR_Type * base,
    struct clkctr_div *divisors)
{
    if (NULL == divisors)
        return CLKCTR_Status_InvalidArgument;

    divisors->clkctr_fclk_div    = CLKCTR_GetFClkDiv(base);
    divisors->clkctr_sysclk_div  = CLKCTR_GetSysClkDiv(base);
    divisors->clkctr_gnssclk_div = CLKCTR_GetGNSSClkDiv(base);
    divisors->clkctr_qspiclk_div = CLKCTR_GetQSPIClkDiv(base);
    divisors->clkctr_i2sclk_div  = CLKCTR_GetI2SClkDiv(base);
    divisors->clkctr_mco_div     = CLKCTR_GetMCODiv(base);

    return CLKCTR_Status_Ok;
}

uint32_t CLKCTR_GetXTIClk(CLKCTR_Type * base)
{
    return CLKCTR_GetXTI(base);
}

uint32_t CLKCTR_GetHFIClk(CLKCTR_Type * base)
{
    return (uint32_t) (CLKCTR_GetHFI(base) *
        (1.0 + delta_HFITRIM[CLKCTR_GetHFITrim(base)] / 100.0));
}

uint32_t CLKCTR_GetRTCClk(CLKCTR_Type * base)
{
    uint32_t rtc_div;             /* Делитель частоты RTCCLK */
    enum rwc_rtcclk_type rtc_src; /* Источник частоты RTCCLK */

    if (RWC_Status_Ok != RWC_GetRTCClkParam(&rtc_div, &rtc_src))
        return CLKCTR_FREQ_NOT_SET;

    switch (rtc_src) {
        case CLKCTR_RTCClkTypeLFI:
            /* Тут есть калибровка, но какая - не говорят, пока без нее. */
            return CLKCTR_GetLFI(base) / rtc_div;
            break;
        case CLKCTR_RTCClkTypeLFE:
            return CLKCTR_GetXTI32(base) / rtc_div;
            break;
        default:
            return CLKCTR_FREQ_NOT_SET;
    }
}

uint32_t CLKCTR_GetLPClk(CLKCTR_Type * base)
{
    switch (CLKCTR_GetSwitchLPClk(base)) {
        case CLKCTR_LPClkTypeRTCClk:
            return CLKCTR_GetRTCClk(base);
            break;
        case CLKCTR_LPClkType500:
            return CLKCTR_GetXTIClk(base) / 500;
            break;
        default:
            return CLKCTR_FREQ_NOT_SET;
    }
}

uint32_t CLKCTR_GetPLLClk(CLKCTR_Type * base)
{
    struct clkctr_pll_cfg config; /* Настройки PLL */

    CLKCTR_GetPLLConfig(base, &config);
    if (config.sel == 0)
        return CLKCTR_GetPLLRef(base);
    if (config.lock == 0)
        return CLKCTR_FREQ_NOT_SET;
    if (config.man) {
        /* Делители */
        uint32_t nf = config.nf_man + 1;
        uint32_t nr = config.nr_man + 1;
        uint32_t od = config.od_man + 1;
        double ref = CLKCTR_GetPLLRef(base);
        return ref * nf / (nr * od);
    } else {
        /* Предустановки */
        if (config.sel >= PLL_MAX_MULTIPLIER)
            return CLKCTR_GetPLLRef(base) * (PLL_MAX_MULTIPLIER + 1);
        else
            return CLKCTR_GetPLLRef(base) * (config.sel + 1);
    }
}

uint32_t CLKCTR_GetPLLRef(CLKCTR_Type * base)
{
    switch (CLKCTR_GetSwitchPLLRef(base)) {
        case CLKCTR_PLLRefTypeHFIClk:
            return CLKCTR_GetHFIClk(base);
            break;
        case CLKCTR_PLLRefTypeXTIClk:
            return CLKCTR_GetXTIClk(base);
            break;
        default:
            return CLKCTR_FREQ_NOT_SET;
    }
}

uint32_t CLKCTR_GetMainClk(CLKCTR_Type * base)
{
    switch (CLKCTR_GetSwitchMainClk(base)) {
        case CLKCTR_MainClkTypeHFIClk:
            return CLKCTR_GetHFIClkForMainClk(base);
            break;
        case CLKCTR_MainClkTypeXTIClk:
            return CLKCTR_GetXTIClk(base);
            break;
        case CLKCTR_MainClkTypePLLClk:
            return CLKCTR_GetPLLClk(base);
            break;
        default:
            return CLKCTR_FREQ_NOT_SET;
    }
}

uint32_t CLKCTR_GetUSBClk(CLKCTR_Type * base)
{
    switch (CLKCTR_GetSwitchUSBClk(base)) {
        case CLKCTR_USBClkTypeHFIClk:
            return CLKCTR_GetHFIClk(base);
            break;
        case CLKCTR_USBClkTypeXTIClk:
            return CLKCTR_GetXTIClk(base);
            break;
        default:
            return CLKCTR_FREQ_NOT_SET;
    }
}

uint32_t CLKCTR_GetFClkInt(CLKCTR_Type * base)
{
    return CLKCTR_GetMainClk(base) / CLKCTR_GetFClkDiv(base);
}

uint32_t CLKCTR_GetFClk(CLKCTR_Type * base)
{
    return CLKCTR_GetFClkInt(base);
}

uint32_t CLKCTR_GetSysClk(CLKCTR_Type * base)
{
    return CLKCTR_GetFClkInt(base) / CLKCTR_GetSysClkDiv(base);
}

uint32_t CLKCTR_GetGNSSClk(CLKCTR_Type * base)
{
    return CLKCTR_GetMainClk(base) / CLKCTR_GetGNSSClkDiv(base);
}

uint32_t CLKCTR_GetQSPIClk(CLKCTR_Type * base)
{
    return CLKCTR_GetMainClk(base) / CLKCTR_GetQSPIClkDiv(base);
}

uint32_t CLKCTR_GetI2SClk(CLKCTR_Type * base)
{
    uint32_t clock; /* Частота тактирования */

    switch (CLKCTR_GetSwitchI2SClk(base)) {
        case CLKCTR_I2SClkTypeSysClk:
            clock = CLKCTR_GetSysClk(base);
            break;
        case CLKCTR_I2SClkTypeI2SClk:
            clock = CLKCTR_GetI2SExtClk(base);
            break;
        default:
            return CLKCTR_FREQ_NOT_SET;
    }

    return clock / CLKCTR_GetI2SClkDiv(base);
}

uint32_t CLKCTR_GetMCOClk(CLKCTR_Type * base)
{
    uint32_t freq; /* Выходная частота */

    switch (CLKCTR_GetSwitchMCOClk(base)) {
        case CLKCTR_MCOClkTypeHFIClk:
            freq = CLKCTR_GetHFIClk(base);
            break;
        case CLKCTR_MCOClkTypeRTCClk:
            freq = CLKCTR_GetRTCClk(base);
            break;
        case CLKCTR_MCOClkTypeLPClk:
            freq = CLKCTR_GetLPClk(base);
            break;
        case CLKCTR_MCOClkTypeMainClk:
            freq = CLKCTR_GetMainClk(base);
            break;
        case CLKCTR_MCOClkTypePLLClk:
            freq = CLKCTR_GetPLLClk(base);
            break;
        case CLKCTR_MCOClkTypeSysClk:
            return CLKCTR_GetSysClk(base);
            break;
        case CLKCTR_MCOClkTypeFClkInt:
            freq = CLKCTR_GetFClkInt(base);
            break;
        case CLKCTR_MCOClkTypeFClk:
            freq = CLKCTR_GetFClk(base);
            break;
        default:
            return CLKCTR_FREQ_NOT_SET;
    }

    return freq / CLKCTR_GetMCODiv(base);
}

uint32_t CLKCTR_GetMCODiv(CLKCTR_Type * base)
{
    return ((base->MCODIV & CLKCTR_MCODIV_MCODIV_CUR_Msk)
         >> CLKCTR_MCODIV_MCODIV_CUR_Pos) + 1;
}

enum clkctr_status CLKCTR_SetMCODiv(CLKCTR_Type * base, uint32_t value)
{
    if ((value < (CLKCTR_MIN_MCOCLK_DIV + 1))
        || (value > (CLKCTR_MAX_MCOCLK_DIV + 1)))
    {
        return CLKCTR_Status_InvalidArgument;
    }

    if (0) /* Сделать! 2 Тест допустимости коэффициента */
        return CLKCTR_Status_ConfigureError;

    base->MCODIV = ((value - 1) << CLKCTR_MCODIV_MCODIV_Pos)
        & CLKCTR_MCODIV_MCODIV_Msk;
    while (CLKCTR_GetMCODiv(base) != value)
        ;

    return CLKCTR_Status_Ok;
}

enum clkctr_status CLKCTR_SetSwitchMCOClk(CLKCTR_Type * base, uint32_t value)
{
    uint32_t cfgReg = base->CFG; /* Регистр CGF блока CLKCTR */

    if (value & ~(CLKCTR_CFG_MCO_SEL_Msk >> CLKCTR_CFG_MCO_SEL_Pos))
        return CLKCTR_Status_InvalidArgument;

    if (0) /* Сделать! Тест допустимости переключения */
        return CLKCTR_Status_ConfigureError;

    cfgReg &= ~((CLKCTR_CFG_MCO_EN_Msk) | (CLKCTR_CFG_MCO_SEL_Msk));
    /* Включение тактового сигнала MCO и добавление его источника */
    cfgReg |= (1 << CLKCTR_CFG_MCO_EN_Pos)
        | (value << CLKCTR_CFG_MCO_SEL_Pos);
    base->CFG = cfgReg;

    return CLKCTR_Status_Ok;
}

uint32_t CLKCTR_GetSwitchMCOClk(CLKCTR_Type * base)
{
    return (base->CFG & CLKCTR_CFG_MCO_SEL_Msk)
        >> CLKCTR_CFG_MCO_SEL_Pos;
}

enum clkctr_status CLKCTR_SetSwitchPLLRef(CLKCTR_Type * base, uint32_t value)
{
    uint32_t cfgReg = base->CFG; /* Регистр CGF блока CLKCTR */

    if (value & ~(CLKCTR_CFG_PLLREF_SEL_Msk >> CLKCTR_CFG_PLLREF_SEL_Pos))
        return CLKCTR_Status_InvalidArgument;

    if (0) /* Сделать! Тест допустимости переключения */
        return CLKCTR_Status_ConfigureError;

    cfgReg &= ~(CLKCTR_CFG_PLLREF_SEL_Msk);
    cfgReg |= (value << CLKCTR_CFG_PLLREF_SEL_Pos);
    base->CFG = cfgReg;

    return CLKCTR_Status_Ok;
}

uint32_t CLKCTR_GetSwitchPLLRef(CLKCTR_Type * base)
{
    return (base->CFG & CLKCTR_CFG_PLLREF_SEL_Msk)
        >> CLKCTR_CFG_PLLREF_SEL_Pos;
}

enum clkctr_status CLKCTR_SetSwitchMainClk(CLKCTR_Type * base, uint32_t value)
{
    uint32_t cfgReg = base->CFG; /* Регистр CGF блока CLKCTR */

    if (value > CLKCTR_MainClkTypeMax)
        return CLKCTR_Status_InvalidArgument;

    if (0) /* Сделать! Тест допустимости переключения */
        return CLKCTR_Status_ConfigureError;

    cfgReg &= ~CLKCTR_CFG_MAINCLK_SEL_Msk;
    cfgReg |= (value << CLKCTR_CFG_MAINCLK_SEL_Pos);
    base->CFG = cfgReg;

    return CLKCTR_Status_Ok;
}

uint32_t CLKCTR_GetSwitchMainClk(CLKCTR_Type * base)
{
    return (base->CFG & CLKCTR_CFG_MAINCLK_SEL_Msk)
        >> CLKCTR_CFG_MAINCLK_SEL_Pos;
}

enum clkctr_status CLKCTR_SetSwitchRTCClk(CLKCTR_Type * base, uint32_t value)
{
    UNUSED(base);
    uint32_t rtc_div;             /* Делитель частоты RTCCLK */
    enum rwc_rtcclk_type rtc_src; /* Источник частоты RTCCLK */

    if ((value != CLKCTR_RTCClkTypeLFI) || (value != CLKCTR_RTCClkTypeLFE))
        return CLKCTR_Status_InvalidArgument;

    RWC_GetRTCClkParam(&rtc_div, &rtc_src);
    RWC_SetRTCClkParam(rtc_div, value);

    return CLKCTR_Status_Ok;
}

uint32_t CLKCTR_GetSwitchRTCClk(CLKCTR_Type * base)
{
    uint32_t rtc_div;             /* Делитель частоты RTCCLK */
    enum rwc_rtcclk_type rtc_src; /* Источник частоты RTCCLK */
    UNUSED(base);
    
    RWC_GetRTCClkParam(&rtc_div, &rtc_src);

    return rtc_src;
}

enum clkctr_status CLKCTR_SetSwitchLPClk(CLKCTR_Type * base, uint32_t value)
{
    return CLKCTR_SetSwitchPMUDIS(base, value);
}

uint32_t CLKCTR_GetSwitchLPClk(CLKCTR_Type * base)
{
    return CLKCTR_GetSwitchPMUDIS(base);
}

enum clkctr_status CLKCTR_SetSwitchUSBClk(CLKCTR_Type * base, uint32_t value)
{
    uint32_t cfgReg = base->CFG; /* Регистр CGF блока CLKCTR */

    if (value & ~(CLKCTR_CFG_USBCLK_SEL_Msk >> CLKCTR_CFG_USBCLK_SEL_Pos))
        return CLKCTR_Status_InvalidArgument;

    if (0) /* Сделать! Тест допустимости переключения */
        return CLKCTR_Status_ConfigureError;

    cfgReg &= ~CLKCTR_CFG_USBCLK_SEL_Msk;
    cfgReg |= (value << CLKCTR_CFG_USBCLK_SEL_Pos);
    base->CFG = cfgReg;

    return CLKCTR_Status_Ok;
}

uint32_t CLKCTR_GetSwitchUSBClk(CLKCTR_Type * base)
{
    return (base->CFG & CLKCTR_CFG_USBCLK_SEL_Msk)
        >> CLKCTR_CFG_USBCLK_SEL_Pos;
}

enum clkctr_status CLKCTR_SetSwitchI2SClk(CLKCTR_Type * base, uint32_t value)
{
    uint32_t cfgReg = base->CFG; /* Регистр CGF блока CLKCTR */

    if (value & ~(CLKCTR_CFG_I2SCLK_SEL_Msk >> CLKCTR_CFG_I2SCLK_SEL_Pos))
        return CLKCTR_Status_InvalidArgument;

    if (0) /* Сделать! 2 Тест допустимости переключения */
        return CLKCTR_Status_ConfigureError;

    cfgReg &= ~CLKCTR_CFG_I2SCLK_SEL_Msk;
    cfgReg |= (value << CLKCTR_CFG_I2SCLK_SEL_Pos);
    base->CFG = cfgReg;

    return CLKCTR_Status_Ok;
}

uint32_t CLKCTR_GetSwitchI2SClk(CLKCTR_Type * base)
{
    return (base->CFG & CLKCTR_CFG_I2SCLK_SEL_Msk)
        >> CLKCTR_CFG_I2SCLK_SEL_Pos;
}

enum clkctr_status CLKCTR_SetSwitchPMUDIS(CLKCTR_Type * base, uint32_t value)
{
    UNUSED(base);
    if (value)
        pmudis = 1;
    else
        pmudis = 0;

    return CLKCTR_Status_Ok;
}

uint32_t CLKCTR_GetSwitchPMUDIS(CLKCTR_Type * base)
{
    UNUSED(base);
    return pmudis;
}

uint32_t CLKCTR_GetFClkDiv(CLKCTR_Type * base)
{
    return ((base->FCLKDIV & CLKCTR_FCLKDIV_FCLKDIV_Msk)
        >> CLKCTR_FCLKDIV_FCLKDIV_Pos) + 1;
}

enum clkctr_status CLKCTR_SetFClkDiv(CLKCTR_Type * base, uint32_t value)
{
    if ((value < (CLKCTR_MIN_FCLK_DIV + 1))
        || (value > (CLKCTR_MAX_FCLK_DIV + 1)))
    {
        return CLKCTR_Status_InvalidArgument;
    }

    if (0) /* Сделать! 2 Тест допустимости коэффициента */
        return CLKCTR_Status_ConfigureError;

    base->FCLKDIV = ((value - 1) << CLKCTR_FCLKDIV_FCLKDIV_Pos)
        & CLKCTR_FCLKDIV_FCLKDIV_Msk;
    while (CLKCTR_GetFClkDiv(base) != value)
        ;

    return CLKCTR_Status_Ok;
}

uint32_t CLKCTR_GetSysClkDiv(CLKCTR_Type * base)
{
    return ((base->SYSCLKDIV & CLKCTR_SYSCLKDIV_STSCLKDIV_Msk)
        >> CLKCTR_SYSCLKDIV_STSCLKDIV_Pos) + 1;
}

enum clkctr_status CLKCTR_SetSysClkDiv(CLKCTR_Type * base, uint32_t value)
{
    if ((value < (CLKCTR_MIN_SYSCLK_DIV + 1))
        || (value > (CLKCTR_MAX_SYSCLK_DIV + 1)))
    {
        return CLKCTR_Status_InvalidArgument;
    }

    if (0) /* Сделать! 2 Тест допустимости коэффициента */
        return CLKCTR_Status_ConfigureError;

    base->SYSCLKDIV = ((value - 1)
        << CLKCTR_SYSCLKDIV_STSCLKDIV_Pos)
            & CLKCTR_SYSCLKDIV_STSCLKDIV_Msk;
    while (CLKCTR_GetSysClkDiv(base) != value)
        ;

    return CLKCTR_Status_Ok;
}

uint32_t CLKCTR_GetGNSSClkDiv(CLKCTR_Type * base)
{
    return ((base->GNSSCLKDIV & CLKCTR_GNSSCLKDIV_GNSSCLKDIV_Msk)
        >> CLKCTR_GNSSCLKDIV_GNSSCLKDIV_Pos) + 1;
}

enum clkctr_status CLKCTR_SetGNSSClkDiv(CLKCTR_Type * base, uint32_t value)
{
    if ((value < (CLKCTR_MIN_GNSSCLK_DIV + 1))
        || (value > (CLKCTR_MAX_GNSSCLK_DIV + 1)))
    {
        return CLKCTR_Status_InvalidArgument;
    }

    if (0) /* Сделать! 2 Тест допустимости коэффициента */
        return CLKCTR_Status_ConfigureError;

    base->GNSSCLKDIV = ((value - 1)
        << CLKCTR_GNSSCLKDIV_GNSSCLKDIV_Pos)
            & CLKCTR_GNSSCLKDIV_GNSSCLKDIV_Msk;
#ifdef CLKCTR_NO_GNNS_BUG_ELIOT1RTL_2
    while (CLKCTR_GetGNSSClkDiv(base) != value)
        ;
#endif

    return CLKCTR_Status_Ok;
}

uint32_t CLKCTR_GetQSPIClkDiv(CLKCTR_Type * base)
{
    return ((base->QSPICLKDIV & CLKCTR_QSPICLKDIV_QSPICLKDIV_Msk)
        >> CLKCTR_QSPICLKDIV_QSPICLKDIV_Pos) + 1;
}

enum clkctr_status CLKCTR_SetQSPIClkDiv(CLKCTR_Type * base, uint32_t value)
{
    if ((value < (CLKCTR_MIN_QSPICLK_DIV + 1))
        || (value > (CLKCTR_MAX_QSPICLK_DIV + 1)))
    {
        return CLKCTR_Status_InvalidArgument;
    }

    if (0) /* Сделать! 2 Тест допустимости коэффициента */
        return CLKCTR_Status_ConfigureError;

    base->QSPICLKDIV = ((value - 1)
        << CLKCTR_QSPICLKDIV_QSPICLKDIV_Pos)
            & CLKCTR_QSPICLKDIV_QSPICLKDIV_Msk;
    while (CLKCTR_GetQSPIClkDiv(base) != value)
        ;

    return CLKCTR_Status_Ok;
}

uint32_t CLKCTR_GetI2SClkDiv(CLKCTR_Type * base)
{
    return ((base->I2SCLKDIV & CLKCTR_I2SCLKDIV_I2SCLKDIV_Msk)
        >> CLKCTR_I2SCLKDIV_I2SCLKDIV_Pos) + 1;
}

enum clkctr_status CLKCTR_SetI2SClkDiv(CLKCTR_Type * base, uint32_t value)
{
    if ((value < (CLKCTR_MIN_I2SCLK_DIV + 1))
        || (value > (CLKCTR_MAX_I2SCLK_DIV + 1)))
    {
        return CLKCTR_Status_InvalidArgument;
    }

    if (0) /* Сделать! 2 Тест допустимости коэффициента */
        return CLKCTR_Status_ConfigureError;

    base->I2SCLKDIV = ((value - 1) << CLKCTR_I2SCLKDIV_I2SCLKDIV_Pos)
        & CLKCTR_I2SCLKDIV_I2SCLKDIV_Msk;
    while (CLKCTR_GetI2SClkDiv(base) != value)
        ;

    return CLKCTR_Status_Ok;
}

enum clkctr_status CLKCTR_GetPLLConfig(CLKCTR_Type * base,
    struct clkctr_pll_cfg *config)
{
    uint32_t reg = base->PLLCFG; /* Значение регистра PLLCFG */

    if (NULL == config)
        return CLKCTR_Status_InvalidArgument;

    config->lock   = (reg & CLKCTR_PLLCFG_LOCK_Msk)
        >> CLKCTR_PLLCFG_LOCK_Pos;
    config->nr_man = (reg & CLKCTR_PLLCFG_NR_MAN_Msk)
        >> CLKCTR_PLLCFG_NR_MAN_Pos;
    config->nf_man = (reg & CLKCTR_PLLCFG_NF_MAN_Msk)
        >> CLKCTR_PLLCFG_NF_MAN_Pos;
    config->od_man = (reg & CLKCTR_PLLCFG_OD_MAN_Msk)
        >> CLKCTR_PLLCFG_OD_MAN_Pos;
    config->man    = (reg & CLKCTR_PLLCFG_MAN_Msk)
        >> CLKCTR_PLLCFG_MAN_Pos;
    config->sel    = (reg & CLKCTR_PLLCFG_SEL_Msk)
        >> CLKCTR_PLLCFG_SEL_Pos;

     return CLKCTR_Status_Ok;
}

enum clkctr_status CLKCTR_SetPLLConfig(CLKCTR_Type * base,
    struct clkctr_pll_cfg config)
{
    uint32_t pll_cfg = CLKCTR_PLL_CFG_NR_MAN(config.nr_man) 
        | CLKCTR_PLL_CFG_NF_MAN(config.nf_man) 
        | CLKCTR_PLL_CFG_OD_MAN(config.od_man) 
        | CLKCTR_PLL_CFG_MAN(config.man) 
        | CLKCTR_PLL_CFG_SEL(config.sel); /* Записываемое значение регистра */

    if ((config.nr_man > CLKCTR_NR_MAN_MAX)
        || (config.nf_man > CLKCTR_NF_MAN_MAX)
        || (config.od_man > CLKCTR_OD_MAN_MAX)
        || (config.man > CLKCTR_MAN_MAX)
        || (config.sel > CLKCTR_SEL_MAX))
    {
        return CLKCTR_Status_InvalidArgument;
    }

    if (0) /* сделать проверку корректности установки */
        return CLKCTR_Status_ConfigureError;

    if (config.sel != 0) {
        /* Остановили PLL. */
        base->PLLCFG = 0;
        /* Запускаем PLL. */
        base->PLLCFG = pll_cfg;
        while ((base->PLLCFG & CLKCTR_PLLCFG_LOCK_Msk) == 0)
            ;
    } else {
        base->PLLCFG = pll_cfg;
    }

    return CLKCTR_Status_Ok;
}

enum clkctr_status CLKCTR_SetHFITrim(CLKCTR_Type * base, uint32_t index)
{
    if (index & ~(CLKCTR_HFITRIM_TRIM_Msk >> CLKCTR_HFITRIM_TRIM_Pos))
        return CLKCTR_Status_InvalidArgument;

    if (0) /* Сделать! 2 Тест допустимости коэффициента*/
        return CLKCTR_Status_ConfigureError;

    base->HFITRIM = (index << CLKCTR_HFITRIM_TRIM_Pos)
        & CLKCTR_HFITRIM_TRIM_Msk;
    while (CLKCTR_GetHFITrim(base) != index)
        ;

    return CLKCTR_Status_Ok;
}

uint32_t CLKCTR_GetHFITrim(CLKCTR_Type * base)
{
    return (base->HFITRIM & CLKCTR_HFITRIM_TRIM_Msk)
        >> CLKCTR_HFITRIM_TRIM_Pos;
}

enum clkctr_status CLKCTR_SetFrequency(CLKCTR_Type * base,
    enum clkctr_extern_freq input, uint32_t frequency)
{
    enum clkctr_status result; /* Результат выполнения функции */

    switch (input) {
        case CLKCTR_ExternFreqXTI:
            result = CLKCTR_SetXTI(base, frequency);
            break;
        case CLKCTR_ExternFreqXTI32:
            result = CLKCTR_SetXTI32(base, frequency);
            break;
        case CLKCTR_ExternFreqHFI:
            result = CLKCTR_SetHFI(base, frequency);
            break;
        case CLKCTR_ExternFreqLFI:
            result = CLKCTR_SetLFI(base, frequency);
            break;
        case CLKCTR_ExternFreqI2SExtClk:
            result = CLKCTR_SetI2SExtClk(base, frequency);
            break;
        default:
            result = CLKCTR_Status_InvalidArgument;
    }

    return result;
}

uint32_t CLKCTR_GetFrequency(CLKCTR_Type * base, enum clkctr_extern_freq input)
{
    uint32_t result; /* Полученная частота */
    
    switch (input) {
        case CLKCTR_ExternFreqXTI:
            result = CLKCTR_GetXTI(base);
            break;
        case CLKCTR_ExternFreqXTI32:
            result = CLKCTR_GetXTI32(base);
            break;
        case CLKCTR_ExternFreqHFI:
            result = CLKCTR_GetHFI(base);
            break;
        case CLKCTR_ExternFreqLFI:
            result = CLKCTR_GetLFI(base);
            break;
        case CLKCTR_ExternFreqI2SExtClk:
            result = CLKCTR_GetI2SExtClk(base);
            break;
        default:
            result = CLKCTR_FREQ_NOT_SET;
    }

    return result;
}

enum clkctr_status CLKCTR_SetSwitchClk(CLKCTR_Type * base,
    enum clkctr_int_freq clk, uint32_t value)
{
    enum clkctr_status result; /* Результат выполнения функции */

    switch (clk) {
        case CLKCTR_IntFreqRTCClk:
            result = CLKCTR_SetSwitchRTCClk(base, value);
            break;
        case CLKCTR_IntFreqLPClk:
            result = CLKCTR_SetSwitchLPClk(base, value);
            break;
        case CLKCTR_IntFreqPLLRefClk:
            result = CLKCTR_SetSwitchPLLRef(base, value);
            break;
        case CLKCTR_IntFreqMainClk:
            result = CLKCTR_SetSwitchMainClk(base, value);
            break;
        case CLKCTR_IntFreqUSBClk:
            result = CLKCTR_SetSwitchUSBClk(base, value);
            break;
        case CLKCTR_IntFreqI2SClk:
            result = CLKCTR_SetSwitchI2SClk(base, value);
            break;
        case CLKCTR_IntFreqMCOClk:
            result = CLKCTR_SetSwitchMCOClk(base, value);
            break;
        case CLKCTR_IntFreqHFIForMain:
            result = CLKCTR_SetSwitchPMUDIS(base, value);
            break;
        default:
            result = CLKCTR_Status_InvalidArgument;
    }

    return result;
}

enum clkctr_status CLKCTR_GetSwitchClk(CLKCTR_Type * base,
    enum clkctr_int_freq clk, uint32_t *value)
{
    if (value == NULL)
        return CLKCTR_Status_InvalidArgument;

    switch (clk) {
        case CLKCTR_IntFreqRTCClk:
            *value = CLKCTR_GetSwitchRTCClk(base);
            break;
        case CLKCTR_IntFreqLPClk:
            *value = CLKCTR_GetSwitchLPClk(base);
            break;
        case CLKCTR_IntFreqPLLRefClk:
            *value = CLKCTR_GetSwitchPLLRef(base);
            break;
        case CLKCTR_IntFreqMainClk:
            *value = CLKCTR_GetSwitchMainClk(base);
            break;
        case CLKCTR_IntFreqUSBClk:
            *value = CLKCTR_GetSwitchUSBClk(base);
            break;
        case CLKCTR_IntFreqI2SClk:
            *value = CLKCTR_GetSwitchI2SClk(base);
            break;
        case CLKCTR_IntFreqMCOClk:
            *value = CLKCTR_GetSwitchMCOClk(base);
            break;
        case CLKCTR_IntFreqHFIForMain:
            *value = CLKCTR_GetSwitchPMUDIS(base);
            break;
        default:
            return CLKCTR_Status_InvalidArgument;
    }

    return CLKCTR_Status_Ok;
}

enum clkctr_status CLKCTR_SetDivClk(CLKCTR_Type * base,
    enum clkctr_int_freq clk, uint32_t value)
{
    enum clkctr_status result; /* Результат выполнения функции */

    switch (clk) {
        case CLKCTR_IntFreqFClkInt:
        case CLKCTR_IntFreqFClk:
            result = CLKCTR_SetFClkDiv(base, value);
            break;
        case CLKCTR_IntFreqSysClk:
            result = CLKCTR_SetSysClkDiv(base, value);
            break;
        case CLKCTR_IntFreqGNSSClk:
            result = CLKCTR_SetGNSSClkDiv(base, value);
            break;
        case CLKCTR_IntFreqQSPIClk:
            result = CLKCTR_SetQSPIClkDiv(base, value);
            break;
        case CLKCTR_IntFreqI2SClk:
            result = CLKCTR_SetI2SClkDiv(base, value);
            break;
        case CLKCTR_IntFreqMCOClk:
            result = CLKCTR_SetMCODiv(base, value);
            break;
        default :
            return CLKCTR_Status_InvalidArgument;
    }

    return result;
}

enum clkctr_status CLKCTR_GetDivClk(CLKCTR_Type * base,
    enum clkctr_int_freq clk, uint32_t *value)
{
    if (value == NULL)
        return CLKCTR_Status_InvalidArgument;

    switch (clk) {
        case CLKCTR_IntFreqFClkInt:
        case CLKCTR_IntFreqFClk:
            *value = CLKCTR_GetFClkDiv(base);
            break;
        case CLKCTR_IntFreqSysClk:
            *value = CLKCTR_GetSysClkDiv(base);
            break;
        case CLKCTR_IntFreqGNSSClk:
            *value = CLKCTR_GetGNSSClkDiv(base);
            break;
        case CLKCTR_IntFreqQSPIClk:
            *value = CLKCTR_GetQSPIClkDiv(base);
            break;
        case CLKCTR_IntFreqI2SClk:
            *value = CLKCTR_GetI2SClkDiv(base);
            break;
        case CLKCTR_IntFreqMCOClk:
            *value = CLKCTR_GetMCODiv(base);
            break;
        default:
            return CLKCTR_Status_InvalidArgument;
    }

    return CLKCTR_Status_Ok;
}

enum clkctr_status CLKCTR_GetClk(CLKCTR_Type * base,
    enum clkctr_int_freq clk, uint32_t *value)
{
    if (value == NULL)
        return CLKCTR_Status_InvalidArgument;

    switch (clk) {
        case CLKCTR_IntFreqXTIClk:
            *value = CLKCTR_GetXTIClk(base);
            break;
        case CLKCTR_IntFreqHFIClk:
            *value = CLKCTR_GetHFIClk(base);
            break;
        case CLKCTR_IntFreqRTCClk:
            *value = CLKCTR_GetRTCClk(base);
            break;
        case CLKCTR_IntFreqLPClk:
            *value = CLKCTR_GetLPClk(base);
            break;
        case CLKCTR_IntFreqPLLClk:
            *value = CLKCTR_GetPLLClk(base);
            break;
        case CLKCTR_IntFreqPLLRefClk:
            *value = CLKCTR_GetPLLRef(base);
            break;
        case CLKCTR_IntFreqMainClk:
            *value = CLKCTR_GetMainClk(base);
            break;
        case CLKCTR_IntFreqUSBClk:
            *value = CLKCTR_GetUSBClk(base);
            break;
        case CLKCTR_IntFreqFClkInt:
            *value = CLKCTR_GetFClkInt(base);
            break;
        case CLKCTR_IntFreqFClk:
            *value = CLKCTR_GetFClk(base);
            break;
        case CLKCTR_IntFreqSysClk:
            *value = CLKCTR_GetSysClk(base);
            break;
        case CLKCTR_IntFreqGNSSClk:
            *value = CLKCTR_GetGNSSClk(base);
            break;
        case CLKCTR_IntFreqQSPIClk:
            *value = CLKCTR_GetQSPIClk(base);
            break;
        case CLKCTR_IntFreqI2SClk:
            *value = CLKCTR_GetI2SClk(base);
            break;
        case CLKCTR_IntFreqMCOClk:
            *value = CLKCTR_GetMCOClk(base);
            break;
        case CLKCTR_IntFreqHFIForMain:
            *value = CLKCTR_GetHFIClkForMainClk(base);
            break;
        default:
            return CLKCTR_Status_InvalidArgument;
    }

    return CLKCTR_Status_Ok;
}

uint32_t CLKCTR_GetHFIClkForMainClk(CLKCTR_Type * base)
{
    if (CLKCTR_GetSwitchPMUDIS(base))
        return CLKCTR_GetXTIClk(base);
    else
        return CLKCTR_GetHFIClk(base);
}

uint32_t CLKCTR_GetMCOEn(CLKCTR_Type * base)
{
    return ((base->CFG & CLKCTR_CFG_MCO_EN_Msk) ? 1 : 0);
}

enum clkctr_status CLKCTR_SetMCOEn(CLKCTR_Type * base, uint32_t enable)
{
    if (enable)
        base->CFG |= CLKCTR_CFG_MCO_EN_Msk;
    else
        base->CFG &= ~CLKCTR_CFG_MCO_EN_Msk;

    return CLKCTR_Status_Ok;
}

enum clkctr_status CLKCTR_GetClkForce(CLKCTR_Type * base,
    enum clkctr_device_clk_force device,
    enum clkctr_clk_force_type *clk_type)
{
    if ((device < CLKCTR_SysSysClkForce)
        || (device > CLKCTR_GMSSysClkForce)
        || (clk_type == NULL))
    {
        return CLKCTR_Status_InvalidArgument;
    }

    if (base->CLKFORCE & (CLKCTR_FORCE_CLK_DEV_EN << device))
        *clk_type = CLKCTR_ClkForceTypeForce;
    else
        *clk_type = CLKCTR_ClkForceTypeDynamic;

    return CLKCTR_Status_Ok;
}

enum clkctr_status CLKCTR_SetClkForce(CLKCTR_Type * base,
    enum clkctr_device_clk_force device,
    enum clkctr_clk_force_type clk_type)
{
    if ((device < CLKCTR_SysSysClkForce)
        || (device > CLKCTR_GMSSysClkForce))
    {
        return CLKCTR_Status_InvalidArgument;
    }

    if (clk_type == CLKCTR_ClkForceTypeForce) {
        base->CLKFORCE |= CLKCTR_FORCE_CLK_DEV_EN << device;
        return CLKCTR_Status_Ok;
    } else if (clk_type == CLKCTR_ClkForceTypeDynamic) {
        base->CLKFORCE &= ~(CLKCTR_FORCE_CLK_DEV_EN << device);
        return CLKCTR_Status_Ok;
    } else {
        return CLKCTR_Status_InvalidArgument;
    }
}

/*!
 * @}
 */
