/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




#include "eliot1_board.h"
#include "hal_spi.h"
#include "ad7124.h"
#include "spi_adc.h"

#define AD7124_SPI_MASTER          SPI1_Secure /* Указатель на базовый адрес */
#define AD7124_SPI_MASTER_CLK_FREQ 5000000UL   /* Частота интефейса SPI в Hz */

#define AD7124_CHANNEL_COUNT 6 /* Кол-во используемых каналов */

int main(void)
{
    enum spi_status status = SPI_Status_Ok;
    uint32_t buffer[6];
    uint32_t value = 0;
    double voltage = 0;
    uint32_t time_before = 0;
    uint32_t time_after = 0;
    int i;

    /* Инициализация платы. */
    BOARD_InitAll();

    /* Инициализация spi. */
    status = SPI_AdcInit(AD7124_SPI_MASTER, AD7124_SPI_MASTER_CLK_FREQ);
    if (status != SPI_Status_Ok) {
        printf("SPI Initialization Error\r\n");
        return EXIT_FAILURE;
    }

    /* Сброс значений регистров AD7124. */
    AD7124_Reset(AD7124_SPI_MASTER);
    BOARD_Udelay(100000);

    /* Проверка работоспособности AD7124. */
    if (AD7124_CheckWorking(AD7124_SPI_MASTER) == false) {
        printf("Connection to AD7124 is not established\r\n");
        return EXIT_FAILURE;
    }

    while (1) {
        AD7124_Configuration6ChannelsFullSpeed(AD7124_SPI_MASTER);

        /* После сброса АЦП пропускаем один цикл оцифровки. */
        for (i = 0; i < AD7124_CHANNEL_COUNT; i++) {
            GPIO_PinWrite(GPIO_SPI1_SS, 0); 
            BOARD_Udelay(3);
            value = AD7124_GetDataFast(AD7124_SPI_MASTER);
            GPIO_PinWrite(GPIO_SPI1_SS, 1); 
        }

        /* Засекам момент начала оцифровки. */
        time_before = BOARD_GetTime();

        /* Оцифровываем шесть каналов и записываем их в буфер. */
        for (i = 0; i < AD7124_CHANNEL_COUNT; i++) {
            GPIO_PinWrite(GPIO_SPI1_SS, 0); 
            BOARD_Udelay(3);
            value = AD7124_GetDataFast(AD7124_SPI_MASTER);
            GPIO_PinWrite(GPIO_SPI1_SS, 1); 
            
            buffer[i] = value;             
        }

        /* Засекам момент окончания оцифровки. */
        time_after = BOARD_GetTime();

        /* Преобразуем "сырые" данные из АЦП в вольты и выводим на output. */
        for (i = 0; i < AD7124_CHANNEL_COUNT; i++) {
            voltage = AD7124_ToVoltage(buffer[i], (AD7124_GAIN_1 + 1), 
                    AD7124_INTERNAL_VOLTAGE_REFERENCE, true);
            printf("voltage = %.10lfV raw_data = %ld\r\n", voltage, buffer[i]);
        }

        /* Итоговое время оцифровки 6-ти каналов. */
        printf("%.3lfms\r\n", ((time_after - time_before) / 1000.0));
        printf("--------------------------------------------\r\n");

        /* Сброс значений регистров AD7124. */
        AD7124_Reset(AD7124_SPI_MASTER);
    }

    return 0;
}
