/**
 * Copyright (c) 2021-2025, RnD Center «ELVEES», JSC
 * All rights reserved.
 * Contacts: https://elvees.ru, support@elvees.com
 *
 * Project:		SDK
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 *
 * Разрешается повторное распространение и использование как в виде исходного кода, так и в объектном коде, 
 * с изменениями или без, при соблюдении следующих условий:
 * 
 * 1. При повторном распространении исходного кода должно оставаться указанное выше уведомление об авторском праве, 
 * этот список условий и последующий отказ от гарантий.
 * 2. При повторном распространении двоичного кода должна сохраняться указанная выше информация об авторском праве, 
 * этот список условий и последующий отказ от гарантий в документации и/или в других материалах, поставляемых при 
 * распространении.
 * 3. Ни название организации, ни имена её сотрудников не могут быть использованы в качестве поддержки или 
 * продвижения продуктов, основанных на этом ПО без предварительного письменного разрешения.
 * ЭТА ПРОГРАММА ПРЕДОСТАВЛЕНА ВЛАДЕЛЬЦАМИ АВТОРСКИХ ПРАВ И/ИЛИ ДРУГИМИ СТОРОНАМИ «КАК ОНА ЕСТЬ» 
 * БЕЗ КАКОГО-ЛИБО ВИДА ГАРАНТИЙ, ВЫРАЖЕННЫХ ЯВНО ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ИМИ, 
 * ПОДРАЗУМЕВАЕМЫЕ ГАРАНТИИ КОММЕРЧЕСКОЙ ЦЕННОСТИ И ПРИГОДНОСТИ ДЛЯ КОНКРЕТНОЙ ЦЕЛИ. НИ В КОЕМ СЛУЧАЕ 
 * НИ ОДИН ВЛАДЕЛЕЦ АВТОРСКИХ ПРАВ И НИ ОДНО ДРУГОЕ ЛИЦО, КОТОРОЕ МОЖЕТ ИЗМЕНЯТЬ И/ИЛИ ПОВТОРНО 
 * РАСПРОСТРАНЯТЬ ПРОГРАММУ, КАК БЫЛО СКАЗАНО ВЫШЕ, НЕ НЕСЁТ ОТВЕТСТВЕННОСТИ, ВКЛЮЧАЯ ЛЮБЫЕ ОБЩИЕ, 
 * СЛУЧАЙНЫЕ, СПЕЦИАЛЬНЫЕ ИЛИ ПОСЛЕДОВАВШИЕ УБЫТКИ, ВСЛЕДСТВИЕ ИСПОЛЬЗОВАНИЯ ИЛИ НЕВОЗМОЖНОСТИ ИСПОЛЬЗОВАНИЯ ПРОГРАММЫ 
 * (ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ПОТЕРЕЙ ДАННЫХ, ИЛИ ДАННЫМИ, СТАВШИМИ НЕПРАВИЛЬНЫМИ, ИЛИ ПОТЕРЯМИ, 
 * ПРИНЕСЕННЫМИ ИЗ-ЗА ВАС ИЛИ ТРЕТЬИХ ЛИЦ, ИЛИ ОТКАЗОМ ПРОГРАММЫ РАБОТАТЬ СОВМЕСТНО С ДРУГИМИ ПРОГРАММАМИ), 
 * ДАЖЕ ЕСЛИ ТАКОЙ ВЛАДЕЛЕЦ ИЛИ ДРУГОЕ ЛИЦО БЫЛИ ИЗВЕЩЕНЫ О ВОЗМОЖНОСТИ ТАКИХ УБЫТКОВ.
 *
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided 
 * that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions 
 * and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse 
 * or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */




#include "eliot1_board.h"
#include "ad7124.h"
#include "hal_spi.h"
#include "spi_adc.h"

enum ad7124_status AD7124_SetConfigure(ad7124_setups setup,
    ad7124_ref_sel ref, ad7124_gain gain, bool bipolar,
    ad7124_burnout_current burnout, SPI_Type *base)
{
    uint32_t result        = 0;
    uint8_t src_buffer[3]  = {0};
    uint8_t dest_buffer[3] = {0};
    enum spi_status status = SPI_Status_Ok;

    setup += AD7124_REG_CONFIG_0;

    result = AD7124_CFG_REG_REF_SEL(ref)
        | AD7124_CFG_REG_PGA(gain)
        | (bipolar ? AD7124_CFG_REG_BIPOLAR : 0)
        | AD7124_CFG_REG_BURNOUT(burnout)
        | AD7124_CFG_REG_REF_BUFP
        | AD7124_CFG_REG_REF_BUFM
        | AD7124_CFG_REG_AIN_BUFP
        | AD7124_CFG_REG_AINN_BUFM;

    src_buffer[0] = WR_REGISTER(setup);
    src_buffer[1] = ((result & 0xFF00) >> 8);
    src_buffer[2] = (result & 0x00FF);

    BOARD_Udelay(100);
    status = SPI_DuplexTransfer(base, src_buffer, dest_buffer,
            DIM(src_buffer));
    if (status != SPI_Status_Ok)
        return AD7124_STATUS_FAIL;

    return AD7124_STATUS_OK;
}

enum ad7124_status AD7124_SetFilter(ad7124_filters filter,
    ad7124_filters_type filter_type, ad7124_post_filter_type post_filter,
    uint16_t fs, bool rej60, bool single_cycle, SPI_Type *base)
{
    uint32_t result        = 0;
    uint8_t src_buffer[4]  = {0};
    uint8_t dest_buffer[4] = {0};
    enum spi_status status = SPI_Status_Ok;

    filter += AD7124_REG_FILTER_0;

    result = AD7124_FILT_REG_FILTER((uint32_t) filter_type)
        | AD7124_FILT_REG_POST_FILTER((uint32_t) post_filter)
        | AD7124_FILT_REG_FS(fs)
        | (rej60 ? AD7124_FILT_REG_REJ60 : 0)
        | (single_cycle ? AD7124_FILT_REG_SINGLE_CYCLE : 0);

    src_buffer[0] = WR_REGISTER(filter);
    src_buffer[1] = ((result & 0xFF0000) >> 16);
    src_buffer[2] = ((result & 0x00FF00) >> 8);
    src_buffer[3] = (result & 0x0000FF);

    BOARD_Udelay(100);
    status = SPI_DuplexTransfer(base, src_buffer, dest_buffer,
            DIM(src_buffer));
    if (status != SPI_Status_Ok)
        return AD7124_STATUS_FAIL;

    return AD7124_STATUS_OK;
}

enum ad7124_status AD7124_SetChannel(ad7124_channel channel,
    ad7124_setups setup, ad7124_input_sel ainp,
    ad7124_input_sel ainm, bool enable, SPI_Type *base)
{
    uint32_t result        = 0;
    uint8_t src_buffer[3]  = {0};
    uint8_t dest_buffer[3] = {0};
    enum spi_status status = SPI_Status_Ok;

    channel += AD7124_REG_CHANNEL_0;

    result = AD7124_CH_MAP_REG_SETUP(setup)
        | AD7124_CH_MAP_REG_AINP(ainp)
        | AD7124_CH_MAP_REG_AINM(ainm)
        | (enable ? AD7124_CH_MAP_REG_CH_ENABLE : 0);

    src_buffer[0] = WR_REGISTER(channel);
    src_buffer[1] = ((result & 0xFF00) >> 8);
    src_buffer[2] = (result & 0x00FF);

    BOARD_Udelay(100);
    status = SPI_DuplexTransfer(base, src_buffer, dest_buffer,
            DIM(dest_buffer));
    if (status != SPI_Status_Ok)
        return AD7124_STATUS_FAIL;

    return AD7124_STATUS_OK;
}

enum ad7124_status AD7124_SetControl(ad7124_operating_mode mode,
    ad7124_power_mode power, bool ref_en, bool cont_read,
    bool data_status, ad7124_clk_sourse clk_sel, SPI_Type *base)
{
    uint32_t result        = 0;
    uint8_t src_buffer[3]  = {0};
    uint8_t dest_buffer[3] = {0};
    enum spi_status status = SPI_Status_Ok;

    result = AD7124_ADC_CTRL_REG_MODE(mode)
        | (cont_read ? AD7124_ADC_CTRL_REG_CONT_READ : 0)
        | (data_status ? AD7124_ADC_CTRL_REG_DATA_STATUS : 0) 
        | AD7124_ADC_CTRL_REG_POWER_MODE(power)
        | AD7124_ADC_CTRL_REG_CLK_SEL(clk_sel)
        | (ref_en ? AD7124_ADC_CTRL_REG_REF_EN : 0)
        | AD7124_ADC_CTRL_REG_DOUT_RDY_DEL;

    src_buffer[0] = WR_REGISTER(AD7124_REG_ADC_CONTROL);
    src_buffer[1] = ((result & 0xFF00) >> 8);
    src_buffer[2] = (result & 0x00FF);

    BOARD_Udelay(100);
    status = SPI_DuplexTransfer(base, src_buffer, dest_buffer,
            DIM(dest_buffer));
    if (status != SPI_Status_Ok)
        return AD7124_STATUS_FAIL;

    return AD7124_STATUS_OK;
}

enum ad7124_status AD7124_EnableChannel(ad7124_channel channel, 
    bool enable, SPI_Type *base)
{
    uint32_t result        = 0;
    uint8_t src_buffer[3]  = {0};
    uint8_t dest_buffer[3] = {0};
    enum spi_status status = SPI_Status_Ok;

    channel += AD7124_REG_CHANNEL_0;

    if (enable) {
        result = AD7124_CH_MAP_REG_CH_ENABLE;

        src_buffer[0] = WR_REGISTER(channel);
        src_buffer[1] = ((result & 0xFF00) >> 8);
        src_buffer[2] = (result & 0x00FF);

        BOARD_Udelay(100);
        status = SPI_DuplexTransfer(base, src_buffer,
                dest_buffer, DIM(dest_buffer));
        if (status != SPI_Status_Ok) {
            return AD7124_STATUS_FAIL;
        }
    } else {
        result = AD7124_CH_MAP_REG_CH_ENABLE;

        src_buffer[0] = WR_REGISTER(channel);
        src_buffer[1] = ((result & 0xFF00) >> 8);
        src_buffer[2] = (result & 0x00FF);

        status = SPI_DuplexTransfer(base, src_buffer,
                dest_buffer, DIM(dest_buffer));
        if (status != SPI_Status_Ok) {
            return AD7124_STATUS_FAIL;
        }
    }

    return AD7124_STATUS_OK;
}

enum ad7124_status AD7124_SetMode(ad7124_operating_mode mode, SPI_Type *base)
{
    uint32_t result        = 0;
    uint8_t src_buffer[3]  = {0};
    uint8_t dest_buffer[3] = {0};
    enum spi_status status = SPI_Status_Ok;

    result = AD7124_ADC_CTRL_REG_MODE(mode);

    src_buffer[0] = WR_REGISTER(AD7124_REG_ADC_CONTROL);
    src_buffer[1] = ((result & 0xFF00) >> 8);
    src_buffer[2] = (result & 0x00FF);

    BOARD_Udelay(100);
    status = SPI_DuplexTransfer(base, src_buffer, dest_buffer,
            DIM(dest_buffer));
    if (status != SPI_Status_Ok)
        return AD7124_STATUS_FAIL;

    return AD7124_STATUS_OK;
}

int32_t AD7124_GetData(SPI_Type *base)
{
    int32_t result         = 0;
    uint8_t src_buffer[1]  = {0};
    uint8_t dest_buffer[3] = {0};

    AD7124_CheckStatusRegister(AD7124_STATUS_REG_RDY, base);

    src_buffer[0] = RD_REGISTER(AD7124_REG_DATA);
    SPI_HalfDuplexTransfer(base,
        src_buffer, dest_buffer, DIM(dest_buffer),
        DIM(src_buffer), true);

    result |= (dest_buffer[0] << 16)
        | (dest_buffer[1] << 8)
        | (dest_buffer[2]);

    return result;
}

int32_t AD7124_GetDataFast(SPI_Type *base)
{
    int32_t result         = 0;
    uint8_t src_buffer[1]  = {0};
    uint8_t dest_buffer[3] = {0};

    AD7124_CheckRDY();

    src_buffer[0] = RD_REGISTER(AD7124_REG_DATA);
    SPI_HalfDuplexTransfer(base,
        src_buffer, dest_buffer, DIM(dest_buffer),
        DIM(src_buffer), true);

    result |= (dest_buffer[0] << 16)
        | (dest_buffer[1] << 8)
        | (dest_buffer[2]);

    return result;
}

void AD7124_CheckStatusRegister(int32_t rdy_field, SPI_Type *base)
{
    bool ready;
    uint8_t src_buffer[1]  = {0};
    uint8_t dest_buffer[1] = {0};

    src_buffer[0] = RD_REGISTER(AD7124_REG_STATUS);

    do {
        SPI_HalfDuplexTransfer(base, src_buffer,
            dest_buffer, DIM(dest_buffer), DIM(src_buffer), true);

        /* Проверка RDY бит из регистра статуса. */
        ready = ((dest_buffer[0] & rdy_field) == 0);
    } while (!ready);
}

void AD7124_CheckRDY(void)
{
    GPIO_PinMode_GPIO(GPIO_SPI1_MISO, GPIO_DigitalInput);

    while (GPIO_PinRead(GPIO_SPI1_MISO) == 1) {
        BOARD_Udelay(1);
    }

    while (GPIO_PinRead(GPIO_SPI1_MISO) == 0) {
        BOARD_Udelay(1);
    }

    GPIO_PinMode_Function(GPIO_SPI1_MISO, GPIO_ALT_FUNC_SPI0_SPI1);
}

void AD7124_Reset(SPI_Type *base)
{
    uint32_t i;
    uint8_t src_buffer[8];
    uint8_t dest_buffer[8];

    for (i = 0; i < DIM(src_buffer); i++) {
        src_buffer[i] = 0xFF;
    }

    BOARD_Udelay(100);
    SPI_DuplexTransfer(base, src_buffer, dest_buffer,
        DIM(dest_buffer));
}

uint8_t AD7124_ReadIdRegister(SPI_Type *base)
{
    uint8_t src_buffer[1]  = {0};
    uint8_t dest_buffer[1] = {0};

    src_buffer[0] = RD_REGISTER(AD7124_REG_ID);

    BOARD_Udelay(100);
    SPI_HalfDuplexTransfer(base, src_buffer, dest_buffer,
        DIM(dest_buffer), DIM(src_buffer), true);

    return dest_buffer[0];
}

double AD7124_ToVoltage(int32_t value, uint8_t gain,
    float vref, bool bipolar)
{
    double voltage = (double)value;

    if (bipolar) {
        voltage = voltage / (double) AD7124_MULTIPLIER_BIPOLAR - 1;
    } else {
        voltage = voltage / (double) AD7124_MULTIPLIER_UNIPOLAR;
    }

    voltage = voltage * vref / (double) gain;

    return voltage * 2;
}

bool AD7124_CheckWorking(SPI_Type *base)
{
    uint8_t id;
    bool result;

    id = AD7124_ReadIdRegister(base);

    switch (id) {
        case AD7124_4_ID:
        case AD7124_4B_ID:
        case AD7124_4E_ID:
            result = true;
            break;
        default:
            result = false;
            break;
    }

    return result;
}

uint32_t AD7124_ReadRegister(uint8_t reg, SPI_Type *base)
{
    int32_t result         = 0;
    uint8_t src_buffer[1]  = {0};
    uint8_t dest_buffer[3] = {0};

    src_buffer[0] = RD_REGISTER(reg);
    SPI_HalfDuplexTransfer(base, src_buffer,
            dest_buffer, DIM(dest_buffer),
            DIM(src_buffer), true);

    result |= (dest_buffer[0] << 16)
        | (dest_buffer[1] << 8)
        | (dest_buffer[2]);

    return result;
}

void AD7124_Configuration6ChannelsFullSpeed(SPI_Type *base)
{
    enum ad7124_status ad7124_answer;

    /* Настройка конфигурации */
    ad7124_answer = AD7124_SetConfigure(AD7124_SETUP_0,
        AD7124_REFERENCE_INTERNAL, AD7124_GAIN_1,
        true, AD7124_BURNOUT_CURRENT_OFF, base);
    if (ad7124_answer != AD7124_STATUS_OK) {
        printf("Configuration set error\r\n");
        exit(EXIT_FAILURE);
    }

    /* Настройка канала 0 */
    ad7124_answer = AD7124_SetChannel(AD7124_CHANNEL_0, AD7124_SETUP_0,
        AD7124_AIN0_INPUT, AD7124_DGND, true, base);
    if (ad7124_answer != AD7124_STATUS_OK) {
        printf("Channel_0 set error\r\n");
        exit(EXIT_FAILURE);
    }

    /* Настройка канала 1 */
    ad7124_answer = AD7124_SetChannel(AD7124_CHANNEL_1, AD7124_SETUP_0,
        AD7124_AIN1_INPUT, AD7124_DGND, true, base);
    if (ad7124_answer != AD7124_STATUS_OK) {
        printf("Channel_1 set error\r\n");
        exit(EXIT_FAILURE);
    }

    /* Настройка канала 2 */
    ad7124_answer = AD7124_SetChannel(AD7124_CHANNEL_2, AD7124_SETUP_0,
        AD7124_AIN2_INPUT, AD7124_DGND, true, base);
    if (ad7124_answer != AD7124_STATUS_OK) {
        printf("Channel_2 set error\r\n");
        exit(EXIT_FAILURE);
    }

    /* Настройка канала 3 */
    ad7124_answer = AD7124_SetChannel(AD7124_CHANNEL_3, AD7124_SETUP_0,
        AD7124_AIN3_INPUT, AD7124_DGND, true, base);
    if (ad7124_answer != AD7124_STATUS_OK) {
        printf("Channel_3 set error\r\n");
        exit(EXIT_FAILURE);
    }

    /* Настройка канала 4 */
    ad7124_answer = AD7124_SetChannel(AD7124_CHANNEL_4, AD7124_SETUP_0,
        AD7124_AIN4_INPUT, AD7124_DGND, true, base);
    if (ad7124_answer != AD7124_STATUS_OK) {
        printf("Channel_4 set error\r\n");
        exit(EXIT_FAILURE);
    }

    /* Настройка канала 5 */
    ad7124_answer = AD7124_SetChannel(AD7124_CHANNEL_5, AD7124_SETUP_0,
        AD7124_AIN5_INPUT, AD7124_DGND, true, base);
    if (ad7124_answer != AD7124_STATUS_OK) {
        printf("Channel_5 set error\r\n");
        exit(EXIT_FAILURE);
    }

    /* Настройка фильтрa */
    ad7124_answer = AD7124_SetFilter(AD7124_FILTER_0, AD7124_SINC_4_FILTER, 
        AD7124_NO_POST_FILTER, 1, false, false, base);
    if (ad7124_answer != AD7124_STATUS_OK) {
        printf("Filter set error\r\n");
        exit(EXIT_FAILURE);
    }

    /* Настройка режима работы */
    ad7124_answer = AD7124_SetControl(AD7124_CONTINUOUS_MODE,
            AD7124_FULL_POWER, true, true, true, AD7124_INTERNAL_CLK, base);
    if (ad7124_answer != AD7124_STATUS_OK) {
        printf("Control set error\r\n");
        exit(EXIT_FAILURE);
    }
}
