#!/usr/bin/env bash
# Copyright 2024 RnD Center "ELVEES", JSC

set -euo pipefail

help() {
    echo 'mcom03-set-hostname - Utility for generation hostname from board model and serial-number'
    echo
    echo 'Usage: mcom03-set-hostname -f /proc/device-tree/compatible  # set hostname'
    echo '       echo elvmc03smarc-r1.0-rockpi-n10 | mcom03-set-hostname -t  # generate hostname'
    echo '          and print it'
    echo
    echo 'Options:'
    echo '  -h   Print this help'
    echo '  -f   file with compatibles'
    echo '  -t   do not set hostname, just print it'
    exit 1
}

COMPATIBLE_FILE=""
TEST_MODE=0

while getopts 'htf:' opt; do
    case $opt in
        h) help;;
        f) COMPATIBLE_FILE="$OPTARG";;
        t) TEST_MODE=1;;
        *) help;;
    esac
done
shift $((OPTIND-1))

if [[ -z $COMPATIBLE_FILE ]]; then
    COMPATIBLE=$(cat -)
else
    COMPATIBLE=$(cat $COMPATIBLE_FILE | sed 's/elvees,/ /g' | awk '{print $1}')
fi

if [[ -z $COMPATIBLE ]]; then
    echo 'Compatible is empty' > /dev/stderr
    exit 1
fi

pm_and_ver=$(echo $COMPATIBLE | grep -oE "[a-zA-Z0-9\-]*-[rv][0-9]*" | head -n 1)
pm=$(echo $pm_and_ver | awk -F "-" '{$(NF--)=""; print}' | tr -d ' -')
ver=$(echo $pm_and_ver | awk -F "-" '{print $NF}' | tr -d "rv")

if [[ -z $pm || -z $ver ]]; then
    echo 'Parse compatible error' > /dev/stderr
    exit 1
fi
hostname="${pm}${ver}"

if [[ $TEST_MODE == 1 ]]; then
    echo $hostname
    exit 0
fi

if [ -f /proc/device-tree/serial-number ]; then
    hostname+=-$(tr -d '\0' < /proc/device-tree/serial-number | tr '[:upper:]' '[:lower:]')
fi

# hostnamectl doesn't recreate file if new hostname matches existing, so in that case sed will
# append another comment at the begining of /etc/hostname, and there will be recurring "Hostname
# generated by ..." message. To prevent repeat comment delete /etc/hostname, which will then be
# created by hostnamectl.
rm -f /etc/hostname
echo Set hostname $hostname
hostnamectl hostname $hostname
sed -i '1i# Hostname generated by set-default-hostname service.\
# To set your own hostname, firstly disable the service.' /etc/hostname
