#!/usr/bin/env bash
#
# Copyright 2022 RnD Center "ELVEES", JSC
# SPDX-License-Identifier: BSD-2-Clause
#
# Basic control for pwm channel

set -eo pipefail

BASEDIR=/sys/class/pwm
DEVICENUM=0
PWMCHIP=pwmchip$DEVICENUM
PWMCHANNEL=0
DUTY_CYCLE=50
PERIOD=1000
POLARITY=normal

log() {
    # shellcheck disable=SC2015
    [[ $VERBOSE ]] && echo "$@" || true
}

help() {
    me=$(basename $0)
    echo "$me: sets up parameters for PWM channel"
    echo "Usage: $me [-v] [-h] [-d device] [-c channel] [-s duty_cycle] [-p period] [-P polarity]"
    echo "-d	pwm device, default=$DEVICENUM"
    echo "-c 	pwm channel, default=$PWMCHANNEL"
    echo "-s	duty_cycle, integer %, default=$DUTY_CYCLE"
    echo "-p	period, integer ms, default=$PERIOD"
    echo "-P	polarity, normal or inversed, default=$POLARITY"
    echo "-e 	enable/disable channel, enable=1 disable=0"
    echo "-v	be verbose"
    echo "-h	this message"
}

pwm_export () {
    log "Export pwm channel $PWMCHANNEL"
    if [ ! -d $BASEDIR/$PWMCHIP/pwm$PWMCHANNEL ]; then
        echo "$PWMCHANNEL" > $BASEDIR/$PWMCHIP/export
    fi
}

pwmchannel_set_prop () {
    log "Set pwmchannel: set $1 to $2"
    if [ -e $BASEDIR/$PWMCHIP/pwm$PWMCHANNEL/$1 ]; then
        echo "$2" > $BASEDIR/$PWMCHIP/pwm$PWMCHANNEL/$1
    fi
}

pwm_disable() {
    if [ $1 -eq 1 ]; then
        log "Disable pwm$PWMCHANNEL"
        pwmchannel_set_prop "enable" "0"
    fi
}

pwmchannel_get_prop() {
    log "Get pwmchannel: $1"
    if [ -e $BASEDIR/$PWMCHIP/pwm$PWMCHANNEL/$1 ]; then
        prop=$(cat $BASEDIR/$PWMCHIP/pwm$PWMCHANNEL/$1)
    fi
}

pwm_channel_reset () {
    pwmchannel_set_prop "duty_cycle" 0
}

pwm_channel_setup () {
    pwmchannel_set_prop "period" $1
    pwmchannel_set_prop "duty_cycle" $2
    pwmchannel_set_prop "polarity" $POLARITY
}

while getopts 'vhc:p:s:P:d:e:' opt; do
    case $opt in
        d) DEVICENUM=$OPTARG;;
        s) DUTY_CYCLE=$OPTARG;; # 1...100% in integers
        p) PERIOD=$OPTARG;; # period in milliseconds
        c) PWMCHANNEL=$OPTARG;; # channel
        P) case $OPTARG in
            "normal" | "inversed") POLARITY=$OPTARG;;
            *) echo "Unsupported polarity $OPTARG"; exit 1;;
            esac;;
        v) VERBOSE=1;;
        e) case $OPTARG in
            "1" | "0") EN_CHAN=$OPTARG;;
            *) echo "$OPTARG unsupported, must be 0 or 1"; exit 1;;
            esac;;
        :) echo "option -$OPTARG requires an argument"; exit 1;;
        h) help $0; exit 0;;
        *) help $0; exit 1;;
    esac
done
shift $((OPTIND - 1))

declare -i per=$((PERIOD*1000000))
declare -i duty=$((PERIOD*DUTY_CYCLE*10000))

if [ -z $EN_CHAN ]; then
    echo "-e option is required"
    help $0
    exit 1
fi

pwm_export
pwmchannel_get_prop "enable"
enabled=$prop

if [ $EN_CHAN -eq 1 ]; then
    pwm_disable $enabled
    pwmchannel_get_prop "period"
    if [ $prop -ne 0 ]; then
        pwm_channel_reset
    fi
    log "Setup pwm$PWMCHANNEL: period=$per duty_cycle=$duty polarity=$POLARITY"
    pwm_channel_setup $per $duty
    log "Enable pwm$PWMCHANNEL"
    pwmchannel_set_prop "enable" "$EN_CHAN"
else
    pwm_disable $enabled
fi
