#!/bin/bash
#
# Copyright 2022 RnD Center "ELVEES", JSC

DEFAULT_PLAYLIST=playlist.txt

set -e

print_help() {
    cat << END
    Usage: $0 [action]

    Actions:
        test       - perform ALSA playback/capture tests
        help       - print help and exit
        gen        - generate default playlist.txt for playback
        <filename> - path to file with audio URLs to play by mpv player

    If action doesnt set then play from playback.txt.
    If you need custom audio device, please specify ACARD env variable.
END
}

gen_playlist() {
    # put your URLs here
    cat << END > "$DEFAULT_PLAYLIST"
    https://download.samplelib.com/mp3/sample-15s.mp3
    https://filesamples.com/samples/audio/mp3/sample4.mp3
END
}

play_mpv() {
    local PLAYLIST="${1:-$DEFAULT_PLAYLIST}"

    if [ ! -f "$PLAYLIST" ]; then
        echo "Playlist file '$PLAYLIST' not found"
        print_help
        exit 1
    fi

    local ACARD="${ACARD:-alsa/default:CARD=mfbspi2s}"

    mpv --no-video --loop-playlist=inf --audio-device="$ACARD" --playlist="$PLAYLIST"
}

get_alsa_device() {
    aplay -l | grep analog | sed -n 's/card \(.\):.\+, device \(.\):.\+/hw:\1,\2/p'
}

test_alsa() {
    clean_up() {
        rm -f tmp*.wav
    }
    trap clean_up EXIT

    local ACARD="${ACARD:-$(get_alsa_device)}"
    echo "Will use $ACARD alsa device"

    echo "basic record"
    arecord -D "$ACARD" -f S16_LE -c 2 -r 48000 -d 3 tmp0.wav

    echo "basic play"
    aplay -D "$ACARD" tmp0.wav

    echo "Simultaneous play and record"
    aplay -D "$ACARD" tmp0.wav &
    arecord -D "$ACARD" -c 2 -r 8000 -f S16_LE -d 5 tmp1.wav
    wait

    aplay -D "$ACARD" tmp1.wav &
    arecord -D "$ACARD" -c 2 -r 16000 -f S16_LE -d 5 tmp2.wav
    wait

    aplay -D "$ACARD" tmp2.wav &
    arecord -D "$ACARD" -c 2 -r 44100 -f S16_LE -d 5 tmp3.wav
    wait

    aplay -D "$ACARD" tmp3.wav &
    arecord -D "$ACARD" -f S16_LE -c 2 -r 48000 -d 5 tmp4.wav
    wait
}

case "$1" in
    test)
        test_alsa
        ;;
    help)
        print_help
        ;;
    gen)
        gen_playlist
        # passthrough
        shift
        ;&
    *)
        play_mpv "$@"
esac
