#!/usr/bin/env python3
#
# Copyright 2022 RnD Center "ELVEES", JSC

import argparse
import fcntl
import os
import select
import subprocess
import sys
import threading

from textwrap import dedent

FTRACE_DEPTH = 2
FTRACE_DIR = "/sys/kernel/debug/tracing"
OUTPUT_FILE = sys.stdout
TRACING_FUNCTIONS_LIST = ["*:mod:elcore50"]
NOTRACING_FUNCTIONS_LIST = ["elcore50_runtime_resume", "elcore50_runtime_suspend"]

EVENT_LIST = [
    "elcore50_buf_create",
    "elcore50_buf_release",
    "elcore50_syscall",
    "elcore50_buf_sync",
    "elcore50_mmu_map",
    "elcore50_uptime",
]

done = 0


epilog = dedent(r"""
   For assigning format of functions for '--trace-functions' and '--notrace-functions' options
   see 'set_ftrace_filter' and 'set_ftrace_notrace' in the ftrace documentation:
   https://www.kernel.org/doc/html/v4.19/trace/ftrace.html

   The list of all available functions:

     cat /sys/kernel/debug/tracing/available_filter_functions

   Examples:

     elcore-prof.py -o /tmp/xyram.ftrace \
       cmd elcorecl-run -e /usr/share/elcore50-extra/test-xyram

     elcore-prof.py \
       --trace-functions=qlic_handle_irq,swiotlb_sync_single_for_cpu,swiotlb_sync_single_for_device\
       -o /tmp/xyram.ftrace \
       cmd elcorecl-run -e /usr/share/elcore50-extra/test-xyram
""")


def parse_args():
    parser = argparse.ArgumentParser(
        usage="%(prog)s [options] cmd <command>",
        formatter_class=argparse.RawTextHelpFormatter,
        description="Ftrace based elcore50 driver profiler.",
        epilog=epilog
    )
    parser.add_argument(
        "-o",
        "--output",
        type=argparse.FileType("w"),
        default=sys.stdout,
        help="Write profiling results to file. Default: stdout",
    )
    parser.add_argument(
        "-d", "--depth", type=int, default=0, help="The max tracing depth. Default: 0"
    )
    parser.add_argument(
        "--trace-functions",
        help="Comma separated list of tracing functions by function tracer. Functions of elcore50\n"
             "driver are enabled by default",
    )
    parser.add_argument(
        "--notrace-functions",
        help="Comma separated list of non-tracing functions",
    )

    return parser.parse_args()


def write_tracing_file(name, val, append=False):
    mode = 'a' if append else 'w'
    with open(f"{FTRACE_DIR}/{name}", mode) as file:
        print(val, file=file)


def set_tracing_events():
    for event in EVENT_LIST:
        write_tracing_file(f"events/elcore50/{event}/enable", "1")


def clear_tracing_events():
    for event in EVENT_LIST:
        write_tracing_file(f"events/elcore50/{event}/enable", "0")


def reset_tracing_files():
    write_tracing_file("tracing_on", "0")
    write_tracing_file("current_tracer", "nop")
    write_tracing_file("set_ftrace_pid", " ")
    write_tracing_file("max_graph_depth", "0")

    # Reset tracing filters
    write_tracing_file("set_ftrace_filter", " ")
    write_tracing_file("set_ftrace_notrace", " ")
    write_tracing_file("set_graph_function", " ")
    write_tracing_file("set_graph_notrace", " ")

    # Reset tracing events
    clear_tracing_events()


def set_tracing_depth():
    write_tracing_file("max_graph_depth", str(FTRACE_DEPTH))


def set_tracing_filters():
    for func in TRACING_FUNCTIONS_LIST:
        write_tracing_file("set_ftrace_filter", func, True)

    for func in NOTRACING_FUNCTIONS_LIST:
        write_tracing_file("set_ftrace_notrace", func, True)


def print_column_headers():
    with open(f"{FTRACE_DIR}/trace", "r") as f:
        print(*f, file=OUTPUT_FILE, end="")


def read_pipe():
    trace_file = open(f"{FTRACE_DIR}/trace_pipe", "r")
    fl = fcntl.fcntl(trace_file.fileno(), fcntl.F_GETFL)
    fcntl.fcntl(trace_file.fileno(), fcntl.F_SETFL, fl | os.O_NONBLOCK)
    poller = select.poll()
    poller.register(trace_file.fileno(), select.POLLIN)

    while not done:
        events = poller.poll(100)
        for _ in events:
            for line in trace_file:
                print(line, file=OUTPUT_FILE, end="")
    try:
        for line in trace_file:
            print(line, file=OUTPUT_FILE, end="")
    except Exception:
        # there is no data in pipe
        pass
    finally:
        trace_file.close()


def main():
    global done, OUTPUT_FILE, FTRACE_DEPTH

    try:
        split_idx = sys.argv.index("cmd")
    except Exception:
        parse_args()
        print(
            "Keyword 'cmd' not found\n"
            f"Try '{os.path.basename(sys.argv[0])} --help' for more information."
        )
        exit(1)

    try:
        cmd = sys.argv[split_idx + 1]
    except Exception:
        sys.argv = sys.argv[:split_idx]
        parse_args()
        print(
            "Command is not specified\n"
            f"Try '{os.path.basename(sys.argv[0])} --help' for more information."
        )
        exit(1)

    cmd_args = sys.argv[(split_idx + 2):]
    # Remove cmd <command> from argv list
    sys.argv = sys.argv[:split_idx]
    known = parse_args()

    OUTPUT_FILE = known.output
    FTRACE_DEPTH = known.depth
    if known.trace_functions is not None:
        TRACING_FUNCTIONS_LIST.extend(known.trace_functions.split(','))
    if known.notrace_functions is not None:
        NOTRACING_FUNCTIONS_LIST.extend(known.notrace_functions.split(','))

    reset_tracing_files()
    write_tracing_file("trace", "0")
    set_tracing_depth()

    set_tracing_filters()

    set_tracing_events()
    write_tracing_file("current_tracer", "function_graph")
    # Add linux timestamps in ftrace log
    write_tracing_file("trace_options", "funcgraph-abstime")
    # Add PID to ftrace log
    write_tracing_file("trace_options", "funcgraph-proc")
    # Add funcgraph-tail
    write_tracing_file("options/funcgraph-tail", "1")
    # Do not remove sleep time from log
    write_tracing_file("options/sleep-time", "1")

    thread = threading.Thread(target=read_pipe)
    thread.start()

    print_column_headers()

    write_tracing_file("tracing_on", "1")
    subprocess.call([cmd] + list(cmd_args))
    done = 1
    thread.join()

    write_tracing_file("tracing_on", "0")
    reset_tracing_files()


if __name__ == "__main__":
    main()
