# Copyright 2022 RnD Center "ELVEES", JSC
#
# Based on https://github.com/corakwue/ftrace/tree/master/ftrace

import os
import re
import warnings

from .entry import Entry, EntryList
from .task import Task


class Ftrace(object):

    START_NESTED_PATTERN = re.compile(
        r"""
        (?P<timestamp>\d+\W{1}\d+)
        \s+\|\s+
        (?P<cpu>\d+)\) # cpu#
        \s+
        (?P<name>.+) # task name
        \D
        (?P<pid>\d+) # pid
        \s+\|\s+\|
        (?P<function>\s+\S+) # function
        \s+\W+
        """,
        re.X | re.M,
    )

    END_NESTED_PATTERN = re.compile(
        r"""
        (?P<timestamp>\d+\W{1}\d+)
        \s+\|\s+
        (?P<cpu>\d+)\) # cpu#
        \s+
        (?P<name>.+) # task name
        \D
        (?P<pid>\d+) # pid
        \s+\W{1}\s+\W{1}\s
        (?P<duration>\d+\W{0,1}\d+)\sus # duration [us]
        \s+\W{1}
        (?P<function>.+) # function
        \s+\W+
        """,
        re.X | re.M,
    )

    LEAF_PATTERN = re.compile(
        r"""
        (?P<timestamp>\d+\W{1}\d+)
        \s+\|\s+
        (?P<cpu>\d+)\) # cpu#
        \s+
        (?P<name>.+) # task name
        \D
        (?P<pid>\d+) # pid
        \s+\W{1}\s+\W{1}\s
        (?P<duration>\d+\W{0,1}\d+)\sus # duration [us]
        \s+\W{1}
        (?P<function>\s+\w+) # function
        .+;
        """,
        re.X | re.M,
    )

    EVENT_PATTERN = re.compile(
        r"""
        (?P<timestamp>\d+\W{1}\d+)
        \s+\W\s+
        (?P<cpu>\d+)\) # cpu#
        \s+
        (?P<name>.+) # task name
        \D
        (?P<pid>\d+) # pid
        \s+\W{1}\s+\W{1}\s
        \s+\W{2}\s
        (?P<function>\w.+) # function
        \s\W{2}
        """,
        re.X | re.M,
    )

    def __init__(self, filepath):
        """
        Parser for ftrace output.

        Params:
        -------
        filepath : str
            Path of file to parse
        """
        self.filepath = filepath

        self._raw_start_timestamp = None
        self.entries = None
        self.num_entries = 0
        self.interval = None
        self.functions = set()
        self.seen_cpus = set()

        # tracer metadata
        self.filedir, self.filename = os.path.split(self.filepath)

        success = self._parse_file()
        if success:
            self.interval = self.entries.interval

    def __repr__(self):
        return "Trace(filepath={} num_entries={})".format(
            self.filepath, self.num_entries
        )

    def _parse_file(self):
        """
        Parse input file (lazily), return True if successful, False otherwise.
        """

        try:
            self.entries = EntryList(self._parse_lines())
            return True
        except Exception as e:
            print(e)
            return False

    def _parse_lines(self):
        """
        Parse systrace lines in file.
        """

        entry = None
        non_white_pat = re.compile("[^ ]")

        start_nested_and_events_list = []

        for line in self._line_gen():
            # Save start nested lines to list
            match = re.match(self.START_NESTED_PATTERN, line)
            if match:
                start_nested_and_events_list.append(line)
                if self._raw_start_timestamp is None:
                    match_dict = match.groupdict()
                    self._raw_start_timestamp = float(match_dict["timestamp"])
                self.num_entries += 1
                continue

            # Save events to list
            match = re.match(self.EVENT_PATTERN, line)
            if match:
                start_nested_and_events_list.append(line)
                self.num_entries += 1
                continue

            match = re.match(self.LEAF_PATTERN, line)
            if match:
                match_dict = match.groupdict()
                m = non_white_pat.search(match_dict["function"])
                match_dict["depth"] = int(m.start() / 2)
                match_dict["function"] = match_dict["function"].strip()
                match_dict["timestamp"] = float(match_dict["timestamp"])
                match_dict["duration"] = float(match_dict["duration"])

                if self._raw_start_timestamp is None:
                    self._raw_start_timestamp = match_dict["timestamp"]

                match_dict["task"] = Task(**match_dict)
                entry = Entry(**match_dict)
                # add to seen cpus
                self.seen_cpus.add(entry.cpu)
                self.functions.add(entry.function)
                yield entry
                self.num_entries += 1
                continue

            match = re.match(self.END_NESTED_PATTERN, line)
            if match:
                match_dict = match.groupdict()
                m = non_white_pat.search(match_dict["function"])
                match_dict["depth"] = int(m.start() / 2)
                match_dict["timestamp"] = float(match_dict["timestamp"])
                match_dict["duration"] = float(match_dict["duration"])

                # Search start part and according events
                founds = []
                event = None

                for subline in reversed(start_nested_and_events_list):
                    match = re.match(self.EVENT_PATTERN, subline)
                    if match:
                        submatch_dict = match.groupdict()
                        # TODO: Check depth and pid ?
                        event = submatch_dict["function"]
                        founds.append(subline)
                        continue

                    match = re.match(self.START_NESTED_PATTERN, subline)
                    if match:
                        submatch_dict = match.groupdict()
                        m = non_white_pat.search(submatch_dict["function"])
                        submatch_dict["depth"] = int(m.start() / 2)
                        if (
                            submatch_dict["depth"] == match_dict["depth"]
                            and submatch_dict["pid"] == match_dict["pid"]
                        ):
                            match_dict["timestamp"] = float(submatch_dict["timestamp"])
                            match_dict["function"] = submatch_dict["function"].strip()
                            founds.append(subline)
                            break
                    continue

                # Kernel 4.19 may use end nested format as leaf so we do not raise exception here
                if len(founds) == 0:
                    warnings.warn(
                        "According start line did not found for line: {}".format(line)
                    )

                for found in founds:
                    start_nested_and_events_list.remove(found)

                match_dict["task"] = Task(**match_dict)
                match_dict["event"] = event
                entry = Entry(**match_dict)
                # add to seen cpus
                self.seen_cpus.add(entry.cpu)
                self.functions.add(entry.function)
                yield entry
                self.num_entries += 1

    def _line_gen(self):
        """
        Generator that yields ftrace lines in file.
        """
        yield_trace = False
        with open(self.filepath, "r") as f:
            num_lines = os.fstat(f.fileno()).st_size
            while True:
                line = f.readline().strip()
                if not yield_trace and "TASK/PID" in line:
                    yield_trace = True
                    _ = f.readline()
                    continue
                if yield_trace:
                    yield line
                if not line and f.tell() == num_lines:
                    break
