# Copyright 2022 RnD Center "ELVEES", JSC
#
# Based on https://github.com/corakwue/ftrace/tree/master/ftrace

from collections import namedtuple
import sys

Entrybase = namedtuple(
    "Entry",
    [
        "timestamp",
        "task",
        "pid",
        "cpu",
        "depth",
        "duration",
        "function",
        "event",
    ],
)


class Interval:
    """
    Represents an interval of time defined by two timestamps.

    Parameters:
    -----------

    start: float.
        Starting value.
    end : float
        Ending value.
    """

    __slots__ = ("start", "end")

    def __init__(self, start, end):
        if end < start:
            raise ValueError(
                f"End timestamp:{end} cannot be less than start timestamp:{start}"
            )
        self.start, self.end = float(start), float(end)

    def __repr__(self):
        return "Interval(start={:.3f}ms, end={:.3f}ms, duration={:.3f}ms)".format(
            self.start * 1000, self.end * 1000, self.duration * 1000
        )

    @property
    def duration(self):
        """Returns float"""
        return self.end - self.start


class Entry(Entrybase):

    __slots__ = ()

    def __new__(cls, timestamp, cpu, task, pid, depth, duration, function, event=None, **kwargs):
        cpu = int(cpu)
        duration = float(duration)
        timestamp = float(timestamp)
        event = event

        return super(Entry, cls).__new__(
            cls,
            timestamp=timestamp,
            cpu=cpu,
            task=task,
            pid=pid,
            depth=depth,
            duration=duration,
            function=function,
            event=event,
        )

    def __repr__(self):
        return "Entry(timestamp={:.6f}, task={}, pid={} cpu={}, function={}".format(
            self.timestamp, self.task, self.pid, self.cpu, self.function
        ) + " depth={}, duration={:.3f}us, event={})".format(
            self.depth, self.duration, self.event
        )


class EntryList(list):
    """
    List with objects with timestamps, sorted and sliceable by interval.
    """

    def __init__(self, iterable=None):
        self._timestamps = []
        if iterable:
            for item in iterable:
                self.insert(len(self), item)

    def __repr__(self):
        return "\n".join([item.__repr__() for item in self])

    @property
    def start(self):
        """First timestamp for depth=1 in list"""
        for item in self:
            if item.depth == 1:
                return item.timestamp

    @property
    def end(self):
        """Last timestamp in list"""
        return self[-1].timestamp + (self[-1].duration * 1e-6)

    @property
    def interval(self):
        """Interval for this entry list"""
        try:
            return Interval(start=self.start, end=self.end)
        except Exception as exc:
            print(exc, file=sys.stderr)
            return None

    @property
    def duration(self):
        """Duration of entries in seconds"""
        return self.interval.duration if self.interval else None
