"""Python backend for ElcoreNN."""

import ctypes.util
import logging
import os
from abc import ABCMeta, abstractmethod
from dataclasses import dataclass
from typing import Dict, List, Tuple, Type, Union

import numpy as np

__author__ = 'asomikov'

LOGGING_LEVEL = logging.WARNING
LAYER_NAME_LEN = 250
ELCORENN_DATA_TYPES = {
    'float32': 0,
    'float16': 1,
    'int32': 2,
    'uint32': 3,
    'uint8': 4,
    'int8': 5,
}
HEAP_SIZES = ['64mb', '128mb', '256mb', '512mb', '1gb', '2gb', '3gb']


def create_logger(name: str, level: int) -> logging.Logger:
    """Create logger.

    :param name: name of the logger
    :param level: level of the logging
    :return: logger
    """
    logger = logging.Logger(name)
    logger.setLevel(level)
    formatter = logging.Formatter(
        '%(asctime)s - %(name)s - %(levelname)s - %(message)s',
    )

    stdout_ch = logging.StreamHandler()
    stdout_ch.setLevel(logging.DEBUG)
    stdout_ch.setFormatter(formatter)
    logger.addHandler(stdout_ch)

    return logger


class BaseModel():
    """Base class for NN models."""

    def __init__(self, json_path: str, weights_path: str):
        """Init model.

        :param json_path: path to the model json
        :param weights_path: path to the model weights
        """
        if not os.path.exists(json_path):
            raise OSError(f'Can not find model json: {json_path}')

        if not os.path.exists(weights_path):
            raise OSError(f'Can not find model weights: {weights_path}')

        self.json_path = json_path
        self.weights_path = weights_path

        self.__model_id = None
        self.inputs = None
        self.outputs = None

    def set_id(self, model_id: int) -> None:
        """Set model ID.

        :param model_id: model ID
        """
        self.__model_id = model_id

    def get_id(self) -> int:
        """Get model ID.

        :return: model ID.
        """
        return self.__model_id

    def set_inputs(self, inputs: Dict) -> None:
        """Set model inputs.

        :param model_id: model ID
        """
        self.inputs = inputs

    def set_outputs(self, outputs: Dict) -> None:
        """Set model outputs.

        :param model_id: model ID
        """
        self.outputs = outputs


class KerasModel(BaseModel):
    """Keras model."""


@dataclass
class ModelDescription:
    """Model description."""

    json_path: str
    weights_path: str
    model_class: Type[BaseModel]


class BaseRuntime(metaclass=ABCMeta):
    """Base class for NN Runtime."""

    def __init__(self):
        """Init base runtime class."""
        self._logger = create_logger(type(self).__name__, LOGGING_LEVEL)

    @classmethod
    @abstractmethod
    def is_support(cls) -> bool:
        """Check that the current OS support this NN runtime.

        :return: status
        """
        raise NotImplementedError()

    @abstractmethod
    def load(self, model: BaseModel) -> None:
        """Load model to NN Runtime.

        :param model: model
        :return: none
        """
        raise NotImplementedError()

    @abstractmethod
    def predict(self, model: BaseModel, input_data: Dict) -> Dict:
        """Predict model.

        :param model: model
        :param input_data: input data
        :return: output data
        """
        raise NotImplementedError()


class CLRuntime(BaseRuntime):
    """ElcoreCL NN Runtime."""

    ELCORE_NN_LIBRARY = '/usr/lib/libelcorenn.so'
    ELCORECL_LIBRARY = '/usr/lib/libelcorecl.so'
    ELCORE_NN_DSP_ELF = 'elcorenn_dsp_backend'
    ELCORE_NN_DSP_PATH_NAME = 'ELCORENN_CL_DSP_ELF_PATH'
    ELCORE_NN_DEFAULT_BATCH_PACK_SIZE = 32
    ECL_DEVICE_TYPE_CUSTOM = (1 << 4)

    def __del__(self):
        self.__lib.release_backend(self.__backend_id)

    def __init__(
            self,
            devices_list: List = None,
            heap_size: str = '512mb',
            single_stream: bool = False,
            local_queue: bool = False,
    ):
        """Init CL NN Runtime.

        :param devices_list: List with indices of DSP cores for model inference
        """
        super().__init__()
        self._logger.info('init')

        if not devices_list:
            self.__lib_cl = ctypes.CDLL(self.ELCORECL_LIBRARY)
            platform_id = ctypes.c_ulong()
            ndevs = ctypes.c_uint()

            # eclGetPlatformIDs(1, &platform_id, nullptr);
            self.__lib_cl.eclGetPlatformIDs(
                1,
                ctypes.byref(platform_id),
                None,
            )

            # eclGetDeviceIDs(platform_id,
            # ECL_DEVICE_TYPE_CUSTOM, 0, nullptr, &ndevs);
            self.__lib_cl.eclGetDeviceIDs(
                platform_id,
                self.ECL_DEVICE_TYPE_CUSTOM,
                0,
                None,
                ctypes.byref(ndevs),
            )

            devices_list = list(range(ndevs.value))

        self.__dsp_path_list = ['/usr/share/elcorenn']

        self._logger.debug('open ElcoreNN CL library')
        self.__lib = ctypes.CDLL(self.ELCORE_NN_LIBRARY)
        if heap_size not in HEAP_SIZES:
            err_msg = f'Plese select heap size from the list {HEAP_SIZES}'
            raise AssertionError(err_msg)
        dsp_backend_name = self.ELCORE_NN_DSP_ELF + '_' + heap_size
        dsp_elfpath = self.find_dsp_binary(dsp_backend_name)
        devices_number_c, devices_c = self._set_devices(devices_list)

        self._logger.debug('init ElcoreNN CL library')
        single_stream_c = ctypes.c_uint32(single_stream)
        local_queue_c = ctypes.c_uint32(local_queue)
        self.__backend_id = self.__lib.init_device(
            devices_number_c,
            devices_c,
            ctypes.c_char_p(dsp_elfpath.encode('utf-8')),
            local_queue_c,
            single_stream_c,
        )

        self.__batch_pack_size = self.ELCORE_NN_DEFAULT_BATCH_PACK_SIZE

        self._logger.info('init - success')

    @staticmethod
    def _set_devices(
            devices_list: List,
    ) -> Tuple[ctypes.c_uint32, ctypes.POINTER(ctypes.c_uint32)]:
        """Convert python list of devices to C uint32 number and uint32 array.

        :param devices_list: List with indices of DSP cores for model inference
        :return: Tuple with two elements: uint32 number and uint32 pointer to
                 array
        """
        devices_number = len(devices_list) if devices_list else 0
        devices_number_c = ctypes.c_uint32(devices_number)
        if devices_number:
            devices = np.array(devices_list, dtype=np.uint32)
            devices_c = devices.ctypes.data_as(ctypes.POINTER(ctypes.c_uint32))
        else:
            # nullptr
            devices_c = ctypes.POINTER(ctypes.c_uint32)()

        return devices_number_c, devices_c

    @classmethod
    def is_support(cls) -> bool:
        """Check that the current OS support ElcoreCL NN Runtime.

        :return: status
        """
        return cls.find_library() is not None

    def find_dsp_binary(self, dsp_binary_name: str = ELCORE_NN_DSP_ELF) -> str:
        """Find binary file with DSP kernels.

        :param dsp_binary_name: binary file name
        :return: path to the binary file with DSP kernels.
        """
        self._logger.debug('search the ElcoreNN DSP binary with name %s,',
                           dsp_binary_name)

        if self.ELCORE_NN_DSP_PATH_NAME in os.environ:
            self.__dsp_path_list.insert(
                0, os.environ[self.ELCORE_NN_DSP_PATH_NAME],
            )

        for path in self.__dsp_path_list:
            filepath = f'{path}/{dsp_binary_name}'
            if os.path.exists(filepath):
                self._logger.debug('the ElcoreNN DSP binary '
                                   'was found successfully in the %s', filepath)
                return filepath

        raise OSError(f'Can not find {self.ELCORE_NN_DSP_ELF}')

    def load(self, model: BaseModel, optimization: str) -> None:
        """Load model to the NN Runtime.

        :param model: model
        :return: none
        """
        self._logger.info('load model to runtime')
        json_path_encode = ctypes.c_char_p(
            model.json_path.encode('utf-8'),
        )
        weights_path_encode = ctypes.c_char_p(
            model.weights_path.encode('utf-8'),
        )

        if isinstance(model, KerasModel):  # noqa: SIM106
            model.set_id(self.__lib.get_model_id())
            optimization_c = ctypes.c_uint32(ELCORENN_DATA_TYPES[optimization])
            self.__lib.inc_model_id()
            self.__lib.run_kernel_load_model_keras(
                model.get_id(),
                json_path_encode,
                weights_path_encode,
                optimization_c,
                self.__backend_id,
            )
            inputs_num = self.__lib.get_inputs_number(model.get_id())
            outputs_num = self.__lib.get_outputs_number(model.get_id())
        else:
            raise TypeError(f'Unsupported model type: {type(model)}')

        model.inputs_number = inputs_num
        model.outputs_number = outputs_num

        self._logger.info('load model to runtime - success')

    def _get_input_name(self, model: BaseModel, input_idx: int) -> str:
        """Get model inputs names."""
        self._logger.debug('Loading model inputs name')

        layer_ind_c = ctypes.c_uint32(input_idx)
        layer_name = np.zeros((LAYER_NAME_LEN,), dtype=np.uint8)
        c_char_p = ctypes.POINTER(ctypes.c_char_p)
        layer_name_c = layer_name.ctypes.data_as(c_char_p)

        self.__lib.write_input_name(model.get_id(), layer_ind_c, layer_name_c)

        chars = [chr(a) for a in layer_name if a != 0]
        self._logger.debug('Loading shape name - success')

        return ''.join(chars)

    def _get_input_shape(
            self,
            model: BaseModel,
            input_idx: int,
    ) -> Tuple[int]:
        """Get shape of the input layer."""
        status_msg = f'Get mode input shape for "{input_idx}" input'
        self._logger.debug(status_msg)

        layer_ind_c = ctypes.c_uint32(input_idx)
        shape = np.zeros((6,), dtype=np.uint32)
        shape_p = shape.ctypes.data_as(ctypes.POINTER(ctypes.c_uint32))

        self.__lib.get_input_shape(model.get_id(), layer_ind_c, shape_p)
        if shape[0] > 5:
            raise ValueError(
                f'Unsupported input tensor shape size: {shape[0]}, fot input'
                f'{input_idx}')
        status_msg = f'Loading shape for "{input_idx}" input - success'
        self._logger.debug(status_msg)

        return shape[1: 1 + shape[0]]

    def get_inputs(self, model: BaseModel) -> Dict:
        """
        For each input of the model get its name and shape.

        For example for model with one input "input_1" with shape (1, 2, 3),
        function will return {"input_1": (1, 2, 3)}.

        :param model: Model
        :return: Dictionary with model names and its shape
        """
        inputs = {}
        for input_idx in range(model.inputs_number):
            layer_name = self._get_input_name(model, input_idx)
            layer_shape = self._get_input_shape(model, input_idx)
            inputs[layer_name] = layer_shape

        return inputs

    def _get_output_name(self, model: BaseModel, output_idx: int) -> str:
        """Get model outputs names."""
        self._logger.debug('Loading model outputs name')

        layer_ind_c = ctypes.c_uint32(output_idx)
        layer_name = np.zeros((LAYER_NAME_LEN,), dtype=np.uint8)
        c_char_p = ctypes.POINTER(ctypes.c_char_p)
        layer_name_c = layer_name.ctypes.data_as(c_char_p)

        self.__lib.write_output_name(model.get_id(), layer_ind_c, layer_name_c)

        chars = [chr(a) for a in layer_name if a != 0]

        self._logger.debug('Loading shape name - success')

        return ''.join(chars)

    def _get_output_shape(
            self,
            model: BaseModel,
            output_idx: int,
    ) -> Tuple[int]:
        """Get shape of the output layer."""
        status_msg = f'Get mode input shape for "{output_idx}" input'
        self._logger.debug(status_msg)

        layer_ind_c = ctypes.c_uint32(output_idx)
        shape = np.zeros((6,), dtype=np.uint32)
        shape_p = shape.ctypes.data_as(ctypes.POINTER(ctypes.c_uint32))

        self.__lib.get_output_shape(model.get_id(), layer_ind_c, shape_p)
        if shape[0] > 5:
            raise ValueError(
                f'Unsupported input tensor shape size: {shape[0]}, fot input'
                f'{output_idx}')
        status_msg = f'Loading shape for "{output_idx}" input - success'
        self._logger.debug(status_msg)

        return shape[1: 1 + shape[0]]

    def get_outputs(self, model: BaseModel) -> Dict:
        """
        For each output of the model get its name and shape.

        For example for model with one input "output_1" with shape (1, 2, 3),
        function will return {"input_1": (1, 2, 3)}.

        :param model: Model
        :return: Dictionary with model names and its shape
        """
        outputs = {}
        for output_idx in range(model.outputs_number):
            layer_name = self._get_output_name(model, output_idx)
            layer_shape = self._get_output_shape(model, output_idx)
            outputs[layer_name] = layer_shape

        return outputs

    @staticmethod
    def _aligned(src: np.ndarray, alignment: int = 64) -> np.ndarray:
        """Create new numpy array from source, with aligned address and size.

        :param src: source numpy array
        :param alignment: alignment
        :return: aligned numpy array
        """
        if (src.ctypes.data % alignment) == 0:
            return src
        assert alignment % src.itemsize == 0  # noqa: S001

        size = src.size

        aligned_size = (size + alignment - 1) // alignment
        aligned_size *= alignment
        aligned_size += alignment

        buf = np.empty(aligned_size, dtype=src.dtype)
        ofs = (-buf.ctypes.data % alignment) // src.itemsize
        aa = buf[ofs:ofs + src.size].reshape(src.shape)
        np.copyto(aa, src)
        assert aa.ctypes.data % alignment == 0

        return aa

    def _check_inputs(self, model: KerasModel, input_data: Dict):
        """Check model inputs shape and batch shape is eqaul."""
        for model_input in input_data:
            model_input_shape = model.inputs[model_input][1:]
            data_shape = input_data[model_input].shape[1:]

            if list(model_input_shape) != list(data_shape):
                raise ValueError(
                    f'Input "{model_input}" shape ({model_input_shape}) is not'
                    f' match data shape ({data_shape})')
            batch_size = input_data[model_input].shape[0]

        return batch_size

    def _prepare_inputs(self, model: KerasModel, input_data: Dict):
        """
        Prepare input arguments for ELcoreNN API  functions.

        :return: list with inputs names char pointers, list with data arrays
                 pointers, list with input data type indices
        """
        inputs_data = []
        inputs_names = []
        for model_input in model.inputs:
            inputs_data.append(input_data[model_input])
            inputs_names.append(model_input.encode('utf-8'))

        c_float_p = ctypes.POINTER(ctypes.c_float)
        input_data_p = (c_float_p * model.inputs_number)()
        input_data_p[:] = [inp.ctypes.data_as(c_float_p) for inp in inputs_data]

        c_char_p = ctypes.c_char_p
        input_names_p = (c_char_p * model.inputs_number)()
        input_names_p[:] = [c_char_p(inp) for inp in inputs_names]

        inputs_data_types = []
        for data in inputs_data:
            data_type_index = ELCORENN_DATA_TYPES.get(str(data.dtype), 0)
            inputs_data_types.append(data_type_index)
        data_types = np.array(inputs_data_types, dtype=np.uint32)
        data_types_p = data_types.ctypes.data_as(
            ctypes.POINTER(ctypes.c_uint32),
        )

        return input_names_p, input_data_p, data_types_p

    def _prepare_outputs(
            self,
            model: KerasModel,
            batch_size: int,
    ):
        """Create containers for inference outputs."""
        output_data = []
        output_names = []
        output_dict = {}
        for model_output in model.outputs:
            output_shape = (batch_size, *model.outputs[model_output][1:])
            output_dict[model_output] = np.zeros(output_shape, dtype=np.float32)
            output_data.append(output_dict[model_output])
            output_names.append(model_output.encode('utf-8'))

        c_float_p = ctypes.POINTER(ctypes.c_float)
        output_data_p = (c_float_p * model.outputs_number)()
        output_data_p[:] = [
            inp.ctypes.data_as(c_float_p)
            for inp in output_data
        ]

        c_char_p = ctypes.c_char_p
        output_names_p = (c_char_p * model.outputs_number)()
        output_names_p[:] = [c_char_p(inp) for inp in output_names]

        return output_names_p, output_data_p, output_dict

    def _predict_keras_model(
            self, model: KerasModel, input_data: np.ndarray,
    ) -> np.ndarray:
        """Predict Elcore model.

        :param model: model
        :param input_data:
        :return:
        """
        self._logger.debug('Predict Elcore model')

        batch_size = self._check_inputs(model, input_data)
        _, input_data_p, input_types_p = self._prepare_inputs(model, input_data)

        _, output_data_p, output_dict = self._prepare_outputs(model, batch_size)
        self.__lib.run_predict(model.get_id(), input_data_p, input_types_p,
                               output_data_p, batch_size)
        self._logger.debug('predict Elcore model - success')

        return output_dict

    def predict(self, model: BaseModel, input_data: Dict) -> Dict:
        """Predict model.

        :param model: model
        :param input_data: input data
        :return: output data
        """
        self._logger.debug('predict')
        if not isinstance(model, KerasModel):  # noqa: SIM106
            raise TypeError(f'{type(self)} does not support {type(model)}')

        output_data = self._predict_keras_model(model, input_data)
        self._logger.debug('predict - success')
        return output_data  # noqa: R504

    def set_batch_pack_size(self, pack_size: int) -> None:
        """
        Set batch pack size for the model.

        :param pack_size: batch pack size
        """
        self.__batch_pack_size = pack_size
        self.__lib.set_max_pack_size(self.__backend_id, pack_size)

    def save_model_statistic(self, model: BaseModel, result_path: str) -> None:
        """Save model perfomance statistics.

        :param model: Model
        :param result_path: Path to the result filename without extension
        """
        path_for_c = ctypes.c_char_p(result_path.encode('utf-8'))
        self.__lib.save_layers_statistic(model.get_id(), path_for_c)

    def save_model_tensors(self, model: BaseModel, result_path: str) -> None:
        """
        Save model layers tensors.

        Data are saving in following format: [layer_name]_inner - tensors for
        layer kernel (weights), [layer_name]_in_[ind] - tensors for layer
        input, [layer_name]_output - tensors for layer output.

        :param model: Model
        :param result_path: Path to the result filename without extension
        """
        path_for_c = ctypes.c_char_p(result_path.encode('utf-8'))
        self.__lib.save_layers_tensors(model.get_id(), path_for_c)

    def load_layer_input(
            self,
            model: BaseModel,
            layer_ind: int,
            result_path: str,
    ) -> None:
        """
        Load input data to specific model layer.

        :param model: Model
        :param layer_ind: Model layer indice for loading data
        :param result_path: Path to the result filename without extension
        """
        path_for_c = ctypes.c_char_p(result_path.encode('utf-8'))
        layer_ind_c = ctypes.c_uint32(layer_ind)
        self.__lib.load_layer_input(model.get_id(), layer_ind_c, path_for_c)

    def load_layer_output(
            self,
            model: BaseModel,
            layer_ind: int,
            result_path: str,
    ) -> None:
        """
        Load output data to specific model layer.

        :param model: Model
        :param layer_ind: Model layer indice for loading data
        :param result_path: Path to the result filename without extension
        """
        path_for_c = ctypes.c_char_p(result_path.encode('utf-8'))
        layer_ind_c = ctypes.c_uint32(layer_ind)
        self.__lib.load_layer_output(model.get_id(), layer_ind_c, path_for_c)


class CPURuntime(BaseRuntime):
    """CPU NN Runtime."""

    CPU_NN_LIBRARY = 'libcpunn.so'

    def __init__(self, shared_library_path: str = 'libcpunn.so'):
        """Init CPU NN Runtime."""
        super().__init__()
        self._logger.info('init')

        self._logger.debug('open CPU NN library')
        self.__lib = ctypes.CDLL(shared_library_path)

        self._logger.info('init - success')

    @classmethod
    def is_support(cls) -> bool:
        """Check that the current OS support CPU NN Runtime.

        :return: status
        """
        return cls.find_library() is not None

    def _get_input_name(self, model: BaseModel, input_idx: int) -> str:
        """Get model inputs names."""
        self._logger.debug('Loading model inputs name')

        layer_ind_c = ctypes.c_uint32(input_idx)
        layer_name = np.zeros((LAYER_NAME_LEN,), dtype=np.uint8)
        c_char_p = ctypes.POINTER(ctypes.c_char_p)
        layer_name_c = layer_name.ctypes.data_as(c_char_p)

        self.__lib.write_input_name(model.get_id(), layer_ind_c, layer_name_c)

        chars = [chr(a) for a in layer_name if a != 0]

        self._logger.debug('Loading shape name - success')

        return ''.join(chars)

    def _get_input_shape(
            self,
            model: BaseModel,
            input_idx: int,
    ) -> Tuple[int]:
        """Get shape of the input layer."""
        status_msg = f'Get mode input shape for "{input_idx}" input'
        self._logger.debug(status_msg)

        layer_ind_c = ctypes.c_uint32(input_idx)
        shape = np.zeros((6,), dtype=np.uint32)
        shape_p = shape.ctypes.data_as(ctypes.POINTER(ctypes.c_uint32))

        self.__lib.get_input_shape(model.get_id(), layer_ind_c, shape_p)
        if shape[0] > 5:
            raise ValueError(
                f'Unsupported input tensor shape size: {shape[0]}, fot input'
                f'{input_idx}')
        status_msg = f'Loading shape for "{input_idx}" input - success'
        self._logger.debug(status_msg)

        return shape[1: 1 + shape[0]]

    def get_inputs(self, model: BaseModel) -> Dict:
        """
        For each input of the model get its name and shape.

        For example for model with one input "input_1" with shape (1, 2, 3),
        function will return {"input_1": (1, 2, 3)}.

        :param model: Model
        :return: Dictionary with model names and its shape
        """
        inputs = {}
        for input_idx in range(model.inputs_number):
            layer_name = self._get_input_name(model, input_idx)
            layer_shape = self._get_input_shape(model, input_idx)
            inputs[layer_name] = layer_shape

        return inputs

    def _get_output_name(self, model: BaseModel, output_idx: int) -> str:
        """Get model outputs names."""
        self._logger.debug('Loading model outputs name')

        layer_ind_c = ctypes.c_uint32(output_idx)
        layer_name = np.zeros((LAYER_NAME_LEN,), dtype=np.uint8)
        c_char_p = ctypes.POINTER(ctypes.c_char_p)
        layer_name_c = layer_name.ctypes.data_as(c_char_p)

        self.__lib.write_output_name(model.get_id(), layer_ind_c, layer_name_c)

        chars = [chr(a) for a in layer_name if a != 0]

        self._logger.debug('Loading shape name - success')

        return ''.join(chars)

    def _get_output_shape(
            self,
            model: BaseModel,
            output_idx: int,
    ) -> Tuple[int]:
        """Get shape of the output layer."""
        status_msg = f'Get mode input shape for "{output_idx}" input'
        self._logger.debug(status_msg)

        layer_ind_c = ctypes.c_uint32(output_idx)
        shape = np.zeros((6,), dtype=np.uint32)
        shape_p = shape.ctypes.data_as(ctypes.POINTER(ctypes.c_uint32))

        self.__lib.get_output_shape(model.get_id(), layer_ind_c, shape_p)
        if shape[0] > 5:
            raise ValueError(
                f'Unsupported input tensor shape size: {shape[0]}, fot input'
                f'{output_idx}')
        status_msg = f'Loading shape for "{output_idx}" input - success'
        self._logger.debug(status_msg)

        return shape[1: 1 + shape[0]]

    def get_outputs(self, model: BaseModel) -> Dict:
        """
        For each output of the model get its name and shape.

        For example for model with one input "output_1" with shape (1, 2, 3),
        function will return {"input_1": (1, 2, 3)}.

        :param model: Model
        :return: Dictionary with model names and its shape
        """
        outputs = {}
        for output_idx in range(model.outputs_number):
            layer_name = self._get_output_name(model, output_idx)
            layer_shape = self._get_output_shape(model, output_idx)
            outputs[layer_name] = layer_shape

        return outputs

    def load(self, model: BaseModel, optimization: str) -> None:
        """Load model to the NN Runtime.

        :param model: model
        :return: none
        """
        self._logger.info('load model to runtime')
        json_path_encode = ctypes.c_char_p(
            model.json_path.encode('utf-8'),
        )
        weights_path_encode = ctypes.c_char_p(
            model.weights_path.encode('utf-8'),
        )

        if isinstance(model, KerasModel):  # noqa: SIM106
            model.set_id(self.__lib.get_model_id())
            optimization_c = ctypes.c_uint32(ELCORENN_DATA_TYPES[optimization])
            self.__lib.inc_model_id()
            self.__lib.load_model_keras(
                model.get_id(),
                json_path_encode,
                weights_path_encode,
                optimization_c,
            )
            inputs_num = self.__lib.get_inputs_number(model.get_id())
            outputs_num = self.__lib.get_outputs_number(model.get_id())
        else:
            raise TypeError(f'Unsupported model type: {type(model)}')

        model.inputs_number = inputs_num
        model.outputs_number = outputs_num

        self._logger.info('load model to runtime - success')

    def _check_inputs(self, model: KerasModel, input_data: Dict):
        """Check model inputs shape and batch shape is eqaul."""
        for model_input in input_data:
            model_input_shape = model.inputs[model_input][1:]
            data_shape = input_data[model_input].shape[1:]

            if list(model_input_shape) != list(data_shape):
                raise ValueError(
                    f'Input "{model_input}" shape ({model_input_shape}) is not'
                    f' match data shape ({data_shape})')
            batch_size = input_data[model_input].shape[0]

        return batch_size

    def _prepare_inputs(self, model: KerasModel, input_data: Dict):
        """
        Prepare input arguments for ELcoreNN API  functions.

        :return: list with inputs names char pointers, list with data arrays
                 pointers, list with input data type indices
        """
        inputs_data = []
        inputs_names = []
        for model_input in model.inputs:
            inputs_data.append(input_data[model_input])
            inputs_names.append(model_input.encode('utf-8'))

        c_float_p = ctypes.POINTER(ctypes.c_float)
        data_p = (c_float_p * model.inputs_number)()
        data_p[:] = [inp.ctypes.data_as(c_float_p) for inp in inputs_data]

        c_char_p = ctypes.c_char_p
        names_p = (c_char_p * model.inputs_number)()
        names_p[:] = [c_char_p(inp) for inp in inputs_names]

        inputs_data_types = []
        for data in inputs_data:
            data_type_index = ELCORENN_DATA_TYPES.get(str(data.dtype), 0)
            inputs_data_types.append(data_type_index)
        data_types = np.array(inputs_data_types, dtype=np.uint32)
        data_types_p = data_types.ctypes.data_as(
            ctypes.POINTER(ctypes.c_uint32),
        )

        return names_p, data_p, data_types_p

    def _prepare_outputs(
            self,
            model: KerasModel,
            batch_size: int,
    ):
        """Create containers for inference outputs."""
        output_data = []
        output_names = []
        output_dict = {}
        for model_output in model.outputs:
            output_shape = (batch_size, *model.outputs[model_output][1:])
            output_dict[model_output] = np.zeros(output_shape, dtype=np.float32)
            output_data.append(output_dict[model_output])
            output_names.append(model_output.encode('utf-8'))

        c_float_p = ctypes.POINTER(ctypes.c_float)
        output_data_p = (c_float_p * model.outputs_number)()
        output_data_p[:] = [
            inp.ctypes.data_as(c_float_p)
            for inp in output_data
        ]

        c_char_p = ctypes.c_char_p
        output_names_p = (c_char_p * model.outputs_number)()
        output_names_p[:] = [c_char_p(inp) for inp in output_names]

        return output_names_p, output_data_p, output_dict

    def _predict_keras_model(
            self, model: KerasModel, input_data: np.ndarray,
    ) -> np.ndarray:
        """Predict Elcore model.

        :param model: model
        :param input_data:
        :return:
        """
        self._logger.debug('Predict Elcore model')

        batch_size = self._check_inputs(model, input_data)
        _, input_data_p, input_types_p = self._prepare_inputs(model, input_data)

        _, output_data_p, output_dict = self._prepare_outputs(model, batch_size)
        self.__lib.run_predict(model.get_id(), input_data_p, input_types_p,
                               output_data_p, batch_size)

        self._logger.debug('Predict Elcore model - success')

        return output_dict

    def predict(self, model: BaseModel, input_data: Dict) -> Dict:
        """Predict model.

        :param model: model
        :param input_data: input data
        :return: output data
        """
        self._logger.debug('predict')
        if not isinstance(model, KerasModel):  # noqa: SIM106
            raise TypeError(f'{type(self)} does not support {type(model)}')

        output_data = self._predict_keras_model(model, input_data)
        self._logger.debug('predict - success')
        return output_data  # noqa: R504

    def release_model(self, model: BaseModel) -> None:
        """
        Free memory allocated for the model.

        :param model: Model
        """
        self.__lib.release_model(model.get_id())

    def save_model_statistic(self, model: BaseModel, result_path: str) -> None:
        """
        Save model perfomance statistics.

        :param model: Model
        :param result_path: Path to the result filename without extension
        """
        path_for_c = ctypes.c_char_p(result_path.encode('utf-8'))
        self.__lib.save_layers_statistic(model.get_id(), path_for_c)

    def save_model_tensors(self, model: BaseModel, result_path: str) -> None:
        """
        Save model layers tensors.

        Data are saving in following format: [layer_name]_inner - tensors for
        layer kernel (weights), [layer_name]_in_[ind] - tensors for layer
        input, [layer_name]_output - tensors for layer output.

        :param model: Model
        :param result_path: Path to the result filename without extension
        """
        path_for_c = ctypes.c_char_p(result_path.encode('utf-8'))
        self.__lib.save_layers_tensors(model.get_id(), path_for_c)

    def load_layer_input(
            self,
            model: BaseModel,
            layer_ind: int,
            result_path: str,
    ) -> None:
        """
        Load input data to specific model layer.

        :param model: Model
        :param layer_ind: Model layer index for loading data
        :param result_path: Path to the result filename without extension
        """
        path_for_c = ctypes.c_char_p(result_path.encode('utf-8'))
        layer_ind_c = ctypes.c_uint32(layer_ind)
        self.__lib.load_layer_input(model.get_id(), layer_ind_c, path_for_c)

    def load_layer_output(
            self,
            model: BaseModel,
            layer_ind: int,
            result_path: str,
    ) -> None:
        """
        Load output data to specific model layer.

        :param model: Model
        :param layer_ind: Model layer index for loading data
        :param result_path: Path to the result filename without extension
        """
        path_for_c = ctypes.c_char_p(result_path.encode('utf-8'))
        layer_ind_c = ctypes.c_uint32(layer_ind)
        self.__lib.load_layer_output(model.get_id(), layer_ind_c, path_for_c)


class ElcoreNN:
    """ElcoreNN backend."""

    def __init__(self, runtime: BaseRuntime = None):
        """Init ElcoreNN backend."""
        self.logger = create_logger('ElcoreNN', LOGGING_LEVEL)

        self.__next_model_id = 0
        self.__models: List[BaseModel] = []
        self.__model_types = [KerasModel]
        self.__runtime = runtime

        if not self.__runtime:
            self.logger.info('create CLRuntime')
            self.__runtime = CLRuntime()
            self.logger.info('create CLRuntime - success')

    def load(
            self,
            model: Union[BaseModel, ModelDescription],
            optimization: str = 'float16',
    ) -> BaseModel:
        """Load model.

        :param model: model or model description
        :return: model
        """
        self.logger.info('load model')
        if isinstance(model, ModelDescription):
            if model.model_class not in self.__model_types:
                raise TypeError('Invalid model type')
            model = model.model_class(model.json_path, model.weights_path)

        model.id = self.__next_model_id
        self.__next_model_id += 1

        self.__models.append(model)
        self.__runtime.load(model, optimization)
        model.set_inputs(self.__runtime.get_inputs(model))
        model.set_outputs(self.__runtime.get_outputs(model))

        self.logger.info('load model - success')

        return model

    def predict(
            self, model: BaseModel, input_data: Dict,
    ) -> Dict:
        """Predict model.

        :param model: model
        :param input_data: input data
        :return: output data
        """
        self.logger.info('predict model')

        if not model.get_id() and model not in self.__models:
            raise RuntimeError(f'The {model} is not loaded to ElcoreNN')

        output_data = self.__runtime.predict(model, input_data)

        self.logger.info('predict model - success')
        return output_data  # noqa: R504

    def save_model_statistic(self, model: BaseModel, result_path: str) -> None:
        """
        Save model perfomance statistics.

        :param model: Model
        :param result_path: Path to the result filename without extension
        """
        self.logger.info('saving model statistic')

        if not model.get_id() and model not in self.__models:
            raise RuntimeError(f'The {model} is not loaded to ElcoreNN')

        self.__runtime.save_model_statistic(model, result_path)

        self.logger.info('saving model statistic - success')

    def save_model_tensors(self, model: BaseModel, result_path: str) -> None:
        """
        Save model layers tensors.

        Data are saving in following format: [layer_name]_inner - tensors for
        layer kernel (weights), [layer_name]_in_[ind] - tensors for layer
        input, [layer_name]_output - tensors for layer output.

        :param model: Model
        :param result_path: Path to the result filename without extension
        """
        self.logger.info('saving model tensors')

        if not model.get_id() and model not in self.__models:
            raise RuntimeError(f'The {model} is not loaded to ElcoreNN')

        self.__runtime.save_model_tensors(model, result_path)

        self.logger.info('saving model tensors - success')

    def load_layer_input(
            self,
            model: BaseModel,
            layer_ind: int,
            result_path: str,
    ) -> None:
        """
        Load input data to specific model layer.

        :param model: Model
        :param layer_ind: Model layer index for loading data
        :param result_path: Path to the result filename without extension
        """
        self.logger.info('Loading layer %d inputs to the model', layer_ind)

        if not model.get_id() and model not in self.__models:
            raise RuntimeError(f'The {model} is not loaded to ElcoreNN')

        self.__runtime.load_layer_input(model, layer_ind, result_path)

        self.logger.info('Loading layer %d input - success', layer_ind)

    def load_layer_output(
            self,
            model: BaseModel,
            layer_ind: int,
            result_path: str,
    ) -> None:
        """
        Load output data to specific model layer.

        :param model: Model
        :param layer_ind: Model layer index for loading data
        :param result_path: Path to the result filename without extension
        """
        self.logger.info('Loading layer %d inputs to the model', layer_ind)

        if not model.get_id() and model not in self.__models:
            raise RuntimeError(f'The {model} is not loaded to ElcoreNN')

        self.__runtime.load_layer_output(model, layer_ind, result_path)

        self.logger.info('Loading layer %d input - success', layer_ind)
