#!/bin/bash
#
# Copyright 2023 RnD Center "ELVEES", JSC

set -eu

set_default() {
    COUNT=1000
    MODE="0"
    INPUT="/tmp/01.264"
    PLANE_ID=31
    SENSOR="IMX219_0"
}

set_default

help() {
    cat << EOF
mcom03-vpu-stress - tool to run GStreamer stress tests
Usage: mcom03-vpu-stress [options]
Options:
    -c          Loop count (default: $COUNT)
    -h          Show help
    -i          Input file (default: $INPUT)
    -m          Mode:
                    0 - raw H.264 video -> VPU decoder -> HDMI output via KMS
                    1 - ISP -> VPU encoder -> file
                    (default: $MODE)
    -p          Parameter "plane-id" for kmssink. Only for test with mode=0.
                (set default for kernel 5.10: $PLANE_ID. For kernel 4.19 should be 28)
    -s          Sensor type:
                    IMX219, OV2718, OV5647, OV10823 sensors are supported
                    (default: $SENSOR)
EOF
}

while getopts "c:hi:m:p:s:" arg; do
    case $arg in
        c) COUNT=$((OPTARG));;
        h) set_default; help; exit 0;;
        i) INPUT=$OPTARG;;
        m) MODE=$OPTARG;;
        p) PLANE_ID=$OPTARG;;
        s) SENSOR=$OPTARG;;
        ?) exit 1;;
    esac
done

modprobe dmabuf_exporter

if [ "$MODE" = "0" ]; then
    # Check if input file exists
    if [ ! -f "$INPUT" ]; then
        echo "Error: file $INPUT doesn't exist" >&2
        exit 1
    fi
elif [ "$MODE" = "1" ]; then
    # Adjust SETUPFILE and SENSORMODE
    if [[ "$SENSOR" = IMX219* ]]; then
        SETUPFILE="/etc/felix/imx219/imx219.cfg"
        SENSORMODE="2"
    elif [[ "$SENSOR" = OV2718* ]]; then
        SETUPFILE="/etc/felix/ov2718/ov2718.cfg"
        SENSORMODE="0"
    elif [[ "$SENSOR" = OV5647* ]]; then
        SETUPFILE="/etc/felix/ov5647/ov5647-arducam.cfg"
        SENSORMODE="0"
    elif [[ "$SENSOR" = OV10823* ]]; then
        SETUPFILE="/etc/felix/ov10823/ov10823.cfg"
        SENSORMODE="0"
    else
        echo "Error: unknown sensor name $SENSOR selected" >&2
        exit 1
    fi

    OUTPUT="/tmp/gst-out-$$.264"
    # shellcheck disable=SC2064
    trap "rm -f $OUTPUT" EXIT
else
    echo "Error: unknown mode ($MODE)" >&2
    exit 1
fi

for ((i = 1; i <= COUNT; ++i)); do
    echo "Starting $i iteration"
    if [ "$MODE" = "0" ]; then
        gst-launch-1.0 filesrc location=$INPUT ! h264parse ! omxh264dec ! \
            kmssink driver-name=mali-dp plane-id=$PLANE_ID
        sleep 5
    elif [ "$MODE" = "1" ]; then
        gst-launch-1.0 felixsrc setup-file=$SETUPFILE sensor=$SENSOR \
            sensor-mode=$SENSORMODE exposure-auto=true awb-enable=true awb-algorithm=pid \
            awb-mode=high-lum ! queue max-size-buffers=1 ! video/x-raw,format=NV12 ! omxh264enc ! \
            filesink location=$OUTPUT &
        sleep 10
        kill $!

        # Check if output file is not empty - the best we can do now
        if [ ! -s $OUTPUT ]; then
            echo "Error: output file is empty" >&2
            exit 1
        fi
        sleep 10
        rm -f $OUTPUT
    fi
done
echo
