#!/usr/bin/env python3

# Copyright 2022 RnD Center "ELVEES", JSC

import signal
import subprocess
import sys
from enum import Enum

from luma.core.interface.serial import spi
from luma.core.render import canvas
from luma.oled.device import ssd1322

"""
A test script to check NGFW-CP connected tp NGFW-CB.
When user presses the NGFW-CP button it's name will be shown on the screen
and buzzer will be enabled while button is pressed.
"""


class ControlPanel:
    GPIO_CHIP = 10
    BUZZER_PIN = 5

    class Buttons(Enum):
        RIGHT = 0
        UP = 1
        OK = 2
        DOWN = 3
        LEFT = 4


class Display:
    SPI_PORT = 2
    SPI_DEVICE = 0

    GPIO_CHIP = 6
    RST_PIN = 2
    DC_PIN = 3


def gpioset(chip: int, pin: int, value: int):
    subprocess.run([f"gpioset {chip} {pin}={value}"], check=True, shell=True)


def gpioget(chip: int, pin: int) -> int:
    return int(subprocess.getoutput(f"gpioget {chip} {pin}"))


# RPi.GPIO compatible GPIO interface
class GPIO:
    OUT = 1
    LOW = 0
    HIGH = 1

    def setup(self, pin, direction):
        pass

    def output(self, pin, value):
        gpioset(Display.GPIO_CHIP, pin, value)


def draw_button(device, button: str):
    with canvas(device) as draw:
        draw.rectangle(device.bounding_box, outline="white", fill="black")

        if button == "":
            draw.text((30, 40), "Press any button", fill="white")
        else:
            draw.text((30, 40), f"Button {button} pressed", fill="white")


def main():
    signal.signal(signal.SIGINT, lambda x, y: sys.exit(0))

    serial = spi(
        gpio=GPIO(),
        port=Display.SPI_PORT,
        device=Display.SPI_DEVICE,
        gpio_RST=Display.RST_PIN,
        gpio_DC=Display.DC_PIN,
        reset_hold_time=0.1,
        reset_release_time=0.1,
        transfer_size=64,
    )
    device = ssd1322(serial)

    # Pin 2 on gpiochip6 resets both display and GPIO expander for NGFW-CP buttons and
    # buzzer. For some reason we can't set buzzer pin after reset until reading this pin.
    # See issue #MCOM03SW-1974.
    gpioget(ControlPanel.GPIO_CHIP, ControlPanel.BUZZER_PIN)

    while True:
        buzzer_disable = True

        for button in ControlPanel.Buttons:
            if gpioget(ControlPanel.GPIO_CHIP, button.value) == GPIO.LOW:
                draw_button(device, button.name)
                gpioset(ControlPanel.GPIO_CHIP, ControlPanel.BUZZER_PIN, GPIO.HIGH)
                buzzer_disable = False
                break

        if buzzer_disable:
            draw_button(device, "")
            gpioset(ControlPanel.GPIO_CHIP, ControlPanel.BUZZER_PIN, GPIO.LOW)


if __name__ == "__main__":
    sys.exit(main())
