#!/usr/bin/env bash
#
# Copyright 2022 RnD Center "ELVEES", JSC
# SPDX-License-Identifier: BSD-2-Clause
#
# Shows property value for sysfs power_supply device class.
# For property names please see sysfs-class-power in Linux kernel
# documentation.
#

set -eo pipefail

BASEDIR=/sys/class/power_supply
PROP=capacity

log() {
    # shellcheck disable=SC2015
    [[ $VERBOSE ]] && echo "$@" || true
}

help() {
    echo "$0: shows property values for power_supply devices"
    echo "Usage: $0 [-v] [-l] [-d device] [-p property]"
    echo "-d    show property value for device"
    echo "-l    show all devices with property"
    echo "-p    property of device, default=capacity"
    echo "-v    be verbose"
    exit 1
}

while getopts 'vld:p:' opt; do
    case $opt in
        d) DEVICE=$OPTARG;;
        l) SHOW_LIST=1;;
        v) VERBOSE=1;;
        p) PROP=$OPTARG;;
        :) echo "option -$OPTARG requires an argument"; exit 1;;
        *) help $0;;
    esac
done
shift $((OPTIND - 1))

show_all_devices () {
    log "Devices with property $1:"
    for dir in "$BASEDIR"/*; do
        [[ -e "$dir" ]] || break
        if test -e $dir/$1 ; then
            basename $dir
        fi
    done
}

show_dev_prop () {
    if test -e $BASEDIR/$1/$2; then
        log "$1 $2"
        cat $BASEDIR/$1/$2
    else
        echo "$1: no such device or device do not have $2 property"
        exit 1
    fi
}

if [ -z $DEVICE ] && [ -z $SHOW_LIST ]; then
    echo "-d or -l option required"
    help
elif [[ $SHOW_LIST ]]; then
    show_all_devices $PROP
    exit 0
else
    show_dev_prop $DEVICE $PROP
    exit 0
fi
